begin sierra ThreadedRing
  define direction axial with vector 0 0 1

  begin material aluminum
    density = 2.5e-4
   
    begin parameters for model elastic
      youngs modulus = 10e6
      poissons ratio = 0.33
    end 
  end

  begin finite element model canister
    database name = threaded_ring.g

    begin parameters for block cylinder hat cap 
      material = aluminum
      model = elastic
    end 

    begin parameters for block ring ring_cylinder_preload ring_cap_preload
      material = aluminum
      model = elastic
    end 
  end

  begin function func_preload_cap
    type = piecewise linear
    begin values
      0.0  0.0
      0.5  -0.5  # shrink
      0.6  -0.1  # grow
      1.0  0.02  # preload some more
    end
  end

  begin function func_preload_cyl
    type = piecewise linear
    begin values
      0.0  0.0
      0.5  0.04
      1.0  0.04 
    end
  end

  #
  # This fancy function allows us to preload a threaded joint by targeting
  # a specific region in space to apply artificial strain.  We use the ternary
  # operator to apply "if-then" logic in the function.  By nesting these, we can 
  # create complex logic.  The function below says:
  #    if (z-coord between 5 and 7):
  #        apply constant strain
  #    else if (z-coord >= 7):
  #        ramp strain down to zero
  #    else if (z-coord <= 5): 
  #        ramp strain down to zero
  #    else:
  #        apply zero strain
  #
  # Note that because we plan to use this function in the ARTIFICIAL STRAIN 
  # boundary condition, we can access element variables such as "CENTROID" and
  # use them in the analytic function.
  #
  # We could also apply this distribution of values as a pre-processing step
  # added to the mesh via an external code (e.g. Python, MATLAB), and that may
  # be preferable in some cases.
  #
  begin function func_preload_hat
    type = analytic
    expression variable: t = global time
    expression variable: c = element_vector centroid
    evaluate expression = "(c_z > 5 && c_z < 7) ? -0.002*t :  \#
                             ( \#
                               (c_z >= 7) ? -t*max(0.002*(1 - (c_z - 7)), 0.0) : \#
                               ( \#
                                 (c_z <= 5) ? -t*max(0.002*(1 - (5 - c_z)), 0.0) : 0.0 \#
                               ) \#
                             )"
  end

  begin adagio procedure adagio_proc1
    begin time control
      begin time stepping block tb_preload_cylinder
        start time = 0.0
        begin parameters for adagio region region1
          number of time steps = 5
        end 
      end 

      begin time stepping block tb_preload_cap
        start time = 0.5
        begin parameters for adagio region region1
          number of time steps = 5
        end 
      end 

      termination time = 1.0
    end time control

    begin adagio region region1
      use finite element model canister

      begin fixed displacement
        surface = cylinder_end
        components = x y z
      end

      begin fixed displacement
        surface = y_symmetry
        component = y
      end

      # This boundary condition is here strictly to help accelerate
      # convergence towards axisymmetric solutions.  It can be removed
      # or kept as needed.  This BC prevents radial displacement on the
      # exterior of the assembly.
      begin fixed displacement
        surface = assembly_od, cap_top
        components = x y 
      end

# Use this APREPRO variable to configure how the preload is
# applied:
#   preload_hat = 0: apply preload strains to the lock ring
#   preload_hat = 1: apply preload strains to the hat
#
# preload_hat = {preload_hat = 0}
#
{if(preload_hat == 0)}
      begin artificial strain
        block = ring_cap_preload
        direction axial function = func_preload_cap
      end

      begin artificial strain
        block = ring_cylinder_preload
        direction axial function = func_preload_cyl
      end
{elseif(preload_hat == 1)}
      begin artificial strain
        block = hat
        direction axial function = func_preload_hat
      end
{endif}

      begin user output
        include all blocks
        extrapolate element variable von_mises to nodal variable von_mises
        extrapolate element variable log_strain to nodal variable log_strain
      end
      begin user output
        surface = ring_to_cap_surf
        compute global ring_to_cap_force as sum of nodal force_contact(z)
      end
      begin user output
        surface = ring_to_cylinder_surf
        compute global ring_to_cylinder_force as sum of nodal force_contact(z)
      end

      begin results output  
        database name = output.e
        at step 0 increment = 1
        nodal displacement
        nodal von_mises
        nodal log_strain
        nodal contact_status
        nodal force_contact
        element artificial_strain
        global ring_to_cap_force
        global ring_to_cylinder_force
      end 

      begin contact definition
        skin all blocks = on

        begin constant friction model friction
          friction coefficient = 0.5
        end

        begin remove initial overlap
        end

        begin interaction defaults
          general contact = on
          self contact = off
          friction model = friction
          al penalty = 0.1
        end
      end
       
      begin solver
        begin cg
          target residual = 100
          minimum iterations = 1
          begin full tangent preconditioner
            small number of iterations = 15
          end
        end
        begin control contact
          target residual = 200
          minimum iterations = 3
        end
      end
    
      begin user output
        block = hat
        compute at every step
        compute global max_vm as max of element von_mises
      end
      begin solution verification
        skip times = 0 to 0.9
        completion file = verifErr
        verify global max_vm < 12e3
        verify global max_vm > 8e3
      end

    end #region
  end #procedure
end #sierra

