begin sierra preload_pad
  # These functions are used to apply element-based artificial strains.  They look complicated,
  # but are relatively simple piecewise-linear ramps for the most part (see "ternary" operator 
  # in user manual for explanation of syntax.  Basically,
  #
  #    Result = (condition) ? <value_if_true> : <value_if_false>
  #
  begin function strain_radial
    type = analytic
    expression variable: t = global time
    evaluate expression = "(t <= 10 ? t/10.0 :               \#
                              (t < 20.0 ? -t/10. + 2 : 0.0)  \#
                           ) * (1-5)/5."
  end

  # Note that in this example, the cylindrical strain is a function of the radial position.
  # Elements located closer to the axis must grow, whereas elements located further must shrink
  # to achieve the desired deformation.
  begin function strain_theta
    type = analytic
    expression variable: t = global time
    expression variable: c = element_vector centroid
    evaluate expression = "r = sqrt(c_x^2 + c_y^2); \#
                           r0 = 38.5;               \#
                           eps = 2.0/r;             \#
                           (t <= 10 ? t/10.0 :               \#
                              (t < 20.0 ? -t/10. + 2 : 0.0)  \#
                           ) * ( ((r0 - r)/2.5)*eps  )"
  end

  # These functions are used to populate the local cylindrical axes for each element variable.
  # We could have alternatively used a "user output" transformation, or pre-processed these fields
  # using an external script.  Here, we use functions to highlight the flexibility they provide.

  # Local radial (r) unit vectors
  begin function radial_x
    type = analytic
    expression variable: c = element_vector centroid
    evaluate expression = "theta = atan2(c_y, c_x); \#
                           cos(theta)"
  end
  begin function radial_y
    type = analytic
    expression variable: c = element_vector centroid
    evaluate expression = "theta = atan2(c_y, c_x); \#
                           sin(theta)"
  end
  begin function radial_z
    type = analytic
    evaluate expression = "0.0"
  end

  # Local circumferential (theta) unit vectors
  begin function theta_x
    type = analytic
    expression variable: c = element_vector centroid
    evaluate expression = "theta = atan2(c_y, c_x); \#
                           -sin(theta)"
  end
  begin function theta_y
    type = analytic
    expression variable: c = element_vector centroid
    evaluate expression = "theta = atan2(c_y, c_x); \#
                           cos(theta)"
  end
  begin function theta_z
    type = analytic
    evaluate expression = "0.0"
  end

  # Some generic material properties.  
  begin material steel
    density         = 7.3240e-4
    begin parameters for model elastic
      youngs modulus  = 30.e6
      poissons ratio  = 0.3
    end parameters for model elastic
  end
  begin material pad
    density         = 1.0e-4
    begin parameters for model elastic
      youngs modulus  = 5000
      poissons ratio  = 0.0
    end 
  end

  begin finite element model mesh1
    database name = oversize_cylinder_pad_r1.g

    begin parameters for block block_1 block_3
      material = steel
      model = elastic
      hourglass stiffness = 0.2  # needed to combat hourglassing from contact forces
      hourglass viscosity = 0.2  # needed to combat hourglassing from contact forces
    end
    begin parameters for block block_2
      material = pad
      model = elastic
      hourglass stiffness = 0.3  # needed to combat hourglassing from contact forces
      hourglass viscosity = 0.2  # needed to combat hourglassing from contact forces
      effective moduli model = probed
    end
  end 

  begin adagio procedure PreloadCurvedPad
    begin time control
        begin time stepping block shrink_pad
          start time = 0.0
          begin parameters for adagio region AdagioRegion1
            time increment = 1.0
          end 
        end
        begin time stepping block grow_pad
          start time = 10.0
          begin parameters for adagio region AdagioRegion1
            time increment = 5.0
          end 
        end
        termination time = 20
    end time control

    begin adagio region AdagioRegion1
      use finite element model mesh1

      # Fix exterior surfaces of the constraining steel cylinders
      begin fixed displacement
        surface = sideset_1, sideset_2
        component = x y z
      end

      # Temporarily apply kinematic constraints to ensure the desired displacement 
      # results from the applied strain (displacement given strain is not unique)
      begin fixed displacement
        surface = sideset_10
        component = y z
        active periods = shrink_pad
      end
      begin fixed displacement
        surface = sideset_11
        component = x z
        active periods = shrink_pad
      end
      begin fixed displacement
        nodeset = nodeset_1
        component = x y z
        active periods = shrink_pad
      end

      # Establish element fields for use with artificial strain assembly          
      begin user variable radial_dir
        type = element vector
        initial value = 1. 0. 0.
      end
      begin user variable theta_dir
        type = element vector
        initial value = 0. 1. 0.
      end

      # Populate the X/Y/Z components of the "radial" and "theta" direction strains
      begin user output
        compute element radial_dir(x) as function radial_x
        compute element radial_dir(y) as function radial_y
        compute element radial_dir(z) as function radial_z
        compute element theta_dir(x) as function theta_x
        compute element theta_dir(y) as function theta_y
        compute element theta_dir(z) as function theta_z
        compute at every step
      end

      # Apply the artificial strains in "radial" and "theta" directions.
      begin artificial strain
        block = block_2
        strain type = engineering
        direction field radial_dir function = strain_radial
        direction field theta_dir  function = strain_theta
      end

      # Use a simple contact definition, active only during pad growth phase.
      # Note that contact does not conflict with artificial strain the way it does
      # with "harder" kinematic constraints, so contact can be "slowly activated"
      # over multiple steps
      begin contact definition
        skin all blocks = on
        begin interaction defaults
          general contact = on
          friction model = rubber_to_steel
          constraint formulation = node_face
        end
        begin constant friction model rubber_to_steel
          friction coefficient = 0.9
        end
        begin interaction int1
          side a = block_1, block_3
          side b = block_2
        end

        active periods = grow_pad
      end

      # We use a few non-default settings here to assist the solver.
      begin solver
        begin loadstep predictor
          type = scale_factor
          scale factor = 0
        end
        begin cg
          target relative residual = 0.5e-3
          acceptable relative residual = 100
          maximum iterations = 200
          preconditioner = probe
        end
        begin control contact
          target relative residual = 1.0e-3
        end
      end

      begin results output
        database name = output_strain_based.e
        at step 0, increment = 1
        nodal displacement
        nodal velocity
        nodal force_contact
        element log_strain
        element von_mises
        element stress
        element artificial_strain
        element centroid
        element radial_dir, theta_dir
        global contact_force_norm
        global net_contact_force
      end 

      begin user output
        block = block_2
        compute at every step
        compute global contact_force_norm as l2norm of nodal force_contact
        compute global max_vm as max of element von_mises
      end
      begin user output
        surface = pad_inside
        compute global net_contact_force as sum of nodal force_contact
      end

    end #region 
  end #procedure
end #sierra
