###############################################################################
#  E1      = { E1      = 207.85e6 }
#  E2      = { E2      = 207.85e6 }
#  A1      = { A1      = 3.33e15  }
#  A2      = { A2      = 4.99e13  }
#  vo2_2   = { vo2_2   = 0.092    }
#  deltaH1 = { deltaH1 = 250000   }
#  deltaH2 = { deltaH2 = 0        }
#  SA_VOL = { SA_VOL = 1000.0 } # 1/pore diameter assuming pore diam ~1mm
#  coeff = { coeff=0.00001 } # flow coupling penalty coeff
#
#  This example problem demonstrates an inverted foam in a can decomposition
#  including solid conduction, porous-fluid coupled flow, enthalpy, and species
#  transport, and enclosure radiation between the can and porous medium through
#  the fluid filled region. All equations are solved in the same Aria region,
#  monolithically in the same equation system.
#
################################################################################

Begin Sierra myJob
 
  Begin Finite Element Model all
    Database Name = ./fic2D_inverted_quad_all_R0.g
    #Database Name = ./fic2D_inverted_tri_all_R0.g
    Coordinate System Is cartesian      
    Use Material 304ss For block_1  # Top plate
    Use Material 321ss For block_2  # Sides
    Use Material 304ss For block_3  # Bottom plate
    Use Material 304ss For block_4  # Slug
    Use Material foam For block_5
    Use Material fluid for block_6

    decomposition method = rcb
  End Finite Element Model all
  
  begin tpetra equation solver gmres_teko
    begin gmres solver
      convergence tolerance = 1e-8
      begin teko preconditioner
        define matrix subblock number=1  dof=pressure
        define matrix subblock number=1  dof=enthalpy

        define matrix subblock number=2  dof=solid_phase_temperature
        define matrix subblock number=2  dof=gas_phase_temperature
        define matrix subblock number=2  dof=gas_phase_pressure

        define matrix subblock number=3  dof=velocity
        define matrix subblock number=4  dof=temperature

        define matrix subblock number=5 dof=gas_phase_mass_fraction_co2
        define matrix subblock number=5 dof=gas_phase_mass_fraction_organic
        define matrix subblock number=5 dof=mass_fraction_co2
        define matrix subblock number=5 dof=mass_fraction_organic
        define matrix subblock number=5 dof=solid_phase_density
        define matrix subblock number=5 dof=solid_phase_mass_fraction_foama
        define matrix subblock number=5 dof=solid_phase_mass_fraction_foamb

        begin teko subblock solver ddilu_solver
          begin gmres solver
            convergence tolerance = 1e-2
            maximum iterations = 30
            restart iterations = 30
            begin dd-ilu preconditioner
            end
          end
        end

        begin teko subblock preconditioner ddilu_pc
          begin dd-ilu preconditioner
          end
        end

        begin teko subblock preconditioner jacobi_pc
          begin jacobi preconditioner
          end
        end

        begin block gauss-seidel solver
          use inverse ddilu_solver for subblock 1
          use inverse ddilu_solver for subblock 2
          use inverse ddilu_pc     for subblock 3
          use inverse ddilu_pc     for subblock 4
          use inverse jacobi_pc    for subblock 5
        end
      end
    end
  end tpetra equation solver

  Begin Global Constants
    Stefan Boltzmann Constant = 5.670373e-8
  end Global Constants

  Begin Aria Material 304SS  # 304 Stainless Steel, Isotropic 
        Density              = constant  rho = 7901.  # [kg/m^3]
        Thermal Conductivity = user_function  name=f_steelcond X=Temperature  # [W/m-K]
        Specific Heat        = user_function  name=f_steelspheat X=Temperature  # [J/kg-K]
        Emissivity           = constant  e=0.8
        Heat Conduction = basic                                                 
  end Aria Material 304SS  # 304 Stainless Steel, Isotropic 

  Begin Aria Material 321SS  # Copy properties from 304SS
        Density              = constant  rho = 7901.  # [kg/m^3]
        Thermal Conductivity = user_function  name=f_steelcond X=Temperature  # [W/m-K]
        Specific Heat        = user_function  name=f_steelspheat X=Temperature  # [J/kg-K]
        Emissivity           = constant  e=0.8
        Heat Conduction = basic
  end Aria Material 321SS  # Copy properties from 304SS

  BEGIN ARIA MATERIAL fluid
    species names = CO2 Organic N2

    Density            = General_Ideal_Gas  R=8.314472
    Viscosity          = Constant  mu=2.0e-5
    Specific_heat      = Constant  cp=1000  # Needed by temperature model
    Specific_heat_cp   = Constant  value=1000  # Needed by thermal_diffusivity model

    molecular_weight of CO2     = constant  M=44e-3  # units: kg/mol
    molecular_weight of N2      = constant  M=28e-3
    molecular_weight of Organic = constant  M=100e-3

    mass_diffusivity of CO2     = constant  D=2.0e-5
    mass_diffusivity of N2      = constant  D=2.0e-5
    mass_diffusivity of ORGANIC = constant  D=2.0e-5

    thermal_conductivity = constant  k=0.0224  # FIXME: Compute from k=rho*cp*D to match porous?

    Bulk_Viscosity = Constant  kappa=0.0
    Momentum Stress = Formal_Newtonian
    mass_flux = darcy

    Mass Fraction Diffusive Flux of CO2      = basic
    Mass Fraction Diffusive Flux of Organic  = basic
    Energy Diffusive Flux                    = basic
  end ARIA MATERIAL fluid

  Begin Aria Material foam
    volumetric_heat_transfer_coefficient = constant  h=1.E5
    Emissivity = constant  e=0.7

    User_Expression = Scalar_String_Function user_tag = foamA_decomp_rate \$
      f = "SOLID_PHASE_DENSITY*SOLID_PHASE_MASS_FRACTION_FOAMA*{A1}*exp(-{E1}/(8314*SOLID_PHASE_TEMPERATURE))"

    User_Expression = Scalar_String_Function user_tag = foamB_decomp_rate \$
      f = "SOLID_PHASE_DENSITY*SOLID_PHASE_MASS_FRACTION_FOAMB*{A2}*exp(-{E2}/(8314*SOLID_PHASE_TEMPERATURE))"

    User_Expression = Scalar_String_Function user_tag = char_density_ratio \$
      f = "SOLID_PHASE_BULK_DENSITY_CHAR / SOLID_PHASE_BULK_DENSITY_FOAMB"

    User_Expression = Scalar_String_Function user_tag = foam_mass_src \$
      f = "-foamA_decomp_rate - (1 - char_density_ratio)*foamB_decomp_rate"

    User_Expression = Scalar_String_Function user_tag = co2_gas_mass_src \$
      f = "foamA_decomp_rate + {vo2_2}*(1-char_density_ratio)*foamB_decomp_rate"

    User_Expression = Scalar_String_Function user_tag = organic_gas_mass_src \$
      f = "(1-{vo2_2})*(1-char_density_ratio)*foamB_decomp_rate"

    User_Expression = Scalar_String_Function user_tag = gas_mass_src \$
      f = "co2_gas_mass_src + organic_gas_mass_src"

    User_Expression = Scalar_String_Function user_tag = gas_enthalpy_src \$
      f = "gas_mass_src*GAS_PHASE_SPECIFIC_HEAT*SOLID_PHASE_TEMPERATURE"

    User_Expression = Scalar_String_Function user_tag = condensed_enthalpy_src \$
      f = "- foamA_decomp_rate*({deltaH1} + SOLID_PHASE_ENTHALPY_FOAMA) \$
           - foamB_decomp_rate*({deltaH2} + SOLID_PHASE_ENTHALPY_FOAMB) \$
           + char_density_ratio*foamB_decomp_rate*SOLID_PHASE_ENTHALPY_CHAR"

    Begin Material Phase gas_phase
      SPECIES NAMES = CO2 Organic N2

      density = general_ideal_gas  R=8.314472
      velocity = darcy  GX=0 GY=-9.8 GZ=0
      relative_permeability = Constant  K=1.

      specific_heat = constant  cp=1000.  # [J/kg-K]
      viscosity     = constant  mu=2.0e-5

      molecular_weight of CO2     = constant  M=44e-3  # units: kg/mol
      molecular_weight of N2      = constant  M=28e-3
      molecular_weight of ORGANIC = constant  M=100e-3

      mass_diffusivity of CO2     = constant  D=2.0e-5
      mass_diffusivity of N2      = constant  D=2.0e-5
      mass_diffusivity of ORGANIC = constant  D=2.0e-5
      mixture_mass_diffusivity = mixture_average

      bulk_mass_density            = porous_density                # d(phi*rho)/dt term
      bulk_mass_density of CO2     = mass_fraction_porous_density  # d(phi*rho*Y_i)/dt term
      bulk_mass_density of Organic = mass_fraction_porous_density  # d(phi*rho*Y_i)/dt term

      mass balance advective flux of CO2      = porous  # d(rho*u_j*Y_i)/dx_j term
      mass_balance_advective_flux of Organic  = porous  # d(rho*u_j*Y_i)/dx_j term

      mass_balance_diffusive_flux of CO2      = porous  # d(q_j)/dx_j term
      mass_balance_diffusive_flux of Organic  = porous  # d(q_j)/dx_j term

      transported_enthalpy = porous                          # d(phi*rho*h)/dt term
      enthalpy_advection   = porous                          # d(mdot*h)/dx term
      heat_conduction      = porous_simplified_diffusive_enthalpy_flux  # d(q)/dx term
    end Material Phase gas_phase

    Begin Material Phase solid_phase
      SPECIES NAMES = FoamA FoamB Char

      intrinsic_permeability_scaling of FoamA = T_Exponent  K_ref=1.0E-9 T_ref=300. n=0.0
      intrinsic_permeability_scaling of FoamB = T_Exponent  K_ref=1.0E-9 T_ref=300. n=0.0
      intrinsic_permeability_scaling of Char  = T_Exponent  K_ref=1.0e-8 T_ref=300. n=0.0
      intrinsic_permeability of FoamA = constant  XX=1.0 YY=1.0 ZZ=1.0
      intrinsic_permeability of FoamB = constant  XX=1.0 YY=1.0 ZZ=1.0
      intrinsic_permeability of Char  = constant  XX=1.0 YY=1.0 ZZ=1.0
      intrinsic_permeability = volume_average

      specific_heat of FoamA = T_Exponent  cp_ref=1400. T_ref=300. n=0.0  # [J/kg-K]
      specific_heat of FoamB = T_Exponent  cp_ref=1400. T_ref=300. n=0.0  # [J/kg-K]
      specific_heat of Char  = T_Exponent  cp_ref=1400. T_ref=300. n=0.0  # [J/kg-K]
      specific_heat = mass_average  # Only used for post-processing
      enthalpy = mass_average

      bulk_conductivity of FoamA = T_Exponent  kbulk_ref=0.05 T_ref=300. n=1.6
      bulk_conductivity of FoamB = T_Exponent  kbulk_ref=0.05 T_ref=300. n=1.6
      bulk_conductivity of Char  = T_Exponent  kbulk_ref=0.05 T_ref=300. n=1.6
      bulk_conductivity = volume_average

      radiative_conductivity of FoamA = Optically_Thick  n=1.0 beta_r=187.5 
      radiative_conductivity of FoamB = Optically_Thick  n=1.0 beta_r=187.5
      radiative_conductivity of Char  = Optically_Thick  n=1.0 beta_r=187.5  
      radiative_conductivity = volume_average

      thermal_conductivity = summed  contributions="bulk_conductivity radiative_conductivity"

      bulk_density of FoamA = T_Exponent  rho_ref=26.5 T_ref=300. n=0.0
      bulk_density of FoamB = T_Exponent  rho_ref=26.5 T_ref=300. n=0.0
      bulk_density of Char  = T_Exponent  rho_ref=17.961 T_ref=300. n=0.0

      solid_density of FoamA = constant rho_solid=883.3
      solid_density of FoamB = constant rho_solid=883.3
      solid_density of Char  = constant rho_solid=883.3
      volume_fraction of FoamA = from_mass_fraction
      volume_fraction of FoamB = from_mass_fraction
      volume_fraction of Char = from_mass_fraction
      porosity = one_minus_volume_fractions

      # Needed for the solid phase enthalpy convection contribution at the porous-fluid interface
      # At present only the generic models like constant are available, in the future we can add a model
      # to update this as the porosity changes.
      specific_surface_area = constant value={SA_VOL}

      bulk_mass_density           = density                # d(rho)/dt term
      bulk_mass_density of FoamA  = mass_fraction_density  # d(rho*Y_i)/dt term
      bulk_mass_density of FoamB  = mass_fraction_density  # d(rho*Y_i)/dt term

      transported_enthalpy = standard        # d(rho*h)/dt term
      heat_conduction      = fouriers_law    # d(q)/dx term
    end Material Phase solid_phase
  End Aria Material foam

  Begin Definition for Function f_steelcond  # AISI-304 stainless steel conductivity
    Type is PIECEWISE LINEAR
    Begin Values
      #Temperature    Conductivity (W/mK) from W.Gill
      50.0            14.5
      300.0           14.5     
      330.0           15      
      344.5           15.33
      372.8           15.77
      415.5           16.68
      470.1           17.56
      511.7           18.26
      562.2           19.17
      606.5           19.97
      743.1           22.2
      783.2           22.91
      840.6           23.91
      880.1           24.6
      2000.0          24.6     # Extrapolated value
   End Values
  End Definition for Function f_steelcond
  
  Begin Definition for Function f_steelspheat  # AISI-304 stainless steel specific heat 
    Type is PIECEWISE LINEAR
    Begin Values
      #Temperature    Specific Heat (J/kgK) from W.Gill
      50.0            460.31 
      273.5           460.31   
      293.3           472.86
      300.3           477.05
      350.3           497.97
      400.3           514.71
      450.3           527.26
      500.3           539.82
      550.3           548.19
      600.3           556.56
      650.3           564.92
      700.3           569.11
      750.3           577.48
      800.3           581.66
      850.3           585.85
      900.3           594.22
      950.3           602.59
      1000.3          610.96
      1100.3          623.51
      1200.3          640.25
      1300.3          652.8
      1400.3          665.36
      1500.3          682.09
    End Values
  End Definition for Function f_steelspheat

  BEGIN DEFINITION FOR FUNCTION T1 # Temperature history of cup bottom
    # test T1- tdi, tall, upright, 900- same as t2
    type is piecewise linear
    begin values
      #t(s)     T (K) v1
      0,        302.02
      3,        303.36
      100,      646.58
      250,      1135.16
      270,      1182.28
      277,      1183.82
      287,      1180.63
      324,      1160.31
      424,      1159.18
      800,      1159.18
    end values
  END DEFINITION FOR FUNCTION T1

  Begin Output Scheduler forResults
    At Step 0, Increment = 1  # Rapid output for testing
    #At Time 0, Increment = 1.0
  end Output Scheduler forResults

  Begin Procedure myAriaProcedure

    Begin Solution Control Description
      Use System Main

      Begin Initialize init
        #TODO: Remove these transfers
        Transfer solid_to_porous
        Transfer solid_to_fluid
        Transfer porous_to_solid
        Transfer fluid_to_solid
        Advance all
      end Initialize init

      Begin System Main
        Use Initialize init

        Begin Transient The_Transient_Block
          #TODO: Remove these transfers
          Transfer solid_to_porous
          Transfer solid_to_fluid
          Transfer porous_to_solid
          Transfer fluid_to_solid
          Advance all
        end Transient The_Transient_Block

        Simulation Start Time = 0.0
        Simulation Termination Time = 1e-4
        #Simulation Termination Time = 1.
        #Simulation Termination Time = 10.
        #Simulation Termination Time = 600.
      end System Main

      Begin Parameters For Transient The_Transient_Block
        Begin Parameters For Aria Region all
          Initial Time Step Size = 1.e-4
          maximum time step size = 5.0
          Time Step Variation = adaptive
          Time Integration Method = BDF2
          predictor order = 1
          predictor-corrector tolerance = 1.0e-3
          predictor-corrector field normalization for gas_phase_mass_fraction_CO2     = USER  1.0
          predictor-corrector field normalization for gas_phase_mass_fraction_Organic = USER  1.0
          courant limit = 10.0
        end Parameters For Aria Region all
      end Parameters For Transient The_Transient_Block
    End solution control description

    # CHT BCs (arrow indicates what is *sent* to the other region):
    #   Solid T <--> Porous Flux
    #   Solid T <--> Fluid Flux

    begin TRANSFER solid_to_porous
      interpolate surface nodes from all to all
      send block surface_5 to surface_5
      send field solution->temperature state new to bc_wall_temp_solid state none
      send field solution->temperature state new to solution->solid_phase_temperature state new
      send field solution->temperature state new to solution->gas_phase_temperature state new
      geometric tolerance = 0.01
    end TRANSFER solid_to_porous

    begin TRANSFER solid_to_fluid
      interpolate surface nodes from all to all
      send block surface_7 surface_8 to surface_70 surface_80
      send field solution->Temperature state new to nodal_cht_Tw state none
      geometric tolerance = 0.0001
    end TRANSFER solid_to_fluid

    begin TRANSFER fluid_to_solid
      interpolate surface nodes from all to all
      send block surface_70 surface_80 to surface_7 surface_8
      send field nodal_cht_Too state none to bc_ref_temp_fluid state none
      send field nodal_cht_H state none to bc_conv_coeff_fluid state none
      geometric tolerance = 0.0001
    end TRANSFER fluid_to_solid

    begin TRANSFER porous_to_solid
      interpolate surface nodes from all to all
      send block surface_5 to surface_5
      send field bc_ref_temp_porous_solid state none to bc_ref_temp_porous state none
      send field bc_conv_coeff_porous_solid state none to bc_conv_coeff_porous state none
      geometric tolerance = 0.01
    end TRANSFER porous_to_solid

    BEGIN ARIA REGION all
        
      Use Linear Solver gmres_teko
      Nonlinear Residual Tolerance   = 1e-6
      Nonlinear Correction Tolerance = 1.0e-6
      Nonlinear Relaxation Factor    = 1.0
      Maximum Nonlinear Iterations   = 30
      Nonlinear Solution Strategy    = line_search

      use finite element model all

      Mesh Group fluid            = block_6
      Mesh Group fluid_solid_interface  = surface_70 surface_80
      Mesh Group fluid_porous_interface = surface_6

      Mesh Group porous          = block_5
      Mesh Group porous_solid_interface = surface_5
      Mesh Group porous_fluid_interface = surface_6

      # For solid
      Mesh Group canister               = block_1 block_2 block_3 block_4
      Mesh Group solid_porous_interface = surface_5
      Mesh Group solid_fluid_interface  = surface_7 surface_8
      Mesh Group heated_surface         = surface_1
      Mesh Group convective_surface     = surface_2 surface_3 surface_4

        EQ energy FOR temperature ON canister USING Q1 WITH mass diff

        # Interface boundary with fluid region
        USER FIELD REAL NODE SCALAR bc_conv_coeff_fluid on solid_fluid_interface
        USER FIELD REAL NODE SCALAR bc_ref_temp_fluid on solid_fluid_interface
        Begin Convective Flux Boundary Condition cht_fluid
          add surface solid_fluid_interface
          Reference Temperature Node Variable is bc_ref_temp_fluid
          Convective Coefficient Node Variable is bc_conv_coeff_fluid
        End

        # Interface boundary with porous region
        User Field real node scalar bc_conv_coeff_porous on solid_porous_interface    
        User Field real node scalar bc_ref_temp_porous on solid_porous_interface    
        Begin Convective Flux Boundary Condition cht_porous
          add surface solid_porous_interface    
          Reference Temperature Node Variable is bc_ref_temp_porous
          Convective Coefficient Node Variable is bc_conv_coeff_porous
        End

        # Heated lower surface
        BC dirichlet FOR temperature ON heated_surface = user_function  name=T1 X=Time

        # Outer convective boundary condition
        BC flux FOR energy ON convective_surface = nat_conv  H=12.56 T_Ref=299.

        Postprocess Heat_Flux ON canister

        # Enclosure radiation definitions begin here.
        # They may be placed in either the solid or porous
        # equation system.
        Begin Viewfactor Calculation vf_calc
          Compute Rule = Hemicube
          #Compute Rule            = Pairwise
          BSP Tree Max Depth = 12 and min list length = 25
          Geometric Tolerance = 1.0E-7
          Hemicube Resolution = 50
          Hemicube Max Subdivides = 5
          Output Rule = Summary
        End

        Begin Viewfactor Smoothing smooth
          Method = Least-Squares # None
          Convergence Tolerance = 1.0E-08
          Maximum Iterations = 150
          Weight Power = 2.0
          Reciprocity Rule = Average # Subtraction
          Output Rule = Summary
        End

        Begin Radiosity Solver Rad_Solv
          Coupling = Mason
          Solver = Chaparral GMRES
          Convergence Tolerance = 1.0E-8
          Maximum Iterations = 1000
          Output Rule = verbose
        End 

        Begin Enclosure Definition Enc1
          add surface surface_7 surface_8 surface_600
          meshed enclosure is block_6 # Cannot use mesh groups here unfortunately
          Blocking Surfaces
          Use Viewfactor Calculation vf_calc
          Use Viewfactor Smoothing smooth
          Use Radiosity Solver Rad_Solv
          Matched flux on surface_600 = porous_enthalpy phase = solid_phase
        End
        # End of radiation definitions

      # CONDENSED PHASE MASS BALANCE and species
      #

        EQ mass_balance FOR density IN solid_phase ON porous USING Q1 WITH lumped_mass src
        Source FOR mass_balance IN solid_phase ON porous = scalar_string_function f="foam_mass_src"

        EQ mass_balance FOR mass_fraction OF FoamA IN solid_phase ON porous USING Q1 WITH lumped_mass src
        Source FOR mass_balance OF FoamA IN solid_phase ON porous = scalar_string_function f="-foamA_decomp_rate"

        EQ mass_balance FOR mass_fraction OF FoamB IN solid_phase ON porous USING Q1 WITH lumped_mass src
        Source FOR mass_balance OF FoamB IN solid_phase ON porous = scalar_string_function f = "-foamB_decomp_rate"

        EQ mass_balance FOR mass_fraction OF Char IN solid_phase ON porous USING Q1 WITH fracbal 

        # POSTPROCESSORS
        Postprocess porosity IN solid_phase ON porous
        Postprocess intrinsic_permeability_XX IN solid_phase ON porous

      # GAS PHASE MASS BALANCE

        EQ mass_balance FOR pressure IN gas_phase ON porous USING Q1 WITH lumped_mass adv src
        Source FOR mass_balance IN gas_phase ON porous = scalar_string_function f="gas_mass_src"

        # Flow coupling BC in the porous region
        BC flux for mass_balance in gas_phase on porous_fluid_interface = fluid_robin_coupled_one_region coeff_scaling=0.000001

        # POSTPROCESSORS
        Postprocess density IN gas_phase ON porous
        Postprocess velocity IN gas_phase ON porous
        Postprocess mass_balance_source in gas_phase on porous

        EQ cvfem_continuity for pressure on fluid using Q1 with LUMPED_MASS ADV
        EQ cvfem_momentum for velocity on fluid using Q1 with LUMPED_MASS DIFF ADV SRC

        SOURCE for cvfem_momentum on fluid = hydrostatic  ref_density=1.123 gx=0 gy=-9.8 gz=0

        BC Dirichlet for velocity on fluid_solid_interface = Constant  X=0.0 Y=0.0

        # Flow coupling BC
        BC flux for cvfem_continuity on fluid_porous_interface = porous_robin_one_region
        BC Disting for cvfem_momentum on fluid_porous_interface = porous_robin_one_region alpha=0.1

        Postprocess Density on fluid


        EQ cvfem_energy for enthalpy on fluid using Q1 with lumped_mass adv diff src
        EQ cvfem_mass_fraction for mass_fraction of CO2     on fluid using Q1 with lumped_mass adv diff
        EQ cvfem_mass_fraction for mass_fraction of Organic on fluid using Q1 with lumped_mass adv diff
        EQ cvfem_mass_fraction for mass_fraction of N2      on fluid using Q1 with fracbal
  
        #SOURCE for cvfem_energy on fluid = viscous_dissipation
        SOURCE for cvfem_energy on fluid = pressure_material_derivative

        # Interface boundary with solid region
        BC disting for cvfem_energy on fluid_solid_interface = from_cht_temperature_cpt

        # Enthalpy coupling BC with porous
        BC flux for cvfem_energy on fluid_porous_interface = porous_robin_coupled_with_solid_phase_convection_one_region

        # Mass fraction coupling BCs with porous
        BC flux for cvfem_mass_fraction of CO2 on fluid_porous_interface = porous_robin_coupled_one_region
        BC flux for cvfem_mass_fraction of Organic on fluid_porous_interface = porous_robin_coupled_one_region

        Postprocess Temperature on fluid
        Postprocess specific_heat on fluid

      # CONDENSED AND GAS PHASE ENERGY 
      #

        EQ porous_enthalpy FOR temperature IN solid_phase ON porous USING Q1 WITH lumped_mass diff src
        SOURCE FOR porous_enthalpy IN solid_phase ON porous = volumetric_transfer  remote_phase="gas_phase"
        SOURCE FOR porous_enthalpy IN solid_phase ON porous = scalar_string_function f="condensed_enthalpy_src"

        EQ porous_enthalpy FOR temperature IN gas_phase ON porous USING Q1 WITH lumped_mass adv diff src
        SOURCE FOR porous_enthalpy IN gas_phase ON porous = compressible_pressure  # (dp/dt + ui*dp/dxi) term
        SOURCE FOR porous_enthalpy IN gas_phase ON porous = volumetric_transfer  remote_phase="solid_phase"
        SOURCE FOR porous_enthalpy IN gas_phase ON porous = scalar_string_function f="gas_enthalpy_src"

        # Enthalpy coupling BC in the porous region.
        BC flux for porous_enthalpy in gas_phase on porous_fluid_interface = fluid_robin_coupled_with_solid_convection_one_region
        BC flux for porous_enthalpy in solid_phase on porous_fluid_interface = fluid_solid_convection_coupled_one_region

        # Interface boundary with solid region
        USER FIELD REAL NODE SCALAR bc_wall_temp_solid on porous_solid_interface
        BC XFER Dirichlet on porous_solid_interface solid_phase_temperature
        BC XFER Dirichlet on porous_solid_interface gas_phase_temperature
        BC disting for porous_enthalpy in solid_phase on porous_solid_interface = from_cht_temperature  interface_name="solid" Twall_field="bc_wall_temp_solid" Tref_field="bc_ref_temp_porous_solid" H_field="bc_conv_coeff_porous_solid"

        # POSTPROCESSORS
        Postprocess porous_enthalpy_source IN solid_phase ON porous
        Postprocess porous_enthalpy_source IN gas_phase ON porous
        Postprocess specific_heat IN solid_phase ON porous
        Postprocess bulk_conductivity IN solid_phase ON porous
        Postprocess radiative_conductivity IN solid_phase ON porous
        Postprocess thermal_conductivity IN solid_phase ON porous

      # GAS PHASE SPECIES
      #

        EQ mass_balance FOR mass_fraction OF CO2 IN gas_phase ON porous USING Q1 WITH lumped_mass adv diff src
        Source FOR mass_balance OF CO2 IN gas_phase ON porous = scalar_string_function f="co2_gas_mass_src"

        EQ mass_balance FOR mass_fraction OF organic IN gas_phase ON porous USING Q1 WITH lumped_mass adv diff src
        Source FOR mass_balance OF organic IN gas_phase ON porous = scalar_string_function f="organic_gas_mass_src"

        EQ mass_balance FOR mass_fraction OF N2 IN gas_phase ON porous USING Q1 WITH fracbal

        # Mass fraction coupling BCs in the porous region.
        # Needs the xfer mass fraction and diffusive flux
        BC flux for mass_balance of CO2 in gas_phase on porous_fluid_interface = mass_fraction_fluid_robin_coupled_one_region
        BC flux for mass_balance of Organic in gas_phase on porous_fluid_interface = mass_fraction_fluid_robin_coupled_one_region

        Postprocess mass_balance_source of CO2 in gas_phase on porous
        Postprocess mass_balance_source of Organic in gas_phase on porous

      IC for Pressure on fluid = constant  value=100000.0
      IC for velocity on fluid = constant x=0. y=0.

      IC for Enthalpy on fluid  = Constant H=300000.0
      IC for Mass_Fraction of CO2     on fluid = Constant value=0.0
      IC for Mass_Fraction of Organic on fluid = Constant value=0.0
      IC for Mass_Fraction of N2      on fluid = fracbal

      # Solid temp
      IC const FOR canister Temperature = 300

      IC FOR density IN solid_phase ON porous = mass_average
      IC FOR mass_fraction OF FoamA IN solid_phase ON porous = constant  value=0.25
      IC FOR mass_fraction OF FoamB IN solid_phase ON porous = constant  value=0.75
      IC FOR mass_fraction OF Char  IN solid_phase ON porous = fracbal
      IC FOR temperature IN solid_phase ON porous = constant  value=300.

      IC FOR pressure IN gas_phase ON porous = constant  value=100000.
      IC FOR mass_fraction OF CO2     IN gas_phase ON porous = constant  value=0.0
      IC FOR mass_fraction OF organic IN gas_phase ON porous = constant  value=0.0
      IC FOR mass_fraction OF N2      IN gas_phase ON porous = fracbal
      IC FOR temperature IN gas_phase ON porous = constant  value=300.

      Predictor Fields = not solid_phase_mass_fraction_Char
      Predictor Fields = not gas_phase_mass_fraction_N2
      predictor fields = not velocity

      BEGIN SOLUTION OPTIONS
        # Needed when not using enthalpy equation to prevent utility creation
        omit enthalpy adjustment after temperature clipping

        BEGIN CVFEM ALGORITHM SPECIFICATION
          PRESSURE STABILIZATION ORDER   = second_order
          PRESSURE STABILIZATION SCALING = characteristic
          FIRST ORDER UPWIND FACTOR = 1.0 FOR EQUATION cvfem_momentum
          FIRST ORDER UPWIND FACTOR = 1.0 FOR EQUATION cvfem_energy
          FIRST ORDER UPWIND FACTOR = 1.0 FOR EQUATION cvfem_mass_fraction
        end CVFEM ALGORITHM SPECIFICATION

        BEGIN TURBULENCE MODEL SPECIFICATION
          TURBULENCE MODEL = LAMINAR
        end TURBULENCE MODEL SPECIFICATION
      END SOLUTION OPTIONS

      BEGIN RESULTS OUTPUT LABEL fluid output
        database Name = omd_fic_enclosure_all.e
        Use Output Scheduler forResults
        title Aria uniform velocity test
        nodal variables = solution->VELOCITY    as v
        nodal variables = solution->pressure    as P
        nodal variables = solution->enthalpy    as h
        nodal variables = solution->mass_fraction_CO2     as Y_CO2
        nodal variables = solution->mass_fraction_N2      as Y_N2
        nodal variables = solution->mass_fraction_Organic as Y_Organic

        nodal variables = pp->Density as rho
        nodal variables = pp->Viscosity as mu
        nodal variables = pp->Temperature as T
        nodal variables = pp->specific_heat_cp as Cpp
        nodal variables = pp->specific_heat as Cp

        Node Variables = solution->solid_phase_density as rho_bar
        Node Variables = solution->solid_phase_temperature as T_bar
        Node Variables = solution->solid_phase_mass_fraction_FoamA as Y_FoamA
        Node Variables = solution->solid_phase_mass_fraction_FoamB as Y_FoamB
        Node Variables = solution->solid_phase_mass_fraction_Char as Y_Char

        Node Variables = solution->gas_phase_temperature as T_g
        Node Variables = solution->gas_phase_pressure as P_g
        Node Variables = solution->gas_phase_mass_fraction_CO2 as Yg_CO2
        Node Variables = solution->gas_phase_mass_fraction_N2 as Yg_N2
        Node Variables = solution->gas_phase_mass_fraction_Organic as Yg_Organic

        Node Variables = pp->solid_phase_porosity as phi_bar
        Node Variables = pp->solid_phase_porosity_FoamA as phi_FoamA
        Node Variables = pp->solid_phase_porosity_FoamB as phi_FoamB
        Node Variables = pp->solid_phase_porosity_Char as phi_Char

        Node Variables = pp->gas_phase_velocity as v_g
        Node Variables = pp->solid_phase_heat_flux as q_bar

        nodal variables = bc_conv_coeff_fluid
        nodal variables = bc_ref_temp_fluid
        nodal variables = bc_conv_coeff_porous
        nodal variables = bc_ref_temp_porous

        nodal variables = pp->solid_phase_intrinsic_permeability_xx as intrinsic_perm_xx
        nodal variables = pp->solid_phase_thermal_conductivity as solid_thermal_cond
        nodal variables = pp->solid_phase_bulk_conductivity as solid_bulk_cond
        nodal variables = pp->solid_phase_radiative_conductivity as solid_radiative_cond
        nodal variables = pp->solid_phase_enthalpy_source as cond_enth_source

        nodal variables = pp->gas_phase_enthalpy_source as gas_enth_source
        nodal variables = pp->gas_phase_mass_balance_source as gas_source
        nodal variables = pp->gas_phase_mass_balance_source_CO2 as CO2_source
        nodal variables = pp->gas_phase_mass_balance_source_Organic as Organic_source

        # Solid variables
        Nodal Variables = solution->TEMPERATURE as T_solid
        Nodal Variables = pp->heat_flux as q
      END RESULTS OUTPUT LABEL fluid output
    End aria region all
  End 
End 

