function [ emissions, lats2D, longs2D ] = getEmissionsFromFile( edir, periods )
%getEmissionsFromFile Get 1 deg x 1deg Gurney emissions from files over
%periods
%   getEmissionsFromFile( edir, periods ) Get emissions as a 2D matrix,
%   over the 2D domain with cell-center (50.5N, -126.5W) <-> (24.5N, -63.5W),
%   leading to a 27 x 64 matrix. The emission files contain gurney
%   emissions at 1 deg resolution, averaged over 8 days and in
%   microMoles/m^2/sec.
%
% INPUTS:
%   edir ; the directory where the emission files are 
%   periods : [start_period, end_period]
%
% OUTPUTS:
%   emissions ; a 2D matrix, ncells X nperiods in size. ncells = 27*64
%   lats      : lats of the cell centers where emissions resize. matrix,
%                27 X 64 in size
%   longs     : ditto, longitudes
%
% Jaideep Ray, 5/18/2012
% =========================================================================

% ---- Get lats and longs
lats2D  = load([edir, '/EightDayAvSourcesLatitiudes.dat']) ;
longs2D = load([edir, '/EightDayAvSourcesLongitudes.dat']) ;

% ---- Allocate emissions matrix
nPeriods = periods(2) - periods(1) + 1 ;
emissions = zeros( numel(lats2D), nPeriods) ;

count = 1 ;
for r = periods(1) : periods(2)
    % Make the filename
    str = sprintf('/EightDayAvSources%02d.dat', r) ;
    filename = [edir, str] ;
    
    % Read file
    d = load(filename) ;
    
    % Stick into matrix
    emissions(:, count) = d(:) ;
    count = count + 1;
end
end

