function [ G ] = computeGain( prob )
%computeGain Function to compute Gain = H * Phi
%   [ G ] = computeGain( prob ) computes the Gain matrix for the problem so
%   that we can write y = Gx, x being the wavelet coefficients.
%
%   INPUTS :
%   prob.H   : cell array of H matrices for the towers. H{i} is a 2D matrix, of
%              size (nObsPerPeriod*nPeriods) x (nUSCells*nPeriods)
%   prob.Phi : wavelet bases. nUSCells x ncoeffsPerPeriod in size
%
%   OUTPUTS:
%   G        : cell array of gain matrices for towers. G{i} is a matrix, 
%              (nObsPerPeriod*nPeriods) x (ncoeffsPerPeriod*nPeriods) in
%              size
%
%   Jaideep Ray, 3/21/2012
%
% -------------------------------------------------------------------------

for i = 1 : prob.info.nTowers % loop over all towers
    G{i} = scaleMatrices(prob.H{i}, prob.Phi, prob.info) ;
end
end


function G = scaleMatrices(H, Phi, info)
% Function to scale the sensitivity matrix
%
% H, for 12 8-day periods is
% H = [ H_11  H_12 .... H_1,12
%       H_21  H_22      H_2,12
%       H_31            H_3,12]
%    
% H_i,j is  nObsPerPeriod x nUSCells in size. 
%
% G = [ H_11 Phi  H_12 Phi     H_1,12 Phi
%       H_21 Phi  H_22 Phi     H_2,12 Phi
%                              H_3,12 Phi ]
%
% -------------------------------------------------------------------------
ncoeffsPerPeriod = size(Phi, 2) ;
nUSCells         = info.nUSCells ;
nObsPerPeriod    = info.nObsPerPeriod ;
nPeriods         = size(H, 2) / nUSCells ;

c1 = 1 : ncoeffsPerPeriod : ncoeffsPerPeriod*nPeriods ;
G = zeros(size(H, 1), ncoeffsPerPeriod*nPeriods) ;
for r = 1 : nObsPerPeriod : size(H, 1)
    i = 1 ;
    for c = 1 : nUSCells : size(H, 2)
        G(r : r + nObsPerPeriod-1, c1(i) : c1(i) + ncoeffsPerPeriod-1) = ...
           H( r : r + nObsPerPeriod-1, c : c + nUSCells-1) * Phi ;
       i = i + 1 ;
    end
end
G = sparse(G) ;
end
