function [ csmeas ] = calcOceanEmissions(prob, sol) 
%calcOceanEmissions Calculate the compressively sensed ffCO2 emissions over 
%                   oceans in the wavelet coeff vector sol
%   [ csmeas ] = calcOceanEmissions(prob, sol)  Calculates the comressively sensed
%   ffCO2 emissions, as estimated, over oceans, Canada and Mexico, in fact all
%   areas in North America which are not the Lower 48 states. This number
%   should be zero; its deviation from zero is a measure of error in the
%   solution
%
%  INPUTS :
%   prob : a struct containing info about the buffer we use as we all the
%          gain matrix for the compressively sampled region (where we are
%          supposed to have zero ffCO2 emissions)
%   sol :  a long vector, contain the wavelet coeffs for all the periods of
%          interest as well as buffer regions.
%
%  OUTPUTS 
%    csmeas : a vector which is a CS measurement of the emissions over all
%             grid cells not in Lower48
%
% Jaideep Ray, 6/27/2012
% -------------------------------------------------------------------------

% ---- Extract the useful soln i.e. wavelet coeff outside of buffer periods
% and cast them into a 2D matrix, nwavelets x nUsefulPeriods in size

info = prob.info ;
usefulSol = sol(info.buffer*size(prob.Phi,2)+1 : end - info.buffer*size(prob.Phi,2)) ;

% ---- Multiplying wavelet solution by GCS gives the compressively sensed
% measurements of emissions in the region CLower48. This should be zero
GCS       = prob.GCS(:, ....
               info.buffer*size(prob.Phi,2)+1 : end - info.buffer*size(prob.Phi,2)) ;
csmeas = GCS * usefulSol ;

fprintf('\n\nNorm of the %d CS emissions in CLower48, over %d periods is %f\n', ...
        length(csmeas), info.nPeriods, norm(csmeas)) ;
end

