function [ wcv ] = extractMaskedWaveletCoeffs( wcs, level, index )
%extractWaveletCoeffs Extract wavelet coefficients from a WCS
%
%   wcv = extractWaveletCoeffs(wcs, level, index) is used to extract the
%   wavelet coefficients from a WCS on a level-by-level basis. Specifying
%   the index (1, 2 or 3) gives you the wavelet coeffs for the x-, y- or
%   xy- direction trends. wcv is a 1D array.
%
%   wcv = extractWaveletCoeffs(wcs, level) returns ALL the wavelet
%   coefficients on level 'level'.
%
%  INPUTS :
%   wcs   A complete WCS structure. It has wcs.coeffs filled in.
%
%   level The level (of the wavelet decomposition) whose coeffs you are
%         interested in. 0 <= level <= wcs.nlevels.
%
%   index 1, 2 or 3, to get the wavelet coeffs (on level 'level')
%         controlling variations in x-, y-, or xt- directions. If you don't
%         specify index, you get all 3 type of wavelet coeffs.
%
% OUTPUTS
%  wcv    The wavelet coeff vector. It is actually a 2^{level-1} X
%         2^{level-1} 2D array, reshaped into a 1D array. If you did not
%         specify the index, the array is 3 times as long since the X, Y
%         and XY coefficients are concatenated into the same 1D array, in
%         that sequence.
%
% Jaideep Ray, SNL, CA, 2/10/2011.
% =========================================================================

% ---- If zeroth level, return mother wavelet
if level == 0
    wcv = wcs.baseVal ; % baseval has already been masked
    return ;
end

% ---- If index specified, just return that.
if (nargin == 3) % Somebody specified an index
    tmp = wcs.coeffs{level, index} ;
    ind = wcs.maskNonZeroIndices{level, index} ;
    wcv = tmp(ind) ;
    return ;
end

if (level > wcs.nlevels) 
    wcv = 0 ;
    return 
end

% ---- If no index specified, return wavelet coeffs in x-, y- and xy-
% directions.
tmpA = wcs.coeffs{level, 1} ; tmpB = wcs.maskNonZeroIndices{level, 1} ;
tmp1 = tmpA(tmpB) ;

tmpA = wcs.coeffs{level, 2} ; tmpB = wcs.maskNonZeroIndices{level, 2} ;
tmp2 = tmpA(tmpB) ;

tmpA = wcs.coeffs{level, 3} ; tmpB = wcs.maskNonZeroIndices{level, 3} ;
tmp3 = tmpA(tmpB) ;

wcv = cat(1, tmp1, tmp2, tmp3) ;

end

