/**
   \file graphDB.cpp

   Implementation of the graph DB

   Jaideep Ray

   7/25/2011
*/

#include <iostream>
#include <sstream>
#include <iomanip>
#include <math.h>
#include <cassert>
#include "converter.h"
#include "graphDB.h"

#define CHUNKSIZE 100

graphDB::graphDB() 
{
  allGraphs_.reserve(CHUNKSIZE) ;
}

graphDB::~graphDB()
{
  clear() ;
}

int graphDB::insertGraph(struct graph *pG)
{
  /* Allocate a copy */
  struct graph *pTmp = new struct graph ;
  if (pTmp == 0)
  {
    std::cerr << " graphDB::insertGraph() in " << __FILE__ << "; could not allocate graph. Return with error "
	      << std::endl ;
    return (-1) ;
  }
  alloc_gr(pTmp, pG->V, pG->E) ; // Creates an empty data structure

  /* Fill it up */
  for (int i = 0; i < pG->V + 1; i++) pTmp->ptr[i] = pG->ptr[i] ;
  for (int i = 0; i < pG->E*2  ; i++) pTmp->ind[i] = pG->ind[i] ;

  /* Insert into vector */
  if ( allGraphs_.size() == allGraphs_.capacity() ) 
    allGraphs_.reserve( allGraphs_.size() + CHUNKSIZE ) ;
  allGraphs_.push_back(pTmp) ;

  return (0) ;
}

int graphDB::getNumGraphsInDB() 
{ 
  return( allGraphs_.size() ) ; 
}

struct graph * graphDB::extractGraph(int i) 
{ 
  return( allGraphs_[i] ) ; 
}

struct graph * graphDB::extractCopyOfGraph(int i)
{

  // The one whose copy we'll make
  struct graph *pG = allGraphs_[i] ;

  // The copy
  struct graph *tmp = new struct graph ;
  alloc_gr(tmp, pG->V, pG->E) ;

  // Fill up the copy
  for (int i = 0; i < pG->V + 1; i++) tmp->ptr[i] = pG->ptr[i] ;
  for (int i = 0; i < pG->E*2  ; i++) tmp->ind[i] = pG->ind[i] ; 

  return (tmp) ;
}

int graphDB::writeGraphsToFiles(std::string filestub)
{
  /*
    The files containing the graphs' edgelists will be called file.000, file.001, ...
    We need to know how many zeros to add
  */
  long int tmp = lrint( ceil( log10( allGraphs_.size() ) ) );
  double denom = pow(10.0, tmp) ;

  /*
    Loop over all graphs in DB; write them out as edgelists
  */
  for (int i = 0; i < allGraphs_.size(); i++)
  {
    std::stringstream ss  ;
    ss << filestub << "_" << std::fixed << std::setw(tmp+1) << std::setprecision(tmp) << i/denom ;
    std::cout << " graphDB::writeGraphsToFiles() in " << __FILE__ << "; writing out file "
	      << ss.str() << std::endl ;
    
    int err = writeEdgelistToFile( allGraphs_[i], ss.str().c_str(), Text ) ;
    assert( err == 0 ) ;
  }

  return (0) ;
}

int graphDB::createThinnedVersionOfDatabase(graphDB &ndb, int thin_factor)
{
  if (thin_factor > getNumGraphsInDB())
  {
    std::cerr << " graphDB::createThinnedVersionOfDatabase() : Thinning factor larger than the chain itself!"
	      << " Doing nothing and return with -1 " << std::endl ;
    return (-1) ;
  }

  /* 
     Loop over graphs, extracting every "thin_factor"-th instance
  */
  for (int i = 0; i < getNumGraphsInDB(); i += thin_factor)
    ndb.insertGraph( extractGraph(i) ) ;

  return(0) ;
}

void graphDB::clear()
{
  /* Loop over all entries in the vector, deleteing each */
  for (int i = 0; i < allGraphs_.size(); i++)
  {
    struct graph *p = allGraphs_[i] ;
    free(p->ind) ; 
    free(p->ptr) ; 
    delete p ; 
    allGraphs_[i] = 0 ;
  }
  allGraphs_.clear() ;
}
