#ifndef CONVERTERHSEEN
#define  CONVERTERHSEEN
/**
   \file converter.h

   Header file for the converter functions
*/

// Definition of struct graph
extern "C" {
#include "graph.h"
}

/**
   Given a graph in the compressed representatin in struct G, compute
   its adjacency matrix.

   @param G, the graph in compressed form. (INPUT)
   @param numNodes, the number of nodes in that graph (OUTPUT)
   @param numEdges, the number of edges in that graph (OUTPUT)
   @param row, column : arrays, 2*numEdges long, that will be allocated in this function
                        and returned to caller. It's the caller's responsibility to 
			delete the memory. These will contain the (row, col) indices of
			the adjacency matrix which are occupied. (OUTPUT)
   @param indices     : an array, 2*numEdges long, that will contain the indices of the entries
                        in the adjacency matrix that are occupied.  (OUTPUT)
                        indices[i] = row[i] + col[i] * numNodes, i = 0 : numEdges-1
   @return  Zero if all OK, +ve if something went bad.  (OUTPUT)

   The adjacency matrix can be represented in 2 forms

  1) as M (row, col) pairs, indicating the M locations in an adjacency matrix that are
     occupied. N, the number of nodes in the graph is also needed for a full description.

  2) as M integers, which are the indices into the vector form of the 2D adjacency matrix. 
     We will follow the matlab way of indexing i.e., if the (r, c) entry in an N x N adjacency
     metrics is occupied, its corresponding index number indx = r + c*N. Both r and c are counted
     from zero, as in C and not one, as in fortran or matlab.
     
  The adjacency matrix A(r, c) is defined as a N x N boolean matrix. The rows of A
  indicate the emanating node and the columns, the terminating node. Thus, is A(r, c) == true,
  an edge going from node 'r' and ending in node 'c' exists.
  
  The adjacency matrix will be symmetric i.e., we will only deal with undirected graphs.
  So, if we report numEdges edges, matrix A will have 2*numEdges entries, as will the 
  row, col and indices vectors. We will NOT count upon filling up half the adjacency matrix and
  assume that the caller knows which half is filled up.
*/
int convert2AdjacencyMatrix(struct graph *G, int *numNodes, int *numEdges, 
			    int **row, int **column, long int **indices) ;


/**
   What kind of an output file do you want?
*/

typedef enum {ErrorFileType = -1, Binary, Text} FileType ;

/**
   Given a graph, write out its edgelist to file. The output file can be 
   text or binary.

   @param G, the incoming graph
   @param filename, where to dump the adjacency matrix
   @param fileType, Binary or Text
   @return 0 if ok, else +ve number

   If Text, the file contains:
         numNodes   numEdges
	 Then 2*numEdges rows of [emanatingNode  terminatingNode] pairs
   
   If Binary, the file contains
         numNodes  numEdges  (4 byte integers, each)
	 emanatingNodes, an array 2*numEdges long. (These are 4-byte integers)
	 terminatingNodes, an array 2*numEdges long
*/
int writeEdgelistToFile(struct graph *G, const char *filename, FileType a) ;
#endif
