/*
  \file converters.cpp

  This is a file that contains code that are used to convert the sparse graph
  structure in graph.h (struct graph) into an adjacency matrix as well as write the 
  matrix out to file. The adjacency matrix can be stored in 2 forms

  1) as M (row, col) pairs, indicating the M locations in an adjacency matrix that are
     occupied. N, the number of nodes in the graph is also needed for a full description.

  2) as M integers, which are the indices into the vector form of the 2D adjacency matrix. 
     We will follow the matlab way of indexing i.e., if the (r, c) entry in an N x N adjacency
     metrics is occupied, its corresponding index number indx = r + c*N. Both r and c are counted
     from zero, as in C and not one, as in fortran or matlab.

*/

// Include definition of struct graph
#include <iostream>
#include <string.h> // For memset()
#include <fstream>
#include "converter.h"

/**
   Given a graph in the compressed representatin in struct G, compute
   its adjacency matrix.

   @param G, the graph in compressed form. (INPUT)
   @param numNodes, the number of nodes in that graph (OUTPUT)
   @param numEdges, the number of edges in that graph (OUTPUT)
   @param row, column : arrays, 2*numEdges long, that will be allocated in this function
                        and returned to caller. It's the caller's responsibility to 
			delete the memory. These will contain the (row, col) indices of
			the adjacency matrix which are occupied. (OUTPUT)
   @param indices     : an array, 2*numEdges long, that will contain the indices of the entries
                        in the adjacency matrix that are occupied.  (OUTPUT)
                        indices[i] = row[i] + col[i] * numNodes, i = 0 : numEdges-1
   @return  Zero if all OK, +ve if something went bad.  (OUTPUT)

   The adjacency matrix A(r, c) is defined as a N x N boolean matrix. The rows of A
   indicate the emanating node and the columns, the terminating node. Thus, is A(r, c) == true,
   an edge going from node 'r' and ending in node 'c' exists.
   
   The adjacency matrix will be symmetric i.e., we will only deal with undirected graphs.
   So, if we report numEdges edges, matrix A will have 2*numEdges entries, as will the 
   row, col and indices vectors. We will NOT count upon filling up half the adjacency matrix and
   assume that the caller knows which half is filled up.
*/
int convert2AdjacencyMatrix(struct graph *G, int *numNodes, int *numEdges, 
			    int **row, int **column, long int **indices)
{
  *numNodes = G->V ;
  *numEdges = G->E ;

  // Allocate arrays and zero them out
  *row      = new int [ 2 * (*numEdges) ] ; 
  *column   = new int [ 2 * (*numEdges) ] ; 
  *indices  = new long int [ 2 * (*numEdges) ] ; 

  memset((*row),     0, sizeof(int) * (2 * (*numEdges))) ;
  memset((*column),  0, sizeof(int) * (2 * (*numEdges))) ;
  memset((*indices), 0, sizeof(long int) * (2 * (*numEdges))) ;

  /* Loop over all the emanating nodes in the graph */
  int sumEdges = 0 ; // How many edges did we fill up in row, column, and indices ?
  for (int enode = 0; enode < (*numNodes); enode++)
  {
    /* 
       The node 'enode' has myDegree edges emanating from it. The 
       edges are numbered consecutively, from startingEdgeNo : (startingEdgeNo+myDegree-1),
       both numbers inclusive. Find the startingEdgeNo and degree of this enode
    */
    int startingEdgeNo = G->ptr[enode] ;
    int myDegree       = G->ptr[enode+1] - G->ptr[enode] ;
    
    /*
      The nodes that I am connected to are stored in G->ind[startingEdgeNo : (startingEdgeNo+myDegree-1)].
      Extract them and fill up the rows, cols and indices
    */
    for (int myEdges = 0; myEdges < myDegree; myEdges++)
    {
      int edgeNo         = startingEdgeNo + myEdges ;
      (*row)[edgeNo]     = enode ;
      (*column)[edgeNo]  = G->ind[edgeNo] ;
      (*indices)[edgeNo] = (*row)[edgeNo] + (*column)[edgeNo] * (*numNodes) ;
      sumEdges++ ;
    }
  }
  
  if (sumEdges != 2*(*numEdges))
  {
    std::cerr << "convert2AdjacencyMatrix() in file " << __FILE__ << ", in line " << __LINE__
	      << " : The adjacency matrix should have " << 2*(*numEdges) << " but found "
	      << sumEdges << " edges. ERROR !! Return with error code " << std::endl ;
    return(1) ;
  }

  return(0) ;
}

/**
   Given a graph, write out its edgelist to file. The output file can be 
   text or binary.

   @param G, the incoming graph
   @param filename, where to dump the adjacency matrix
   @param fileType, Binary or Text
   @return 0 if ok, else +ve number.

   If Text, the file contains:
         numNodes   numEdges
	 Then 2*numEdges rows of [emanatingNode  terminatingNode] pairs
   
   If Binary, the file contains
         numNodes  numEdges
	 emanatingNodes, an array 2*numEdges long. These are 4-byte integers
	 terminatingNodes, an array 2*numEdges long
*/
int writeEdgelistToFile(struct graph *G, const char *filename, FileType a)
{
  // Convert the graph to row, col form
  int *row, *col;
  long int *indx ;
  int numNodes, numEdges ;
  convert2AdjacencyMatrix(G, &numNodes, &numEdges, &row, &col, &indx) ;

  // Write out edge list as text
  if ( a == Text )
  {
    std::ofstream ofile(filename) ;
    ofile << G->V << "     " << G->E << std::endl ;
    for (int edge = 0; edge < 2*(G->E); edge++)
      ofile << row[edge] << "     " << col[edge] << std::endl ;
    ofile.close() ;
  }

  // Write out row and col arrays as binary
  if ( a == Binary )
  {
    FILE *ofile = fopen(filename, "wb");

    int tmp[2] = {G->V, G->E} ;
    fwrite(tmp, sizeof(int), 2, ofile) ;

    // emanating nodes first
    fwrite(row, sizeof(int), 2*G->E, ofile) ;

    // terminating nodes then
    fwrite(col, sizeof(int), 2*G->E, ofile) ;
    
    fclose(ofile);
  }

  // Clean up
  delete [] row ;
  delete [] col ;
  delete [] indx ;

  return(0);
}
