#include "mcconv.h"
/*
  indtest                                                  
                                                                      
  This subroutine tests for an independence chain against a first-    
  order Markov chain using the log-linear modeling formulation.  In   
  our case the independence model is the [1][2][3] model, while the   
  first-order model is the [12][23] model.  Both the [1][2][3] and    
  the [12][23] models have closed form estimates - see Bishop,        
  Feinberg and Holland (1975).                                        

  Inputs:                                                             
                                                                      
    data     = an integer vector containing the series of 0's and 1's 
               for which this subroutine is to determine whether an   
               independence chain is sufficient or whether a first-   
               order Markov chain is needed to model the data.  There 
               must be at least datacnt elements in the data vector.  
                                                                      
    datacnt  = an integer containing the number of elements in the    
               data argument.                                         
                                                                      
                                                                      
  Outputs:                                                            
                                                                      
    g2       = a double precision number in which this subroutine is  
               to return the log likelihood ratio statistic for       
               testing a first-order Markov chain against simply an   
               independence chain.  Bishop, Feinberg and Holland      
               denote this statistic as G2.                           
                                                                      
    bic      = a double precision number in which this subroutine is  
               to return the BIC value corresponding to the log       
               likelihood ratio statistic, g2.                        
                                                                      
*/

void indtest(int *data, int n, double *g2, double *bic)
{

  double fitted, focus, dcm1 ;
  int    tran[4]={0,0,0,0}, i1, i2 ;
  int    n1=2, n2=2; 

  /*  
    Count up the number of occurrences of each possible type of        
    transition.  Keep these counts in the transition counts array.      
  */
  for ( i1 = 1; i1<n; i1++ )
    tran[ MATINDX(data[i1-1],data[i1],0,n1,n2,1) ]++;

  /*
    Compute the log likelihood ratio statistic for testing a first-     
    order Markov chain against simply an independence chain.  This is   
    Bishop, Feinberg and Holland's G2 statistic.                        
  */                                                                   
  dcm1 = (double) ( n-1 ) ;
  (*g2) = 0.0 ;

  for ( i1=0; i1<n1; i1++ )
    for ( i2=0; i2<n2; i2++ )
      if ( tran[MATINDX(i1,i2,0,n1,n2,1)] != 0 )
      { 
        fitted = ((double) ( (tran[MATINDX(i1,0,0,n1,n2,1)] 
			      + tran[MATINDX(i1,1,0,n1,n2,1)] )
			     *(tran[MATINDX(0,i2,0,n1,n2,1)] 
			       + tran[MATINDX(1,i2,0,n1,n2,1)]) 
			   )
		 ) / dcm1 ;

	focus = (double) tran[MATINDX(i1,i2,0,n1,n2,1)]  ;
	(*g2) += log( focus / fitted ) * focus ;
        //printf("%f %f %f\n",fitted,focus,*g2);
      }

  (*g2) *= 2.0 ;

  /*
  Finally calculate the associated bic statistic and return to the    
  caller.  Note that the first-order Markov chain model contains just 
  one more parameter than does the independence chain model, so p=1.  
  */
  (*bic) = (*g2) - log( dcm1 ) ;

  return ;

}
