#include "mcconv.h"
#include "limits.h"
/*
  gibbmain
                                                                      
  This program calculates the number of iterations required in a run  
  of MCMC.  The user has to specify the precision required.  This     
  subroutine returns the number of iterations required to estimate    
  the posterior cdf of the q-quantile of the quantity of interest (a  
  function of the parameters) to within +-r with probability s.  It   
  also gives the number of "burn-in" iterations required for the      
  conditional distribution given any starting point (of the derived   
  two-state process) to be within epsilon of the actual equilibrium   
  distribution.                                                       
                                                                      
  If q<=0, then gibbmain is to treat the original input series as a   
  vector of 0-1 outcome variables.  In this case no quantile needs to 
  be found.  Instead, this subroutine just needs to calculate kthin,  
  nburn, nprec and kmind tuning parameters such that a MCMC run based 
  on these tuning parameters should be adequate for estimating the    
  probability of an outcome of 1 within the prescribed (by r, s, and  
  epsilon) probability.                                               
                                                                      
  Inputs:                                                             
                                                                      
    original = a double precision vector containing the original MCMC 
               generated series of parameter estimates.  This vector  
               contains iteracnt elements.                            
                                                                      
    iteracnt = an integer containing the number of actual iterations  
               provided in the sample MCMC output series, original.   
                                                                      
    q,r,s    = double precision numbers in which the caller specifies 
               the required precision:  the q-quantile is to be       
               estimated to within r with probability s.              
                                                                      
    epsilon  = a double precision number containing the half width of 
               the tolerance interval required for the q-quantile.    
                                                                      
    work     = an integer vector passed to various subroutines to     
               hold a number of internal vectors.  There must be at   
               least (iteracnt * 2) elements in this vector.          
                                                                      
  Outputs:                                                            
                                                                      
    nmin     = an integer which will be set to the minimum number of  
               independent Gibbs iterates required to achieve the     
               specified accuracy for the q-quantile.                 
                                                                      
    kthin    = an integer which will be set to the skip parameter     
               sufficient to produce a first-order Markov chain.      
                                                                      
    nburn    = an integer which will be set to the number of          
               iterations to be discarded at the beginning of the     
               simulation, i.e. the number of burn-in iterations.     
                                                                      
    nprec    = an integer which will be set to the number of          
               iterations not including the burn-in iterations which  
               need to be obtained in order to attain the precision   
               specified by the values of the q, r and s input        
               parameters.                                            
                                                                      
    kmind    = an integer which will be set to the minimum skip       
               parameter sufficient to produce an independence chain. 
                                                                      
    r15      = an integer valued error return code.  This variable    
               is set to 0 if no errors were encountered.             
               Otherwise, r15 can assume the following values:        
                                                                      
                 12 = the original input vector contains something    
                      other than a 0 or 1 even though q<=0.           
                 13 = too short a chain to get a 1st order Markov process
*/

void gibbmain(double *original, int n, double q, double r,double s,
              double epsilon, double *dwrk, int *iwrk,
              int *nmin, int *kthin, int *nburn, int *nprec, int *kmind,
              int *r15)
{
  /*
  The following variables hold various temporary values used in this  
  subroutine.  This includes any do-loop counters and similar such    
  temporary subscripts and indices.                                   
                                                                      
    cutpt    - the q-th empirical quantile                            
    qhat     - when q=0, proportion of 1's in the input data vector,  
               when q>0, qhat is set equal to the passed value of q   
    g2       - G2 for the test of first-order vs second-order Markov  
    bic      - the corresponding BIC value                            
    phi      - \PHI^{-1} ((s+1)/2)                                    
    alpha    - probability of moving from below the cutpt to above    
    beta     - probability of moving from above the cutpt to below    
    probsum  - sum of alpha + beta                                    
                                                                      
  The first iteracnt elements of the work vector will be used to      
  store a binary 0-1 series indicating which elements are less than   
  or equal to the cutpt (set to 1) and which elements are less than   
  the cutpt (set to 0).                                               
                                                                      
  The remaining iteracnt elements of the work vector are to be used   
  to hold thinned versions of the 0-1 series, where the amount of     
  thinning is determined by the current value of kthin (or kmind).    
  That is, for each proposed value of kthin (or kmind), only every    
  kthin-th (or kmind-th) element of the 0-1 series is copied to this  
  thinned copy of the series.                                         
                                                                      
  ixkstart is the subscript of the first element of the thinned       
  series.  That is, ixkstart = iteracnt + 1.                          
                                                                      
  thincnt is the current length of the thinned series.                

  */

  double cutpt, qhat, g2, bic, phi, alpha, beta, probsum ;
  double tmp1, tmp2 ;
  int ixkstart, thincnt, i1, rc ;
  /*                                                                     
  If the q argument is a postive number, interpret it as the quantile 
  which is to be ascertained using MCMC.  It should be a positive     
  number less than 1.  Set qhat to the passed value of q (we will use 
  qhat later when we calculate nmin).                                 
  */                                                                      

  if ( q > 0.0 )
  {
    qhat = q ;
    /*  Find the q-th quantile of the original MCMC series of parameter 
        estimates. */
    cutpt = empquant(original,n,qhat,dwrk) ;

    /*
      Calculate a binary 0-1 series indicating which elements are less    
      than or equal to the cutpt (set to 1) and which elements are        
      greater than the cutpt (set to 0).  The resulting series is stored  
      in the work vector.                                                 
    */
    dichot(original,n,cutpt,iwrk) ;

  }
  else
  {
    /*
      Otherwise treat the original input series as a binary 0-1 series of 
      outcomes whose probability needs to be estimated using MCMC.  This  
      is easily accomplished by copying the input series into the first   
      iteracnt elements of the work vector, converting the double preci-  
      sion input into an equivalent integer vector of 0's and 1's.  For   
      this case we will also need to set qhat equal to the proportion of  
      1's in the original input data vector.                              
    */
    qhat = 0.0 ;
    for (i1=0; i1<n; i1++)
    {
      if (original[i1]==0.0 || original[i1] == 1.0)
      {
        iwrk[i1] = (int) original[i1] ;
        qhat += original[i1] ;
      }
      else
      {
        (*r15) = 12 ;
        return ;
      }
    }

    qhat /= (double) n ;

  }

  /*
    Find kthin, the degree of thinning at which the indicator series is
    first-order Markov.                                                 
  */
  ixkstart = n ;
  (*kthin) = 1 ;
  do 
  {
    thin(iwrk,n,(*kthin),&iwrk[ixkstart],&thincnt) ;
    mctest(&iwrk[ixkstart],thincnt,&g2,&bic) ;
    (*kthin)++ ;
  } while ( (bic > 0.0) && ((*kthin) < n ) ) ;
	    
  /* Check if we got a good value of (*kthin). Else, exit */
  if (bic > 0.0)
  {
    printf(" --> gibbmain(): Chain too short to make a first-order Markov process\n") ;
    *nprec = INT_MAX ;
    *nburn = INT_MAX ;
    *nmin  = INT_MAX ;
    *kthin = INT_MAX ;
    *kmind = INT_MAX ;
    *r15   = 13 ;
    return ;
  }
  (*kthin)-- ;

  /*
    Calculate both the alpha and beta transition probabilities (in the 
    Cox & Miller parametrization) of the two state first-order Markov   
    chain determined above.                                             
  */

  mcest(&iwrk[ixkstart],thincnt,&alpha,&beta) ;
  (*kmind) = (*kthin)-1 ;

  /*
  Now compute just how big the spacing needs to be so that a thinned  
  chain would no longer be a Markov chain, but rather would be an     
  independence chain.  This thinning parameter must be at least as    
  large as the thinning parameter required for a first-order Markov   
  chain.                                                              
  */
  do 
  {
    (*kmind)++ ;
    indtest(&iwrk[ixkstart],thincnt,&g2,&bic) ;
    if ( bic > 0 ) 
      thin(iwrk,n,(*kmind)+1,&iwrk[ixkstart],&thincnt) ;
     
  } while ( (bic > 0.0) && ((*kmind) < n) ) ;

  /* Did we get an independence chain ?*/
  if (bic > 0.0)
    printf(" gibbmain(): Chain too short to make an independence chain \n") ;

  /*
    Estimate the first-order Markov chain parameters and find the 
    burn-in and precision number of required iterations.          
  */
  probsum = alpha + beta ;
  tmp1 = log(probsum * epsilon / MAX(alpha,beta)) /
         log( fabs(1.0 - probsum) ) ;
  (*nburn) = ((int)( tmp1 + 1.0 )) * (*kthin) ;

  /*                                                                   
    Note:  ppnd7 is the routine that implements AS algorithm 241.       
    It calculates the specified percentile of the Normal distribution.
  */
  phi = ppnd7( ((s + 1.0) / 2.0), &rc ) ;
  tmp2 = (2.0 - probsum) * alpha * beta * phi*phi 
       / (pow(probsum,3) *r*r) ;
  (*nprec) = ((int)( tmp2 + 1.0 )) * (*kthin) ;
  (*nmin)  = (int) ( ((1.0-qhat) * qhat * phi*phi / (r*r)) + 1.0 ) ;

  /*
  At this point we have calculated nmin, kthin, nburn, nprec and 
  kmind, so we can return to the calling program.                
  */
  (*r15) = 0 ;

  return ;

}
