/*
  \file ex02.cpp

  This file contains a program that will read in a graph, change some of its 
  edges without destroying the JOINT DEGREE DISTRIBUTION  and store it into a database.
  At the end it will check the db for convergence using GIBBSIT

  Jaideep Ray, jairay@sandia.gov
  7/17/2012
*/

#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <cassert>
#include <unistd.h>

extern "C" {
#include "mix.h"
}
#include "converter.h"
#include "graphDB.h"
#include "convergence.h"

#define MAX(A, B) ( (A) > (B) ? (A) : (B))

void usage()
{

  std::cout << "Program invocation : " << std::endl << std::endl 
            << " ex02.exe -G <InputGrap> -N <num_realizations> -S <Output Stub>"  
	    << std::endl
	    << " where " << std::endl 
	    << " <InputGraphFile>, string, is the name of the elist file with the starting graph; no default. "
	    << std::endl 
	    << std::endl 
	    << " <num_realizations>, int. num_realization*G.E is the number of MC steps to go"
	    << " Default: 100000." 
	    << std::endl 
	    << std::endl 
	    << " <Output Stub>, string, is the Output_Stub.<instance_no> filename of output "
	    << " graph instances."
	    << " By default, ex02."
	    << std::endl
	    << std::endl
	    << "  Example : " << std::endl
	    << "     ./ex02.exe -G lesmis.elist -N 10000000 -S lesmis" << std::endl ;
}

/* Make a copy of a graph */
void copyGraph(struct graph *fromG, struct graph *toGraph);

int main(int argc, char *argv[])
{
 /* Required copyright banner */
  std::cout << std::endl
    << " ==================================================================== " << std::endl
    << " Copyright 2012 Sandia Corporation. Under the terms of Contract" << std::endl
    << " DE-AC04-94AL85000, there is a non-exclusive license for use of this" << std::endl
    << " work by or on behalf of the U.S. Government. Export of this program" << std::endl
    << " may require a license from the United States Government." << std::endl << std::endl
    << " ==================================================================== " << std::endl << std::endl;

  if (argc < 3) 
  {
    usage() ;
    exit(0) ;
  }

  int c ;
  int numRealizations = 100000 ;
  std::string inputGraphName("UNKNOWN") ;
  std::string outputFileStub("ex02") ;
  struct graph G;
  int **J, jN;

  while ((c = getopt (argc, argv, "G:N:S:ct:s:")) != -1)
  {
    switch(c)
    {
    case 'G':   // Initial graph to start with
      inputGraphName.assign( optarg );
      break ;
    case 'N':   // How many realizations to take
      numRealizations = atoi( optarg ) ;
      break ;
    case 'S' : // What is the stub of the output file?
      outputFileStub.assign( optarg ) ;
      break ;
    default :
      std::cerr << " Input error! Argument not understood; terminate! " << std::endl ;
      std::terminate();
    }
  }

  if ( inputGraphName == std::string("UNKNOWN") ) 
  {
    std::cout << " ex02.cpp: You forgot to specify an input file ! Exit " << std::endl ;
    std::terminate() ;
  }
  
  /* Print the input */
  std::cout << " ex02.cpp - Input graph name = " << inputGraphName << std::endl ;
  std::cout << " ex02.cpp - Output graph filename stub = " << outputFileStub << std::endl ;
  std::cout << " ex02.cpp - Number of graph realizations to try = " << numRealizations  
	    << std::endl ;

  /* Set the seed for the RNG */
  srand48(2098342342);

   /* Read edge-list and make a graph */
  edges2gr(const_cast<char *> (inputGraphName.c_str()), &G);

  /* Check the graph so formed */
  check_simple(&G);

 /* Construct a joint-degree-distribution from the graph read in */
  jN = make_JDD(&G, &J);

  /* Print out some stats */
  std::cout << " The graph has " << G.V << " nodes and " << G.E << " edges " << std::endl ;

  /* DB where we'll store the graphs */
  graphDB db ; 
  db.insertGraph(&G) ;

  /* This is a graph we'll use to store copies */
  struct graph gCopy ;
  gCopy.V = 0; gCopy.E = 0; gCopy.ind = 0; gCopy.ptr = 0;
  
  /* MC loop */
  for ( int istep = 0 ; istep  < (numRealizations * (G.E)); istep++ )\
  {
    /* Every G.E steps, plot a point */
    if ( istep%(100*(G.E)) == 0 ) std::cout << "." << std::flush;

    /* Keep a copy of the graph */
    copyGraph(&G, &gCopy) ;

    /* Mix the graph one edge */
    mix_jdd(&G, 1); 

    /* With probability 0.5, discard the new graph, and restore G */
    if ( drand48() < 0.5 ) copyGraph(&gCopy, &G) ;

    /* if we've gone G.E steps, save to dbase */
    if (istep%(G.E) == 0) db.insertGraph(&G) ;
  }

  /* Check for convergence at end of run */
  double r = 0.01 ;
  double s = 0.95 ;
  std::cout << "\n ex02 : Check if the Markov chain was long enough at end of run ... " << std::endl;
  std::cout << " ex02 : Will enforce a Markov chain long enough that edge-means are estimated within "
	    << "+/- " << r << " with " << s << " probability " << std::endl ;
  int Nprec, kthin, kmind, unique_edges ;
  bool longEnough = checkConvergence(db, G.V, G.E, r, s, Nprec, kthin, kmind, unique_edges) ;

  /* Write out the graphs */
  std::cout << " Chain length that would be sufficient = " << Nprec 
	    << "; length actually run = " << db.getNumGraphsInDB() << std::endl ;
  if (longEnough == false)
  {
    std::cout << " FAILURE !! CHAIN IS TOO SHORT FOR USE TO CHECK FOR INDEPENDENCE" << std::endl ;
    std::cout << " Set -N " << Nprec << " at least to generate a long enough Markov chain" << std::endl ;
    std::cout << " Exit (-1) " << std::endl ;
    exit(-1) ;
  }
  else
    std::cout << " SUCCESS !! CHAIN IS LONG ENOUGH FOR CHECKING FOR INDEPENDENCE!!" << std::endl ;

  /* Stats of what's in the DB */
  std::cout << " ex02.cpp : Number of graphs in the DB = " <<  db.getNumGraphsInDB() << std::endl ;

  if ( longEnough == true )
  {
    std::cout << " Kthin = " << kthin << ", Kmind = " << kmind << std::endl ;
    std::cout << " The saved chain in DB needs to be thinned by a factor of " 
	      << MAX(kthin, kmind) << " |G.E| " << std::endl ;

    int goodThinning =  MAX(kthin, kmind) ;
    graphDB ndb ;
    int ans = db. createThinnedVersionOfDatabase(ndb,  goodThinning) ;
    assert(ans == 0) ;
    ndb.writeGraphsToFiles(outputFileStub) ;
  }
  
  /* Clean up */
  free(G.ptr) ; free(G.ind) ;
  free(gCopy.ptr) ; free(gCopy.ind) ;
  for (int i = 0; i < jN; i++) free(J[i]) ;  free(J) ;
  return (0) ;
}

/*
  Make a copy of a graph. 

  INPUT: fromGraph, pointer to a strucg graph that is a true graph
  OUTPUT toGraph, pointer to a struct graph that is either 
                    - initialized to zero, in which case we'll allocate memory
                      for its edges and conections
                  pointer to a struct graph that already has memory allocated and
                  we'll simply to a copy
*/

void copyGraph(struct graph *fromGraph, struct graph *toGraph)
{

  /* Does the incoming toGraph needs memory to be allocated? */
  if ( (toGraph->ind == 0) && (toGraph->ptr == 0)) 
  {
    toGraph->V   = fromGraph->V ;
    toGraph->E   = fromGraph->E ;
    toGraph->ind = new int [ 2 * (toGraph->E) ] ;
    toGraph->ptr = new int [ (toGraph->V) + 1 ] ;
  }

  /* Some assertions; the 2 graphs should have the same number of edges and nodes */
  assert(toGraph->V == fromGraph->V) ;
  assert(toGraph->E == fromGraph->E) ;
  
  /* Do the copy */
  for (int i = 0; i < 2*(toGraph->E)  ; i++) toGraph->ind[i] = fromGraph->ind[i] ;
  for (int i = 0; i < (toGraph->V) + 1; i++) toGraph->ptr[i] = fromGraph->ptr[i] ;
}
