#ifndef CONVERGENCEHSEEN
#define CONVERGENCEHSEEN
/**
   \file convergence.h

   Header file for the functions used to judge the convergence (lack
   of auto-correlations) of a Markov Chain of graphs. The top level
   functions are implemented in convergence.cpp
*/


#include <vector>
#include <string>

#include "graph.h"   /* In EdgeFlip/ */
#include "graphDB.h"
#include "converter.h" /* For FileType */

/**
   This function takes in a graphDB, containing a time-series of
   numGraph graphs, stored in Ali's compressed format and extracts a
   time-history of a specified edge

   @param db          reference to a graph Database
   @param fromNode    An edge is specified as emanating from a fromNode ...
   @param toNode      ... and terminating at a toNode
   @param startGraph  We don't have to create a time-history of edge
                      [fromNode, toNode] starting from the first
                      graph. We can start at an arbitrary point in the
                      graph Markov chain. startGraph >= 0
   @param endGraph    End of the Markov chain of graphs to be
                      interrogated. startGraph <= endGraph <=
                      numGraphs.
   @param edgeTS      A vector containing 1s and 0s indicating whether
                      the edge [fromNode, toNode] exists or not. The
                      vector is allocated by the caller, but the
                      function fills it up.
   @return            0 if all OK, -ve number if something screws up
   
*/
int extractEdgeTimeHistory( graphDB &db, int fromNode, int toNode, int startGraph,
			    int endGraph, std::vector<int> &edgeTS ) ;

/**
  Given a sequence of graphs stored in a DB, check out each of the
  edges in the graph.  Identify, Kthin, the thinning factor that turns
  the binary time-series into a first-order Markov chain. Then, find
  the thinning factor, Kmind, that turns the time-series into
  independent draws from a binary distribution. Since (Kthin, Kmind)
  will be different for different each edge in the MC of graphs, take
  the worst-case scenario i.e., maximum values of each, over all edge time-series.

  @param db the graph database with graphs in it
  @param numNodes, nodes in the graph
  @param numEdges, number of edges in the graph
  @param r, I want a chain long enought that my edge means will be estimated within 
            +/- this tolerance.Usually, 0.01
  @param s, The chain will be long enough that the edge means will be
            estimated within +/- r with s confidence. Usually, s = 0.95	    
  @param maxNprec, the number of graphs we must have so that we can
          estimate a 1st-order Markov chain or independent draws correctly.
  @param maxKthin, the factor by which we have to thin the chain to get an uncorrelated chain
  @param maxKmind, the factor by which we have to thin to get an independent chain. 
  @param unique_edge_instance, in the chain of graphs in the DB, how many unique edges did we find?
  @return true or false i.e. do we have enough graphs in DB, so that
                        thinning by maxKmind will give us atleast 2 independent realizations?
*/
bool checkConvergence(graphDB &db, int numNodes, int numEdges, double r, double s, 
		      int &maxNprec, int &maxKthin, int &maxKmind, int &unique_edge_instance) ;
#endif
