#include "mix.h" 


/* ----------------------------------------------------------------------- */
/*
  make_JDD(struct graph *G,  int ***J)
  
  Given a graph G, construct the joint-degree-distribution and stick it into
  J. The matrix J, nrows x 3 cols, is allocated here, but someone else has to
  deallocate it.

  G : incoming graph
  J ; matrix allocated here but the caller has to deallocate it.
*/
int make_JDD(struct graph *G,  int ***J)
{
  int  t,u,v,k,N,i,j,*d, *dind, *dptr, *mark; 

  N = G->V; /* number of nodes */
#ifdef MIX_VERBOSE
  printf("From make_JDD() N = %d \n", N);
#endif

  /* 
     Allocate & zero out a vector that'll contain the starting edge_ids 
     of nodes
  */
  dptr = (int *) malloc(sizeof(int) * (N+1));
  memset ( dptr, 0, sizeof(int)*(N+1) );

  /* dind[i] will contain the node_id of the starting node of edge i */
  dind = (int *) malloc(sizeof(int) * N);

  /* d[i] will contain the # of edges that node i has */
  d    = (int *) malloc(sizeof(int) * N);

  /* Loop over nodes, figure out how many edges each has and update degree distribution */
  for ( i = 0; i < N; i++)
  {
    d[i] = G->ptr[i+1] - G->ptr[i];
    dptr[ d[i] ]++;
  }

  /* Change the degree distribution to starting edge_ids. Cumulative sum */
  /* JR_BUG 2011-11-28
     Given the malloc on line 28 this loop is guaranteed to go out of bounds.
     shouldn't the last value be dumped in dptr[N], since the array goes from
     dptr(0:N)? Ask Ali. Also, this loop will never touch dptr[0].
     for ( i = 1; i <= N; i++) dptr[i+1] += dptr[i];
  */
  for ( i = 1; i < N; i++) dptr[i+1] += dptr[i];
  dptr[0]=0;

  /* Loop over nodes and stick in the emanating nodes for the starting edge_ids */
  for( i = 0; i < N; i++)
    if (d[i] > 0) 
    {
     /*  printf ("%d %d %d \n",i, d[i],  dptr[d[i]-1]); */
      dind[ dptr[ d[i]-1 ] ] = i;
      dptr[ d[i] - 1 ]++;
    }

  /* Shift the pointers back */
  for ( i = N; i > 0; i-- ) dptr[i] = dptr[i-1];
  dptr[0] = 0;

  /*
    Find out the biggest degree in the graph. The joint-degree-distribution
    array will have to have that many rows
  */
    
  /* An array we'll use to accummulate # of nodes of degree i */
  mark = (int *) malloc( sizeof(int)*N );
  memset( mark, 0, sizeof(int)*N );

  t = 0; /* Var to hold the number of distinct degrees we have in a graph */
  
  /* Loop over all nodes */
  for ( i = 0; i < N; i++) 
  { 
    /* Loop over all edges associated with node 'i' */
    for( j = dptr[i]; j < dptr[i+1]; j++ ) 
    {
      v = dind[j]; /* Emanating node of edge j */

      /*
	Identify the terminating nodes of the edges emanating from v, 
	find their degrees and increment the corresponding element in mark[]
      */
      for (k = G->ptr[v]; k < G->ptr[v+1]; k++) 
	mark[ d[ G->ind[k] ] ]++;
    } 
  
    /* Loop over all edges */
    for(j = dptr[i]; j < dptr[i+1]; j++) 
    {
      v = dind[j]; /* terminating node if that edge */
      for ( k = G->ptr[v]; k < G->ptr[v+1]; k++ ) 
      {
	u = G->ind[k];
	if ( d[u] >= d[v] && mark[ d[u] ] )
	  t++;
	mark[ d[u] ]=0;
      }    
    }
  }

  /* Allocate JDD matrix */
  (*J) = (int **) malloc( sizeof(int *) * t );
  for ( i = 0; i < t; i++) (*J)[i] = (int *) malloc( sizeof(int)*3 );
  
  t = 0;
  for ( i = 0; i < N; i++) 
  {
    for( j = dptr[i]; j < dptr[i+1]; j++) 
    {
      v = dind[j];
      for ( k = G->ptr[v]; k < G->ptr[v+1]; k++ ) 
	mark[ d[ G->ind[k] ] ]++;
    } /* end of loop 'j' */
    
    for(j = dptr[i]; j < dptr[i+1]; j++) 
    {
      v = dind[j];
      for ( k = G->ptr[v]; k < G->ptr[v+1]; k++) 
      {
	u = G->ind[k];
	if ( d[u] >= d[v]  && mark[d[u]] )
        {	
	  /* 	  printf(" add %d %d %d \n", d[v],d[u],mark[d[u]]); */
	  (*J)[t][0] = d[v];
	  (*J)[t][1] = d[u];
	  if (d[u] == d[v])
	    (*J)[t][2] = mark[ d[u] ] / 2;
	  else 
	    (*J)[t][2] = mark[ d[u] ];
	  t++;
	} 
	mark[ d[u] ]=0;
      } /* End of if */   
    } /* end of loop over 'j' */
  }  /* End of loop 'i' over nodes */

  free(dptr);
  free(dind);
  free(d) ;
  free(mark);
  return(t);
}

/* ----------------------------------------------------------------------- */

void  Make_small_first_JDD(int **J,int jlen)
{
  int i,t; 
  for (i=0;i<jlen;i++)
    if (J[i][0] >J[i][1]) {
      t=J[i][0];
      J[i][0]=J[i][1];
      J[i][1]=t;
    } 
}

/* ----------------------------------------------------------------------- */

void SortJDD(int **JDD, int jN, int *t)
{
  int i;
  Make_small_first_JDD(JDD,jN);
  for (i=0;i<jN;i++)
    t[i]=i;
#ifdef MIX_MAC
  qsort_r(t, jN, sizeof(int),(void *)JDD, comp); 
#endif
}

int compareJDD(int **J, int jN, int *t, int **J2, int jN2, int *t2)
{
  int i, r;
  r=0;
 
  if (jN-jN2) {
    printf(" lengths of the tables do not match %d vs %d\n",jN,jN2);
    return(1);
  }
  for(i=0;i<jN;i++)
    if (J[t[i]][0] != J2[t2[i]][0] || J[t[i]][1] != J2[t2[i]][1]) {
      r=1;
      printf(" different edge (%d %d) vs (%d %d)\n", 
	     J[t[i]][0], J[t[i]][1], J2[t2[i]][0], J2[t2[i]][1]);
    }
    else if (J[t[i]][2] != J2[t2[i]][2]) { 
      r=1; 
      printf(" different counts for the edge (%d %d) %d vs %d\n",    
	     J[t[i]][0], J[t[i]][1], J[t[i]][2], J2[t2[i]][2]);
    }
  return (r);
}

/* ----------------------------------------------------------------------- */

int  compareG2JDD(struct graph *G, int **J, int jN)
{
  int jN2, **J2,  *t, *t2;;
  jN2=make_JDD(G,&J2);
  if (jN2 ==jN) {
    t= (int *) malloc(sizeof(int)*2*jN);
    t2=t+jN;
    SortJDD(J,jN,t);
    SortJDD(J2,jN2,t2);
    return(compareJDD(J,jN,t,J2,jN2,t2));
  }
  else {
    printf("Dimensions of the two JDDs do not match (%d %d)\n",jN,jN2); 
    return(2);
  }
}

/* ----------------------------------------------------------------------- */

void compare_JDD2(struct graph *G,  int **J, int jlen)
{
  int  t,u,v,k,N,i,j,*d, *dind, *dptr, *mark; 

  N=G->V;
  dptr=(int *) malloc(sizeof(int)*N);
  memset(dptr,0,sizeof(int)*N);

  dind=(int *) malloc(sizeof(int)*N);
  d= (int *) malloc(sizeof(int)*N);

  for (i=0;i<N; i++){
    d[i]=G->ptr[i+1]-G->ptr[i];
    dptr[d[i]]++;
  }
  for (i=1;i<N;i++)
    dptr[i+1]+=dptr[i];

  for(i=0;i<N;i++){
    dind[dptr[d[i]]]=i;
    dptr[d[i]]++;
  }
  for (i=N;i>0;i--)
    dptr[i]=dptr[i-1];
  
  mark=(int *)malloc(sizeof(int)*N);
  memset(mark,0,sizeof(int)*N);
  
  t=0;
  for (i=1;i<N;i++) 
    for(j=dptr[i];j<dptr[i+1];j++) {
      v=dind[j];
      for (k=G->ptr[v];k<G->ptr[v+1];k++) 
	mark[d[G->ind[k]]]++;
    } 
  
    for(j=dptr[i];j<dptr[i+1];j++) {
      v=dind[j];
      for (k=G->ptr[v];k<G->ptr[v+1];k++) {
	u=G->ind[k];
	if (d[u]>=d[v] && mark[d[u]]){	
	  J[t][0]=d[v];
	  J[t][1]=d[u];
	  J[t][2]=mark[d[v]];
	} 
	mark[d[u]]=0;
      }    
    }
    free(dptr);
    free(dind);
    free(mark);
}


/* ----------------------------------------------------------------------- */

/*
  ComputeDegrees() Function to compute degrees of N nodes of a graph,
                   given a JDD

  JDM  : Joint degree matrix, read-only
  N    : number of nodes in the graph
  jlen : number of rows in the JDM; num. columns is always 3
  d    : array to hold degrees of each node. Caller supplies array, of 
         size N
*/

int  ComputeDegrees(int **JDM, int N, int jlen, int *d)
{
  int i;

  /* Initialize the supplied degree array to 0*/
  memset(d,0,sizeof(int)*N);
  
  /* Make degree distribution. Loop over each degree
     entry in the JDM and update the number of nodes in
     in each bin.

     i = a valid degree; d[i] is the number of edges starting/terminating
     at a node with degree i and d[i] / i gives me the number of nodes
     with degree 'i'.
  */
  for(i = 0; i< jlen; i++){
    d[ JDM[i][0] ] += JDM[i][2];
    d[ JDM[i][1] ] += JDM[i][2];
  }
  
  /* fill up d[i]. d[i] will contain the number of nodes with degree i */
  for (i = 1; i < N; i++){
    if ((d[i]/i)*i != d[i]) {
      printf("From ComputeDegrees() Error! Degrees are not integral\n");
      return(-1);
    }
    else
      d[i] = d[i] / i;
  }
}

/* ----------------------------------------------------------------------- */

/*
  NumEdges() Given a JDD, compute total number of edges in the graph represented
  by the JDD.
  JDM    : 2D array containing the JDD
  jN     : number of rows in the joint degree matrix
*/

int NumEdges(int **JDM, int jN)
{
  int i, sum;
  for (i = sum = 0;i < jN; i++)
    sum+=JDM[i][2];
  return(sum);
}

/* ----------------------------------------------------------------------- */

/*
  FindNext(int *sptr,int *rptr, int i)

  A strategy to indicate the last/highest node_id of nodes
  with degree i. sptr[i] contains the starting node_id of all
  nodes with degree i. rptr[i] is the ending one.
*/

int  FindNext(int *sptr,int *rptr, int i)
{ 
  if (rptr[i]==sptr[i+1]-1)
    rptr[i] = sptr[i];
  else
    rptr[i]++;
 
    return(rptr[i]);
}

/* ----------------------------------------------------------------------- */

/*
  AddEdge(struct graph  *G, int u, int v) 

  Add an edge between nodes of node_id equal to u and v 
*/

void AddEdge(struct graph  *G, int u, int v) 
{
  /*  printf("add edge %d--%d\n", u,v); */
  G->ind[ G->ptr[u] ] = v;
  G->ptr[u]++;
  G->ind[ G->ptr[v] ] = u;
  G->ptr[v]++;
}


/* ----------------------------------------------------------------------- */

/*

  make_dd(struct  graph *G, int *d, int *ind, int N, int r)
  
  Make a degree distribution
  G     the graph we're taking about
  d[]   an array, N long. d[j] contains the # of edges impinging on node j
  ind[] an array, N long, containing node_ids. ind[j] is the node_id of node j
  r     an int stating the # of nodes that will have one more edge than others.
*/

int make_dd(struct  graph *G, int *d, int *ind, int N, int r)
{
  int i; 
  
  /* Loop over all nodes */
  for ( i = 0; i < N; i++ )
  {
    /* For the node 'i' picked, loop over all the d[i] edges emanating from it */
    for ( ; d[i] > 0 ; d[i]-- )
    {
      /* 
	 if I am one of the early nodes, connect me to preferentially to the nodes
	 that have one extra edge, else start connecting me preferentially with 
	 the highest node_ids
      */
      if ( r <= i) r = N-1;
      AddEdge(G, ind[i], ind[r]);

      /* 
	 Decrement the degree of the node I got connected to; my own degree will
	 be decremented in the loop
      */
      d[r]--; 

      /* Next time around, connect the node to the next lower node_id */
      r--;
    } /* End loop over edges of node 'i' */
  }  /* End loop over all N nodes */
}


/* ----------------------------------------------------------------------- */

/*
  make_bdd(struct  graph *G, int *sptr, int *rptr, int u, int v, int m)

  Add m edges in a graph between nodes of degree u and nodes of degree v
  
  G       The graph
  sptr[]  Array, where sptr[i] contains the starting node_id of nodes of degree i
  rptr[]  Array, where rptr[i] contains the endinf node_id of nodes with degree i
  u, v    We want to add edges between nodes of degree u and v
  m       We want to add this many edges
*/

int make_bdd(struct  graph *G, int *sptr, int *rptr, int u, int v, int m)
{
  int t, i, j, x, y, f, r, cnt; 

  /*  printf(" %d %d %d \n", u,v,m); */

  /* How many nodes are emanating edges? */
  cnt = sptr[u+1] - sptr[u];
  f   = m / cnt;    /* How many edges emanating per node? */
  r   = m - cnt*f;  /* Not an integral multiple of cnt - so what's the remainder ?*/

  /* Loop over the nodes emanating (f+1) edges */
  for ( i = 0; i < r; i++) 
  {
    /* Find node_id of a node with degree u */
    x = FindNext(sptr, rptr, u);

    /* Loop over the (f+1) edges emanating from the picked node */
    for ( j = 0; j <= f; j++) 
    {
      /* Find the next node with degree v, to terminate the edge */
      y = FindNext(sptr, rptr, v);

      AddEdge(G, x, y); /* Connect with edge */
    } /* End of loop over f edges */
  } /* End of loop over cnt nodes */

  /* What's the highest node_id connected so far? */
  t = rptr[u];  

  /* If we do have graphs with degree 2 or more */
  if ( f > 0 )
  {
    /* Loop over the nodes with f edges emanating from them */
    for ( ; i < cnt; i++) 
    {
      x = FindNext(sptr, rptr, u); /* get the next node */

      for ( j = 0; j < f; j++)     /* loop over the f edges emanating from it */
      {

	y = FindNext(sptr, rptr, v); /* get the node to terminate the edge at */

	AddEdge(G, x, y); /* Connect */

      } /* End of loop over f edges */
    }  /* End of loop over cnt nodes */
  } /* End of if */

  /* Highest node_id to date */
  rptr[u] = t;
}

/* ----------------------------------------------------------------------- */

/*
  GenEdges (struct graph *G, int **JDM, int jlen, int *d, int N, int M)

  Given a Joint Degree Matrix JDM and a degree distribution vector,
  create a graph

  JDM:  Represents the (jlen x 3) Joint Degree Matrix
        JDM[i][0] and JDM[i][1] correspond to the  degrees of the ith edge's end vertices
        JDM[i][2] is the multiplicity of sucgh an edge    
    
  jlen:  is the  #rows in the JDM matrix 
  d   : Degree distribution vector  d[i] is the number of vertces of degree i.  
  N   : number of vertices.
  M   : number of edges 
*/

int  GenEdges (struct graph *G, int **JDM, int jlen, int *d, int N, int M)
{
  int i, j, cnt, f, r, last, *sptr, *rptr, *tmp, *tind;

  /* temporaries */
  tmp  = (int *) malloc(sizeof(int) * N);
  tind = (int *) malloc(sizeof(int) * N);

  /* 
     A static pointer that point to vertex list  based on degrees. 
     We are going to number the vertices based on their degrees. d[1] contains
     the number of vertices with degree 1, d[2], number of vertices with degree 2 
     and so on. So node_ids = [ 0 : (d[1]-1) ] are node_ids of nodes with degree 1,
     [ d[1] : (d[1] + d[2] -1) ] node_ids of degree 2 and so on. So
     sptr[0] = 0 ;     #  Starting node_id of degree 1 nodes
     sptr[1] = d[1] ;  #  Starting node_id of degree 2 nodes
     sptr[2] = d[1] + d[2] = sptr[1] + d[2] ; # Starting node_id of degree 3 nodesl
  */
  sptr = (int *) malloc(sizeof(int)*(N+1));

  /* 
     A running pointer that  helps  using vertices in around robin way.
     rptr[i] will contain the last/highest node_id of the range of node_ids with degree i.
     Initialize rptr[] as if there were no self-loops.
 */
  rptr= (int *) malloc(sizeof(int)*(N+1));

  for ( i = 0, rptr[i] = 0, sptr[0] = 0; i < N; i++ )
  {
    sptr[i+1] = sptr[i] + d[i];
    rptr[i]   = sptr[i+1] - 1;
  }
#ifdef MIX_VERBOSE
  printf(" Starting the main loop \n");
#endif

  /*  
      for (j=0;j<N; j++)
      printf("%d ", rptr[j]);
      printf("\n"); 
  */
  for(i = 0; i < jlen; i++)
    if ( JDM[i][0] == JDM[i][1] ) 
    {
#ifdef MIX_VERBOSE  
      printf(" if \n");
#endif
      /* How many nodes have degree JDM[i][0] ? */
      cnt = sptr[JDM[i][0]+1] - sptr[JDM[i][0]] ;

      /*
	How many edges exist between nodes that have degree JDM[i][0]? 2*JDM[i][2].
	But if there are self-loops, 2*JDM[i][2] will not be an integral multiple of 
	cnt. So figure out how to distribute 2*JDM[i][2] edges over cnt nodes. 

	One way: divide 2*JDM[i][2] edges evenly over cnt nodes. What's left is allocated
	allocated to some of the cnt nodes again, leading to some nodes having one more
	edge than others. Note, that in doing so, some nodes which should have had JDM[i][0]
	degree, don't have it. Will have to fix it later.
      */

      /* All nodes with degree i should has this many f edges. f should be equal to i */
      f   = 2*JDM[i][2] / cnt;  

      r   = 2*JDM[i][2] - cnt*f; /* So many edges are left over */

      /* 
	 Loop over r nodes and allocate the an extra edge. tind[j] will contain the 
	 node_id of the terminating node of the extra edges j = [0 ...r-1]
      */
      for( j = 0; j < r; j++) 
      {
	tmp[j]  = f + 1;
        tind[j] = FindNext(sptr, rptr, JDM[i][0]);
      }
      last = rptr[JDM[i][0]];

      /* Now allocate f edges to the remaining nodes */
      for( ; j < cnt; j++) 
      {
	tmp[j] = f;
	if ( last == sptr[JDM[i][0]+1] - 1 )
	  last = sptr[JDM[i][0]];
	else
	  last++;
	tind[j] = last;
      }

      /* 
	 OK, tmp[0:cnt] contain the number of edges emanating from cnt nodes
	 of degree JDM[i][0]. tind[j] contains the node_ids of the emanating nodes.
	 Now, make a degree distribution out of it i.e fill up G->ind[].
       */
      make_dd(G, tmp, tind, cnt, r); 		           
    }  
    else 
    {
#ifdef MIX_VERBOSE
      printf("else  \n");
#endif
      make_bdd(G, sptr, rptr, JDM[i][0], JDM[i][1], JDM[i][2]);
      /*
	cnt=sptr[JDM[i][0]+1]-sptr[JDM[i][0]];
	f=JDM[i][2]/cnt;
	r=JDM[i][2]-cnt*f;
	for(j=0;j<r;j++)
	tmp[j]=f+1;
	for(;j<cnt;j++)
	tmp[j]=f;
	cnt2=sptr[JDM[i][1]+1]-sptr[JDM[i][1]];
	f2=JDM[i][2]/cnt2;
	r2=JDM[i][2]-cnt*f2;
	for(;j<cnt+r2;j++)
	tmp[j]=f+1;
	for(;j<cnt+cnt2;j++)
	tmp[j]=f;
	make_bdd(G,tmp,cnt,cnt2,r, r2,sptr[JDM[i][0]],sptr[JDM[i][1]]);
	
	for (j=0; j<JDM[i][2];j++) {
	x=FindNext(sptr,rptr,JDM[i][0]);
	y=FindNext(sptr,rptr,JDM[i][1]);
	printf(" add %d %d \n", x,y);
	AddEdge(G,x,y);
	}
      */
      /* 
	 for (j=0;j<N; j++)
	 printf("%d ", rptr[j]);
	 printf("\n");
      */
    } /* End of if statement */

  /* Clean up */
  free (sptr) ;
  free (rptr) ;
  free (tmp)  ;
  free (tind) ;
}

/* ----------------------------------------------------------------------- */

int BinSearch(int *A,int N, int x)
{ 
  int l,m,u;
  l=0;
  u=N;
   while (l<u-1) {
     m= (u+l)/2;
    if (x<A[m])
      u=m;
    else
      l=m;
  }
  return(l);  
}

/* ----------------------------------------------------------------------- */

/*
  gen_jdd() 

  Function to generate a graph, when we are only given 
  - a joint-degree-distribution in a matrix J
  - jN, the number of rows in the matrix J. It has 3 columns.
  - N, the number of nodes in the graph.
  G will contain the final graph
*/

void gen_jdd(struct graph *G, int **J, int jN, int N)
{
  int i, j, k, E, *d;
  
  /* Compute the number of edges in the graph represented by the JDD in JDM */
  E = NumEdges(J, jN);

#ifdef MIX_VERBOSE
  printf("From gen_jdd() Num Edges = %d  Num nodes = %d \n", E, N);
#endif

  /* Allocate array to hold degree distribution. Since we do
     not know what the max degree is, make it as large as the number of
     nodes.

     Then, compute degree distribution.
  */
  d = (int *) malloc(sizeof(int) * (N+1) );  /* degree distribution */

  ComputeDegrees(J, N,jN, d); /* Fill up array d */

#ifdef MIX_VERBOSE
  printf("From gen_jdd() degrees distribution \n");
  for (i=0;i<N;i++) /* Print out degree of each node */
    printf("  Degree =  %d, num nodes = %d\n", i, d[i]);
  printf("\n");
#endif

  /* 
     Fill up a struct that contains a graph that agrees with the 
     JDD and the degree dsitribution in d. G->ptr[node_id] will contain
     an edge_id. The range [ G->ptr[node_id-1] : G->ptr[node_id] ] 
     contains the edge_ids of edges incident on node 'node_id'

     G->ptr[0] = 0 since all edge_ids start at 0
     node_ids start at 1 

  */
  alloc_gr(G,N,E); /* Allocate the graph structure */
  for(G->ptr[0] = 0, k = 0, i = 1; k < N; i++) /* loop over all possible degrees 'i'; stop when we have n nodes*/
    for(j = 0; j < d[i]; j++)  /* loop over all d[i] nodes with degree 'i' */
    {
      G->ptr[k+1] =G->ptr[k] + i;
      k++;
    }

#ifdef MIX_VERBOSE
  /* Print cumulative edge_ids */
  for (i = 0; i <= N; i++)
     printf("From gen_jdd() Node id = %d, cumulative edge_id = %d \n", i, G->ptr[i]);
 
  /*
    Generate edges i.e.even though we have filled up G->ptr[], i.e. edge_ids and the node_ids
     from where they start, we haven't filled up G->ind[] to indicate when the edges end
  */
  printf("From gen_jdd() generating edges \n");  
#endif
  GenEdges (G, J, jN, d, N, E);

  /* Shift the prt back one element */
  for (i = N-1; i>0; i--)
    G->ptr[i] = G->ptr[i-1];

  /*  
      for (i=0;i<=N;i++)
      printf("%d ", G->ptr[i]);
      printf("\n"); 
  */ 
  G->ptr[0] = 0;
  
}

/* ----------------------------------------------------------------------- */

int count(struct graph *G, int x,int y) {
  int i, c,j,u;
  c=0;
  for (i=0;i<G->V;i++)
    if (G->ptr[i+1]-G->ptr[i] == x)
      for (j=G->ptr[i];j<G->ptr[i+1];j++) {
	u=G->ind[j];
	if (G->ptr[u+1]-G->ptr[u] ==y)
	  c++;
      }
  printf(" #(%d,%d)=%d \n",x,y,c);
  return(c);
}

/* ----------------------------------------------------------------------- */

/*
  read_JDM_vector() 

  Function to read in a file that specifies a joint degree distribution.
  
  (*J)    : Pointer to a 2D array, to serve as the holder for the joint degree distribution.
            This array is allocated in this function, but has to be deallocated elsewhere.

  (*jlen) : number of rows of joint degree distribution array. number of columns is always 3.
            The number is read from file and returned to the caller.

  (*N)    : Number of nodes in the graph. Also returned to the caller.

  fname   : String; filename where the joint degree distribution (JDD) can be found.
            Supplied by the caller.
*/
void read_JDM_vector (int ***J, int *jlen, int *N, char *fname)
{
  
  FILE *fp; 
  int i ;

#ifdef MIX_ORIG
  int *T; 
#endif

  /* Open the file and read in the joint degree distribution */
  printf("From read_JDM_vector(): reading vector from file %s\n", fname);
  fp = fopen(fname,"r");
  fscanf(fp,"%d %d", N, jlen); /* N = number of nodes in graph, jlen = number of lines in file */
  printf("From read_JDM_vector() Num nodes in graph = %d\n", (*N)) ;

#ifdef MIX_ORIG
  T=(int *) malloc(sizeof(int *)*3*(*jlen));
  (*J)=(int **)malloc(sizeof(int *)* (*jlen)); 
  (*J)=(int **) T;
  for(i=0;i< *jlen;i++)
    (*J)[i]=(int *) malloc(sizeof(int)*3); 
  printf("N=%d \n", *jlen);
#endif

  /* Allocate the array to hold the joint degree distribution */
  (*J) = (int **) malloc( sizeof(int *) * (*jlen) ) ; /* number of rows in matrix */
  for ( i = 0; i < (*jlen); i++)
    (*J)[i] = (int *) malloc( sizeof(int) * 3 ) ; /* Each row has 3 columns */

  /* Read in file */
  for(i=0;i<*jlen;i++)
    fscanf(fp,"%d %d %d",&((*J)[i][0]), &((*J)[i][1]),&((*J)[i][2]));

  /* Clean up and close  file */
  fclose(fp);

  printf("From read_JDM_vector() Finished reading vector from file %s\n\n", fname);
}

/* ----------------------------------------------------------------------- */
/*

  edges2gr (char *fname, struct graph *G)

  Given a list of edges i.e. M pairs like [emanating_node_id, terminating_node_id],
  create a graph in the format that we have.

  fname : file with the edge list. The top 2 number are numNodes and numEdges

  G     : The graph struct that will be contructed with the edge list
*/
void edges2gr (char *fname, struct graph *G)
{
  int i, *to, *from, N,M;
  float x,y;
  FILE *fp;
  
  /* Read the number of nodes (N) and edges (M) */
  printf("edges2gr() Reading edge-list from file %s\n", fname) ;
  fp = fopen(fname,"r");
  fscanf(fp, "%d %d",&N,&M);
  printf("edges2gr() Num nodes = %d, num edges = %d\n", N, M) ;

  /* 
     Allocate the arrays to store the terminating node ids and 
     enamanting node ids and fill from file
  */
  to   = (int *) malloc(sizeof(int) * 2*M);
  from = (int *) malloc(sizeof(int) * 2*M);
  for (i = 0; i < 2*M; i++) 
  {
    fscanf(fp, "%e %e",&x, &y);
    to[i]   = (int ) (x-1); /* Node_ids are read in as double precision numbers */
    from[i] = (int ) (y-1);
    if ( to[i] == from[i] ) printf(" self loop\n");
    /* printf("%d %f %d %f %d\n", i, x,to[i],y,from[i]); */
  }
  fclose(fp);

  /* Allocate and fill up the graph */
  alloc_gr(G,N,M);
  memset (G->ptr, 0, sizeof(int)*(N+1)); /* zero out the ptr array */
  
  /* Fill up ptr[node_id] to contain the # of edges terminating at node node_id */
  for(i = 0;i < 2*M; i++)    G->ptr[ to[i] ]++;

  /* 
     Replace that number with the starting edge_id of the edges terminating at node node_id 
     Number the edges, starting with edge_id = 0 for the first node's first edge.
  */
  for(i = 0; i < N; i++) G->ptr[ i+1 ] += G->ptr[i];

  /* Loop over all edges and fill in the emanating node_ids in ind[] */
  for(i = 0; i < 2*M; i++) 
  {
    G->ptr[ to[i] ]--;
    G->ind[ G->ptr[ to[i]] ] = from[i];
  }
#ifdef MIX_VERBOSE
  printf("edges2gr() G->ptr[0] = %d\n",G->ptr[0]); 
#endif

  /*  write_graph(G); */

  /* Clean up */
  free(to); 
  free(from); 
}
/* ----------------------------------------------------------------------- */


