c     Copyright (c) 2006, Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000
c     with Sandia Corporation, the U.S. Governement retains certain rights in this software.
c
c     All rights reserved.
c     
c     Redistribution and use in source and binary forms, with or without 
c     modification, are permitted provided that the following conditions are met:
c     
c     * Redistributions of source code must retain the above copyright notice, 
c     this list of conditions and the following disclaimer.
c     * Redistributions in binary form must reproduce the above copyright notice, 
c     this list of conditions and the following disclaimer in the documentation 
c     and/or other materials provided with the distribution.
c     * Neither the name of the Sandia National Laboratories nor the names of 
c     its contributors may be used to endorse or promote products derived 
c     from this software without specific prior written permission.
c     
c     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
c     ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
c     WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
c     IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
c     INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
c     BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
c     DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
c     LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
c     OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
c     OF THE POSSIBILITY OF SUCH DAMAGE.
c     

c
c     Version 0.1, June 2006.
c     Authors : Christopher Kennedy, M. H. Carpenter and Jaideep Ray.
c     Maintainer: Jaideep Ray, Advanced Software R. & D., Sandia National Laboratories, Livermore, CA, USA.
c                 jairay@ca.sandia.gov
c
c

      subroutine expfil_coeff(iord,disi,disb,ierror,error)
!
!  ======================================================================
!! EXPFIL_COEFF defines the dissipation filter coefficients.
!! 
!  Filters basically work like this: Feed in the vector that 
!  you would like to have filtered, say U, and do this to it -
!
!  U_filt = (I - alpha_D*D)*U_unfilt
!
!  where alpha_D is just for scaling and D is the dissipation matrix.
!  D is based on finite-difference derivatives like d^{2n}/dx^{2n}.
!  A 10th-order filter uses a D based on d^{10}/dx^{10} stencils.
!
!  alpha    - Scaling constant for Dissipation matrix coefficients
!  disb     - Dissipation matrix coefficients for the boundary grid points 
!  disi     - Dissipation matrix coefficients for the interior grid points 
!  iord     - Filter order  
!  iordd2   - Filter order divided by 2 
!  iordd2p1 - Filter order divided by 2 + 1
! 
!  HIGH-ORDER SHAPIRO?(Lars ERIKSSON?) FILTERS
!
!  NOTE: Pascal's Triangle/Binomial expansion
!                                    1
!                                 1     1
!  disi(2nd)                   1     2     1
!                           1     3     3     1
!  disi(4th)             1     4     6     4     1
!                     1     5    10    10     5     1
!  disi(6th)       1     6    15    20    15     6     1
!               1     7    21    35    35     21    7     1
!  disi(8th) 1     8    28    56    70    56    28     8     1
!
!
!                        EXAMPLE
!    -1   2  -1   0   0   0   0 (2nd)
!     2  -5   4  -1   0   0   0 (2nd) Note symmetry about the diagonal
!    -1   4  -6   4  -1   0   0 (4th)  and sum of coeff's =0.
!     0  -1   4  -6   4  -1   0
!     0   0  -1   4  -6   4  -1
!     0   0   0  -1   4  -5   2
!     0   0   0   0  -1   2  -1
!
! Dissipation matrix will be symmetric. Symmetric matrices have real
! eigenvalues. If the scheme is stable then all eigenvalues occur
! in quadrants 2 & 3 of the complex plane (LHP). Hence (Ui.Dij.Uj)=
! U1*U1*lambda1 +U2*U2*lambda2 +U3*U3*lambda3 + .... < 0 since all
! lambda's are less than zero and Ui*Ui is always > 0.
!
! For references, see NASA TP-3484, December 1997 or Applied
! Numer. Math., Vol 14, 4 (1994) 397-433. 
!
! ERROR MESSAGES
!   ierror =  0: No errors
!   ierror <  0: Errors
!
!   ierror = -10 => The filter must be of even order
!  ======================================================================
!
      implicit none

      real*8    alpha
!     JR_CHANGE Making changes for 14th - 20th order filters. December 17th, 2005.
!      real*8    disb(6,12)
!      real*8    disi(7)
!     JR_CHANGE, 04/15/06. Moving from 20th-order to 50th order stencils.
!      real*8    disb(10,20)
!      real*8    disi(11)
      real*8    disb(25,50)
      real*8    disi(26)
!
      integer i
      integer ierror
      integer ii
      integer iord
      integer iordd2
      integer iordd2p1
      integer j
      integer jj
!
      character*60 error
!
      ierror = 0
!
      if( mod(iord,2) .ne. 0) then
        ierror = -10
        error = 'Expfil_coeff: Filter order is set to an odd number'
        return
      endif
!
      if( iord .le. 0) then
        ierror = -20
        error = 'Expfil_coeff: Filter order is set wrong'
        return
      endif

!
!  Form coefficient alpha
!
      iordd2 = iord / 2
      iordd2p1 = iordd2 + 1
      alpha = ( - 1.0)**iordd2 / real ( 2**iord )
!
!  Initialize dissipation matrix; Interior vector and boundary matrix
!
!     JR_CHANGE dissib is now(10,20), not (6,12)
!     JR_CHANGE dissib is now(25, 50), not (10,20) and disi is now disi(26), not
!     disi(11). 04/15/06
!      do i = 1, 6
!      do i = 1, 10
      do i = 1, 25
        disi(i) = 0.0
!     JR_CHANGE, 12/17/05  
!        do j = 1, 12
!     JR_CHANGE 05/15/06
!        do j = 1, 20
        do j = 1, 50
          disb(i,j) = 0.0
        end do
      end do
!     JR_CHANGE, 12/17/05
!      disi(7) = 0.0
!     JR_CHANGE, 04/15/06
!      disi(11) = 0.0
      disi(26) = 0.0
!
! Create the upper left corner of the dissipation matrix for orders 2 <-> 12)
!
      if ( iord .eq. 2 ) then
!
! 2nd order inner with 1st order boundaries
!
        disi(2) = -1.0
        disi(1) = +2.0
!
        disb(1,1) = +1.0
        disb(1,2) = -1.0
!
      else if ( iord .eq. 4 ) then
!
! 4th order inner with 2nd order boundaries
!
        disi(3) = -1.0
        disi(2) = +4.0
        disi(1) = -6.0
!
        disb(1,1) = -1.0
        disb(1,2) = +2.0
        disb(1,3) = -1.0
!
        disb(2,1) = +2.0
        disb(2,2) = -5.0
        disb(2,3) = +4.0
        disb(2,4) = -1.0
!
      else if ( iord .eq. 6 ) then
!
! 6th order inner with 3rd order boundaries
!
        disi(4) = -1.0
        disi(3) = +6.0
        disi(2) = -15.0
        disi(1) = +20.0
!
        disb(1,1) =  1.0
        disb(1,2) = -3.0
        disb(1,3) =  3.0
        disb(1,4) = -1.0
!
        disb(2,1) = -3.0
        disb(2,2) = +10.0
        disb(2,3) = -12.0
        disb(2,4) = +6.0
        disb(2,5) = -1.0
!
        disb(3,1) = +3.0
        disb(3,2) = -12.0
        disb(3,3) = +19.0
        disb(3,4) = -15.0
        disb(3,5) = +6.0
        disb(3,6) = -1.0
!
      else if ( iord .eq. 8 ) then
!
! 8th order inner with 4th order boundaries
!
        disi(5) = -1.0
        disi(4) = +8.0
        disi(3) = -28.0
        disi(2) = +56.0
        disi(1) = -70.0
!
        disb(1,1) = -1.0
        disb(1,2) = +4.0
        disb(1,3) = -6.0
        disb(1,4) = +4.0
        disb(1,5) = -1.0
!
        disb(2,1) = +4.0
        disb(2,2) = -17.0
        disb(2,3) = +28.0
        disb(2,4) = -22.0
        disb(2,5) = +8.0
        disb(2,6) = -1.0
!
        disb(3,1) = -6.0
        disb(3,2) = +28.0
        disb(3,3) = -53.0
        disb(3,4) = +52.0
        disb(3,5) = -28.0
        disb(3,6) = +8.0
        disb(3,7) = -1.0
!
        disb(4,1) = +4.0
        disb(4,2) = -22.0
        disb(4,3) = +52.0
        disb(4,4) = -69.0
        disb(4,5) = +56.0
        disb(4,6) = -28.0
        disb(4,7) = +8.0
        disb(4,8) = -1.0
!
      else if ( iord .eq. 10 ) then
!
! 10th order inner with 5th order boundaries
!
        disi(6) = -1.0
        disi(5) = +10.0
        disi(4) = -45.0
        disi(3) = +120.0
        disi(2) = -210.0
        disi(1) = +252.0
!
        disb(1,1) = +1.0
        disb(1,2) = -5.0
        disb(1,3) = +10.0
        disb(1,4) = -10.0
        disb(1,5) = +5.0
        disb(1,6) = -1.0
!
        disb(2,1) = -5.0
        disb(2,2) = +26.0
        disb(2,3) = -55.0
        disb(2,4) = +60.0
        disb(2,5) = -35.0
        disb(2,6) = +10.0
        disb(2,7) = -1.0
!
        disb(3,1) = +10.0
        disb(3,2) = -55.0
        disb(3,3) = +126.0
        disb(3,4) = -155.0
        disb(3,5) = +110.0
        disb(3,6) = -45.0
        disb(3,7) = +10.0
        disb(3,8) = -1.0
!
        disb(4,1) = -10.0
        disb(4,2) = +60.0
        disb(4,3) = -155.0
        disb(4,4) = +226.0
        disb(4,5) = -205.0
        disb(4,6) = +120.0
        disb(4,7) = -45.0
        disb(4,8) = +10.0
        disb(4,9) = -1.0
!
        disb(5,1) = +5.0
        disb(5,2) = -35.0
        disb(5,3) = +110.0
        disb(5,4) = -205.0
        disb(5,5) = +251.0
        disb(5,6) = -210.0
        disb(5,7) = +120.0
        disb(5,8) = -45.0
        disb(5,9) = +10.0
        disb(5,10) = -1.0
!
      else if ( iord .eq. 12 ) then
!
! 12th order inner with 6th order boundaries (may be bug here)
!
        disi(7) = -1.0
        disi(6) = +12.0
        disi(5) = -66.0
        disi(4) = +220.0
        disi(3) = -495.0
        disi(2) = +792.0
        disi(1) = -924.0
!
        disb(1,1) = -1.0
        disb(1,2) = +6.0
        disb(1,3) = -15.0
        disb(1,4) = +20.0
        disb(1,5) = -15.0
        disb(1,6) = +6.0
        disb(1,7) = -1.0
!
        disb(2,1) = +6.0
        disb(2,2) = -37.0
        disb(2,3) = +96.0
        disb(2,4) = -135.0
        disb(2,5) = +110.0
        disb(2,6) = -51.0
        disb(2,7) = +12.0
        disb(2,8) = -1.0
!
        disb(3,1) = -15.0
        disb(3,2) = +96.0
        disb(3,3) = -262.0
        disb(3,4) = +396.0
        disb(3,5) = -360.0
        disb(3,6) = +200.0
        disb(3,7) = -66.0
        disb(3,8) = +12.0
        disb(3,9) = -1.0
!
        disb(4,1) = +20.0
        disb(4,2) = -135.0
        disb(4,3) = +396.0
        disb(4,4) = -662.0
        disb(4,5) = +696.0
        disb(4,6) = -480.0
        disb(4,7) = +220.0
        disb(4,8) = -66.0
        disb(4,9) = +12.0
        disb(4,10) = -1.0
!
        disb(5,1) = -15.0
        disb(5,2) = +110.0
        disb(5,3) = -360.0
        disb(5,4) = +696.0
        disb(5,5) = -887.0
        disb(5,6) = +786.0
        disb(5,7) = -495.0
        disb(5,8) = +220.0
        disb(5,9) = -66.0
        disb(5,10) = +12.0
        disb(5,11) = -1.0
!
        disb(6,1) = +6.0
        disb(6,2) = -51.0
        disb(6,3) = +200.0
        disb(6,4) = -480.0
        disb(6,5) = +786.0
        disb(6,6) = -923.0
        disb(6,7) = +792.0
        disb(6,8) = -495.0
        disb(6,9) = +220.0
        disb(6,10) = -66.0
        disb(6,11) = +12.0
        disb(6,12) = -1.0
!      
!     JR_CHANGE Adding these; December 17, 2005
      else if ( iord .eq. 14 ) then
c
c 14th order inner with 7th order boundaries
c ------------------------------------------
c
         disi(1)= + 3432.0d0
         disi(2)= - 3003.0d0
         disi(3)= + 2002.0d0
         disi(4)= - 1001.0d0
         disi(5)= + 364.0d0
         disi(6)= - 91.0d0
         disi(7)= + 14.0d0
         disi(8)= - 1.0d0
c     
         disb(1,1) =+1.0d0
         disb(1,2) =-7.0d0
         disb(1,3) =+21.0d0
         disb(1,4) =-35.0d0
         disb(1,5) =+35.0d0
         disb(1,6) =-21.0d0
         disb(1,7) =+7.0d0
         disb(1,8) =-1.0d0
c     
         disb(2,1) =-7.0d0
         disb(2,2) =+50.0d0
         disb(2,3) =-154.0d0
         disb(2,4) =+266.0d0
         disb(2,5) =-280.0d0
         disb(2,6) =+182.0d0
         disb(2,7) =-70.0d0
         disb(2,8) =+14.0d0
         disb(2,9) =-1.0d0
c     
         disb(3,1) =+21.0d0
         disb(3,2) =-154.0d0
         disb(3,3) =+491.0d0
         disb(3,4) =-889.0d0
         disb(3,5) =+1001.0d0
         disb(3,6) =-721.0d0
         disb(3,7) =+329.0d0
         disb(3,8) =-91.0d0
         disb(3,9) =+14.0d0
         disb(3,10)=-1.0d0
c     
         disb(4,1) =-35.0d0
         disb(4,2) =+266.0d0
         disb(4,3) =-889.0d0
         disb(4,4) =+1716.0d0
         disb(4,5) =-2114.0d0
         disb(4,6) =+1736.0d0
         disb(4,7) =-966.0d0
         disb(4,8) =+364.0d0
         disb(4,9) =-91.0d0
         disb(4,10)=+14.0d0
         disb(4,11)=-1.0d0
c     
         disb(5,1) =+35.0d0
         disb(5,2) =-280.0d0
         disb(5,3) =+1001.0d0
         disb(5,4) =-2114.0d0
         disb(5,5) =+2941.0d0
         disb(5,6) =-2849.0d0
         disb(5,7) =+1981.0d0
         disb(5,8) =-1001.0d0
         disb(5,9) =+364.0d0
         disb(5,10)=-91.0d0
         disb(5,11)=+14.0d0
         disb(5,12)=-1.0d0
c     
         disb(6,1) =-21.0d0
         disb(6,2) =+182.0d0
         disb(6,3) =-721.0d0
         disb(6,4) =+1736.0d0
         disb(6,5) =-2849.0d0
         disb(6,6) =+3382.0d0
         disb(6,7) =-2996.0d0
         disb(6,8) =+2002.0d0
         disb(6,9) =-1001.0d0
         disb(6,10)=+364.0d0
         disb(6,11)=-91.0d0
         disb(6,12)=+14.0d0
         disb(6,13)=-1.0d0
c     
         disb(7,1) =+7.0d0
         disb(7,2) =-70.0d0
         disb(7,3) =+329.0d0
         disb(7,4) =-966.0d0
         disb(7,5) =+1981.0d0
         disb(7,6) =-2996.0d0
         disb(7,7) =+3431.0d0
         disb(7,8) =-3003.0d0
         disb(7,9) =+2002.0d0
         disb(7,10)=-1001.0d0
         disb(7,11)=+364.0d0
         disb(7,12)=-91.0d0
         disb(7,13)=+14.0d0
         disb(7,14)=-1.0d0
c     
      else if ( iord .eq. 16 ) then
c
c 16th order inner with 8th order boundaries
c ------------------------------------------
c
         disi(1)= - 12870.0d0
         disi(2)= + 11440.0d0
         disi(3)= - 8008.0d0
         disi(4)= + 4368.0d0
         disi(5)= - 1820.0d0
         disi(6)= + 560.0d0
         disi(7)= - 120.0d0
         disi(8)= + 16.0d0
         disi(9)= - 1.0d0
c
         disb(1,1) =-1.0d0
         disb(1,2) =+8.0d0
         disb(1,3) =-28.0d0
         disb(1,4) =+56.0d0
         disb(1,5) =-70.0d0
         disb(1,6) =+56.0d0
         disb(1,7) =-28.0d0
         disb(1,8) =+8.0d0
         disb(1,9) =-1.0d0
c     
         disb(2,1) =+8.0d0
         disb(2,2) =-65.0d0
         disb(2,3) =+232.0d0
         disb(2,4) =-476.0d0
         disb(2,5) =+616.0d0
         disb(2,6) =-518.0d0
         disb(2,7) =+280.0d0
         disb(2,8) =-92.0d0
         disb(2,9) =+16.0d0
         disb(2,10)=-1.0d0
c
         disb(3,1) =-28.0d0
         disb(3,2) =+232.0d0
         disb(3,3) =-849.0d0
         disb(3,4) =+1800.0d0
         disb(3,5) =-2436.0d0
         disb(3,6) =+2184.0d0
         disb(3,7) =-1302.0d0
         disb(3,8) =+504.0d0
         disb(3,9) =-120.0d0
         disb(3,10)=+16.0d0
         disb(3,11)=-1.0d0
c
         disb(4,1) =+56.0d0
         disb(4,2) =-476.0d0
         disb(4,3) =+1800.0d0
         disb(4,4) =-3985.0d0
         disb(4,5) =+5720.0d0
         disb(4,6) =-5572.0d0
         disb(4,7) =+3752.0d0
         disb(4,8) =-1750.0d0
         disb(4,9) =+560.0d0
         disb(4,10)=-120.0d0
         disb(4,11)=+16.0d0
         disb(4,12)=-1.0d0
c
         disb(5,1) =-70.0d0
         disb(5,2) =+616.0d0
         disb(5,3) =-2436.0d0
         disb(5,4) =+5720.0d0
         disb(5,5) =-8885.0d0
         disb(5,6) =+9640.0d0
         disb(5,7) =-7532.0d0
         disb(5,8) =+4312.0d0
         disb(5,9) =-1820.0d0
         disb(5,10)=+560.0d0
         disb(5,11)=-120.0d0
         disb(5,12)=+16.0d0
         disb(5,13)=-1.0d0
c
         disb(6,1) =+56.0d0
         disb(6,2) =-518.0d0
         disb(6,3) =+2184.0d0
         disb(6,4) =-5572.0d0
         disb(6,5) =+9640.0d0
         disb(6,6) =-12021.0d0
         disb(6,7) =+11208.0d0
         disb(6,8) =-7980.0d0
         disb(6,9) =+4368.0d0
         disb(6,10)=-1820.0d0
         disb(6,11)=+560.0d0
         disb(6,12)=-120.0d0
         disb(6,13)=+16.0d0
         disb(6,14)=-1.0d0
c
         disb(7,1) =-28.0d0
         disb(7,2) =+280.0d0
         disb(7,3) =-1302.0d0
         disb(7,4) =+3752.0d0
         disb(7,5) =-7532.0d0
         disb(7,6) =+11208.0d0
!     disb(7,7) =-12085.0d0 Error!!!
         disb(7,7) =-12805.0d0
         disb(7,8) =+11432.0d0
         disb(7,9) =-8008.0d0
         disb(7,10)=+4368.0d0
         disb(7,11)=-1820.0d0
         disb(7,12)=+560.0d0
         disb(7,13)=-120.0d0
         disb(7,14)=+16.0d0
         disb(7,15)=-1.0d0
c     
         disb(8,1) =+8.0d0
         disb(8,2) =-92.0d0
         disb(8,3) =+504.0d0
         disb(8,4) =-1750.0d0
         disb(8,5) =+4312.0d0
         disb(8,6) =-7980.0d0
         disb(8,7) =+11432.0d0
         disb(8,8) =-12869.0d0
         disb(8,9) =+11440.0d0
         disb(8,10)=-8008.0d0
         disb(8,11)=+4368.0d0
         disb(8,12)=-1820.0d0
         disb(8,13)=+560.0d0
         disb(8,14)=-120.0d0
         disb(8,15)=+16.0d0
         disb(8,16)=-1.0d0
c
      elseif(iord.eq.18)then
c
c 18th order inner with 9th order boundaries
c ------------------------------------------
c
         disi(1) = + 48620.0d0
         disi(2) = - 43758.0d0
         disi(3) = + 31824.0d0
         disi(4) = - 18564.0d0
         disi(5) = + 8568.0d0
         disi(6) = - 3060.0d0
         disi(7) = + 816.0d0
         disi(8) = - 153.0d0
         disi(9) = + 18.0d0
         disi(10)= - 1.0d0
c     
         disb(1,1) =+1.0d0
         disb(1,2) =-9.0d0
         disb(1,3) =+36.0d0
         disb(1,4) =-84.0d0
         disb(1,5) =+126.0d0
         disb(1,6) =-126.0d0
         disb(1,7) =+84.0d0
         disb(1,8) =-36.0d0
         disb(1,9) =+9.0d0
         disb(1,10)=-1.0d0
c     
         disb(2,1) =-9.0d0
         disb(2,2) =+82.0d0
         disb(2,3) =-333.0d0
         disb(2,4) =+792.0d0
         disb(2,5) =-1218.0d0
         disb(2,6) =+1260.0d0
         disb(2,7) =-882.0d0
         disb(2,8) =+408.0d0
         disb(2,9) =-117.0d0
         disb(2,10)=+18.0d0
         disb(2,11)=-1.0d0
c
         disb(3,1) =+36.0d0
         disb(3,2) =-333.0d0
         disb(3,3) =+1378.0d0
         disb(3,4) =-3357.0d0
         disb(3,5) =+5328.0d0
         disb(3,6) =-5754.0d0
         disb(3,7) =+4284.0d0
         disb(3,8) =-2178.0d0
         disb(3,9) =+732.0d0
         disb(3,10)=-153.0d0
         disb(3,11)=+18.0d0
         disb(3,12)=-1.0d0
c
         disb(4,1) =-84.0d0
         disb(4,2) =+792.0d0
         disb(4,3) =-3357.0d0
         disb(4,4) =+8434.0d0
         disb(4,5) =-13941.0d0
         disb(4,6) =+15912.0d0
         disb(4,7) =-12810.0d0
         disb(4,8) =+7308.0d0
         disb(4,9) =-2934.0d0
         disb(4,10)=+816.0d0
         disb(4,11)=-153.0d0
         disb(4,12)=+18.0d0
         disb(4,13)=-1.0d0
c     
         disb(5,1) =+126.0d0
         disb(5,2) =-1218.0d0
         disb(5,3) =+5328.0d0
         disb(5,4) =-13941.0d0
         disb(5,5) =+24310.0d0
         disb(5,6) =-29817.0d0
         disb(5,7) =+26496.0d0
         disb(5,8) =-17346.0d0
         disb(5,9) =+8442.0d0
         disb(5,10)=-3060.0d0
         disb(5,11)=+816.0d0
         disb(5,12)=-153.0d0
         disb(5,13)=+18.0d0
         disb(5,14)=-1.0d0
c
         disb(6,1) =-126.0d0
         disb(6,2) =+1260.0d0
         disb(6,3) =-5754.0d0
         disb(6,4) =+15912.0d0
         disb(6,5) =-29817.0d0
         disb(6,6) =+40186.0d0
         disb(6,7) =-40401.0d0
         disb(6,8) =+31032.0d0
         disb(6,9) =-18480.0d0
         disb(6,10)=+8568.0d0
         disb(6,11)=-3060.0d0
         disb(6,12)=+816.0d0
         disb(6,13)=-153.0d0
         disb(6,14)=+18.0d0
         disb(6,15)=-1.0d0
c
         disb(7,1) =+84.0d0
         disb(7,2) =-882.0d0
         disb(7,3) =+4284.0d0
         disb(7,4) =-12810.0d0
         disb(7,5) =+26496.0d0
         disb(7,6) =-40401.0d0
         disb(7,7) =+47242.0d0
         disb(7,8) =-43425.0d0
         disb(7,9) =+31788.0d0
         disb(7,10)=-18564.0d0
         disb(7,11)=+8568.0d0
         disb(7,12)=-3060.0d0
         disb(7,13)=+816.0d0
         disb(7,14)=-153.0d0
         disb(7,15)=+18.0d0
         disb(7,16)=-1.0d0
c
         disb(8,1) =-36.0d0
         disb(8,2) =+408.0d0
         disb(8,3) =-2178.0d0
         disb(8,4) =+7308.0d0
         disb(8,5) =-17346.0d0
         disb(8,6) =+31032.0d0
         disb(8,7) =-43425.0d0
         disb(8,8) =+48538.0d0
         disb(8,9) =-43749.0d0
         disb(8,10)=+31824.0d0
         disb(8,11)=-18564.0d0
         disb(8,12)=+8568.0d0
         disb(8,13)=-3060.0d0
         disb(8,14)=+816.0d0
         disb(8,15)=-153.0d0
         disb(8,16)=+18.0d0
         disb(8,17)=-1.0d0
c
         disb(9,1) =+9.0d0
         disb(9,2) =-117.0d0
         disb(9,3) =+732.0d0
         disb(9,4) =-2934.0d0
         disb(9,5) =+8442.0d0
         disb(9,6) =-18480.0d0
         disb(9,7) =+31788.0d0
         disb(9,8) =-43749.0d0
         disb(9,9) =+48619.0d0
         disb(9,10)=-43758.0d0
         disb(9,11)=+31824.0d0
         disb(9,12)=-18564.0d0
         disb(9,13)=+8568.0d0
         disb(9,14)=-3060.0d0
         disb(9,15)=+816.0d0
         disb(9,16)=-153.0d0
         disb(9,17)=+18.0d0
         disb(9,18)=-1.0d0
c
      elseif(iord.eq.20)then
c
c 20th order inner with 10th order boundaries
c ------------------------------------------
c
         disi(1) = - 184756.0d0
         disi(2) = + 167960.0d0
         disi(3) = - 125970.0d0
         disi(4) = + 77520.0d0
         disi(5) = - 38760.0d0
         disi(6) = + 15504.0d0
         disi(7) = - 4845.0d0
         disi(8) = + 1140.0d0
         disi(9) = - 190.0d0
         disi(10)= + 20.0d0
         disi(11)= - 1.0d0
c     
         disb(1,1) =-1.0d0
         disb(1,2) =+10.0d0
         disb(1,3) =-45.0d0
         disb(1,4) =+120.0d0
         disb(1,5) =-210.0d0
         disb(1,6) =+252.0d0
         disb(1,7) =-210.0d0
         disb(1,8) =+120.0d0
         disb(1,9) =-45.0d0
         disb(1,10)=+10.0d0
         disb(1,11)=-1.0d0
c
         disb(2,1) =+10.0d0
         disb(2,2) =-101.0d0
         disb(2,3) =+460.0d0
         disb(2,4) =-1245.0d0
         disb(2,5) =+2220.0d0
         disb(2,6) =-2730.0d0
         disb(2,7) =+2352.0d0
         disb(2,8) =-1410.0d0
         disb(2,9) =+570.0d0
         disb(2,10)=-145.0d0
         disb(2,11)=+20.0d0
         disb(2,12)=-1.0d0
c
         disb(3,1) =-45.0d0
         disb(3,2) =+460.0d0
         disb(3,3) =-2126.0d0
         disb(3,4) =+5860.0d0
         disb(3,5) =-10695.0d0
         disb(3,6) =+13560.0d0
         disb(3,7) =-12180.0d0
         disb(3,8) =+7752.0d0
         disb(3,9) =-3435.0d0
         disb(3,10)=+1020.0d0
         disb(3,11)=-190.0d0
         disb(3,12)=+20.0d0
         disb(3,13)=-1.0d0
c     
         disb(4,1) =+120.0d0
         disb(4,2) =-1245.0d0
         disb(4,3) =+5860.0d0
         disb(4,4) =-16526.0d0
         disb(4,5) =+31060.0d0
         disb(4,6) =-40935.0d0
         disb(4,7) =+38760.0d0
         disb(4,8) =-26580.0d0
         disb(4,9) =+13152.0d0
         disb(4,10)=-4635.0d0
         disb(4,11)=+1140.0d0
         disb(4,12)=-190.0d0
         disb(4,13)=+20.0d0
         disb(4,14)=-1.0d0
c
         disb(5,1) =-210.0d0
         disb(5,2) =+2220.0d0
         disb(5,3) =-10695.0d0
         disb(5,4) =+31060.0d0
         disb(5,5) =-60626.0d0
         disb(5,6) =+83980.0d0
         disb(5,7) =-85035.0d0
         disb(5,8) =+63960.0d0
         disb(5,9) =-36030.0d0
         disb(5,10)=+15252.0d0
         disb(5,11)=-4845.0d0
         disb(5,12)=+1140.0d0
         disb(5,13)=-190.0d0
         disb(5,14)=+20.0d0
         disb(5,15)=-1.0d0
c
         disb(6,1) =+252.0d0
         disb(6,2) =-2730.0d0
         disb(6,3) =+13560.0d0
         disb(6,4) =-40935.0d0
         disb(6,5) =+83980.0d0
         disb(6,6) =-124130.0d0
         disb(6,7) =+136900.0d0
         disb(6,8) =-115275.0d0
         disb(6,9) =+75300.0d0
         disb(6,10)=-38550.0d0
         disb(6,11)=+15504.0d0
         disb(6,12)=-4845.0d0
         disb(6,13)=+1140.0d0
         disb(6,14)=-190.0d0
         disb(6,15)=+20.0d0
         disb(6,16)=-1.
c
         disb(7,1) =-210.0d0
         disb(7,2) =+2352.0d0
         disb(7,3) =-12180.0d0
         disb(7,4) =+38760.0d0
         disb(7,5) =-85035.0d0
         disb(7,6) =+136900.0d0
         disb(7,7) =-168230.0d0
         disb(7,8) =+162100.0d0
         disb(7,9) =-124725.0d0
         disb(7,10)=+77400.0d0
         disb(7,11)=-38760.0d0
         disb(7,12)=+15504.0d0
         disb(7,13)=-4845.0d0
         disb(7,14)=+1140.0d0
         disb(7,15)=-190.0d0
         disb(7,16)=+20.0d0
         disb(7,17)=-1.0d0
c
         disb(8,1) =+120.0d0
         disb(8,2) =-1410.0d0
         disb(8,3) =+7752.0d0
         disb(8,4) =-26580.0d0
         disb(8,5) =+63960.0d0
         disb(8,6) =-115275.0d0
         disb(8,7) =+162100.0d0
         disb(8,8) =-182630.0d0
         disb(8,9) =+167500.0d0
         disb(8,10)=-125925.0d0
         disb(8,11)=+77520.0d0
         disb(8,12)=-38760.0d0
         disb(8,13)=+15504.0d0
         disb(8,14)=-4845.0d0
         disb(8,15)=+1140.0d0
         disb(8,16)=-190.0d0
         disb(8,17)=+20.0d0
         disb(8,18)=-1.0d0
c
         disb(9,1) =-45.0d0
         disb(9,2) =+570.0d0
         disb(9,3) =-3435.0d0
         disb(9,4) =+13152.0d0
         disb(9,5) =-36030.0d0
         disb(9,6) =+75300.0d0
         disb(9,7) =-124725.0d0
         disb(9,8) =+167500.0d0
         disb(9,9) =-184655.0d0
         disb(9,10)=+167950.0d0
         disb(9,11)=-125970.0d0
         disb(9,12)=+77520.0d0
         disb(9,13)=-38760.0d0
         disb(9,14)=+15504.0d0
         disb(9,15)=-4845.0d0
         disb(9,16)=+1140.0d0
         disb(9,17)=-190.0d0
         disb(9,18)=+20.0d0
         disb(9,19)=-1.0d0
c     
         disb(10,1) =+10.0d0
         disb(10,2) =-145.0d0
         disb(10,3) =+1020.0d0
         disb(10,4) =-4635.0d0
         disb(10,5) =+15252.0d0
         disb(10,6) =-38550.0d0
         disb(10,7) =+77400.0d0
         disb(10,8) =-125925.0d0
         disb(10,9) =+167950.0d0
         disb(10,10)=-184755.0d0
         disb(10,11)=+167960.0d0
         disb(10,12)=-125970.0d0
         disb(10,13)=+77520.0d0
         disb(10,14)=-38760.0d0
         disb(10,15)=+15504.0d0
         disb(10,16)=-4845.0d0
         disb(10,17)=+1140.0d0
         disb(10,18)=-190.0d0
         disb(10,19)=+20.0d0
         disb(10,20)=-1.0d0
c
c     JR_CHANGE 04/15/06, Adding order 22-50.
      elseif(iord.eq.22)then
c 
c 22nd order inner with 11th order boundaries
c ------------------------------------------
c
      disi(1) = 705432.d0
      disi(2) = -646646.d0
      disi(3) = 497420.d0
      disi(4) = -319770.d0
      disi(5) = 170544.d0
      disi(6) = -74613.d0
      disi(7) = 26334.d0
      disi(8) = -7315.d0
      disi(9) = 1540.d0
      disi(10) = -231.d0
      disi(11) = 22.d0
      disi(12) = -1.d0
c
      disb(1,1) = 1.d0
      disb(1,2) = -11.d0
      disb(1,3) = 55.d0
      disb(1,4) = -165.d0
      disb(1,5) = 330.d0
      disb(1,6) = -462.d0
      disb(1,7) = 462.d0
      disb(1,8) = -330.d0
      disb(1,9) = 165.d0
      disb(1,10) = -55.d0
      disb(1,11) = 11.d0
      disb(1,12) = -1.d0
c
      disb(2,1) = -11.d0
      disb(2,2) = 122.d0
      disb(2,3) = -616.d0
      disb(2,4) = 1870.d0
      disb(2,5) = -3795.d0
      disb(2,6) = 5412.d0
      disb(2,7) = -5544.d0
      disb(2,8) = 4092.d0
      disb(2,9) = -2145.d0
      disb(2,10) = 770.d0
      disb(2,11) = -176.d0
      disb(2,12) = 22.d0
      disb(2,13) = -1.d0
c
      disb(3,1) = 55.d0
      disb(3,2) = -616.d0
      disb(3,3) = 3147.d0
      disb(3,4) = -9691.d0
      disb(3,5) = 20020.d0
      disb(3,6) = -29205.d0
      disb(3,7) = 30822.d0
      disb(3,8) = -23694.d0
      disb(3,9) = 13167.d0
      disb(3,10) = -5170.d0
      disb(3,11) = 1375.d0
      disb(3,12) = -231.d0
      disb(3,13) = 22.d0
      disb(3,14) = -1.d0
c
      disb(4,1) = -165.d0
      disb(4,2) = 1870.d0
      disb(4,3) = -9691.d0
      disb(4,4) = 30372.d0
      disb(4,5) = -64141.d0
      disb(4,6) = 96250.d0
      disb(4,7) = -105435.d0
      disb(4,8) = 85272.d0
      disb(4,9) = -50919.d0
      disb(4,10) = 22242.d0
      disb(4,11) = -6985.d0
      disb(4,12) = 1540.d0
      disb(4,13) = -231.d0
      disb(4,14) = 22.d0
      disb(4,15) = -1.d0
c
      disb(5,1) = 330.d0
      disb(5,2) = -3795.d0
      disb(5,3) = 20020.d0
      disb(5,4) = -64141.d0
      disb(5,5) = 139272.d0
      disb(5,6) = -216601.d0
      disb(5,7) = 248710.d0
      disb(5,8) = -214335.d0
      disb(5,9) = 139722.d0
      disb(5,10) = -69069.d0
      disb(5,11) = 25872.d0
      disb(5,12) = -7315.d0
      disb(5,13) = 1540.d0
      disb(5,14) = -231.d0
      disb(5,15) = 22.d0
      disb(5,16) = -1.d0
c
      disb(6,1) = -462.d0
      disb(6,2) = 5412.d0
      disb(6,3) = -29205.d0
      disb(6,4) = 96250.d0
      disb(6,5) = -216601.d0
      disb(6,6) = 352716.d0
      disb(6,7) = -430045.d0
      disb(6,8) = 401170.d0
      disb(6,9) = -290565.d0
      disb(6,10) = 165132.d0
      disb(6,11) = -74151.d0
      disb(6,12) = 26334.d0
      disb(6,13) = -7315.d0
      disb(6,14) = 1540.d0
      disb(6,15) = -231.d0
      disb(6,16) = 22.d0
      disb(6,17) = -1.d0
c
      disb(7,1) = 462.d0
      disb(7,2) = -5544.d0
      disb(7,3) = 30822.d0
      disb(7,4) = -105435.d0
      disb(7,5) = 248710.d0
      disb(7,6) = -430045.d0
      disb(7,7) = 566160.d0
      disb(7,8) = -582505.d0
      disb(7,9) = 477400.d0
      disb(7,10) = -315975.d0
      disb(7,11) = 170214.d0
      disb(7,12) = -74613.d0
      disb(7,13) = 26334.d0
      disb(7,14) = -7315.d0
      disb(7,15) = 1540.d0
      disb(7,16) = -231.d0
      disb(7,17) = 22.d0
      disb(7,18) = -1.d0
c
      disb(8,1) = -330.d0
      disb(8,2) = 4092.d0
      disb(8,3) = -23694.d0
      disb(8,4) = 85272.d0
      disb(8,5) = -214335.d0
      disb(8,6) = 401170.d0
      disb(8,7) = -582505.d0
      disb(8,8) = 675060.d0
      disb(8,9) = -636955.d0
      disb(8,10) = 495550.d0
      disb(8,11) = -319605.d0
      disb(8,12) = 170544.d0
      disb(8,13) = -74613.d0
      disb(8,14) = 26334.d0
      disb(8,15) = -7315.d0
      disb(8,16) = 1540.d0
      disb(8,17) = -231.d0
      disb(8,18) = 22.d0
      disb(8,19) = -1.d0
c
      disb(9,1) = 165.d0
      disb(9,2) = -2145.d0
      disb(9,3) = 13167.d0
      disb(9,4) = -50919.d0
      disb(9,5) = 139722.d0
      disb(9,6) = -290565.d0
      disb(9,7) = 477400.d0
      disb(9,8) = -636955.d0
      disb(9,9) = 702285.d0
      disb(9,10) = -646030.d0
      disb(9,11) = 497365.d0
      disb(9,12) = -319770.d0
      disb(9,13) = 170544.d0
      disb(9,14) = -74613.d0
      disb(9,15) = 26334.d0
      disb(9,16) = -7315.d0
      disb(9,17) = 1540.d0
      disb(9,18) = -231.d0
      disb(9,19) = 22.d0
      disb(9,20) = -1.d0
c
      disb(10,1) = -55.d0
      disb(10,2) = 770.d0
      disb(10,3) = -5170.d0
      disb(10,4) = 22242.d0
      disb(10,5) = -69069.d0
      disb(10,6) = 165132.d0
      disb(10,7) = -315975.d0
      disb(10,8) = 495550.d0
      disb(10,9) = -646030.d0
      disb(10,10) = 705310.d0
      disb(10,11) = -646635.d0
      disb(10,12) = 497420.d0
      disb(10,13) = -319770.d0
      disb(10,14) = 170544.d0
      disb(10,15) = -74613.d0
      disb(10,16) = 26334.d0
      disb(10,17) = -7315.d0
      disb(10,18) = 1540.d0
      disb(10,19) = -231.d0
      disb(10,20) = 22.d0
      disb(10,21) = -1.d0
c
      disb(11,1) = 11.d0
      disb(11,2) = -176.d0
      disb(11,3) = 1375.d0
      disb(11,4) = -6985.d0
      disb(11,5) = 25872.d0
      disb(11,6) = -74151.d0
      disb(11,7) = 170214.d0
      disb(11,8) = -319605.d0
      disb(11,9) = 497365.d0
      disb(11,10) = -646635.d0
      disb(11,11) = 705431.d0
      disb(11,12) = -646646.d0
      disb(11,13) = 497420.d0
      disb(11,14) = -319770.d0
      disb(11,15) = 170544.d0
      disb(11,16) = -74613.d0
      disb(11,17) = 26334.d0
      disb(11,18) = -7315.d0
      disb(11,19) = 1540.d0
      disb(11,20) = -231.d0
      disb(11,21) = 22.d0
      disb(11,22) = -1.d0
c
      elseif(iord.eq.24)then
c 
c 24th order inner with 12th order boundaries
c ------------------------------------------
c
      disi(1) = -2704156.d0
      disi(2) = 2496144.d0
      disi(3) = -1961256.d0
      disi(4) = 1307504.d0
      disi(5) = -735471.d0
      disi(6) = 346104.d0
      disi(7) = -134596.d0
      disi(8) = 42504.d0
      disi(9) = -10626.d0
      disi(10) = 2024.d0
      disi(11) = -276.d0
      disi(12) = 24.d0
      disi(13) = -1.d0
c
      disb(1,1) = -1.d0
      disb(1,2) = 12.d0
      disb(1,3) = -66.d0
      disb(1,4) = 220.d0
      disb(1,5) = -495.d0
      disb(1,6) = 792.d0
      disb(1,7) = -924.d0
      disb(1,8) = 792.d0
      disb(1,9) = -495.d0
      disb(1,10) = 220.d0
      disb(1,11) = -66.d0
      disb(1,12) = 12.d0
      disb(1,13) = -1.d0
c
      disb(2,1) = 12.d0
      disb(2,2) = -145.d0
      disb(2,3) = 804.d0
      disb(2,4) = -2706.d0
      disb(2,5) = 6160.d0
      disb(2,6) = -9999.d0
      disb(2,7) = 11880.d0
      disb(2,8) = -10428.d0
      disb(2,9) = 6732.d0
      disb(2,10) = -3135.d0
      disb(2,11) = 1012.d0
      disb(2,12) = -210.d0
      disb(2,13) = 24.d0
      disb(2,14) = -1.d0
c
      disb(3,1) = -66.d0
      disb(3,2) = 804.d0
      disb(3,3) = -4501.d0
      disb(3,4) = 15324.d0
      disb(3,5) = -35376.d0
      disb(3,6) = 58432.d0
      disb(3,7) = -70983.d0
      disb(3,8) = 64152.d0
      disb(3,9) = -43098.d0
      disb(3,10) = 21252.d0
      disb(3,11) = -7491.d0
      disb(3,12) = 1804.d0
      disb(3,13) = -276.d0
      disb(3,14) = 24.d0
      disb(3,15) = -1.d0
c
      disb(4,1) = 220.d0
      disb(4,2) = -2706.d0
      disb(4,3) = 15324.d0
      disb(4,4) = -52901.d0
      disb(4,5) = 124224.d0
      disb(4,6) = -209616.d0
      disb(4,7) = 261712.d0
      disb(4,8) = -245223.d0
      disb(4,9) = 173052.d0
      disb(4,10) = -91498.d0
      disb(4,11) = 35772.d0
      disb(4,12) = -10131.d0
      disb(4,13) = 2024.d0
      disb(4,14) = -276.d0
      disb(4,15) = 24.d0
      disb(4,16) = -1.d0
c
      disb(5,1) = -495.d0
      disb(5,2) = 6160.d0
      disb(5,3) = -35376.d0
      disb(5,4) = 124224.d0
      disb(5,5) = -297926.d0
      disb(5,6) = 516264.d0
      disb(5,7) = -666996.d0
      disb(5,8) = 653752.d0
      disb(5,9) = -490248.d0
      disb(5,10) = 281952.d0
      disb(5,11) = -124168.d0
      disb(5,12) = 41712.d0
      disb(5,13) = -10626.d0
      disb(5,14) = 2024.d0
      disb(5,15) = -276.d0
      disb(5,16) = 24.d0
      disb(5,17) = -1.d0
c
      disb(6,1) = 792.d0
      disb(6,2) = -9999.d0
      disb(6,3) = 58432.d0
      disb(6,4) = -209616.d0
      disb(6,5) = 516264.d0
      disb(6,6) = -925190.d0
      disb(6,7) = 1248072.d0
      disb(6,8) = -1294260.d0
      disb(6,9) = 1045792.d0
      disb(6,10) = -664488.d0
      disb(6,11) = 334224.d0
      disb(6,12) = -133672.d0
      disb(6,13) = 42504.d0
      disb(6,14) = -10626.d0
      disb(6,15) = 2024.d0
      disb(6,16) = -276.d0
      disb(6,17) = 24.d0
      disb(6,18) = -1.d0
c
      disb(7,1) = -924.d0
      disb(7,2) = 11880.d0
      disb(7,3) = -70983.d0
      disb(7,4) = 261712.d0
      disb(7,5) = -666996.d0
      disb(7,6) = 1248072.d0
      disb(7,7) = -1778966.d0
      disb(7,8) = 1979880.d0
      disb(7,9) = -1751640.d0
      disb(7,10) = 1249072.d0
      disb(7,11) = -725472.d0
      disb(7,12) = 345312.d0
      disb(7,13) = -134596.d0
      disb(7,14) = 42504.d0
      disb(7,15) = -10626.d0
      disb(7,16) = 2024.d0
      disb(7,17) = -276.d0
      disb(7,18) = 24.d0
      disb(7,19) = -1.d0
c
      disb(8,1) = 792.d0
      disb(8,2) = -10428.d0
      disb(8,3) = 64152.d0
      disb(8,4) = -245223.d0
      disb(8,5) = 653752.d0
      disb(8,6) = -1294260.d0
      disb(8,7) = 1979880.d0
      disb(8,8) = -2406230.d0
      disb(8,9) = 2371920.d0
      disb(8,10) = -1925880.d0
      disb(8,11) = 1301344.d0
      disb(8,12) = -734976.d0
      disb(8,13) = 346104.d0
      disb(8,14) = -134596.d0
      disb(8,15) = 42504.d0
      disb(8,16) = -10626.d0
      disb(8,17) = 2024.d0
      disb(8,18) = -276.d0
      disb(8,19) = 24.d0
      disb(8,20) = -1.d0
c
      disb(9,1) = -495.d0
      disb(9,2) = 6732.d0
      disb(9,3) = -43098.d0
      disb(9,4) = 173052.d0
      disb(9,5) = -490248.d0
      disb(9,6) = 1045792.d0
      disb(9,7) = -1751640.d0
      disb(9,8) = 2371920.d0
      disb(9,9) = -2651255.d0
      disb(9,10) = 2480820.d0
      disb(9,11) = -1958550.d0
      disb(9,12) = 1307284.d0
      disb(9,13) = -735471.d0
      disb(9,14) = 346104.d0
      disb(9,15) = -134596.d0
      disb(9,16) = 42504.d0
      disb(9,17) = -10626.d0
      disb(9,18) = 2024.d0
      disb(9,19) = -276.d0
      disb(9,20) = 24.d0
      disb(9,21) = -1.d0
c
      disb(10,1) = 220.d0
      disb(10,2) = -3135.d0
      disb(10,3) = 21252.d0
      disb(10,4) = -91498.d0
      disb(10,5) = 281952.d0
      disb(10,6) = -664488.d0
      disb(10,7) = 1249072.d0
      disb(10,8) = -1925880.d0
      disb(10,9) = 2480820.d0
      disb(10,10) = -2699655.d0
      disb(10,11) = 2495340.d0
      disb(10,12) = -1961190.d0
      disb(10,13) = 1307504.d0
      disb(10,14) = -735471.d0
      disb(10,15) = 346104.d0
      disb(10,16) = -134596.d0
      disb(10,17) = 42504.d0
      disb(10,18) = -10626.d0
      disb(10,19) = 2024.d0
      disb(10,20) = -276.d0
      disb(10,21) = 24.d0
      disb(10,22) = -1.d0
c
      disb(11,1) = -66.d0
      disb(11,2) = 1012.d0
      disb(11,3) = -7491.d0
      disb(11,4) = 35772.d0
      disb(11,5) = -124168.d0
      disb(11,6) = 334224.d0
      disb(11,7) = -725472.d0
      disb(11,8) = 1301344.d0
      disb(11,9) = -1958550.d0
      disb(11,10) = 2495340.d0
      disb(11,11) = -2704011.d0
      disb(11,12) = 2496132.d0
      disb(11,13) = -1961256.d0
      disb(11,14) = 1307504.d0
      disb(11,15) = -735471.d0
      disb(11,16) = 346104.d0
      disb(11,17) = -134596.d0
      disb(11,18) = 42504.d0
      disb(11,19) = -10626.d0
      disb(11,20) = 2024.d0
      disb(11,21) = -276.d0
      disb(11,22) = 24.d0
      disb(11,23) = -1.d0
c
      disb(12,1) = 12.d0
      disb(12,2) = -210.d0
      disb(12,3) = 1804.d0
      disb(12,4) = -10131.d0
      disb(12,5) = 41712.d0
      disb(12,6) = -133672.d0
      disb(12,7) = 345312.d0
      disb(12,8) = -734976.d0
      disb(12,9) = 1307284.d0
      disb(12,10) = -1961190.d0
      disb(12,11) = 2496132.d0
      disb(12,12) = -2704155.d0
      disb(12,13) = 2496144.d0
      disb(12,14) = -1961256.d0
      disb(12,15) = 1307504.d0
      disb(12,16) = -735471.d0
      disb(12,17) = 346104.d0
      disb(12,18) = -134596.d0
      disb(12,19) = 42504.d0
      disb(12,20) = -10626.d0
      disb(12,21) = 2024.d0
      disb(12,22) = -276.d0
      disb(12,23) = 24.d0
      disb(12,24) = -1.d0
c
      elseif(iord.eq.26)then
c 
c 26th order inner with 13th order boundaries
c ------------------------------------------
c
      disi(1) = 10400600.d0
      disi(2) = -9657700.d0
      disi(3) = 7726160.d0
      disi(4) = -5311735.d0
      disi(5) = 3124550.d0
      disi(6) = -1562275.d0
      disi(7) = 657800.d0
      disi(8) = -230230.d0
      disi(9) = 65780.d0
      disi(10) = -14950.d0
      disi(11) = 2600.d0
      disi(12) = -325.d0
      disi(13) = 26.d0
      disi(14) = -1.d0
c
      disb(1,1) = 1.d0
      disb(1,2) = -13.d0
      disb(1,3) = 78.d0
      disb(1,4) = -286.d0
      disb(1,5) = 715.d0
      disb(1,6) = -1287.d0
      disb(1,7) = 1716.d0
      disb(1,8) = -1716.d0
      disb(1,9) = 1287.d0
      disb(1,10) = -715.d0
      disb(1,11) = 286.d0
      disb(1,12) = -78.d0
      disb(1,13) = 13.d0
      disb(1,14) = -1.d0
c
      disb(2,1) = -13.d0
      disb(2,2) = 170.d0
      disb(2,3) = -1027.d0
      disb(2,4) = 3796.d0
      disb(2,5) = -9581.d0
      disb(2,6) = 17446.d0
      disb(2,7) = -23595.d0
      disb(2,8) = 24024.d0
      disb(2,9) = -18447.d0
      disb(2,10) = 10582.d0
      disb(2,11) = -4433.d0
      disb(2,12) = 1300.d0
      disb(2,13) = -247.d0
      disb(2,14) = 26.d0
      disb(2,15) = -1.d0
c
      disb(3,1) = 78.d0
      disb(3,2) = -1027.d0
      disb(3,3) = 6254.d0
      disb(3,4) = -23335.d0
      disb(3,5) = 59566.d0
      disb(3,6) = -109967.d0
      disb(3,7) = 151294.d0
      disb(3,8) = -157443.d0
      disb(3,9) = 124410.d0
      disb(3,10) = -74217.d0
      disb(3,11) = 32890.d0
      disb(3,12) = -10517.d0
      disb(3,13) = 2314.d0
      disb(3,14) = -325.d0
      disb(3,15) = 26.d0
      disb(3,16) = -1.d0
c
      disb(4,1) = -286.d0
      disb(4,2) = 3796.d0
      disb(4,3) = -23335.d0
      disb(4,4) = 88050.d0
      disb(4,5) = -227825.d0
      disb(4,6) = 427648.d0
      disb(4,7) = -600743.d0
      disb(4,8) = 642070.d0
      disb(4,9) = -525525.d0
      disb(4,10) = 328900.d0
      disb(4,11) = -156013.d0
      disb(4,12) = 55198.d0
      disb(4,13) = -14235.d0
      disb(4,14) = 2600.d0
      disb(4,15) = -325.d0
      disb(4,16) = 26.d0
      disb(4,17) = -1.d0
c
      disb(5,1) = 715.d0
      disb(5,2) = -9581.d0
      disb(5,3) = 59566.d0
      disb(5,4) = -227825.d0
      disb(5,5) = 599275.d0
      disb(5,6) = -1148030.d0
      disb(5,7) = 1654588.d0
      disb(5,8) = -1827683.d0
      disb(5,9) = 1562275.d0
      disb(5,10) = -1036750.d0
      disb(5,11) = 533390.d0
      disb(5,12) = -211783.d0
      disb(5,13) = 64493.d0
      disb(5,14) = -14950.d0
      disb(5,15) = 2600.d0
      disb(5,16) = -325.d0
      disb(5,17) = 26.d0
      disb(5,18) = -1.d0
c
      disb(6,1) = -1287.d0
      disb(6,2) = 17446.d0
      disb(6,3) = -109967.d0
      disb(6,4) = 427648.d0
      disb(6,5) = -1148030.d0
      disb(6,6) = 2255644.d0
      disb(6,7) = -3356522.d0
      disb(6,8) = 3863080.d0
      disb(6,9) = -3484052.d0
      disb(6,10) = 2482480.d0
      disb(6,11) = -1404832.d0
      disb(6,12) = 633776.d0
      disb(6,13) = -228514.d0
      disb(6,14) = 65780.d0
      disb(6,15) = -14950.d0
      disb(6,16) = 2600.d0
      disb(6,17) = -325.d0
      disb(6,18) = 26.d0
      disb(6,19) = -1.d0
c
      disb(7,1) = 1716.d0
      disb(7,2) = -23595.d0
      disb(7,3) = 151294.d0
      disb(7,4) = -600743.d0
      disb(7,5) = 1654588.d0
      disb(7,6) = -3356522.d0
      disb(7,7) = 5200300.d0
      disb(7,8) = -6301178.d0
      disb(7,9) = 6071572.d0
      disb(7,10) = -4710992.d0
      disb(7,11) = 2973256.d0
      disb(7,12) = -1538680.d0
      disb(7,13) = 656084.d0
      disb(7,14) = -230230.d0
      disb(7,15) = 65780.d0
      disb(7,16) = -14950.d0
      disb(7,17) = 2600.d0
      disb(7,18) = -325.d0
      disb(7,19) = 26.d0
      disb(7,20) = -1.d0
c
      disb(8,1) = -1716.d0
      disb(8,2) = 24024.d0
      disb(8,3) = -157443.d0
      disb(8,4) = 642070.d0
      disb(8,5) = -1827683.d0
      disb(8,6) = 3863080.d0
      disb(8,7) = -6301178.d0
      disb(8,8) = 8144956.d0
      disb(8,9) = -8509670.d0
      disb(8,10) = 7298512.d0
      disb(8,11) = -5201768.d0
      disb(8,12) = 3107104.d0
      disb(8,13) = -1560988.d0
      disb(8,14) = 657800.d0
      disb(8,15) = -230230.d0
      disb(8,16) = 65780.d0
      disb(8,17) = -14950.d0
      disb(8,18) = 2600.d0
      disb(8,19) = -325.d0
      disb(8,20) = 26.d0
      disb(8,21) = -1.d0
c
      disb(9,1) = 1287.d0
      disb(9,2) = -18447.d0
      disb(9,3) = 124410.d0
      disb(9,4) = -525525.d0
      disb(9,5) = 1562275.d0
      disb(9,6) = -3484052.d0
      disb(9,7) = 6071572.d0
      disb(9,8) = -8509670.d0
      disb(9,9) = 9801325.d0
      disb(9,10) = -9429875.d0
      disb(9,11) = 7666594.d0
      disb(9,12) = -5302154.d0
      disb(9,13) = 3123835.d0
      disb(9,14) = -1562275.d0
      disb(9,15) = 657800.d0
      disb(9,16) = -230230.d0
      disb(9,17) = 65780.d0
      disb(9,18) = -14950.d0
      disb(9,19) = 2600.d0
      disb(9,20) = -325.d0
      disb(9,21) = 26.d0
      disb(9,22) = -1.d0
c
      disb(10,1) = -715.d0
      disb(10,2) = 10582.d0
      disb(10,3) = -74217.d0
      disb(10,4) = 328900.d0
      disb(10,5) = -1036750.d0
      disb(10,6) = 2482480.d0
      disb(10,7) = -4710992.d0
      disb(10,8) = 7298512.d0
      disb(10,9) = -9429875.d0
      disb(10,10) = 10312550.d0
      disb(10,11) = -9634365.d0
      disb(10,12) = 7722364.d0
      disb(10,13) = -5311449.d0
      disb(10,14) = 3124550.d0
      disb(10,15) = -1562275.d0
      disb(10,16) = 657800.d0
      disb(10,17) = -230230.d0
      disb(10,18) = 65780.d0
      disb(10,19) = -14950.d0
      disb(10,20) = 2600.d0
      disb(10,21) = -325.d0
      disb(10,22) = 26.d0
      disb(10,23) = -1.d0
c
      disb(11,1) = 286.d0
      disb(11,2) = -4433.d0
      disb(11,3) = 32890.d0
      disb(11,4) = -156013.d0
      disb(11,5) = 533390.d0
      disb(11,6) = -1404832.d0
      disb(11,7) = 2973256.d0
      disb(11,8) = -5201768.d0
      disb(11,9) = 7666594.d0
      disb(11,10) = -9634365.d0
      disb(11,11) = 10394346.d0
      disb(11,12) = -9656673.d0
      disb(11,13) = 7726082.d0
      disb(11,14) = -5311735.d0
      disb(11,15) = 3124550.d0
      disb(11,16) = -1562275.d0
      disb(11,17) = 657800.d0
      disb(11,18) = -230230.d0
      disb(11,19) = 65780.d0
      disb(11,20) = -14950.d0
      disb(11,21) = 2600.d0
      disb(11,22) = -325.d0
      disb(11,23) = 26.d0
      disb(11,24) = -1.d0
c
      disb(12,1) = -78.d0
      disb(12,2) = 1300.d0
      disb(12,3) = -10517.d0
      disb(12,4) = 55198.d0
      disb(12,5) = -211783.d0
      disb(12,6) = 633776.d0
      disb(12,7) = -1538680.d0
      disb(12,8) = 3107104.d0
      disb(12,9) = -5302154.d0
      disb(12,10) = 7722364.d0
      disb(12,11) = -9656673.d0
      disb(12,12) = 10400430.d0
      disb(12,13) = -9657687.d0
      disb(12,14) = 7726160.d0
      disb(12,15) = -5311735.d0
      disb(12,16) = 3124550.d0
      disb(12,17) = -1562275.d0
      disb(12,18) = 657800.d0
      disb(12,19) = -230230.d0
      disb(12,20) = 65780.d0
      disb(12,21) = -14950.d0
      disb(12,22) = 2600.d0
      disb(12,23) = -325.d0
      disb(12,24) = 26.d0
      disb(12,25) = -1.d0
c
      disb(13,1) = 13.d0
      disb(13,2) = -247.d0
      disb(13,3) = 2314.d0
      disb(13,4) = -14235.d0
      disb(13,5) = 64493.d0
      disb(13,6) = -228514.d0
      disb(13,7) = 656084.d0
      disb(13,8) = -1560988.d0
      disb(13,9) = 3123835.d0
      disb(13,10) = -5311449.d0
      disb(13,11) = 7726082.d0
      disb(13,12) = -9657687.d0
      disb(13,13) = 10400599.d0
      disb(13,14) = -9657700.d0
      disb(13,15) = 7726160.d0
      disb(13,16) = -5311735.d0
      disb(13,17) = 3124550.d0
      disb(13,18) = -1562275.d0
      disb(13,19) = 657800.d0
      disb(13,20) = -230230.d0
      disb(13,21) = 65780.d0
      disb(13,22) = -14950.d0
      disb(13,23) = 2600.d0
      disb(13,24) = -325.d0
      disb(13,25) = 26.d0
      disb(13,26) = -1.d0
c
      elseif(iord.eq.28)then
c 
c 28th order inner with 14th order boundaries
c ------------------------------------------
c
      disi(1) = -40116600.d0
      disi(2) = 37442160.d0
      disi(3) = -30421755.d0
      disi(4) = 21474180.d0
      disi(5) = -13123110.d0
      disi(6) = 6906900.d0
      disi(7) = -3108105.d0
      disi(8) = 1184040.d0
      disi(9) = -376740.d0
      disi(10) = 98280.d0
      disi(11) = -20475.d0
      disi(12) = 3276.d0
      disi(13) = -378.d0
      disi(14) = 28.d0
      disi(15) = -1.d0
c
      disb(1,1) = -1.d0
      disb(1,2) = 14.d0
      disb(1,3) = -91.d0
      disb(1,4) = 364.d0
      disb(1,5) = -1001.d0
      disb(1,6) = 2002.d0
      disb(1,7) = -3003.d0
      disb(1,8) = 3432.d0
      disb(1,9) = -3003.d0
      disb(1,10) = 2002.d0
      disb(1,11) = -1001.d0
      disb(1,12) = 364.d0
      disb(1,13) = -91.d0
      disb(1,14) = 14.d0
      disb(1,15) = -1.d0
c
      disb(2,1) = 14.d0
      disb(2,2) = -197.d0
      disb(2,3) = 1288.d0
      disb(2,4) = -5187.d0
      disb(2,5) = 14378.d0
      disb(2,6) = -29029.d0
      disb(2,7) = 44044.d0
      disb(2,8) = -51051.d0
      disb(2,9) = 45474.d0
      disb(2,10) = -31031.d0
      disb(2,11) = 16016.d0
      disb(2,12) = -6097.d0
      disb(2,13) = 1638.d0
      disb(2,14) = -287.d0
      disb(2,15) = 28.d0
      disb(2,16) = -1.d0
c
      disb(3,1) = -91.d0
      disb(3,2) = 1288.d0
      disb(3,3) = -8478.d0
      disb(3,4) = 34412.d0
      disb(3,5) = -96278.d0
      disb(3,6) = 196560.d0
      disb(3,7) = -302302.d0
      disb(3,8) = 356356.d0
      disb(3,9) = -324324.d0
      disb(3,10) = 227656.d0
      disb(3,11) = -122122.d0
      disb(3,12) = 49140.d0
      disb(3,13) = -14378.d0
      disb(3,14) = 2912.d0
      disb(3,15) = -378.d0
      disb(3,16) = 28.d0
      disb(3,17) = -1.d0
c
      disb(4,1) = 364.d0
      disb(4,2) = -5187.d0
      disb(4,3) = 34412.d0
      disb(4,4) = -140974.d0
      disb(4,5) = 398776.d0
      disb(4,6) = -825006.d0
      disb(4,7) = 1289652.d0
      disb(4,8) = -1551550.d0
      disb(4,9) = 1449448.d0
      disb(4,10) = -1053052.d0
      disb(4,11) = 592020.d0
      disb(4,12) = -254618.d0
      disb(4,13) = 82264.d0
      disb(4,14) = -19474.d0
      disb(4,15) = 3276.d0
      disb(4,16) = -378.d0
      disb(4,17) = 28.d0
      disb(4,18) = -1.d0
c
      disb(5,1) = -1001.d0
      disb(5,2) = 14378.d0
      disb(5,3) = -96278.d0
      disb(5,4) = 398776.d0
      disb(5,5) = -1142975.d0
      disb(5,6) = 2402778.d0
      disb(5,7) = -3831009.d0
      disb(5,8) = 4725084.d0
      disb(5,9) = -4557553.d0
      disb(5,10) = 3453450.d0
      disb(5,11) = -2055053.d0
      disb(5,12) = 956384.d0
      disb(5,13) = -345709.d0
      disb(5,14) = 96278.d0
      disb(5,15) = -20475.d0
      disb(5,16) = 3276.d0
      disb(5,17) = -378.d0
      disb(5,18) = 28.d0
      disb(5,19) = -1.d0
c
      disb(6,1) = 2002.d0
      disb(6,2) = -29029.d0
      disb(6,3) = 196560.d0
      disb(6,4) = -825006.d0
      disb(6,5) = 2402778.d0
      disb(6,6) = -5150979.d0
      disb(6,7) = 8414784.d0
      disb(6,8) = -10701873.d0
      disb(6,9) = 10737090.d0
      disb(6,10) = -8565557.d0
      disb(6,11) = 5457452.d0
      disb(6,12) = -2783781.d0
      disb(6,13) = 1138566.d0
      disb(6,14) = -373737.d0
      disb(6,15) = 98280.d0
      disb(6,16) = -20475.d0
      disb(6,17) = 3276.d0
      disb(6,18) = -378.d0
      disb(6,19) = 28.d0
      disb(6,20) = -1.d0
c
      disb(7,1) = -3003.d0
      disb(7,2) = 44044.d0
      disb(7,3) = -302302.d0
      disb(7,4) = 1289652.d0
      disb(7,5) = -3831009.d0
      disb(7,6) = 8414784.d0
      disb(7,7) = -14168988.d0
      disb(7,8) = 18721080.d0
      disb(7,9) = -19719882.d0
      disb(7,10) = 16749096.d0
      disb(7,11) = -11571560.d0
      disb(7,12) = 6550544.d0
      disb(7,13) = -3057054.d0
      disb(7,14) = 1180608.d0
      disb(7,15) = -376740.d0
      disb(7,16) = 98280.d0
      disb(7,17) = -20475.d0
      disb(7,18) = 3276.d0
      disb(7,19) = -378.d0
      disb(7,20) = 28.d0
      disb(7,21) = -1.d0
c
      disb(8,1) = 3432.d0
      disb(8,2) = -51051.d0
      disb(8,3) = 356356.d0
      disb(8,4) = -1551550.d0
      disb(8,5) = 4725084.d0
      disb(8,6) = -10701873.d0
      disb(8,7) = 18721080.d0
      disb(8,8) = -25947612.d0
      disb(8,9) = 29027376.d0
      disb(8,10) = -26590746.d0
      disb(8,11) = 20184528.d0
      disb(8,12) = -12820808.d0
      disb(8,13) = 6862856.d0
      disb(8,14) = -3105102.d0
      disb(8,15) = 1184040.d0
      disb(8,16) = -376740.d0
      disb(8,17) = 98280.d0
      disb(8,18) = -20475.d0
      disb(8,19) = 3276.d0
      disb(8,20) = -378.d0
      disb(8,21) = 28.d0
      disb(8,22) = -1.d0
c
      disb(9,1) = -3003.d0
      disb(9,2) = 45474.d0
      disb(9,3) = -324324.d0
      disb(9,4) = 1449448.d0
      disb(9,5) = -4557553.d0
      disb(9,6) = 10737090.d0
      disb(9,7) = -19719882.d0
      disb(9,8) = 29027376.d0
      disb(9,9) = -34965621.d0
      disb(9,10) = 35039382.d0
      disb(9,11) = -29596749.d0
      disb(9,12) = 21277620.d0
      disb(9,13) = -13094081.d0
      disb(9,14) = 6904898.d0
      disb(9,15) = -3108105.d0
      disb(9,16) = 1184040.d0
      disb(9,17) = -376740.d0
      disb(9,18) = 98280.d0
      disb(9,19) = -20475.d0
      disb(9,20) = 3276.d0
      disb(9,21) = -378.d0
      disb(9,22) = 28.d0
      disb(9,23) = -1.d0
c
      disb(10,1) = 2002.d0
      disb(10,2) = -31031.d0
      disb(10,3) = 227656.d0
      disb(10,4) = -1053052.d0
      disb(10,5) = 3453450.d0
      disb(10,6) = -8565557.d0
      disb(10,7) = 16749096.d0
      disb(10,8) = -26590746.d0
      disb(10,9) = 35039382.d0
      disb(10,10) = -38973625.d0
      disb(10,11) = 37043384.d0
      disb(10,12) = -30325477.d0
      disb(10,13) = 21459802.d0
      disb(10,14) = -13122109.d0
      disb(10,15) = 6906900.d0
      disb(10,16) = -3108105.d0
      disb(10,17) = 1184040.d0
      disb(10,18) = -376740.d0
      disb(10,19) = 98280.d0
      disb(10,20) = -20475.d0
      disb(10,21) = 3276.d0
      disb(10,22) = -378.d0
      disb(10,23) = 28.d0
      disb(10,24) = -1.d0
c
      disb(11,1) = -1001.d0
      disb(11,2) = 16016.d0
      disb(11,3) = -122122.d0
      disb(11,4) = 592020.d0
      disb(11,5) = -2055053.d0
      disb(11,6) = 5457452.d0
      disb(11,7) = -11571560.d0
      disb(11,8) = 20184528.d0
      disb(11,9) = -29596749.d0
      disb(11,10) = 37043384.d0
      disb(11,11) = -39975626.d0
      disb(11,12) = 37407748.d0
      disb(11,13) = -30416568.d0
      disb(11,14) = 21473816.d0
      disb(11,15) = -13123110.d0
      disb(11,16) = 6906900.d0
      disb(11,17) = -3108105.d0
      disb(11,18) = 1184040.d0
      disb(11,19) = -376740.d0
      disb(11,20) = 98280.d0
      disb(11,21) = -20475.d0
      disb(11,22) = 3276.d0
      disb(11,23) = -378.d0
      disb(11,24) = 28.d0
      disb(11,25) = -1.d0
c
      disb(12,1) = 364.d0
      disb(12,2) = -6097.d0
      disb(12,3) = 49140.d0
      disb(12,4) = -254618.d0
      disb(12,5) = 956384.d0
      disb(12,6) = -2783781.d0
      disb(12,7) = 6550544.d0
      disb(12,8) = -12820808.d0
      disb(12,9) = 21277620.d0
      disb(12,10) = -30325477.d0
      disb(12,11) = 37407748.d0
      disb(12,12) = -40108122.d0
      disb(12,13) = 37440872.d0
      disb(12,14) = -30421664.d0
      disb(12,15) = 21474180.d0
      disb(12,16) = -13123110.d0
      disb(12,17) = 6906900.d0
      disb(12,18) = -3108105.d0
      disb(12,19) = 1184040.d0
      disb(12,20) = -376740.d0
      disb(12,21) = 98280.d0
      disb(12,22) = -20475.d0
      disb(12,23) = 3276.d0
      disb(12,24) = -378.d0
      disb(12,25) = 28.d0
      disb(12,26) = -1.d0
c
      disb(13,1) = -91.d0
      disb(13,2) = 1638.d0
      disb(13,3) = -14378.d0
      disb(13,4) = 82264.d0
      disb(13,5) = -345709.d0
      disb(13,6) = 1138566.d0
      disb(13,7) = -3057054.d0
      disb(13,8) = 6862856.d0
      disb(13,9) = -13094081.d0
      disb(13,10) = 21459802.d0
      disb(13,11) = -30416568.d0
      disb(13,12) = 37440872.d0
      disb(13,13) = -40116403.d0
      disb(13,14) = 37442146.d0
      disb(13,15) = -30421755.d0
      disb(13,16) = 21474180.d0
      disb(13,17) = -13123110.d0
      disb(13,18) = 6906900.d0
      disb(13,19) = -3108105.d0
      disb(13,20) = 1184040.d0
      disb(13,21) = -376740.d0
      disb(13,22) = 98280.d0
      disb(13,23) = -20475.d0
      disb(13,24) = 3276.d0
      disb(13,25) = -378.d0
      disb(13,26) = 28.d0
      disb(13,27) = -1.d0
c
      disb(14,1) = 14.d0
      disb(14,2) = -287.d0
      disb(14,3) = 2912.d0
      disb(14,4) = -19474.d0
      disb(14,5) = 96278.d0
      disb(14,6) = -373737.d0
      disb(14,7) = 1180608.d0
      disb(14,8) = -3105102.d0
      disb(14,9) = 6904898.d0
      disb(14,10) = -13122109.d0
      disb(14,11) = 21473816.d0
      disb(14,12) = -30421664.d0
      disb(14,13) = 37442146.d0
      disb(14,14) = -40116599.d0
      disb(14,15) = 37442160.d0
      disb(14,16) = -30421755.d0
      disb(14,17) = 21474180.d0
      disb(14,18) = -13123110.d0
      disb(14,19) = 6906900.d0
      disb(14,20) = -3108105.d0
      disb(14,21) = 1184040.d0
      disb(14,22) = -376740.d0
      disb(14,23) = 98280.d0
      disb(14,24) = -20475.d0
      disb(14,25) = 3276.d0
      disb(14,26) = -378.d0
      disb(14,27) = 28.d0
      disb(14,28) = -1.d0
c
      elseif(iord.eq.30)then
c 
c 30th order inner with 15th order boundaries
c ------------------------------------------
c
      disi(1) = 155117520.d0
      disi(2) = -145422675.d0
      disi(3) = 119759850.d0
      disi(4) = -86493225.d0
      disi(5) = 54627300.d0
      disi(6) = -30045015.d0
      disi(7) = 14307150.d0
      disi(8) = -5852925.d0
      disi(9) = 2035800.d0
      disi(10) = -593775.d0
      disi(11) = 142506.d0
      disi(12) = -27405.d0
      disi(13) = 4060.d0
      disi(14) = -435.d0
      disi(15) = 30.d0
      disi(16) = -1.d0
c
      disb(1,1) = 1.d0
      disb(1,2) = -15.d0
      disb(1,3) = 105.d0
      disb(1,4) = -455.d0
      disb(1,5) = 1365.d0
      disb(1,6) = -3003.d0
      disb(1,7) = 5005.d0
      disb(1,8) = -6435.d0
      disb(1,9) = 6435.d0
      disb(1,10) = -5005.d0
      disb(1,11) = 3003.d0
      disb(1,12) = -1365.d0
      disb(1,13) = 455.d0
      disb(1,14) = -105.d0
      disb(1,15) = 15.d0
      disb(1,16) = -1.d0
c
      disb(2,1) = -15.d0
      disb(2,2) = 226.d0
      disb(2,3) = -1590.d0
      disb(2,4) = 6930.d0
      disb(2,5) = -20930.d0
      disb(2,6) = 46410.d0
      disb(2,7) = -78078.d0
      disb(2,8) = 101530.d0
      disb(2,9) = -102960.d0
      disb(2,10) = 81510.d0
      disb(2,11) = -50050.d0
      disb(2,12) = 23478.d0
      disb(2,13) = -8190.d0
      disb(2,14) = 2030.d0
      disb(2,15) = -330.d0
      disb(2,16) = 30.d0
      disb(2,17) = -1.d0
c
      disb(3,1) = 105.d0
      disb(3,2) = -1590.d0
      disb(3,3) = 11251.d0
      disb(3,4) = -49365.d0
      disb(3,5) = 150255.d0
      disb(3,6) = -336245.d0
      disb(3,7) = 571935.d0
      disb(3,8) = -753753.d0
      disb(3,9) = 777205.d0
      disb(3,10) = -628485.d0
      disb(3,11) = 396825.d0
      disb(3,12) = -193375.d0
      disb(3,13) = 71253.d0
      disb(3,14) = -19215.d0
      disb(3,15) = 3605.d0
      disb(3,16) = -435.d0
      disb(3,17) = 30.d0
      disb(3,18) = -1.d0
c
      disb(4,1) = -455.d0
      disb(4,2) = 6930.d0
      disb(4,3) = -49365.d0
      disb(4,4) = 218276.d0
      disb(4,5) = -670440.d0
      disb(4,6) = 1516620.d0
      disb(4,7) = -2613520.d0
      disb(4,8) = 3499860.d0
      disb(4,9) = -3681678.d0
      disb(4,10) = 3054480.d0
      disb(4,11) = -1994850.d0
      disb(4,12) = 1017900.d0
      disb(4,13) = -400400.d0
      disb(4,14) = 119028.d0
      disb(4,15) = -26040.d0
      disb(4,16) = 4060.d0
      disb(4,17) = -435.d0
      disb(4,18) = 30.d0
      disb(4,19) = -1.d0
c
      disb(5,1) = 1365.d0
      disb(5,2) = -20930.d0
      disb(5,3) = 150255.d0
      disb(5,4) = -670440.d0
      disb(5,5) = 2081501.d0
      disb(5,6) = -4769535.d0
      disb(5,7) = 8348445.d0
      disb(5,8) = -11397295.d0
      disb(5,9) = 12283635.d0
      disb(5,10) = -10513503.d0
      disb(5,11) = 7153575.d0
      disb(5,12) = -3858075.d0
      disb(5,13) = 1638975.d0
      disb(5,14) = -543725.d0
      disb(5,15) = 139503.d0
      disb(5,16) = -27405.d0
      disb(5,17) = 4060.d0
      disb(5,18) = -435.d0
      disb(5,19) = 30.d0
      disb(5,20) = -1.d0
c
      disb(6,1) = -3003.d0
      disb(6,2) = 46410.d0
      disb(6,3) = -336245.d0
      disb(6,4) = 1516620.d0
      disb(6,5) = -4769535.d0
      disb(6,6) = 11099510.d0
      disb(6,7) = -19799550.d0
      disb(6,8) = 27672750.d0
      disb(6,9) = -30721600.d0
      disb(6,10) = 27313650.d0
      disb(6,11) = -19531512.d0
      disb(6,12) = 11252670.d0
      disb(6,13) = -5224440.d0
      disb(6,14) = 1954290.d0
      disb(6,15) = -588770.d0
      disb(6,16) = 142506.d0
      disb(6,17) = -27405.d0
      disb(6,18) = 4060.d0
      disb(6,19) = -435.d0
      disb(6,20) = 30.d0
      disb(6,21) = -1.d0
c
      disb(7,1) = 5005.d0
      disb(7,2) = -78078.d0
      disb(7,3) = 571935.d0
      disb(7,4) = -2613520.d0
      disb(7,5) = 8348445.d0
      disb(7,6) = -19799550.d0
      disb(7,7) = 36149535.d0
      disb(7,8) = -52006725.d0
      disb(7,9) = 59879925.d0
      disb(7,10) = -55771625.d0
      disb(7,11) = 42343665.d0
      disb(7,12) = -26363337.d0
      disb(7,13) = 13529945.d0
      disb(7,14) = -5749965.d0
      disb(7,15) = 2029365.d0
      disb(7,16) = -593775.d0
      disb(7,17) = 142506.d0
      disb(7,18) = -27405.d0
      disb(7,19) = 4060.d0
      disb(7,20) = -435.d0
      disb(7,21) = 30.d0
      disb(7,22) = -1.d0
c
      disb(8,1) = -6435.d0
      disb(8,2) = 101530.d0
      disb(8,3) = -753753.d0
      disb(8,4) = 3499860.d0
      disb(8,5) = -11397295.d0
      disb(8,6) = 27672750.d0
      disb(8,7) = -52006725.d0
      disb(8,8) = 77558760.d0
      disb(8,9) = -93415950.d0
      disb(8,10) = 92087100.d0
      disb(8,11) = -75095930.d0
      disb(8,12) = 51127440.d0
      disb(8,13) = -29291262.d0
      disb(8,14) = 14205620.d0
      disb(8,15) = -5846490.d0
      disb(8,16) = 2035800.d0
      disb(8,17) = -593775.d0
      disb(8,18) = 142506.d0
      disb(8,19) = -27405.d0
      disb(8,20) = 4060.d0
      disb(8,21) = -435.d0
      disb(8,22) = 30.d0
      disb(8,23) = -1.d0
c
      disb(9,1) = 6435.d0
      disb(9,2) = -102960.d0
      disb(9,3) = 777205.d0
      disb(9,4) = -3681678.d0
      disb(9,5) = 12283635.d0
      disb(9,6) = -30721600.d0
      disb(9,7) = 59879925.d0
      disb(9,8) = -93415950.d0
      disb(9,9) = 118967985.d0
      disb(9,10) = -125623125.d0
      disb(9,11) = 111411405.d0
      disb(9,12) = -83879705.d0
      disb(9,13) = 54055365.d0
      disb(9,14) = -29966937.d0
      disb(9,15) = 14302145.d0
      disb(9,16) = -5852925.d0
      disb(9,17) = 2035800.d0
      disb(9,18) = -593775.d0
      disb(9,19) = 142506.d0
      disb(9,20) = -27405.d0
      disb(9,21) = 4060.d0
      disb(9,22) = -435.d0
      disb(9,23) = 30.d0
      disb(9,24) = -1.d0
c
      disb(10,1) = -5005.d0
      disb(10,2) = 81510.d0
      disb(10,3) = -628485.d0
      disb(10,4) = 3054480.d0
      disb(10,5) = -10513503.d0
      disb(10,6) = 27313650.d0
      disb(10,7) = -55771625.d0
      disb(10,8) = 92087100.d0
      disb(10,9) = -125623125.d0
      disb(10,10) = 144018010.d0
      disb(10,11) = -140653140.d0
      disb(10,12) = 118243230.d0
      disb(10,13) = -86156980.d0
      disb(10,14) = 54580890.d0
      disb(10,15) = -30042012.d0
      disb(10,16) = 14307150.d0
      disb(10,17) = -5852925.d0
      disb(10,18) = 2035800.d0
      disb(10,19) = -593775.d0
      disb(10,20) = 142506.d0
      disb(10,21) = -27405.d0
      disb(10,22) = 4060.d0
      disb(10,23) = -435.d0
      disb(10,24) = 30.d0
      disb(10,25) = -1.d0
c
      disb(11,1) = 3003.d0
      disb(11,2) = -50050.d0
      disb(11,3) = 396825.d0
      disb(11,4) = -1994850.d0
      disb(11,5) = 7153575.d0
      disb(11,6) = -19531512.d0
      disb(11,7) = 42343665.d0
      disb(11,8) = -75095930.d0
      disb(11,9) = 111411405.d0
      disb(11,10) = -140653140.d0
      disb(11,11) = 153036019.d0
      disb(11,12) = -144752235.d0
      disb(11,13) = 119609595.d0
      disb(11,14) = -86472295.d0
      disb(11,15) = 54625935.d0
      disb(11,16) = -30045015.d0
      disb(11,17) = 14307150.d0
      disb(11,18) = -5852925.d0
      disb(11,19) = 2035800.d0
      disb(11,20) = -593775.d0
      disb(11,21) = 142506.d0
      disb(11,22) = -27405.d0
      disb(11,23) = 4060.d0
      disb(11,24) = -435.d0
      disb(11,25) = 30.d0
      disb(11,26) = -1.d0
c
      disb(12,1) = -1365.d0
      disb(12,2) = 23478.d0
      disb(12,3) = -193375.d0
      disb(12,4) = 1017900.d0
      disb(12,5) = -3858075.d0
      disb(12,6) = 11252670.d0
      disb(12,7) = -26363337.d0
      disb(12,8) = 51127440.d0
      disb(12,9) = -83879705.d0
      disb(12,10) = 118243230.d0
      disb(12,11) = -144752235.d0
      disb(12,12) = 154899244.d0
      disb(12,13) = -145373310.d0
      disb(12,14) = 119752920.d0
      disb(12,15) = -86492770.d0
      disb(12,16) = 54627300.d0
      disb(12,17) = -30045015.d0
      disb(12,18) = 14307150.d0
      disb(12,19) = -5852925.d0
      disb(12,20) = 2035800.d0
      disb(12,21) = -593775.d0
      disb(12,22) = 142506.d0
      disb(12,23) = -27405.d0
      disb(12,24) = 4060.d0
      disb(12,25) = -435.d0
      disb(12,26) = 30.d0
      disb(12,27) = -1.d0
c
      disb(13,1) = 455.d0
      disb(13,2) = -8190.d0
      disb(13,3) = 71253.d0
      disb(13,4) = -400400.d0
      disb(13,5) = 1638975.d0
      disb(13,6) = -5224440.d0
      disb(13,7) = 13529945.d0
      disb(13,8) = -29291262.d0
      disb(13,9) = 54055365.d0
      disb(13,10) = -86156980.d0
      disb(13,11) = 119609595.d0
      disb(13,12) = -145373310.d0
      disb(13,13) = 155106269.d0
      disb(13,14) = -145421085.d0
      disb(13,15) = 119759745.d0
      disb(13,16) = -86493225.d0
      disb(13,17) = 54627300.d0
      disb(13,18) = -30045015.d0
      disb(13,19) = 14307150.d0
      disb(13,20) = -5852925.d0
      disb(13,21) = 2035800.d0
      disb(13,22) = -593775.d0
      disb(13,23) = 142506.d0
      disb(13,24) = -27405.d0
      disb(13,25) = 4060.d0
      disb(13,26) = -435.d0
      disb(13,27) = 30.d0
      disb(13,28) = -1.d0
c
      disb(14,1) = -105.d0
      disb(14,2) = 2030.d0
      disb(14,3) = -19215.d0
      disb(14,4) = 119028.d0
      disb(14,5) = -543725.d0
      disb(14,6) = 1954290.d0
      disb(14,7) = -5749965.d0
      disb(14,8) = 14205620.d0
      disb(14,9) = -29966937.d0
      disb(14,10) = 54580890.d0
      disb(14,11) = -86472295.d0
      disb(14,12) = 119752920.d0
      disb(14,13) = -145421085.d0
      disb(14,14) = 155117294.d0
      disb(14,15) = -145422660.d0
      disb(14,16) = 119759850.d0
      disb(14,17) = -86493225.d0
      disb(14,18) = 54627300.d0
      disb(14,19) = -30045015.d0
      disb(14,20) = 14307150.d0
      disb(14,21) = -5852925.d0
      disb(14,22) = 2035800.d0
      disb(14,23) = -593775.d0
      disb(14,24) = 142506.d0
      disb(14,25) = -27405.d0
      disb(14,26) = 4060.d0
      disb(14,27) = -435.d0
      disb(14,28) = 30.d0
      disb(14,29) = -1.d0
c
      disb(15,1) = 15.d0
      disb(15,2) = -330.d0
      disb(15,3) = 3605.d0
      disb(15,4) = -26040.d0
      disb(15,5) = 139503.d0
      disb(15,6) = -588770.d0
      disb(15,7) = 2029365.d0
      disb(15,8) = -5846490.d0
      disb(15,9) = 14302145.d0
      disb(15,10) = -30042012.d0
      disb(15,11) = 54625935.d0
      disb(15,12) = -86492770.d0
      disb(15,13) = 119759745.d0
      disb(15,14) = -145422660.d0
      disb(15,15) = 155117519.d0
      disb(15,16) = -145422675.d0
      disb(15,17) = 119759850.d0
      disb(15,18) = -86493225.d0
      disb(15,19) = 54627300.d0
      disb(15,20) = -30045015.d0
      disb(15,21) = 14307150.d0
      disb(15,22) = -5852925.d0
      disb(15,23) = 2035800.d0
      disb(15,24) = -593775.d0
      disb(15,25) = 142506.d0
      disb(15,26) = -27405.d0
      disb(15,27) = 4060.d0
      disb(15,28) = -435.d0
      disb(15,29) = 30.d0
      disb(15,30) = -1.d0
c
      elseif(iord.eq.32)then
c 
c 32nd order inner with 16th order boundaries
c ------------------------------------------
c
      disi(1) = -601080390.d0
      disi(2) = 565722720.d0
      disi(3) = -471435600.d0
      disi(4) = 347373600.d0
      disi(5) = -225792840.d0
      disi(6) = 129024480.d0
      disi(7) = -64512240.d0
      disi(8) = 28048800.d0
      disi(9) = -10518300.d0
      disi(10) = 3365856.d0
      disi(11) = -906192.d0
      disi(12) = 201376.d0
      disi(13) = -35960.d0
      disi(14) = 4960.d0
      disi(15) = -496.d0
      disi(16) = 32.d0
      disi(17) = -1.d0
c
      disb(1,1) = -1.d0
      disb(1,2) = 16.d0
      disb(1,3) = -120.d0
      disb(1,4) = 560.d0
      disb(1,5) = -1820.d0
      disb(1,6) = 4368.d0
      disb(1,7) = -8008.d0
      disb(1,8) = 11440.d0
      disb(1,9) = -12870.d0
      disb(1,10) = 11440.d0
      disb(1,11) = -8008.d0
      disb(1,12) = 4368.d0
      disb(1,13) = -1820.d0
      disb(1,14) = 560.d0
      disb(1,15) = -120.d0
      disb(1,16) = 16.d0
      disb(1,17) = -1.d0
c
      disb(2,1) = 16.d0
      disb(2,2) = -257.d0
      disb(2,3) = 1936.d0
      disb(2,4) = -9080.d0
      disb(2,5) = 29680.d0
      disb(2,6) = -71708.d0
      disb(2,7) = 132496.d0
      disb(2,8) = -191048.d0
      disb(2,9) = 217360.d0
      disb(2,10) = -195910.d0
      disb(2,11) = 139568.d0
      disb(2,12) = -77896.d0
      disb(2,13) = 33488.d0
      disb(2,14) = -10780.d0
      disb(2,15) = 2480.d0
      disb(2,16) = -376.d0
      disb(2,17) = 32.d0
      disb(2,18) = -1.d0
c
      disb(3,1) = -120.d0
      disb(3,2) = 1936.d0
      disb(3,3) = -14657.d0
      disb(3,4) = 69136.d0
      disb(3,5) = -227480.d0
      disb(3,6) = 553840.d0
      disb(3,7) = -1032668.d0
      disb(3,8) = 1505296.d0
      disb(3,9) = -1735448.d0
      disb(3,10) = 1590160.d0
      disb(3,11) = -1156870.d0
      disb(3,12) = 663728.d0
      disb(3,13) = -296296.d0
      disb(3,14) = 100688.d0
      disb(3,15) = -25180.d0
      disb(3,16) = 4400.d0
      disb(3,17) = -496.d0
      disb(3,18) = 32.d0
      disb(3,19) = -1.d0
c
      disb(4,1) = 560.d0
      disb(4,2) = -9080.d0
      disb(4,3) = 69136.d0
      disb(4,4) = -328257.d0
      disb(4,5) = 1088336.d0
      disb(4,6) = -2673560.d0
      disb(4,7) = 5038320.d0
      disb(4,8) = -7439068.d0
      disb(4,9) = 8712496.d0
      disb(4,10) = -8141848.d0
      disb(4,11) = 6074640.d0
      disb(4,12) = -3602950.d0
      disb(4,13) = 1682928.d0
      disb(4,14) = -609896.d0
      disb(4,15) = 167888.d0
      disb(4,16) = -34140.d0
      disb(4,17) = 4960.d0
      disb(4,18) = -496.d0
      disb(4,19) = 32.d0
      disb(4,20) = -1.d0
c
      disb(5,1) = -1820.d0
      disb(5,2) = 29680.d0
      disb(5,3) = -227480.d0
      disb(5,4) = 1088336.d0
      disb(5,5) = -3640657.d0
      disb(5,6) = 9038096.d0
      disb(5,7) = -17248120.d0
      disb(5,8) = 25859120.d0
      disb(5,9) = -30862468.d0
      disb(5,10) = 29533296.d0
      disb(5,11) = -22716408.d0
      disb(5,12) = 14024400.d0
      disb(5,13) = -6915350.d0
      disb(5,14) = 2702128.d0
      disb(5,15) = -828296.d0
      disb(5,16) = 197008.d0
      disb(5,17) = -35960.d0
      disb(5,18) = 4960.d0
      disb(5,19) = -496.d0
      disb(5,20) = 32.d0
      disb(5,21) = -1.d0
c
      disb(6,1) = 4368.d0
      disb(6,2) = -71708.d0
      disb(6,3) = 553840.d0
      disb(6,4) = -2673560.d0
      disb(6,5) = 9038096.d0
      disb(6,6) = -22720081.d0
      disb(6,7) = 44017040.d0
      disb(6,8) = -67218040.d0
      disb(6,9) = 82075280.d0
      disb(6,10) = -80832388.d0
      disb(6,11) = 64512240.d0
      disb(6,12) = -41795832.d0
      disb(6,13) = 21974160.d0
      disb(6,14) = -9361430.d0
      disb(6,15) = 3226288.d0
      disb(6,16) = -898184.d0
      disb(6,17) = 201376.d0
      disb(6,18) = -35960.d0
      disb(6,19) = 4960.d0
      disb(6,20) = -496.d0
      disb(6,21) = 32.d0
      disb(6,22) = -1.d0
c
      disb(7,1) = -8008.d0
      disb(7,2) = 132496.d0
      disb(7,3) = -1032668.d0
      disb(7,4) = 5038320.d0
      disb(7,5) = -17248120.d0
      disb(7,6) = 44017040.d0
      disb(7,7) = -86848145.d0
      disb(7,8) = 135628560.d0
      disb(7,9) = -170281000.d0
      disb(7,10) = 173686800.d0
      disb(7,11) = -144960452.d0
      disb(7,12) = 99491184.d0
      disb(7,13) = -56370392.d0
      disb(7,14) = 26458640.d0
      disb(7,15) = -10322390.d0
      disb(7,16) = 3354416.d0
      disb(7,17) = -906192.d0
      disb(7,18) = 201376.d0
      disb(7,19) = -35960.d0
      disb(7,20) = 4960.d0
      disb(7,21) = -496.d0
      disb(7,22) = 32.d0
      disb(7,23) = -1.d0
c
      disb(8,1) = 11440.d0
      disb(8,2) = -191048.d0
      disb(8,3) = 1505296.d0
      disb(8,4) = -7439068.d0
      disb(8,5) = 25859120.d0
      disb(8,6) = -67218040.d0
      disb(8,7) = 135628560.d0
      disb(8,8) = -217721745.d0
      disb(8,9) = 282861360.d0
      disb(8,10) = -301154600.d0
      disb(8,11) = 265298320.d0
      disb(8,12) = -194930372.d0
      disb(8,13) = 120311984.d0
      disb(8,14) = -62776792.d0
      disb(8,15) = 27831440.d0
      disb(8,16) = -10505430.d0
      disb(8,17) = 3365856.d0
      disb(8,18) = -906192.d0
      disb(8,19) = 201376.d0
      disb(8,20) = -35960.d0
      disb(8,21) = 4960.d0
      disb(8,22) = -496.d0
      disb(8,23) = 32.d0
      disb(8,24) = -1.d0
c
      disb(9,1) = -12870.d0
      disb(9,2) = 217360.d0
      disb(9,3) = -1735448.d0
      disb(9,4) = 8712496.d0
      disb(9,5) = -30862468.d0
      disb(9,6) = 82075280.d0
      disb(9,7) = -170281000.d0
      disb(9,8) = 282861360.d0
      disb(9,9) = -383358645.d0
      disb(9,10) = 430094160.d0
      disb(9,11) = -404217560.d0
      disb(9,12) = 321514480.d0
      disb(9,13) = -218353772.d0
      disb(9,14) = 127519184.d0
      disb(9,15) = -64321192.d0
      disb(9,16) = 28037360.d0
      disb(9,17) = -10518300.d0
      disb(9,18) = 3365856.d0
      disb(9,19) = -906192.d0
      disb(9,20) = 201376.d0
      disb(9,21) = -35960.d0
      disb(9,22) = 4960.d0
      disb(9,23) = -496.d0
      disb(9,24) = 32.d0
      disb(9,25) = -1.d0
c
      disb(10,1) = 11440.d0
      disb(10,2) = -195910.d0
      disb(10,3) = 1590160.d0
      disb(10,4) = -8141848.d0
      disb(10,5) = 29533296.d0
      disb(10,6) = -80832388.d0
      disb(10,7) = 173686800.d0
      disb(10,8) = -301154600.d0
      disb(10,9) = 430094160.d0
      disb(10,10) = -514232245.d0
      disb(10,11) = 521705680.d0
      disb(10,12) = -454187480.d0
      disb(10,13) = 342335280.d0
      disb(10,14) = -224760172.d0
      disb(10,15) = 128891984.d0
      disb(10,16) = -64504232.d0
      disb(10,17) = 28048800.d0
      disb(10,18) = -10518300.d0
      disb(10,19) = 3365856.d0
      disb(10,20) = -906192.d0
      disb(10,21) = 201376.d0
      disb(10,22) = -35960.d0
      disb(10,23) = 4960.d0
      disb(10,24) = -496.d0
      disb(10,25) = 32.d0
      disb(10,26) = -1.d0
c
      disb(11,1) = -8008.d0
      disb(11,2) = 139568.d0
      disb(11,3) = -1156870.d0
      disb(11,4) = 6074640.d0
      disb(11,5) = -22716408.d0
      disb(11,6) = 64512240.d0
      disb(11,7) = -144960452.d0
      disb(11,8) = 265298320.d0
      disb(11,9) = -404217560.d0
      disb(11,10) = 521705680.d0
      disb(11,11) = -578360309.d0
      disb(11,12) = 556684624.d0
      disb(11,13) = -468762040.d0
      disb(11,14) = 346819760.d0
      disb(11,15) = -225721132.d0
      disb(11,16) = 129020112.d0
      disb(11,17) = -64512240.d0
      disb(11,18) = 28048800.d0
      disb(11,19) = -10518300.d0
      disb(11,20) = 3365856.d0
      disb(11,21) = -906192.d0
      disb(11,22) = 201376.d0
      disb(11,23) = -35960.d0
      disb(11,24) = 4960.d0
      disb(11,25) = -496.d0
      disb(11,26) = 32.d0
      disb(11,27) = -1.d0
c
      disb(12,1) = 4368.d0
      disb(12,2) = -77896.d0
      disb(12,3) = 663728.d0
      disb(12,4) = -3602950.d0
      disb(12,5) = 14024400.d0
      disb(12,6) = -41795832.d0
      disb(12,7) = 99491184.d0
      disb(12,8) = -194930372.d0
      disb(12,9) = 321514480.d0
      disb(12,10) = -454187480.d0
      disb(12,11) = 556684624.d0
      disb(12,12) = -597439733.d0
      disb(12,13) = 564634384.d0
      disb(12,14) = -471208120.d0
      disb(12,15) = 347343920.d0
      disb(12,16) = -225791020.d0
      disb(12,17) = 129024480.d0
      disb(12,18) = -64512240.d0
      disb(12,19) = 28048800.d0
      disb(12,20) = -10518300.d0
      disb(12,21) = 3365856.d0
      disb(12,22) = -906192.d0
      disb(12,23) = 201376.d0
      disb(12,24) = -35960.d0
      disb(12,25) = 4960.d0
      disb(12,26) = -496.d0
      disb(12,27) = 32.d0
      disb(12,28) = -1.d0
c
      disb(13,1) = -1820.d0
      disb(13,2) = 33488.d0
      disb(13,3) = -296296.d0
      disb(13,4) = 1682928.d0
      disb(13,5) = -6915350.d0
      disb(13,6) = 21974160.d0
      disb(13,7) = -56370392.d0
      disb(13,8) = 120311984.d0
      disb(13,9) = -218353772.d0
      disb(13,10) = 342335280.d0
      disb(13,11) = -468762040.d0
      disb(13,12) = 564634384.d0
      disb(13,13) = -600752133.d0
      disb(13,14) = 565653584.d0
      disb(13,15) = -471426520.d0
      disb(13,16) = 347373040.d0
      disb(13,17) = -225792840.d0
      disb(13,18) = 129024480.d0
      disb(13,19) = -64512240.d0
      disb(13,20) = 28048800.d0
      disb(13,21) = -10518300.d0
      disb(13,22) = 3365856.d0
      disb(13,23) = -906192.d0
      disb(13,24) = 201376.d0
      disb(13,25) = -35960.d0
      disb(13,26) = 4960.d0
      disb(13,27) = -496.d0
      disb(13,28) = 32.d0
      disb(13,29) = -1.d0
c
      disb(14,1) = 560.d0
      disb(14,2) = -10780.d0
      disb(14,3) = 100688.d0
      disb(14,4) = -609896.d0
      disb(14,5) = 2702128.d0
      disb(14,6) = -9361430.d0
      disb(14,7) = 26458640.d0
      disb(14,8) = -62776792.d0
      disb(14,9) = 127519184.d0
      disb(14,10) = -224760172.d0
      disb(14,11) = 346819760.d0
      disb(14,12) = -471208120.d0
      disb(14,13) = 565653584.d0
      disb(14,14) = -601065733.d0
      disb(14,15) = 565720784.d0
      disb(14,16) = -471435480.d0
      disb(14,17) = 347373600.d0
      disb(14,18) = -225792840.d0
      disb(14,19) = 129024480.d0
      disb(14,20) = -64512240.d0
      disb(14,21) = 28048800.d0
      disb(14,22) = -10518300.d0
      disb(14,23) = 3365856.d0
      disb(14,24) = -906192.d0
      disb(14,25) = 201376.d0
      disb(14,26) = -35960.d0
      disb(14,27) = 4960.d0
      disb(14,28) = -496.d0
      disb(14,29) = 32.d0
      disb(14,30) = -1.d0
c
      disb(15,1) = -120.d0
      disb(15,2) = 2480.d0
      disb(15,3) = -25180.d0
      disb(15,4) = 167888.d0
      disb(15,5) = -828296.d0
      disb(15,6) = 3226288.d0
      disb(15,7) = -10322390.d0
      disb(15,8) = 27831440.d0
      disb(15,9) = -64321192.d0
      disb(15,10) = 128891984.d0
      disb(15,11) = -225721132.d0
      disb(15,12) = 347343920.d0
      disb(15,13) = -471426520.d0
      disb(15,14) = 565720784.d0
      disb(15,15) = -601080133.d0
      disb(15,16) = 565722704.d0
      disb(15,17) = -471435600.d0
      disb(15,18) = 347373600.d0
      disb(15,19) = -225792840.d0
      disb(15,20) = 129024480.d0
      disb(15,21) = -64512240.d0
      disb(15,22) = 28048800.d0
      disb(15,23) = -10518300.d0
      disb(15,24) = 3365856.d0
      disb(15,25) = -906192.d0
      disb(15,26) = 201376.d0
      disb(15,27) = -35960.d0
      disb(15,28) = 4960.d0
      disb(15,29) = -496.d0
      disb(15,30) = 32.d0
      disb(15,31) = -1.d0
c
      disb(16,1) = 16.d0
      disb(16,2) = -376.d0
      disb(16,3) = 4400.d0
      disb(16,4) = -34140.d0
      disb(16,5) = 197008.d0
      disb(16,6) = -898184.d0
      disb(16,7) = 3354416.d0
      disb(16,8) = -10505430.d0
      disb(16,9) = 28037360.d0
      disb(16,10) = -64504232.d0
      disb(16,11) = 129020112.d0
      disb(16,12) = -225791020.d0
      disb(16,13) = 347373040.d0
      disb(16,14) = -471435480.d0
      disb(16,15) = 565722704.d0
      disb(16,16) = -601080389.d0
      disb(16,17) = 565722720.d0
      disb(16,18) = -471435600.d0
      disb(16,19) = 347373600.d0
      disb(16,20) = -225792840.d0
      disb(16,21) = 129024480.d0
      disb(16,22) = -64512240.d0
      disb(16,23) = 28048800.d0
      disb(16,24) = -10518300.d0
      disb(16,25) = 3365856.d0
      disb(16,26) = -906192.d0
      disb(16,27) = 201376.d0
      disb(16,28) = -35960.d0
      disb(16,29) = 4960.d0
      disb(16,30) = -496.d0
      disb(16,31) = 32.d0
      disb(16,32) = -1.d0
c
      elseif(iord.eq.34)then
c 
c 34th order inner with 17th order boundaries
c ------------------------------------------
c
      disi(1) = 2333606220.d0
      disi(2) = -2203961430.d0
      disi(3) = 1855967520.d0
      disi(4) = -1391975640.d0
      disi(5) = 927983760.d0
      disi(6) = -548354040.d0
      disi(7) = 286097760.d0
      disi(8) = -131128140.d0
      disi(9) = 52451256.d0
      disi(10) = -18156204.d0
      disi(11) = 5379616.d0
      disi(12) = -1344904.d0
      disi(13) = 278256.d0
      disi(14) = -46376.d0
      disi(15) = 5984.d0
      disi(16) = -561.d0
      disi(17) = 34.d0
      disi(18) = -1.d0
c
      disb(1,1) = 1.d0
      disb(1,2) = -17.d0
      disb(1,3) = 136.d0
      disb(1,4) = -680.d0
      disb(1,5) = 2380.d0
      disb(1,6) = -6188.d0
      disb(1,7) = 12376.d0
      disb(1,8) = -19448.d0
      disb(1,9) = 24310.d0
      disb(1,10) = -24310.d0
      disb(1,11) = 19448.d0
      disb(1,12) = -12376.d0
      disb(1,13) = 6188.d0
      disb(1,14) = -2380.d0
      disb(1,15) = 680.d0
      disb(1,16) = -136.d0
      disb(1,17) = 17.d0
      disb(1,18) = -1.d0
c
      disb(2,1) = -17.d0
      disb(2,2) = 290.d0
      disb(2,3) = -2329.d0
      disb(2,4) = 11696.d0
      disb(2,5) = -41140.d0
      disb(2,6) = 107576.d0
      disb(2,7) = -216580.d0
      disb(2,8) = 342992.d0
      disb(2,9) = -432718.d0
      disb(2,10) = 437580.d0
      disb(2,11) = -354926.d0
      disb(2,12) = 229840.d0
      disb(2,13) = -117572.d0
      disb(2,14) = 46648.d0
      disb(2,15) = -13940.d0
      disb(2,16) = 2992.d0
      disb(2,17) = -425.d0
      disb(2,18) = 34.d0
      disb(2,19) = -1.d0
c
      disb(3,1) = 136.d0
      disb(3,2) = -2329.d0
      disb(3,3) = 18786.d0
      disb(3,4) = -94809.d0
      disb(3,5) = 335376.d0
      disb(3,6) = -882708.d0
      disb(3,7) = 1790712.d0
      disb(3,8) = -2861508.d0
      disb(3,9) = 3649152.d0
      disb(3,10) = -3738878.d0
      disb(3,11) = 3082508.d0
      disb(3,12) = -2038062.d0
      disb(3,13) = 1071408.d0
      disb(3,14) = -441252.d0
      disb(3,15) = 139128.d0
      disb(3,16) = -32436.d0
      disb(3,17) = 5304.d0
      disb(3,18) = -561.d0
      disb(3,19) = 34.d0
      disb(3,20) = -1.d0
c
      disb(4,1) = -680.d0
      disb(4,2) = 11696.d0
      disb(4,3) = -94809.d0
      disb(4,4) = 481186.d0
      disb(4,5) = -1713209.d0
      disb(4,6) = 4543216.d0
      disb(4,7) = -9298388.d0
      disb(4,8) = 15015352.d0
      disb(4,9) = -19392308.d0
      disb(4,10) = 20179952.d0
      disb(4,11) = -16963518.d0
      disb(4,12) = 11498188.d0
      disb(4,13) = -6245902.d0
      disb(4,14) = 2689808.d0
      disb(4,15) = -903652.d0
      disb(4,16) = 231608.d0
      disb(4,17) = -43996.d0
      disb(4,18) = 5984.d0
      disb(4,19) = -561.d0
      disb(4,20) = 34.d0
      disb(4,21) = -1.d0
c
      disb(5,1) = 2380.d0
      disb(5,2) = -41140.d0
      disb(5,3) = 335376.d0
      disb(5,4) = -1713209.d0
      disb(5,5) = 6145586.d0
      disb(5,6) = -16440649.d0
      disb(5,7) = 33998096.d0
      disb(5,8) = -55584628.d0
      disb(5,9) = 72873152.d0
      disb(5,10) = -77250108.d0
      disb(5,11) = 66466192.d0
      disb(5,12) = -46418398.d0
      disb(5,13) = 26225628.d0
      disb(5,14) = -11910302.d0
      disb(5,15) = 4308208.d0
      disb(5,16) = -1227332.d0
      disb(5,17) = 272068.d0
      disb(5,18) = -46376.d0
      disb(5,19) = 5984.d0
      disb(5,20) = -561.d0
      disb(5,21) = 34.d0
      disb(5,22) = -1.d0
c
      disb(6,1) = -6188.d0
      disb(6,2) = 107576.d0
      disb(6,3) = -882708.d0
      disb(6,4) = 4543216.d0
      disb(6,5) = -16440649.d0
      disb(6,6) = 44436930.d0
      disb(6,7) = -93023337.d0
      disb(6,8) = 154342320.d0
      disb(6,9) = -206014908.d0
      disb(6,10) = 223303432.d0
      disb(6,11) = -197594332.d0
      disb(6,12) = 143048880.d0
      disb(6,13) = -84709742.d0
      disb(6,14) = 40953068.d0
      disb(6,15) = -16118142.d0
      disb(6,16) = 5149776.d0
      disb(6,17) = -1332528.d0
      disb(6,18) = 278256.d0
      disb(6,19) = -46376.d0
      disb(6,20) = 5984.d0
      disb(6,21) = -561.d0
      disb(6,22) = 34.d0
      disb(6,23) = -1.d0
c
      disb(7,1) = 12376.d0
      disb(7,2) = -216580.d0
      disb(7,3) = 1790712.d0
      disb(7,4) = -9298388.d0
      disb(7,5) = 33998096.d0
      disb(7,6) = -93023337.d0
      disb(7,7) = 197602306.d0
      disb(7,8) = -333711785.d0
      disb(7,9) = 455202880.d0
      disb(7,10) = -506875468.d0
      disb(7,11) = 463991880.d0
      disb(7,12) = -350759708.d0
      disb(7,13) = 219631568.d0
      disb(7,14) = -114164622.d0
      disb(7,15) = 49368748.d0
      disb(7,16) = -17801278.d0
      disb(7,17) = 5360168.d0
      disb(7,18) = -1344904.d0
      disb(7,19) = 278256.d0
      disb(7,20) = -46376.d0
      disb(7,21) = 5984.d0
      disb(7,22) = -561.d0
      disb(7,23) = 34.d0
      disb(7,24) = -1.d0
c
      disb(8,1) = -19448.d0
      disb(8,2) = 342992.d0
      disb(8,3) = -2861508.d0
      disb(8,4) = 15015352.d0
      disb(8,5) = -55584628.d0
      disb(8,6) = 154342320.d0
      disb(8,7) = -333711785.d0
      disb(8,8) = 575827010.d0
      disb(8,9) = -806492665.d0
      disb(8,10) = 927983760.d0
      disb(8,11) = -885100172.d0
      disb(8,12) = 704680328.d0
      disb(8,13) = -471103932.d0
      disb(8,14) = 265917808.d0
      disb(8,15) = -127389262.d0
      disb(8,16) = 52013676.d0
      disb(8,17) = -18131894.d0
      disb(8,18) = 5379616.d0
      disb(8,19) = -1344904.d0
      disb(8,20) = 278256.d0
      disb(8,21) = -46376.d0
      disb(8,22) = 5984.d0
      disb(8,23) = -561.d0
      disb(8,24) = 34.d0
      disb(8,25) = -1.d0
c
      disb(9,1) = 24310.d0
      disb(9,2) = -432718.d0
      disb(9,3) = 3649152.d0
      disb(9,4) = -19392308.d0
      disb(9,5) = 72873152.d0
      disb(9,6) = -206014908.d0
      disb(9,7) = 455202880.d0
      disb(9,8) = -806492665.d0
      disb(9,9) = 1166803110.d0
      disb(9,10) = -1397468765.d0
      disb(9,11) = 1400764640.d0
      disb(9,12) = -1185960732.d0
      disb(9,13) = 855110608.d0
      disb(9,14) = -528961732.d0
      disb(9,15) = 282448608.d0
      disb(9,16) = -130695422.d0
      disb(9,17) = 52426946.d0
      disb(9,18) = -18156204.d0
      disb(9,19) = 5379616.d0
      disb(9,20) = -1344904.d0
      disb(9,21) = 278256.d0
      disb(9,22) = -46376.d0
      disb(9,23) = 5984.d0
      disb(9,24) = -561.d0
      disb(9,25) = 34.d0
      disb(9,26) = -1.d0
c
      disb(10,1) = -24310.d0
      disb(10,2) = 437580.d0
      disb(10,3) = -3738878.d0
      disb(10,4) = 20179952.d0
      disb(10,5) = -77250108.d0
      disb(10,6) = 223303432.d0
      disb(10,7) = -506875468.d0
      disb(10,8) = 927983760.d0
      disb(10,9) = -1397468765.d0
      disb(10,10) = 1757779210.d0
      disb(10,11) = -1870249645.d0
      disb(10,12) = 1701625200.d0
      disb(10,13) = -1336391012.d0
      disb(10,14) = 912968408.d0
      disb(10,15) = -545492532.d0
      disb(10,16) = 285754768.d0
      disb(10,17) = -131108692.d0
      disb(10,18) = 52451256.d0
      disb(10,19) = -18156204.d0
      disb(10,20) = 5379616.d0
      disb(10,21) = -1344904.d0
      disb(10,22) = 278256.d0
      disb(10,23) = -46376.d0
      disb(10,24) = 5984.d0
      disb(10,25) = -561.d0
      disb(10,26) = 34.d0
      disb(10,27) = -1.d0
c
      disb(11,1) = 19448.d0
      disb(11,2) = -354926.d0
      disb(11,3) = 3082508.d0
      disb(11,4) = -16963518.d0
      disb(11,5) = 66466192.d0
      disb(11,6) = -197594332.d0
      disb(11,7) = 463991880.d0
      disb(11,8) = -885100172.d0
      disb(11,9) = 1400764640.d0
      disb(11,10) = -1870249645.d0
      disb(11,11) = 2136003914.d0
      disb(11,12) = -2110938093.d0
      disb(11,13) = 1821969424.d0
      disb(11,14) = -1382677252.d0
      disb(11,15) = 926193048.d0
      disb(11,16) = -548137460.d0
      disb(11,17) = 286085384.d0
      disb(11,18) = -131128140.d0
      disb(11,19) = 52451256.d0
      disb(11,20) = -18156204.d0
      disb(11,21) = 5379616.d0
      disb(11,22) = -1344904.d0
      disb(11,23) = 278256.d0
      disb(11,24) = -46376.d0
      disb(11,25) = 5984.d0
      disb(11,26) = -561.d0
      disb(11,27) = 34.d0
      disb(11,28) = -1.d0
c
      disb(12,1) = -12376.d0
      disb(12,2) = 229840.d0
      disb(12,3) = -2038062.d0
      disb(12,4) = 11498188.d0
      disb(12,5) = -46418398.d0
      disb(12,6) = 143048880.d0
      disb(12,7) = -350759708.d0
      disb(12,8) = 704680328.d0
      disb(12,9) = -1185960732.d0
      disb(12,10) = 1701625200.d0
      disb(12,11) = -2110938093.d0
      disb(12,12) = 2289169290.d0
      disb(12,13) = -2187520781.d0
      disb(12,14) = 1851424304.d0
      disb(12,15) = -1391092932.d0
      disb(12,16) = 927876184.d0
      disb(12,17) = -548347852.d0
      disb(12,18) = 286097760.d0
      disb(12,19) = -131128140.d0
      disb(12,20) = 52451256.d0
      disb(12,21) = -18156204.d0
      disb(12,22) = 5379616.d0
      disb(12,23) = -1344904.d0
      disb(12,24) = 278256.d0
      disb(12,25) = -46376.d0
      disb(12,26) = 5984.d0
      disb(12,27) = -561.d0
      disb(12,28) = 34.d0
      disb(12,29) = -1.d0
c
      disb(13,1) = 6188.d0
      disb(13,2) = -117572.d0
      disb(13,3) = 1071408.d0
      disb(13,4) = -6245902.d0
      disb(13,5) = 26225628.d0
      disb(13,6) = -84709742.d0
      disb(13,7) = 219631568.d0
      disb(13,8) = -471103932.d0
      disb(13,9) = 855110608.d0
      disb(13,10) = -1336391012.d0
      disb(13,11) = 1821969424.d0
      disb(13,12) = -2187520781.d0
      disb(13,13) = 2327460634.d0
      disb(13,14) = -2202248221.d0
      disb(13,15) = 1855632144.d0
      disb(13,16) = -1391934500.d0
      disb(13,17) = 927981380.d0
      disb(13,18) = -548354040.d0
      disb(13,19) = 286097760.d0
      disb(13,20) = -131128140.d0
      disb(13,21) = 52451256.d0
      disb(13,22) = -18156204.d0
      disb(13,23) = 5379616.d0
      disb(13,24) = -1344904.d0
      disb(13,25) = 278256.d0
      disb(13,26) = -46376.d0
      disb(13,27) = 5984.d0
      disb(13,28) = -561.d0
      disb(13,29) = 34.d0
      disb(13,30) = -1.d0
c
      disb(14,1) = -2380.d0
      disb(14,2) = 46648.d0
      disb(14,3) = -441252.d0
      disb(14,4) = 2689808.d0
      disb(14,5) = -11910302.d0
      disb(14,6) = 40953068.d0
      disb(14,7) = -114164622.d0
      disb(14,8) = 265917808.d0
      disb(14,9) = -528961732.d0
      disb(14,10) = 912968408.d0
      disb(14,11) = -1382677252.d0
      disb(14,12) = 1851424304.d0
      disb(14,13) = -2202248221.d0
      disb(14,14) = 2333125034.d0
      disb(14,15) = -2203866621.d0
      disb(14,16) = 1855955824.d0
      disb(14,17) = -1391974960.d0
      disb(14,18) = 927983760.d0
      disb(14,19) = -548354040.d0
      disb(14,20) = 286097760.d0
      disb(14,21) = -131128140.d0
      disb(14,22) = 52451256.d0
      disb(14,23) = -18156204.d0
      disb(14,24) = 5379616.d0
      disb(14,25) = -1344904.d0
      disb(14,26) = 278256.d0
      disb(14,27) = -46376.d0
      disb(14,28) = 5984.d0
      disb(14,29) = -561.d0
      disb(14,30) = 34.d0
      disb(14,31) = -1.d0
c
      disb(15,1) = 680.d0
      disb(15,2) = -13940.d0
      disb(15,3) = 139128.d0
      disb(15,4) = -903652.d0
      disb(15,5) = 4308208.d0
      disb(15,6) = -16118142.d0
      disb(15,7) = 49368748.d0
      disb(15,8) = -127389262.d0
      disb(15,9) = 282448608.d0
      disb(15,10) = -545492532.d0
      disb(15,11) = 926193048.d0
      disb(15,12) = -1391092932.d0
      disb(15,13) = 1855632144.d0
      disb(15,14) = -2203866621.d0
      disb(15,15) = 2333587434.d0
      disb(15,16) = -2203959101.d0
      disb(15,17) = 1855967384.d0
      disb(15,18) = -1391975640.d0
      disb(15,19) = 927983760.d0
      disb(15,20) = -548354040.d0
      disb(15,21) = 286097760.d0
      disb(15,22) = -131128140.d0
      disb(15,23) = 52451256.d0
      disb(15,24) = -18156204.d0
      disb(15,25) = 5379616.d0
      disb(15,26) = -1344904.d0
      disb(15,27) = 278256.d0
      disb(15,28) = -46376.d0
      disb(15,29) = 5984.d0
      disb(15,30) = -561.d0
      disb(15,31) = 34.d0
      disb(15,32) = -1.d0
c
      disb(16,1) = -136.d0
      disb(16,2) = 2992.d0
      disb(16,3) = -32436.d0
      disb(16,4) = 231608.d0
      disb(16,5) = -1227332.d0
      disb(16,6) = 5149776.d0
      disb(16,7) = -17801278.d0
      disb(16,8) = 52013676.d0
      disb(16,9) = -130695422.d0
      disb(16,10) = 285754768.d0
      disb(16,11) = -548137460.d0
      disb(16,12) = 927876184.d0
      disb(16,13) = -1391934500.d0
      disb(16,14) = 1855955824.d0
      disb(16,15) = -2203959101.d0
      disb(16,16) = 2333605930.d0
      disb(16,17) = -2203961413.d0
      disb(16,18) = 1855967520.d0
      disb(16,19) = -1391975640.d0
      disb(16,20) = 927983760.d0
      disb(16,21) = -548354040.d0
      disb(16,22) = 286097760.d0
      disb(16,23) = -131128140.d0
      disb(16,24) = 52451256.d0
      disb(16,25) = -18156204.d0
      disb(16,26) = 5379616.d0
      disb(16,27) = -1344904.d0
      disb(16,28) = 278256.d0
      disb(16,29) = -46376.d0
      disb(16,30) = 5984.d0
      disb(16,31) = -561.d0
      disb(16,32) = 34.d0
      disb(16,33) = -1.d0
c
      disb(17,1) = 17.d0
      disb(17,2) = -425.d0
      disb(17,3) = 5304.d0
      disb(17,4) = -43996.d0
      disb(17,5) = 272068.d0
      disb(17,6) = -1332528.d0
      disb(17,7) = 5360168.d0
      disb(17,8) = -18131894.d0
      disb(17,9) = 52426946.d0
      disb(17,10) = -131108692.d0
      disb(17,11) = 286085384.d0
      disb(17,12) = -548347852.d0
      disb(17,13) = 927981380.d0
      disb(17,14) = -1391974960.d0
      disb(17,15) = 1855967384.d0
      disb(17,16) = -2203961413.d0
      disb(17,17) = 2333606219.d0
      disb(17,18) = -2203961430.d0
      disb(17,19) = 1855967520.d0
      disb(17,20) = -1391975640.d0
      disb(17,21) = 927983760.d0
      disb(17,22) = -548354040.d0
      disb(17,23) = 286097760.d0
      disb(17,24) = -131128140.d0
      disb(17,25) = 52451256.d0
      disb(17,26) = -18156204.d0
      disb(17,27) = 5379616.d0
      disb(17,28) = -1344904.d0
      disb(17,29) = 278256.d0
      disb(17,30) = -46376.d0
      disb(17,31) = 5984.d0
      disb(17,32) = -561.d0
      disb(17,33) = 34.d0
      disb(17,34) = -1.d0
c
      elseif(iord.eq.36)then
c 
c 36th order inner with 18th order boundaries
c ------------------------------------------
c
      disi(1) = -9075135300.d0
      disi(2) = 8597496600.d0
      disi(3) = -7307872110.d0
      disi(4) = 5567902560.d0
      disi(5) = -3796297200.d0
      disi(6) = 2310789600.d0
      disi(7) = -1251677700.d0
      disi(8) = 600805296.d0
      disi(9) = -254186856.d0
      disi(10) = 94143280.d0
      disi(11) = -30260340.d0
      disi(12) = 8347680.d0
      disi(13) = -1947792.d0
      disi(14) = 376992.d0
      disi(15) = -58905.d0
      disi(16) = 7140.d0
      disi(17) = -630.d0
      disi(18) = 36.d0
      disi(19) = -1.d0
c
      disb(1,1) = -1.d0
      disb(1,2) = 18.d0
      disb(1,3) = -153.d0
      disb(1,4) = 816.d0
      disb(1,5) = -3060.d0
      disb(1,6) = 8568.d0
      disb(1,7) = -18564.d0
      disb(1,8) = 31824.d0
      disb(1,9) = -43758.d0
      disb(1,10) = 48620.d0
      disb(1,11) = -43758.d0
      disb(1,12) = 31824.d0
      disb(1,13) = -18564.d0
      disb(1,14) = 8568.d0
      disb(1,15) = -3060.d0
      disb(1,16) = 816.d0
      disb(1,17) = -153.d0
      disb(1,18) = 18.d0
      disb(1,19) = -1.d0
c
      disb(2,1) = 18.d0
      disb(2,2) = -325.d0
      disb(2,3) = 2772.d0
      disb(2,4) = -14841.d0
      disb(2,5) = 55896.d0
      disb(2,6) = -157284.d0
      disb(2,7) = 342720.d0
      disb(2,8) = -591396.d0
      disb(2,9) = 819468.d0
      disb(2,10) = -918918.d0
      disb(2,11) = 836264.d0
      disb(2,12) = -616590.d0
      disb(2,13) = 365976.d0
      disb(2,14) = -172788.d0
      disb(2,15) = 63648.d0
      disb(2,16) = -17748.d0
      disb(2,17) = 3570.d0
      disb(2,18) = -477.d0
      disb(2,19) = 36.d0
      disb(2,20) = -1.d0
c
      disb(3,1) = -153.d0
      disb(3,2) = 2772.d0
      disb(3,3) = -23734.d0
      disb(3,4) = 127620.d0
      disb(3,5) = -483021.d0
      disb(3,6) = 1366800.d0
      disb(3,7) = -2997576.d0
      disb(3,8) = 5211792.d0
      disb(3,9) = -7286370.d0
      disb(3,10) = 8258328.d0
      disb(3,11) = -7613892.d0
      disb(3,12) = 5705336.d0
      disb(3,13) = -3456882.d0
      disb(3,14) = 1676880.d0
      disb(3,15) = -640968.d0
      disb(3,16) = 188496.d0
      disb(3,17) = -41157.d0
      disb(3,18) = 6324.d0
      disb(3,19) = -630.d0
      disb(3,20) = 36.d0
      disb(3,21) = -1.d0
c
      disb(4,1) = 816.d0
      disb(4,2) = -14841.d0
      disb(4,3) = 127620.d0
      disb(4,4) = -689590.d0
      disb(4,5) = 2624580.d0
      disb(4,6) = -7474509.d0
      disb(4,7) = 16515024.d0
      disb(4,8) = -28965960.d0
      disb(4,9) = 40918320.d0
      disb(4,10) = -46960290.d0
      disb(4,11) = 43964856.d0
      disb(4,12) = -33582276.d0
      disb(4,13) = 20853560.d0
      disb(4,14) = -10448370.d0
      disb(4,15) = 4173840.d0
      disb(4,16) = -1306824.d0
      disb(4,17) = 313344.d0
      disb(4,18) = -55845.d0
      disb(4,19) = 7140.d0
      disb(4,20) = -630.d0
      disb(4,21) = 36.d0
      disb(4,22) = -1.d0
c
      disb(5,1) = -3060.d0
      disb(5,2) = 55896.d0
      disb(5,3) = -483021.d0
      disb(5,4) = 2624580.d0
      disb(5,5) = -10053190.d0
      disb(5,6) = 28842660.d0
      disb(5,7) = -64280349.d0
      disb(5,8) = 113896464.d0
      disb(5,9) = -162865440.d0
      disb(5,10) = 189695520.d0
      disb(5,11) = -180859770.d0
      disb(5,12) = 141346296.d0
      disb(5,13) = -90388116.d0
      disb(5,14) = 47071640.d0
      disb(5,15) = -19811970.d0
      disb(5,16) = 6670800.d0
      disb(5,17) = -1775004.d0
      disb(5,18) = 368424.d0
      disb(5,19) = -58905.d0
      disb(5,20) = 7140.d0
      disb(5,21) = -630.d0
      disb(5,22) = 36.d0
      disb(5,23) = -1.d0
c
      disb(6,1) = 8568.d0
      disb(6,2) = -157284.d0
      disb(6,3) = 1366800.d0
      disb(6,4) = -7474509.d0
      disb(6,5) = 28842660.d0
      disb(6,6) = -83463814.d0
      disb(6,7) = 187899012.d0
      disb(6,8) = -336948381.d0
      disb(6,9) = 488815008.d0
      disb(6,10) = -579441600.d0
      disb(6,11) = 564614064.d0
      disb(6,12) = -453527802.d0
      disb(6,13) = 300402648.d0
      disb(6,14) = -163798740.d0
      disb(6,15) = 73289720.d0
      disb(6,16) = -26803458.d0
      disb(6,17) = 7981704.d0
      disb(6,18) = -1929228.d0
      disb(6,19) = 376992.d0
      disb(6,20) = -58905.d0
      disb(6,21) = 7140.d0
      disb(6,22) = -630.d0
      disb(6,23) = 36.d0
      disb(6,24) = -1.d0
c
      disb(7,1) = -18564.d0
      disb(7,2) = 342720.d0
      disb(7,3) = -2997576.d0
      disb(7,4) = 16515024.d0
      disb(7,5) = -64280349.d0
      disb(7,6) = 187899012.d0
      disb(7,7) = -428085910.d0
      disb(7,8) = 778679748.d0
      disb(7,9) = -1149271893.d0
      disb(7,10) = 1391396688.d0
      disb(7,11) = -1391765112.d0
      disb(7,12) = 1155394800.d0
      disb(7,13) = -798149898.d0
      disb(7,14) = 459459000.d0
      disb(7,15) = -220604580.d0
      disb(7,16) = 88437944.d0
      disb(7,17) = -29643750.d0
      disb(7,18) = 8315856.d0
      disb(7,19) = -1947792.d0
      disb(7,20) = 376992.d0
      disb(7,21) = -58905.d0
      disb(7,22) = 7140.d0
      disb(7,23) = -630.d0
      disb(7,24) = 36.d0
      disb(7,25) = -1.d0
c
      disb(8,1) = 31824.d0
      disb(8,2) = -591396.d0
      disb(8,3) = 5211792.d0
      disb(8,4) = -28965960.d0
      disb(8,5) = 113896464.d0
      disb(8,6) = -336948381.d0
      disb(8,7) = 778679748.d0
      disb(8,8) = -1440852886.d0
      disb(8,9) = 2171234340.d0
      disb(8,10) = -2696554773.d0
      disb(8,11) = 2783951280.d0
      disb(8,12) = -2404532088.d0
      disb(8,13) = 1746175536.d0
      disb(8,14) = -1070817930.d0
      disb(8,15) = 556840440.d0
      disb(8,16) = -246572964.d0
      disb(8,17) = 93307016.d0
      disb(8,18) = -30216582.d0
      disb(8,19) = 8347680.d0
      disb(8,20) = -1947792.d0
      disb(8,21) = 376992.d0
      disb(8,22) = -58905.d0
      disb(8,23) = 7140.d0
      disb(8,24) = -630.d0
      disb(8,25) = 36.d0
      disb(8,26) = -1.d0
c
      disb(9,1) = -43758.d0
      disb(9,2) = 819468.d0
      disb(9,3) = -7286370.d0
      disb(9,4) = 40918320.d0
      disb(9,5) = -162865440.d0
      disb(9,6) = 488815008.d0
      disb(9,7) = -1149271893.d0
      disb(9,8) = 2171234340.d0
      disb(9,9) = -3355615450.d0
      disb(9,10) = 4298748300.d0
      disb(9,11) = -4611317337.d0
      disb(9,12) = 4176505872.d0
      disb(9,13) = -3216855600.d0
      disb(9,14) = 2121094080.d0
      disb(9,15) = -1204717410.d0
      disb(9,16) = 592546968.d0
      disb(9,17) = -253267938.d0
      disb(9,18) = 94094660.d0
      disb(9,19) = -30260340.d0
      disb(9,20) = 8347680.d0
      disb(9,21) = -1947792.d0
      disb(9,22) = 376992.d0
      disb(9,23) = -58905.d0
      disb(9,24) = 7140.d0
      disb(9,25) = -630.d0
      disb(9,26) = 36.d0
      disb(9,27) = -1.d0
c
      disb(10,1) = 48620.d0
      disb(10,2) = -918918.d0
      disb(10,3) = 8258328.d0
      disb(10,4) = -46960290.d0
      disb(10,5) = 189695520.d0
      disb(10,6) = -579441600.d0
      disb(10,7) = 1391396688.d0
      disb(10,8) = -2696554773.d0
      disb(10,9) = 4298748300.d0
      disb(10,10) = -5719519850.d0
      disb(10,11) = 6426262260.d0
      disb(10,12) = -6158600217.d0
      disb(10,13) = 5079087552.d0
      disb(10,14) = -3633431760.d0
      disb(10,15) = 2269871280.d0
      disb(10,16) = -1244391330.d0
      disb(10,17) = 599985828.d0
      disb(10,18) = -254143098.d0
      disb(10,19) = 94143280.d0
      disb(10,20) = -30260340.d0
      disb(10,21) = 8347680.d0
      disb(10,22) = -1947792.d0
      disb(10,23) = 376992.d0
      disb(10,24) = -58905.d0
      disb(10,25) = 7140.d0
      disb(10,26) = -630.d0
      disb(10,27) = 36.d0
      disb(10,28) = -1.d0
c
      disb(11,1) = -43758.d0
      disb(11,2) = 836264.d0
      disb(11,3) = -7613892.d0
      disb(11,4) = 43964856.d0
      disb(11,5) = -180859770.d0
      disb(11,6) = 564614064.d0
      disb(11,7) = -1391765112.d0
      disb(11,8) = 2783951280.d0
      disb(11,9) = -4611317337.d0
      disb(11,10) = 6426262260.d0
      disb(11,11) = -7634282414.d0
      disb(11,12) = 7818816852.d0
      disb(11,13) = -6970923729.d0
      disb(11,14) = 5454006096.d0
      disb(11,15) = -3767331240.d0
      disb(11,16) = 2305577808.d0
      disb(11,17) = -1251086304.d0
      disb(11,18) = 600773472.d0
      disb(11,19) = -254186856.d0
      disb(11,20) = 94143280.d0
      disb(11,21) = -30260340.d0
      disb(11,22) = 8347680.d0
      disb(11,23) = -1947792.d0
      disb(11,24) = 376992.d0
      disb(11,25) = -58905.d0
      disb(11,26) = 7140.d0
      disb(11,27) = -630.d0
      disb(11,28) = 36.d0
      disb(11,29) = -1.d0
c
      disb(12,1) = 31824.d0
      disb(12,2) = -616590.d0
      disb(12,3) = 5705336.d0
      disb(12,4) = -33582276.d0
      disb(12,5) = 141346296.d0
      disb(12,6) = -453527802.d0
      disb(12,7) = 1155394800.d0
      disb(12,8) = -2404532088.d0
      disb(12,9) = 4176505872.d0
      disb(12,10) = -6158600217.d0
      disb(12,11) = 7818816852.d0
      disb(12,12) = -8647049390.d0
      disb(12,13) = 8409597588.d0
      disb(12,14) = -7243591761.d0
      disb(12,15) = 5551387536.d0
      disb(12,16) = -3793299624.d0
      disb(12,17) = 2310446880.d0
      disb(12,18) = -1251659136.d0
      disb(12,19) = 600805296.d0
      disb(12,20) = -254186856.d0
      disb(12,21) = 94143280.d0
      disb(12,22) = -30260340.d0
      disb(12,23) = 8347680.d0
      disb(12,24) = -1947792.d0
      disb(12,25) = 376992.d0
      disb(12,26) = -58905.d0
      disb(12,27) = 7140.d0
      disb(12,28) = -630.d0
      disb(12,29) = 36.d0
      disb(12,30) = -1.d0
c
      disb(13,1) = -18564.d0
      disb(13,2) = 365976.d0
      disb(13,3) = -3456882.d0
      disb(13,4) = 20853560.d0
      disb(13,5) = -90388116.d0
      disb(13,6) = 300402648.d0
      disb(13,7) = -798149898.d0
      disb(13,8) = 1746175536.d0
      disb(13,9) = -3216855600.d0
      disb(13,10) = 5079087552.d0
      disb(13,11) = -6970923729.d0
      disb(13,12) = 8409597588.d0
      disb(13,13) = -8991671486.d0
      disb(13,14) = 8568653940.d0
      disb(13,15) = -7300397601.d0
      disb(13,16) = 5566535760.d0
      disb(13,17) = -3796139916.d0
      disb(13,18) = 2310781032.d0
      disb(13,19) = -1251677700.d0
      disb(13,20) = 600805296.d0
      disb(13,21) = -254186856.d0
      disb(13,22) = 94143280.d0
      disb(13,23) = -30260340.d0
      disb(13,24) = 8347680.d0
      disb(13,25) = -1947792.d0
      disb(13,26) = 376992.d0
      disb(13,27) = -58905.d0
      disb(13,28) = 7140.d0
      disb(13,29) = -630.d0
      disb(13,30) = 36.d0
      disb(13,31) = -1.d0
c
      disb(14,1) = 8568.d0
      disb(14,2) = -172788.d0
      disb(14,3) = 1676880.d0
      disb(14,4) = -10448370.d0
      disb(14,5) = 47071640.d0
      disb(14,6) = -163798740.d0
      disb(14,7) = 459459000.d0
      disb(14,8) = -1070817930.d0
      disb(14,9) = 2121094080.d0
      disb(14,10) = -3633431760.d0
      disb(14,11) = 5454006096.d0
      disb(14,12) = -7243591761.d0
      disb(14,13) = 8568653940.d0
      disb(14,14) = -9065082110.d0
      disb(14,15) = 8594872020.d0
      disb(14,16) = -7307389089.d0
      disb(14,17) = 5567846664.d0
      disb(14,18) = -3796294140.d0
      disb(14,19) = 2310789600.d0
      disb(14,20) = -1251677700.d0
      disb(14,21) = 600805296.d0
      disb(14,22) = -254186856.d0
      disb(14,23) = 94143280.d0
      disb(14,24) = -30260340.d0
      disb(14,25) = 8347680.d0
      disb(14,26) = -1947792.d0
      disb(14,27) = 376992.d0
      disb(14,28) = -58905.d0
      disb(14,29) = 7140.d0
      disb(14,30) = -630.d0
      disb(14,31) = 36.d0
      disb(14,32) = -1.d0
c
      disb(15,1) = -3060.d0
      disb(15,2) = 63648.d0
      disb(15,3) = -640968.d0
      disb(15,4) = 4173840.d0
      disb(15,5) = -19811970.d0
      disb(15,6) = 73289720.d0
      disb(15,7) = -220604580.d0
      disb(15,8) = 556840440.d0
      disb(15,9) = -1204717410.d0
      disb(15,10) = 2269871280.d0
      disb(15,11) = -3767331240.d0
      disb(15,12) = 5551387536.d0
      disb(15,13) = -7300397601.d0
      disb(15,14) = 8594872020.d0
      disb(15,15) = -9074445710.d0
      disb(15,16) = 8597368980.d0
      disb(15,17) = -7307857269.d0
      disb(15,18) = 5567901744.d0
      disb(15,19) = -3796297200.d0
      disb(15,20) = 2310789600.d0
      disb(15,21) = -1251677700.d0
      disb(15,22) = 600805296.d0
      disb(15,23) = -254186856.d0
      disb(15,24) = 94143280.d0
      disb(15,25) = -30260340.d0
      disb(15,26) = 8347680.d0
      disb(15,27) = -1947792.d0
      disb(15,28) = 376992.d0
      disb(15,29) = -58905.d0
      disb(15,30) = 7140.d0
      disb(15,31) = -630.d0
      disb(15,32) = 36.d0
      disb(15,33) = -1.d0
c
      disb(16,1) = 816.d0
      disb(16,2) = -17748.d0
      disb(16,3) = 188496.d0
      disb(16,4) = -1306824.d0
      disb(16,5) = 6670800.d0
      disb(16,6) = -26803458.d0
      disb(16,7) = 88437944.d0
      disb(16,8) = -246572964.d0
      disb(16,9) = 592546968.d0
      disb(16,10) = -1244391330.d0
      disb(16,11) = 2305577808.d0
      disb(16,12) = -3793299624.d0
      disb(16,13) = 5566535760.d0
      disb(16,14) = -7307389089.d0
      disb(16,15) = 8597368980.d0
      disb(16,16) = -9075111566.d0
      disb(16,17) = 8597493828.d0
      disb(16,18) = -7307871957.d0
      disb(16,19) = 5567902560.d0
      disb(16,20) = -3796297200.d0
      disb(16,21) = 2310789600.d0
      disb(16,22) = -1251677700.d0
      disb(16,23) = 600805296.d0
      disb(16,24) = -254186856.d0
      disb(16,25) = 94143280.d0
      disb(16,26) = -30260340.d0
      disb(16,27) = 8347680.d0
      disb(16,28) = -1947792.d0
      disb(16,29) = 376992.d0
      disb(16,30) = -58905.d0
      disb(16,31) = 7140.d0
      disb(16,32) = -630.d0
      disb(16,33) = 36.d0
      disb(16,34) = -1.d0
c
      disb(17,1) = -153.d0
      disb(17,2) = 3570.d0
      disb(17,3) = -41157.d0
      disb(17,4) = 313344.d0
      disb(17,5) = -1775004.d0
      disb(17,6) = 7981704.d0
      disb(17,7) = -29643750.d0
      disb(17,8) = 93307016.d0
      disb(17,9) = -253267938.d0
      disb(17,10) = 599985828.d0
      disb(17,11) = -1251086304.d0
      disb(17,12) = 2310446880.d0
      disb(17,13) = -3796139916.d0
      disb(17,14) = 5567846664.d0
      disb(17,15) = -7307857269.d0
      disb(17,16) = 8597493828.d0
      disb(17,17) = -9075134975.d0
      disb(17,18) = 8597496582.d0
      disb(17,19) = -7307872110.d0
      disb(17,20) = 5567902560.d0
      disb(17,21) = -3796297200.d0
      disb(17,22) = 2310789600.d0
      disb(17,23) = -1251677700.d0
      disb(17,24) = 600805296.d0
      disb(17,25) = -254186856.d0
      disb(17,26) = 94143280.d0
      disb(17,27) = -30260340.d0
      disb(17,28) = 8347680.d0
      disb(17,29) = -1947792.d0
      disb(17,30) = 376992.d0
      disb(17,31) = -58905.d0
      disb(17,32) = 7140.d0
      disb(17,33) = -630.d0
      disb(17,34) = 36.d0
      disb(17,35) = -1.d0
c
      disb(18,1) = 18.d0
      disb(18,2) = -477.d0
      disb(18,3) = 6324.d0
      disb(18,4) = -55845.d0
      disb(18,5) = 368424.d0
      disb(18,6) = -1929228.d0
      disb(18,7) = 8315856.d0
      disb(18,8) = -30216582.d0
      disb(18,9) = 94094660.d0
      disb(18,10) = -254143098.d0
      disb(18,11) = 600773472.d0
      disb(18,12) = -1251659136.d0
      disb(18,13) = 2310781032.d0
      disb(18,14) = -3796294140.d0
      disb(18,15) = 5567901744.d0
      disb(18,16) = -7307871957.d0
      disb(18,17) = 8597496582.d0
      disb(18,18) = -9075135299.d0
      disb(18,19) = 8597496600.d0
      disb(18,20) = -7307872110.d0
      disb(18,21) = 5567902560.d0
      disb(18,22) = -3796297200.d0
      disb(18,23) = 2310789600.d0
      disb(18,24) = -1251677700.d0
      disb(18,25) = 600805296.d0
      disb(18,26) = -254186856.d0
      disb(18,27) = 94143280.d0
      disb(18,28) = -30260340.d0
      disb(18,29) = 8347680.d0
      disb(18,30) = -1947792.d0
      disb(18,31) = 376992.d0
      disb(18,32) = -58905.d0
      disb(18,33) = 7140.d0
      disb(18,34) = -630.d0
      disb(18,35) = 36.d0
      disb(18,36) = -1.d0
c
      elseif(iord.eq.38)then
c 
c 38th order inner with 19th order boundaries
c ------------------------------------------
c
      disi(1) = 35345263800.d0
      disi(2) = -33578000610.d0
      disi(3) = 28781143380.d0
      disi(4) = -22239974430.d0
      disi(5) = 15471286560.d0
      disi(6) = -9669554100.d0
      disi(7) = 5414950296.d0
      disi(8) = -2707475148.d0
      disi(9) = 1203322288.d0
      disi(10) = -472733756.d0
      disi(11) = 163011640.d0
      disi(12) = -48903492.d0
      disi(13) = 12620256.d0
      disi(14) = -2760681.d0
      disi(15) = 501942.d0
      disi(16) = -73815.d0
      disi(17) = 8436.d0
      disi(18) = -703.d0
      disi(19) = 38.d0
      disi(20) = -1.d0
c
      disb(1,1) = 1.d0
      disb(1,2) = -19.d0
      disb(1,3) = 171.d0
      disb(1,4) = -969.d0
      disb(1,5) = 3876.d0
      disb(1,6) = -11628.d0
      disb(1,7) = 27132.d0
      disb(1,8) = -50388.d0
      disb(1,9) = 75582.d0
      disb(1,10) = -92378.d0
      disb(1,11) = 92378.d0
      disb(1,12) = -75582.d0
      disb(1,13) = 50388.d0
      disb(1,14) = -27132.d0
      disb(1,15) = 11628.d0
      disb(1,16) = -3876.d0
      disb(1,17) = 969.d0
      disb(1,18) = -171.d0
      disb(1,19) = 19.d0
      disb(1,20) = -1.d0
c
      disb(2,1) = -19.d0
      disb(2,2) = 362.d0
      disb(2,3) = -3268.d0
      disb(2,4) = 18582.d0
      disb(2,5) = -74613.d0
      disb(2,6) = 224808.d0
      disb(2,7) = -527136.d0
      disb(2,8) = 984504.d0
      disb(2,9) = -1486446.d0
      disb(2,10) = 1830764.d0
      disb(2,11) = -1847560.d0
      disb(2,12) = 1528436.d0
      disb(2,13) = -1032954.d0
      disb(2,14) = 565896.d0
      disb(2,15) = -248064.d0
      disb(2,16) = 85272.d0
      disb(2,17) = -22287.d0
      disb(2,18) = 4218.d0
      disb(2,19) = -532.d0
      disb(2,20) = 38.d0
      disb(2,21) = -1.d0
c
      disb(3,1) = 171.d0
      disb(3,2) = -3268.d0
      disb(3,3) = 29603.d0
      disb(3,4) = -168967.d0
      disb(3,5) = 681378.d0
      disb(3,6) = -2063001.d0
      disb(3,7) = 4864380.d0
      disb(3,8) = -9143484.d0
      disb(3,9) = 13909026.d0
      disb(3,10) = -17283084.d0
      disb(3,11) = 17627402.d0
      disb(3,12) = -14772082.d0
      disb(3,13) = 10144784.d0
      disb(3,14) = -5672526.d0
      disb(3,15) = 2554284.d0
      disb(3,16) = -910860.d0
      disb(3,17) = 250971.d0
      disb(3,18) = -51528.d0
      disb(3,19) = 7467.d0
      disb(3,20) = -703.d0
      disb(3,21) = 38.d0
      disb(3,22) = -1.d0
c
      disb(4,1) = -969.d0
      disb(4,2) = 18582.d0
      disb(4,3) = -168967.d0
      disb(4,4) = 968564.d0
      disb(4,5) = -3924811.d0
      disb(4,6) = 11948910.d0
      disb(4,7) = -28353909.d0
      disb(4,8) = 53690352.d0
      disb(4,9) = -82382442.d0
      disb(4,10) = 103423308.d0
      disb(4,11) = -106797366.d0
      disb(4,12) = 90866360.d0
      disb(4,13) = -63598054.d0
      disb(4,14) = 36435692.d0
      disb(4,15) = -16940058.d0
      disb(4,16) = 6310128.d0
      disb(4,17) = -1849821.d0
      disb(4,18) = 416670.d0
      disb(4,19) = -69939.d0
      disb(4,20) = 8436.d0
      disb(4,21) = -703.d0
      disb(4,22) = 38.d0
      disb(4,23) = -1.d0
c
      disb(5,1) = 3876.d0
      disb(5,2) = -74613.d0
      disb(5,3) = 681378.d0
      disb(5,4) = -3924811.d0
      disb(5,5) = 15991940.d0
      disb(5,6) = -48994939.d0
      disb(5,7) = 117112542.d0
      disb(5,8) = -223657797.d0
      disb(5,9) = 346646184.d0
      disb(5,10) = -440439570.d0
      disb(5,11) = 461480436.d0
      disb(5,12) = -399753198.d0
      disb(5,13) = 286170248.d0
      disb(5,14) = -168761686.d0
      disb(5,15) = 81505820.d0
      disb(5,16) = -31963434.d0
      disb(5,17) = 10065972.d0
      disb(5,18) = -2512617.d0
      disb(5,19) = 490314.d0
      disb(5,20) = -73815.d0
      disb(5,21) = 8436.d0
      disb(5,22) = -703.d0
      disb(5,23) = 38.d0
      disb(5,24) = -1.d0
c
      disb(6,1) = -11628.d0
      disb(6,2) = 224808.d0
      disb(6,3) = -2063001.d0
      disb(6,4) = 11948910.d0
      disb(6,5) = -48994939.d0
      disb(6,6) = 151202324.d0
      disb(6,7) = -364485835.d0
      disb(6,8) = 703024206.d0
      disb(6,9) = -1102525293.d0
      disb(6,10) = 1420817568.d0
      disb(6,11) = -1514610954.d0
      disb(6,12) = 1340347932.d0
      disb(6,13) = -985664862.d0
      disb(6,14) = 601661144.d0
      disb(6,15) = -303972070.d0
      disb(6,16) = 126575948.d0
      disb(6,17) = -43230966.d0
      disb(6,18) = 12054360.d0
      disb(6,19) = -2733549.d0
      disb(6,20) = 501942.d0
      disb(6,21) = -73815.d0
      disb(6,22) = 8436.d0
      disb(6,23) = -703.d0
      disb(6,24) = 38.d0
      disb(6,25) = -1.d0
c
      disb(7,1) = 27132.d0
      disb(7,2) = -527136.d0
      disb(7,3) = 4864380.d0
      disb(7,4) = -28353909.d0
      disb(7,5) = 117112542.d0
      disb(7,6) = -364485835.d0
      disb(7,7) = 887347748.d0
      disb(7,8) = -1731613051.d0
      disb(7,9) = 2753715030.d0
      disb(7,10) = -3608925189.d0
      disb(7,11) = 3927217464.d0
      disb(7,12) = -3565301778.d0
      disb(7,13) = 2707475148.d0
      disb(7,14) = -1721810286.d0
      disb(7,15) = 917152040.d0
      disb(7,16) = -409135702.d0
      disb(7,17) = 152866856.d0
      disb(7,18) = -47870538.d0
      disb(7,19) = 12569868.d0
      disb(7,20) = -2760681.d0
      disb(7,21) = 501942.d0
      disb(7,22) = -73815.d0
      disb(7,23) = 8436.d0
      disb(7,24) = -703.d0
      disb(7,25) = 38.d0
      disb(7,26) = -1.d0
c
      disb(8,1) = -50388.d0
      disb(8,2) = 984504.d0
      disb(8,3) = -9143484.d0
      disb(8,4) = 53690352.d0
      disb(8,5) = -223657797.d0
      disb(8,6) = 703024206.d0
      disb(8,7) = -1731613051.d0
      disb(8,8) = 3426298292.d0
      disb(8,9) = -5540038867.d0
      disb(8,10) = 7408457694.d0
      disb(8,11) = -8263667853.d0
      disb(8,12) = 7735643280.d0
      disb(8,13) = -6104252322.d0
      disb(8,14) = 4074602364.d0
      disb(8,15) = -2307721950.d0
      disb(8,16) = 1112455928.d0
      disb(8,17) = -457961674.d0
      disb(8,18) = 161483204.d0
      disb(8,19) = -48827910.d0
      disb(8,20) = 12620256.d0
      disb(8,21) = -2760681.d0
      disb(8,22) = 501942.d0
      disb(8,23) = -73815.d0
      disb(8,24) = 8436.d0
      disb(8,25) = -703.d0
      disb(8,26) = 38.d0
      disb(8,27) = -1.d0
c
      disb(9,1) = 75582.d0
      disb(9,2) = -1486446.d0
      disb(9,3) = 13909026.d0
      disb(9,4) = -82382442.d0
      disb(9,5) = 346646184.d0
      disb(9,6) = -1102525293.d0
      disb(9,7) = 2753715030.d0
      disb(9,8) = -5540038867.d0
      disb(9,9) = 9138937016.d0
      disb(9,10) = -12522152863.d0
      disb(9,11) = 14390571690.d0
      disb(9,12) = -13976306577.d0
      disb(9,13) = 11544069096.d0
      disb(9,14) = -8154943146.d0
      disb(9,15) = 4953469860.d0
      disb(9,16) = -2600677782.d0
      disb(9,17) = 1185694886.d0
      disb(9,18) = -470886196.d0
      disb(9,19) = 162919262.d0
      disb(9,20) = -48903492.d0
      disb(9,21) = 12620256.d0
      disb(9,22) = -2760681.d0
      disb(9,23) = 501942.d0
      disb(9,24) = -73815.d0
      disb(9,25) = 8436.d0
      disb(9,26) = -703.d0
      disb(9,27) = 38.d0
      disb(9,28) = -1.d0
c
      disb(10,1) = -92378.d0
      disb(10,2) = 1830764.d0
      disb(10,3) = -17283084.d0
      disb(10,4) = 103423308.d0
      disb(10,5) = -440439570.d0
      disb(10,6) = 1420817568.d0
      disb(10,7) = -3608925189.d0
      disb(10,8) = 7408457694.d0
      disb(10,9) = -12522152863.d0
      disb(10,10) = 17672631900.d0
      disb(10,11) = -21055847747.d0
      disb(10,12) = 21372685686.d0
      disb(10,13) = -18631049241.d0
      disb(10,14) = 14050468992.d0
      disb(10,15) = -9229114530.d0
      disb(10,16) = 5311526988.d0
      disb(10,17) = -2690192064.d0
      disb(10,18) = 1201491524.d0
      disb(10,19) = -472641378.d0
      disb(10,20) = 163011640.d0
      disb(10,21) = -48903492.d0
      disb(10,22) = 12620256.d0
      disb(10,23) = -2760681.d0
      disb(10,24) = 501942.d0
      disb(10,25) = -73815.d0
      disb(10,26) = 8436.d0
      disb(10,27) = -703.d0
      disb(10,28) = 38.d0
      disb(10,29) = -1.d0
c
      disb(11,1) = 92378.d0
      disb(11,2) = -1847560.d0
      disb(11,3) = 17627402.d0
      disb(11,4) = -106797366.d0
      disb(11,5) = 461480436.d0
      disb(11,6) = -1514610954.d0
      disb(11,7) = 3927217464.d0
      disb(11,8) = -8263667853.d0
      disb(11,9) = 14390571690.d0
      disb(11,10) = -21055847747.d0
      disb(11,11) = 26206326784.d0
      disb(11,12) = -28037961743.d0
      disb(11,13) = 26027428350.d0
      disb(11,14) = -21137449137.d0
      disb(11,15) = 15124640376.d0
      disb(11,16) = -9587171658.d0
      disb(11,17) = 5401041270.d0
      disb(11,18) = -2705988702.d0
      disb(11,19) = 1203246706.d0
      disb(11,20) = -472733756.d0
      disb(11,21) = 163011640.d0
      disb(11,22) = -48903492.d0
      disb(11,23) = 12620256.d0
      disb(11,24) = -2760681.d0
      disb(11,25) = 501942.d0
      disb(11,26) = -73815.d0
      disb(11,27) = 8436.d0
      disb(11,28) = -703.d0
      disb(11,29) = 38.d0
      disb(11,30) = -1.d0
c
      disb(12,1) = -75582.d0
      disb(12,2) = 1528436.d0
      disb(12,3) = -14772082.d0
      disb(12,4) = 90866360.d0
      disb(12,5) = -399753198.d0
      disb(12,6) = 1340347932.d0
      disb(12,7) = -3565301778.d0
      disb(12,8) = 7735643280.d0
      disb(12,9) = -13976306577.d0
      disb(12,10) = 21372685686.d0
      disb(12,11) = -28037961743.d0
      disb(12,12) = 31918965508.d0
      disb(12,13) = -31846387559.d0
      disb(12,14) = 28078119174.d0
      disb(12,15) = -22016316633.d0
      disb(12,16) = 15417596208.d0
      disb(12,17) = -9660410616.d0
      disb(12,18) = 5413965792.d0
      disb(12,19) = -2707424760.d0
      disb(12,20) = 1203322288.d0
      disb(12,21) = -472733756.d0
      disb(12,22) = 163011640.d0
      disb(12,23) = -48903492.d0
      disb(12,24) = 12620256.d0
      disb(12,25) = -2760681.d0
      disb(12,26) = 501942.d0
      disb(12,27) = -73815.d0
      disb(12,28) = 8436.d0
      disb(12,29) = -703.d0
      disb(12,30) = 38.d0
      disb(12,31) = -1.d0
c
      disb(13,1) = 50388.d0
      disb(13,2) = -1032954.d0
      disb(13,3) = 10144784.d0
      disb(13,4) = -63598054.d0
      disb(13,5) = 286170248.d0
      disb(13,6) = -985664862.d0
      disb(13,7) = 2707475148.d0
      disb(13,8) = -6104252322.d0
      disb(13,9) = 11544069096.d0
      disb(13,10) = -18631049241.d0
      disb(13,11) = 26027428350.d0
      disb(13,12) = -31846387559.d0
      disb(13,13) = 34457916052.d0
      disb(13,14) = -33213514775.d0
      disb(13,15) = 28664030838.d0
      disb(13,16) = -22211620521.d0
      disb(13,17) = 15466422180.d0
      disb(13,18) = -9669026964.d0
      disb(13,19) = 5414923164.d0
      disb(13,20) = -2707475148.d0
      disb(13,21) = 1203322288.d0
      disb(13,22) = -472733756.d0
      disb(13,23) = 163011640.d0
      disb(13,24) = -48903492.d0
      disb(13,25) = 12620256.d0
      disb(13,26) = -2760681.d0
      disb(13,27) = 501942.d0
      disb(13,28) = -73815.d0
      disb(13,29) = 8436.d0
      disb(13,30) = -703.d0
      disb(13,31) = 38.d0
      disb(13,32) = -1.d0
c
      disb(14,1) = -27132.d0
      disb(14,2) = 565896.d0
      disb(14,3) = -5672526.d0
      disb(14,4) = 36435692.d0
      disb(14,5) = -168761686.d0
      disb(14,6) = 601661144.d0
      disb(14,7) = -1721810286.d0
      disb(14,8) = 4074602364.d0
      disb(14,9) = -8154943146.d0
      disb(14,10) = 14050468992.d0
      disb(14,11) = -21137449137.d0
      disb(14,12) = 28078119174.d0
      disb(14,13) = -33213514775.d0
      disb(14,14) = 35194061476.d0
      disb(14,15) = -33529005671.d0
      disb(14,16) = 28769194470.d0
      disb(14,17) = -22237911429.d0
      disb(14,18) = 15471061752.d0
      disb(14,19) = -9669542472.d0
      disb(14,20) = 5414950296.d0
      disb(14,21) = -2707475148.d0
      disb(14,22) = 1203322288.d0
      disb(14,23) = -472733756.d0
      disb(14,24) = 163011640.d0
      disb(14,25) = -48903492.d0
      disb(14,26) = 12620256.d0
      disb(14,27) = -2760681.d0
      disb(14,28) = 501942.d0
      disb(14,29) = -73815.d0
      disb(14,30) = 8436.d0
      disb(14,31) = -703.d0
      disb(14,32) = 38.d0
      disb(14,33) = -1.d0
c
      disb(15,1) = 11628.d0
      disb(15,2) = -248064.d0
      disb(15,3) = 2554284.d0
      disb(15,4) = -16940058.d0
      disb(15,5) = 81505820.d0
      disb(15,6) = -303972070.d0
      disb(15,7) = 917152040.d0
      disb(15,8) = -2307721950.d0
      disb(15,9) = 4953469860.d0
      disb(15,10) = -9229114530.d0
      disb(15,11) = 15124640376.d0
      disb(15,12) = -22016316633.d0
      disb(15,13) = 28664030838.d0
      disb(15,14) = -33529005671.d0
      disb(15,15) = 35329271860.d0
      disb(15,16) = -33574075799.d0
      disb(15,17) = 28780462002.d0
      disb(15,18) = -22239899817.d0
      disb(15,19) = 15471282684.d0
      disb(15,20) = -9669554100.d0
      disb(15,21) = 5414950296.d0
      disb(15,22) = -2707475148.d0
      disb(15,23) = 1203322288.d0
      disb(15,24) = -472733756.d0
      disb(15,25) = 163011640.d0
      disb(15,26) = -48903492.d0
      disb(15,27) = 12620256.d0
      disb(15,28) = -2760681.d0
      disb(15,29) = 501942.d0
      disb(15,30) = -73815.d0
      disb(15,31) = 8436.d0
      disb(15,32) = -703.d0
      disb(15,33) = 38.d0
      disb(15,34) = -1.d0
c
      disb(16,1) = -3876.d0
      disb(16,2) = 85272.d0
      disb(16,3) = -910860.d0
      disb(16,4) = 6310128.d0
      disb(16,5) = -31963434.d0
      disb(16,6) = 126575948.d0
      disb(16,7) = -409135702.d0
      disb(16,8) = 1112455928.d0
      disb(16,9) = -2600677782.d0
      disb(16,10) = 5311526988.d0
      disb(16,11) = -9587171658.d0
      disb(16,12) = 15417596208.d0
      disb(16,13) = -22211620521.d0
      disb(16,14) = 28769194470.d0
      disb(16,15) = -33574075799.d0
      disb(16,16) = 35344295236.d0
      disb(16,17) = -33577831643.d0
      disb(16,18) = 28781124798.d0
      disb(16,19) = -22239973461.d0
      disb(16,20) = 15471286560.d0
      disb(16,21) = -9669554100.d0
      disb(16,22) = 5414950296.d0
      disb(16,23) = -2707475148.d0
      disb(16,24) = 1203322288.d0
      disb(16,25) = -472733756.d0
      disb(16,26) = 163011640.d0
      disb(16,27) = -48903492.d0
      disb(16,28) = 12620256.d0
      disb(16,29) = -2760681.d0
      disb(16,30) = 501942.d0
      disb(16,31) = -73815.d0
      disb(16,32) = 8436.d0
      disb(16,33) = -703.d0
      disb(16,34) = 38.d0
      disb(16,35) = -1.d0
c
      disb(17,1) = 969.d0
      disb(17,2) = -22287.d0
      disb(17,3) = 250971.d0
      disb(17,4) = -1849821.d0
      disb(17,5) = 10065972.d0
      disb(17,6) = -43230966.d0
      disb(17,7) = 152866856.d0
      disb(17,8) = -457961674.d0
      disb(17,9) = 1185694886.d0
      disb(17,10) = -2690192064.d0
      disb(17,11) = 5401041270.d0
      disb(17,12) = -9660410616.d0
      disb(17,13) = 15466422180.d0
      disb(17,14) = -22237911429.d0
      disb(17,15) = 28780462002.d0
      disb(17,16) = -33577831643.d0
      disb(17,17) = 35345234197.d0
      disb(17,18) = -33577997342.d0
      disb(17,19) = 28781143209.d0
      disb(17,20) = -22239974430.d0
      disb(17,21) = 15471286560.d0
      disb(17,22) = -9669554100.d0
      disb(17,23) = 5414950296.d0
      disb(17,24) = -2707475148.d0
      disb(17,25) = 1203322288.d0
      disb(17,26) = -472733756.d0
      disb(17,27) = 163011640.d0
      disb(17,28) = -48903492.d0
      disb(17,29) = 12620256.d0
      disb(17,30) = -2760681.d0
      disb(17,31) = 501942.d0
      disb(17,32) = -73815.d0
      disb(17,33) = 8436.d0
      disb(17,34) = -703.d0
      disb(17,35) = 38.d0
      disb(17,36) = -1.d0
c
      disb(18,1) = -171.d0
      disb(18,2) = 4218.d0
      disb(18,3) = -51528.d0
      disb(18,4) = 416670.d0
      disb(18,5) = -2512617.d0
      disb(18,6) = 12054360.d0
      disb(18,7) = -47870538.d0
      disb(18,8) = 161483204.d0
      disb(18,9) = -470886196.d0
      disb(18,10) = 1201491524.d0
      disb(18,11) = -2705988702.d0
      disb(18,12) = 5413965792.d0
      disb(18,13) = -9669026964.d0
      disb(18,14) = 15471061752.d0
      disb(18,15) = -22239899817.d0
      disb(18,16) = 28781124798.d0
      disb(18,17) = -33577997342.d0
      disb(18,18) = 35345263438.d0
      disb(18,19) = -33578000591.d0
      disb(18,20) = 28781143380.d0
      disb(18,21) = -22239974430.d0
      disb(18,22) = 15471286560.d0
      disb(18,23) = -9669554100.d0
      disb(18,24) = 5414950296.d0
      disb(18,25) = -2707475148.d0
      disb(18,26) = 1203322288.d0
      disb(18,27) = -472733756.d0
      disb(18,28) = 163011640.d0
      disb(18,29) = -48903492.d0
      disb(18,30) = 12620256.d0
      disb(18,31) = -2760681.d0
      disb(18,32) = 501942.d0
      disb(18,33) = -73815.d0
      disb(18,34) = 8436.d0
      disb(18,35) = -703.d0
      disb(18,36) = 38.d0
      disb(18,37) = -1.d0
c
      disb(19,1) = 19.d0
      disb(19,2) = -532.d0
      disb(19,3) = 7467.d0
      disb(19,4) = -69939.d0
      disb(19,5) = 490314.d0
      disb(19,6) = -2733549.d0
      disb(19,7) = 12569868.d0
      disb(19,8) = -48827910.d0
      disb(19,9) = 162919262.d0
      disb(19,10) = -472641378.d0
      disb(19,11) = 1203246706.d0
      disb(19,12) = -2707424760.d0
      disb(19,13) = 5414923164.d0
      disb(19,14) = -9669542472.d0
      disb(19,15) = 15471282684.d0
      disb(19,16) = -22239973461.d0
      disb(19,17) = 28781143209.d0
      disb(19,18) = -33578000591.d0
      disb(19,19) = 35345263799.d0
      disb(19,20) = -33578000610.d0
      disb(19,21) = 28781143380.d0
      disb(19,22) = -22239974430.d0
      disb(19,23) = 15471286560.d0
      disb(19,24) = -9669554100.d0
      disb(19,25) = 5414950296.d0
      disb(19,26) = -2707475148.d0
      disb(19,27) = 1203322288.d0
      disb(19,28) = -472733756.d0
      disb(19,29) = 163011640.d0
      disb(19,30) = -48903492.d0
      disb(19,31) = 12620256.d0
      disb(19,32) = -2760681.d0
      disb(19,33) = 501942.d0
      disb(19,34) = -73815.d0
      disb(19,35) = 8436.d0
      disb(19,36) = -703.d0
      disb(19,37) = 38.d0
      disb(19,38) = -1.d0
c
      elseif(iord.eq.40)then
c 
c 40th order inner with 20th order boundaries
c ------------------------------------------
c
      disi(1) = -137846528820.d0
      disi(2) = 131282408400.d0
      disi(3) = -113380261800.d0
      disi(4) = 88732378800.d0
      disi(5) = -62852101650.d0
      disi(6) = 40225345056.d0
      disi(7) = -23206929840.d0
      disi(8) = 12033222880.d0
      disi(9) = -5586853480.d0
      disi(10) = 2311801440.d0
      disi(11) = -847660528.d0
      disi(12) = 273438880.d0
      disi(13) = -76904685.d0
      disi(14) = 18643560.d0
      disi(15) = -3838380.d0
      disi(16) = 658008.d0
      disi(17) = -91390.d0
      disi(18) = 9880.d0
      disi(19) = -780.d0
      disi(20) = 40.d0
      disi(21) = -1.d0
c
      disb(1,1) = -1.d0
      disb(1,2) = 20.d0
      disb(1,3) = -190.d0
      disb(1,4) = 1140.d0
      disb(1,5) = -4845.d0
      disb(1,6) = 15504.d0
      disb(1,7) = -38760.d0
      disb(1,8) = 77520.d0
      disb(1,9) = -125970.d0
      disb(1,10) = 167960.d0
      disb(1,11) = -184756.d0
      disb(1,12) = 167960.d0
      disb(1,13) = -125970.d0
      disb(1,14) = 77520.d0
      disb(1,15) = -38760.d0
      disb(1,16) = 15504.d0
      disb(1,17) = -4845.d0
      disb(1,18) = 1140.d0
      disb(1,19) = -190.d0
      disb(1,20) = 20.d0
      disb(1,21) = -1.d0
c
      disb(2,1) = 20.d0
      disb(2,2) = -401.d0
      disb(2,3) = 3820.d0
      disb(2,4) = -22990.d0
      disb(2,5) = 98040.d0
      disb(2,6) = -314925.d0
      disb(2,7) = 790704.d0
      disb(2,8) = -1589160.d0
      disb(2,9) = 2596920.d0
      disb(2,10) = -3485170.d0
      disb(2,11) = 3863080.d0
      disb(2,12) = -3543956.d0
      disb(2,13) = 2687360.d0
      disb(2,14) = -1676370.d0
      disb(2,15) = 852720.d0
      disb(2,16) = -348840.d0
      disb(2,17) = 112404.d0
      disb(2,18) = -27645.d0
      disb(2,19) = 4940.d0
      disb(2,20) = -590.d0
      disb(2,21) = 40.d0
      disb(2,22) = -1.d0
c
      disb(3,1) = -190.d0
      disb(3,2) = 3820.d0
      disb(3,3) = -36501.d0
      disb(3,4) = 220420.d0
      disb(3,5) = -943540.d0
      disb(3,6) = 3043800.d0
      disb(3,7) = -7679325.d0
      disb(3,8) = 15519504.d0
      disb(3,9) = -25523460.d0
      disb(3,10) = 34509320.d0
      disb(3,11) = -38588810.d0
      disb(3,12) = 35775480.d0
      disb(3,13) = -27478256.d0
      disb(3,14) = 17416160.d0
      disb(3,15) = -9040770.d0
      disb(3,16) = 3798480.d0
      disb(3,17) = -1269390.d0
      disb(3,18) = 329004.d0
      disb(3,19) = -63745.d0
      disb(3,20) = 8740.d0
      disb(3,21) = -780.d0
      disb(3,22) = 40.d0
      disb(3,23) = -1.d0
c
      disb(4,1) = 1140.d0
      disb(4,2) = -22990.d0
      disb(4,3) = 220420.d0
      disb(4,4) = -1336101.d0
      disb(4,5) = 5743720.d0
      disb(4,6) = -18618100.d0
      disb(4,7) = 47230200.d0
      disb(4,8) = -96052125.d0
      disb(4,9) = 159125304.d0
      disb(4,10) = -216997860.d0
      disb(4,11) = 245131160.d0
      disb(4,12) = -230063210.d0
      disb(4,13) = 179381280.d0
      disb(4,14) = -115851056.d0
      disb(4,15) = 61602560.d0
      disb(4,16) = -26715330.d0
      disb(4,17) = 9321780.d0
      disb(4,18) = -2568990.d0
      disb(4,19) = 545604.d0
      disb(4,20) = -86545.d0
      disb(4,21) = 9880.d0
      disb(4,22) = -780.d0
      disb(4,23) = 40.d0
      disb(4,24) = -1.d0
c
      disb(5,1) = -4845.d0
      disb(5,2) = 98040.d0
      disb(5,3) = -943540.d0
      disb(5,4) = 5743720.d0
      disb(5,5) = -24810126.d0
      disb(5,6) = 80860600.d0
      disb(5,7) = -206410300.d0
      disb(5,8) = 422814600.d0
      disb(5,9) = -706376775.d0
      disb(5,10) = 972891504.d0
      disb(5,11) = -1112140680.d0
      disb(5,12) = 1058897360.d0
      disb(5,13) = -840387860.d0
      disb(5,14) = 554965680.d0
      disb(5,15) = -303643256.d0
      disb(5,16) = 136719440.d0
      disb(5,17) = -50189355.d0
      disb(5,18) = 14845080.d0
      disb(5,19) = -3489540.d0
      disb(5,20) = 642504.d0
      disb(5,21) = -91390.d0
      disb(5,22) = 9880.d0
      disb(5,23) = -780.d0
      disb(5,24) = 40.d0
      disb(5,25) = -1.d0
c
      disb(6,1) = 15504.d0
      disb(6,2) = -314925.d0
      disb(6,3) = 3043800.d0
      disb(6,4) = -18618100.d0
      disb(6,5) = 80860600.d0
      disb(6,6) = -265184142.d0
      disb(6,7) = 681795640.d0
      disb(6,8) = -1408280380.d0
      disb(6,9) = 2375853480.d0
      disb(6,10) = -3310428615.d0
      disb(6,11) = 3837348528.d0
      disb(6,12) = -3716192520.d0
      disb(6,13) = 3011936240.d0
      disb(6,14) = -2042257940.d0
      disb(6,15) = 1155900720.d0
      disb(6,16) = -544017272.d0
      disb(6,17) = 211836320.d0
      disb(6,18) = -67863915.d0
      disb(6,19) = 17790840.d0
      disb(6,20) = -3799620.d0
      disb(6,21) = 658008.d0
      disb(6,22) = -91390.d0
      disb(6,23) = 9880.d0
      disb(6,24) = -780.d0
      disb(6,25) = 40.d0
      disb(6,26) = -1.d0
c
      disb(7,1) = -38760.d0
      disb(7,2) = 790704.d0
      disb(7,3) = -7679325.d0
      disb(7,4) = 47230200.d0
      disb(7,5) = -206410300.d0
      disb(7,6) = 681795640.d0
      disb(7,7) = -1767521742.d0
      disb(7,8) = 3686470840.d0
      disb(7,9) = -6290877580.d0
      disb(7,10) = 8885983080.d0
      disb(7,11) = -10471571175.d0
      disb(7,12) = 10347478128.d0
      disb(7,13) = -8598789720.d0
      disb(7,14) = 6016611440.d0
      disb(7,15) = -3544595540.d0
      disb(7,16) = 1756835760.d0
      disb(7,17) = -731809472.d0
      disb(7,18) = 256022720.d0
      disb(7,19) = -75228315.d0
      disb(7,20) = 18566040.d0
      disb(7,21) = -3838380.d0
      disb(7,22) = 658008.d0
      disb(7,23) = -91390.d0
      disb(7,24) = 9880.d0
      disb(7,25) = -780.d0
      disb(7,26) = 40.d0
      disb(7,27) = -1.d0
c
      disb(8,1) = 77520.d0
      disb(8,2) = -1589160.d0
      disb(8,3) = 15519504.d0
      disb(8,4) = -96052125.d0
      disb(8,5) = 422814600.d0
      disb(8,6) = -1408280380.d0
      disb(8,7) = 3686470840.d0
      disb(8,8) = -7776872142.d0
      disb(8,9) = 13451665240.d0
      disb(8,10) = -19311136780.d0
      disb(8,11) = 23208268200.d0
      disb(8,12) = -23491830375.d0
      disb(8,13) = 20112672528.d0
      disb(8,14) = -14608140120.d0
      disb(8,15) = 9021286640.d0
      disb(8,16) = -4746465620.d0
      disb(8,17) = 2132420160.d0
      disb(8,18) = -820182272.d0
      disb(8,19) = 270751520.d0
      disb(8,20) = -76778715.d0
      disb(8,21) = 18643560.d0
      disb(8,22) = -3838380.d0
      disb(8,23) = 658008.d0
      disb(8,24) = -91390.d0
      disb(8,25) = 9880.d0
      disb(8,26) = -780.d0
      disb(8,27) = 40.d0
      disb(8,28) = -1.d0
c
      disb(9,1) = -125970.d0
      disb(9,2) = 2596920.d0
      disb(9,3) = -25523460.d0
      disb(9,4) = 159125304.d0
      disb(9,5) = -706376775.d0
      disb(9,6) = 2375853480.d0
      disb(9,7) = -6290877580.d0
      disb(9,8) = 13451665240.d0
      disb(9,9) = -23645313042.d0
      disb(9,10) = 34609586440.d0
      disb(9,11) = -42584850100.d0
      disb(9,12) = 44366189400.d0
      disb(9,13) = -39360271275.d0
      disb(9,14) = 29877866928.d0
      disb(9,15) = -19490737320.d0
      disb(9,16) = 10974325520.d0
      disb(9,17) = -5356790270.d0
      disb(9,18) = 2276025960.d0
      disb(9,19) = -844116572.d0
      disb(9,20) = 273270920.d0
      disb(9,21) = -76904685.d0
      disb(9,22) = 18643560.d0
      disb(9,23) = -3838380.d0
      disb(9,24) = 658008.d0
      disb(9,25) = -91390.d0
      disb(9,26) = 9880.d0
      disb(9,27) = -780.d0
      disb(9,28) = 40.d0
      disb(9,29) = -1.d0
c
      disb(10,1) = 167960.d0
      disb(10,2) = -3485170.d0
      disb(10,3) = 34509320.d0
      disb(10,4) = -216997860.d0
      disb(10,5) = 972891504.d0
      disb(10,6) = -3310428615.d0
      disb(10,7) = 8885983080.d0
      disb(10,8) = -19311136780.d0
      disb(10,9) = 34609586440.d0
      disb(10,10) = -51855874642.d0
      disb(10,11) = 65641204200.d0
      disb(10,12) = -70795411700.d0
      disb(10,13) = 65524110600.d0
      disb(10,14) = -52380530475.d0
      disb(10,15) = 36387996528.d0
      disb(10,16) = -22094789160.d0
      disb(10,17) = 11788091720.d0
      disb(10,18) = -5548264670.d0
      disb(10,19) = 2307938360.d0
      disb(10,20) = -847475772.d0
      disb(10,21) = 273438880.d0
      disb(10,22) = -76904685.d0
      disb(10,23) = 18643560.d0
      disb(10,24) = -3838380.d0
      disb(10,25) = 658008.d0
      disb(10,26) = -91390.d0
      disb(10,27) = 9880.d0
      disb(10,28) = -780.d0
      disb(10,29) = 40.d0
      disb(10,30) = -1.d0
c
      disb(11,1) = -184756.d0
      disb(11,2) = 3863080.d0
      disb(11,3) = -38588810.d0
      disb(11,4) = 245131160.d0
      disb(11,5) = -1112140680.d0
      disb(11,6) = 3837348528.d0
      disb(11,7) = -10471571175.d0
      disb(11,8) = 23208268200.d0
      disb(11,9) = -42584850100.d0
      disb(11,10) = 65641204200.d0
      disb(11,11) = -85990654178.d0
      disb(11,12) = 96672821960.d0
      disb(11,13) = -94069125020.d0
      disb(11,14) = 79846395720.d0
      disb(11,15) = -59541673035.d0
      disb(11,16) = 39252453552.d0
      disb(11,17) = -22989931980.d0
      disb(11,18) = 11998713560.d0
      disb(11,19) = -5583368310.d0
      disb(11,20) = 2311633480.d0
      disb(11,21) = -847660528.d0
      disb(11,22) = 273438880.d0
      disb(11,23) = -76904685.d0
      disb(11,24) = 18643560.d0
      disb(11,25) = -3838380.d0
      disb(11,26) = 658008.d0
      disb(11,27) = -91390.d0
      disb(11,28) = 9880.d0
      disb(11,29) = -780.d0
      disb(11,30) = 40.d0
      disb(11,31) = -1.d0
c
      disb(12,1) = 167960.d0
      disb(12,2) = -3543956.d0
      disb(12,3) = 35775480.d0
      disb(12,4) = -230063210.d0
      disb(12,5) = 1058897360.d0
      disb(12,6) = -3716192520.d0
      disb(12,7) = 10347478128.d0
      disb(12,8) = -23491830375.d0
      disb(12,9) = 44366189400.d0
      disb(12,10) = -70795411700.d0
      disb(12,11) = 96672821960.d0
      disb(12,12) = -114201215778.d0
      disb(12,13) = 117830743160.d0
      disb(12,14) = -107089384220.d0
      disb(12,15) = 86356525320.d0
      disb(12,16) = -62145724875.d0
      disb(12,17) = 40066219752.d0
      disb(12,18) = -23181406380.d0
      disb(12,19) = 12030625960.d0
      disb(12,20) = -5586727510.d0
      disb(12,21) = 2311801440.d0
      disb(12,22) = -847660528.d0
      disb(12,23) = 273438880.d0
      disb(12,24) = -76904685.d0
      disb(12,25) = 18643560.d0
      disb(12,26) = -3838380.d0
      disb(12,27) = 658008.d0
      disb(12,28) = -91390.d0
      disb(12,29) = 9880.d0
      disb(12,30) = -780.d0
      disb(12,31) = 40.d0
      disb(12,32) = -1.d0
c
      disb(13,1) = -125970.d0
      disb(13,2) = 2687360.d0
      disb(13,3) = -27478256.d0
      disb(13,4) = 179381280.d0
      disb(13,5) = -840387860.d0
      disb(13,6) = 3011936240.d0
      disb(13,7) = -8598789720.d0
      disb(13,8) = 20112672528.d0
      disb(13,9) = -39360271275.d0
      disb(13,10) = 65524110600.d0
      disb(13,11) = -94069125020.d0
      disb(13,12) = 117830743160.d0
      disb(13,13) = -130069656678.d0
      disb(13,14) = 127595937560.d0
      disb(13,15) = -111971981420.d0
      disb(13,16) = 88309564200.d0
      disb(13,17) = -62756049525.d0
      disb(13,18) = 40209825552.d0
      disb(13,19) = -23205340680.d0
      disb(13,20) = 12033145360.d0
      disb(13,21) = -5586853480.d0
      disb(13,22) = 2311801440.d0
      disb(13,23) = -847660528.d0
      disb(13,24) = 273438880.d0
      disb(13,25) = -76904685.d0
      disb(13,26) = 18643560.d0
      disb(13,27) = -3838380.d0
      disb(13,28) = 658008.d0
      disb(13,29) = -91390.d0
      disb(13,30) = 9880.d0
      disb(13,31) = -780.d0
      disb(13,32) = 40.d0
      disb(13,33) = -1.d0
c
      disb(14,1) = 77520.d0
      disb(14,2) = -1676370.d0
      disb(14,3) = 17416160.d0
      disb(14,4) = -115851056.d0
      disb(14,5) = 554965680.d0
      disb(14,6) = -2042257940.d0
      disb(14,7) = 6016611440.d0
      disb(14,8) = -14608140120.d0
      disb(14,9) = 29877866928.d0
      disb(14,10) = -52380530475.d0
      disb(14,11) = 79846395720.d0
      disb(14,12) = -107089384220.d0
      disb(14,13) = 127595937560.d0
      disb(14,14) = -136079007078.d0
      disb(14,15) = 130600612760.d0
      disb(14,16) = -113173851500.d0
      disb(14,17) = 88685148600.d0
      disb(14,18) = -62844422325.d0
      disb(14,19) = 40224554352.d0
      disb(14,20) = -23206891080.d0
      disb(14,21) = 12033222880.d0
      disb(14,22) = -5586853480.d0
      disb(14,23) = 2311801440.d0
      disb(14,24) = -847660528.d0
      disb(14,25) = 273438880.d0
      disb(14,26) = -76904685.d0
      disb(14,27) = 18643560.d0
      disb(14,28) = -3838380.d0
      disb(14,29) = 658008.d0
      disb(14,30) = -91390.d0
      disb(14,31) = 9880.d0
      disb(14,32) = -780.d0
      disb(14,33) = 40.d0
      disb(14,34) = -1.d0
c
      disb(15,1) = -38760.d0
      disb(15,2) = 852720.d0
      disb(15,3) = -9040770.d0
      disb(15,4) = 61602560.d0
      disb(15,5) = -303643256.d0
      disb(15,6) = 1155900720.d0
      disb(15,7) = -3544595540.d0
      disb(15,8) = 9021286640.d0
      disb(15,9) = -19490737320.d0
      disb(15,10) = 36387996528.d0
      disb(15,11) = -59541673035.d0
      disb(15,12) = 86356525320.d0
      disb(15,13) = -111971981420.d0
      disb(15,14) = 130600612760.d0
      disb(15,15) = -137581344678.d0
      disb(15,16) = 131201547800.d0
      disb(15,17) = -113361643700.d0
      disb(15,18) = 88729335000.d0
      disb(15,19) = -62851786725.d0
      disb(15,20) = 40225329552.d0
      disb(15,21) = -23206929840.d0
      disb(15,22) = 12033222880.d0
      disb(15,23) = -5586853480.d0
      disb(15,24) = 2311801440.d0
      disb(15,25) = -847660528.d0
      disb(15,26) = 273438880.d0
      disb(15,27) = -76904685.d0
      disb(15,28) = 18643560.d0
      disb(15,29) = -3838380.d0
      disb(15,30) = 658008.d0
      disb(15,31) = -91390.d0
      disb(15,32) = 9880.d0
      disb(15,33) = -780.d0
      disb(15,34) = 40.d0
      disb(15,35) = -1.d0
c
      disb(16,1) = 15504.d0
      disb(16,2) = -348840.d0
      disb(16,3) = 3798480.d0
      disb(16,4) = -26715330.d0
      disb(16,5) = 136719440.d0
      disb(16,6) = -544017272.d0
      disb(16,7) = 1756835760.d0
      disb(16,8) = -4746465620.d0
      disb(16,9) = 10974325520.d0
      disb(16,10) = -22094789160.d0
      disb(16,11) = 39252453552.d0
      disb(16,12) = -62145724875.d0
      disb(16,13) = 88309564200.d0
      disb(16,14) = -113173851500.d0
      disb(16,15) = 131201547800.d0
      disb(16,16) = -137821718694.d0
      disb(16,17) = 131276664680.d0
      disb(16,18) = -113379318260.d0
      disb(16,19) = 88732280760.d0
      disb(16,20) = -62852096805.d0
      disb(16,21) = 40225345056.d0
      disb(16,22) = -23206929840.d0
      disb(16,23) = 12033222880.d0
      disb(16,24) = -5586853480.d0
      disb(16,25) = 2311801440.d0
      disb(16,26) = -847660528.d0
      disb(16,27) = 273438880.d0
      disb(16,28) = -76904685.d0
      disb(16,29) = 18643560.d0
      disb(16,30) = -3838380.d0
      disb(16,31) = 658008.d0
      disb(16,32) = -91390.d0
      disb(16,33) = 9880.d0
      disb(16,34) = -780.d0
      disb(16,35) = 40.d0
      disb(16,36) = -1.d0
c
      disb(17,1) = -4845.d0
      disb(17,2) = 112404.d0
      disb(17,3) = -1269390.d0
      disb(17,4) = 9321780.d0
      disb(17,5) = -50189355.d0
      disb(17,6) = 211836320.d0
      disb(17,7) = -731809472.d0
      disb(17,8) = 2132420160.d0
      disb(17,9) = -5356790270.d0
      disb(17,10) = 11788091720.d0
      disb(17,11) = -22989931980.d0
      disb(17,12) = 40066219752.d0
      disb(17,13) = -62756049525.d0
      disb(17,14) = 88685148600.d0
      disb(17,15) = -113361643700.d0
      disb(17,16) = 131276664680.d0
      disb(17,17) = -137845192719.d0
      disb(17,18) = 131282187980.d0
      disb(17,19) = -113380238810.d0
      disb(17,20) = 88732377660.d0
      disb(17,21) = -62852101650.d0
      disb(17,22) = 40225345056.d0
      disb(17,23) = -23206929840.d0
      disb(17,24) = 12033222880.d0
      disb(17,25) = -5586853480.d0
      disb(17,26) = 2311801440.d0
      disb(17,27) = -847660528.d0
      disb(17,28) = 273438880.d0
      disb(17,29) = -76904685.d0
      disb(17,30) = 18643560.d0
      disb(17,31) = -3838380.d0
      disb(17,32) = 658008.d0
      disb(17,33) = -91390.d0
      disb(17,34) = 9880.d0
      disb(17,35) = -780.d0
      disb(17,36) = 40.d0
      disb(17,37) = -1.d0
c
      disb(18,1) = 1140.d0
      disb(18,2) = -27645.d0
      disb(18,3) = 329004.d0
      disb(18,4) = -2568990.d0
      disb(18,5) = 14845080.d0
      disb(18,6) = -67863915.d0
      disb(18,7) = 256022720.d0
      disb(18,8) = -820182272.d0
      disb(18,9) = 2276025960.d0
      disb(18,10) = -5548264670.d0
      disb(18,11) = 11998713560.d0
      disb(18,12) = -23181406380.d0
      disb(18,13) = 40209825552.d0
      disb(18,14) = -62844422325.d0
      disb(18,15) = 88729335000.d0
      disb(18,16) = -113379318260.d0
      disb(18,17) = 131282187980.d0
      disb(18,18) = -137846492319.d0
      disb(18,19) = 131282404580.d0
      disb(18,20) = -113380261610.d0
      disb(18,21) = 88732378800.d0
      disb(18,22) = -62852101650.d0
      disb(18,23) = 40225345056.d0
      disb(18,24) = -23206929840.d0
      disb(18,25) = 12033222880.d0
      disb(18,26) = -5586853480.d0
      disb(18,27) = 2311801440.d0
      disb(18,28) = -847660528.d0
      disb(18,29) = 273438880.d0
      disb(18,30) = -76904685.d0
      disb(18,31) = 18643560.d0
      disb(18,32) = -3838380.d0
      disb(18,33) = 658008.d0
      disb(18,34) = -91390.d0
      disb(18,35) = 9880.d0
      disb(18,36) = -780.d0
      disb(18,37) = 40.d0
      disb(18,38) = -1.d0
c
      disb(19,1) = -190.d0
      disb(19,2) = 4940.d0
      disb(19,3) = -63745.d0
      disb(19,4) = 545604.d0
      disb(19,5) = -3489540.d0
      disb(19,6) = 17790840.d0
      disb(19,7) = -75228315.d0
      disb(19,8) = 270751520.d0
      disb(19,9) = -844116572.d0
      disb(19,10) = 2307938360.d0
      disb(19,11) = -5583368310.d0
      disb(19,12) = 12030625960.d0
      disb(19,13) = -23205340680.d0
      disb(19,14) = 40224554352.d0
      disb(19,15) = -62851786725.d0
      disb(19,16) = 88732280760.d0
      disb(19,17) = -113380238810.d0
      disb(19,18) = 131282404580.d0
      disb(19,19) = -137846528419.d0
      disb(19,20) = 131282408380.d0
      disb(19,21) = -113380261800.d0
      disb(19,22) = 88732378800.d0
      disb(19,23) = -62852101650.d0
      disb(19,24) = 40225345056.d0
      disb(19,25) = -23206929840.d0
      disb(19,26) = 12033222880.d0
      disb(19,27) = -5586853480.d0
      disb(19,28) = 2311801440.d0
      disb(19,29) = -847660528.d0
      disb(19,30) = 273438880.d0
      disb(19,31) = -76904685.d0
      disb(19,32) = 18643560.d0
      disb(19,33) = -3838380.d0
      disb(19,34) = 658008.d0
      disb(19,35) = -91390.d0
      disb(19,36) = 9880.d0
      disb(19,37) = -780.d0
      disb(19,38) = 40.d0
      disb(19,39) = -1.d0
c
      disb(20,1) = 20.d0
      disb(20,2) = -590.d0
      disb(20,3) = 8740.d0
      disb(20,4) = -86545.d0
      disb(20,5) = 642504.d0
      disb(20,6) = -3799620.d0
      disb(20,7) = 18566040.d0
      disb(20,8) = -76778715.d0
      disb(20,9) = 273270920.d0
      disb(20,10) = -847475772.d0
      disb(20,11) = 2311633480.d0
      disb(20,12) = -5586727510.d0
      disb(20,13) = 12033145360.d0
      disb(20,14) = -23206891080.d0
      disb(20,15) = 40225329552.d0
      disb(20,16) = -62852096805.d0
      disb(20,17) = 88732377660.d0
      disb(20,18) = -113380261610.d0
      disb(20,19) = 131282408380.d0
      disb(20,20) = -137846528819.d0
      disb(20,21) = 131282408400.d0
      disb(20,22) = -113380261800.d0
      disb(20,23) = 88732378800.d0
      disb(20,24) = -62852101650.d0
      disb(20,25) = 40225345056.d0
      disb(20,26) = -23206929840.d0
      disb(20,27) = 12033222880.d0
      disb(20,28) = -5586853480.d0
      disb(20,29) = 2311801440.d0
      disb(20,30) = -847660528.d0
      disb(20,31) = 273438880.d0
      disb(20,32) = -76904685.d0
      disb(20,33) = 18643560.d0
      disb(20,34) = -3838380.d0
      disb(20,35) = 658008.d0
      disb(20,36) = -91390.d0
      disb(20,37) = 9880.d0
      disb(20,38) = -780.d0
      disb(20,39) = 40.d0
      disb(20,40) = -1.d0
c
      elseif(iord.eq.42)then
c 
c 42nd order inner with 21st order boundaries
c ------------------------------------------
c
      disi(1) = 538257874440.d0
      disi(2) = -513791607420.d0
      disi(3) = 446775310800.d0
      disi(4) = -353697121050.d0
      disi(5) = 254661927156.d0
      disi(6) = -166509721602.d0
      disi(7) = 98672427616.d0
      disi(8) = -52860229080.d0
      disi(9) = 25518731280.d0
      disi(10) = -11058116888.d0
      disi(11) = 4280561376.d0
      disi(12) = -1471442973.d0
      disi(13) = 445891810.d0
      disi(14) = -118030185.d0
      disi(15) = 26978328.d0
      disi(16) = -5245786.d0
      disi(17) = 850668.d0
      disi(18) = -111930.d0
      disi(19) = 11480.d0
      disi(20) = -861.d0
      disi(21) = 42.d0
      disi(22) = -1.d0
c
      disb(1,1) = 1.d0
      disb(1,2) = -21.d0
      disb(1,3) = 210.d0
      disb(1,4) = -1330.d0
      disb(1,5) = 5985.d0
      disb(1,6) = -20349.d0
      disb(1,7) = 54264.d0
      disb(1,8) = -116280.d0
      disb(1,9) = 203490.d0
      disb(1,10) = -293930.d0
      disb(1,11) = 352716.d0
      disb(1,12) = -352716.d0
      disb(1,13) = 293930.d0
      disb(1,14) = -203490.d0
      disb(1,15) = 116280.d0
      disb(1,16) = -54264.d0
      disb(1,17) = 20349.d0
      disb(1,18) = -5985.d0
      disb(1,19) = 1330.d0
      disb(1,20) = -210.d0
      disb(1,21) = 21.d0
      disb(1,22) = -1.d0
c
      disb(2,1) = -21.d0
      disb(2,2) = 442.d0
      disb(2,3) = -4431.d0
      disb(2,4) = 28140.d0
      disb(2,5) = -127015.d0
      disb(2,6) = 433314.d0
      disb(2,7) = -1159893.d0
      disb(2,8) = 2496144.d0
      disb(2,9) = -4389570.d0
      disb(2,10) = 6376020.d0
      disb(2,11) = -7700966.d0
      disb(2,12) = 7759752.d0
      disb(2,13) = -6525246.d0
      disb(2,14) = 4567220.d0
      disb(2,15) = -2645370.d0
      disb(2,16) = 1255824.d0
      disb(2,17) = -481593.d0
      disb(2,18) = 146034.d0
      disb(2,19) = -33915.d0
      disb(2,20) = 5740.d0
      disb(2,21) = -651.d0
      disb(2,22) = 42.d0
      disb(2,23) = -1.d0
c
      disb(3,1) = 210.d0
      disb(3,2) = -4431.d0
      disb(3,3) = 44542.d0
      disb(3,4) = -283731.d0
      disb(3,5) = 1284990.d0
      disb(3,6) = -4400305.d0
      disb(3,7) = 11828754.d0
      disb(3,8) = -25578693.d0
      disb(3,9) = 45229044.d0
      disb(3,10) = -66114870.d0
      disb(3,11) = 80446380.d0
      disb(3,12) = -81771326.d0
      disb(3,13) = 69485052.d0
      disb(3,14) = -49258146.d0
      disb(3,15) = 28986020.d0
      disb(3,16) = -14040810.d0
      disb(3,17) = 5529114.d0
      disb(3,18) = -1738443.d0
      disb(3,19) = 425334.d0
      disb(3,20) = -78015.d0
      disb(3,21) = 10150.d0
      disb(3,22) = -861.d0
      disb(3,23) = 42.d0
      disb(3,24) = -1.d0
c
      disb(4,1) = -1330.d0
      disb(4,2) = 28140.d0
      disb(4,3) = -283731.d0
      disb(4,4) = 1813442.d0
      disb(4,5) = -8243781.d0
      disb(4,6) = 28349160.d0
      disb(4,7) = -76571425.d0
      disb(4,8) = 166481154.d0
      disb(4,9) = -296220393.d0
      disb(4,10) = 436155944.d0
      disb(4,11) = -535227150.d0
      disb(4,12) = 549558660.d0
      disb(4,13) = -472698226.d0
      disb(4,14) = 340126752.d0
      disb(4,15) = -203910546.d0
      disb(4,16) = 101157140.d0
      disb(4,17) = -41104980.d0
      disb(4,18) = 13489164.d0
      disb(4,19) = -3507343.d0
      disb(4,20) = 704634.d0
      disb(4,21) = -105945.d0
      disb(4,22) = 11480.d0
      disb(4,23) = -861.d0
      disb(4,24) = 42.d0
      disb(4,25) = -1.d0
c
      disb(5,1) = 5985.d0
      disb(5,2) = -127015.d0
      disb(5,3) = 1284990.d0
      disb(5,4) = -8243781.d0
      disb(5,5) = 37633667.d0
      disb(5,6) = -130032546.d0
      disb(5,7) = 353119200.d0
      disb(5,8) = -772507225.d0
      disb(5,9) = 1384368804.d0
      disb(5,10) = -2055391443.d0
      disb(5,11) = 2547161204.d0
      disb(5,12) = -2646232410.d0
      disb(5,13) = 2308729710.d0
      disb(5,14) = -1690585876.d0
      disb(5,15) = 1036062552.d0
      disb(5,16) = -528680586.d0
      disb(5,17) = 222945905.d0
      disb(5,18) = -76925205.d0
      disb(5,19) = 21449214.d0
      disb(5,20) = -4764193.d0
      disb(5,21) = 830319.d0
      disb(5,22) = -111930.d0
      disb(5,23) = 11480.d0
      disb(5,24) = -861.d0
      disb(5,25) = 42.d0
      disb(5,26) = -1.d0
c
      disb(6,1) = -20349.d0
      disb(6,2) = 433314.d0
      disb(6,3) = -4400305.d0
      disb(6,4) = 28349160.d0
      disb(6,5) = -130032546.d0
      disb(6,6) = 451715468.d0
      disb(6,7) = -1234250682.d0
      disb(6,8) = 2719300920.d0
      disb(6,9) = -4913325235.d0
      disb(6,10) = 7365550374.d0
      disb(6,11) = -9232809327.d0
      disb(6,12) = 9724579088.d0
      disb(6,13) = -8627413980.d0
      disb(6,14) = 6449547720.d0
      disb(6,15) = -4056767596.d0
      disb(6,16) = 2140280688.d0
      disb(6,17) = -942762387.d0
      disb(6,18) = 344734670.d0
      disb(6,19) = -103989375.d0
      disb(6,20) = 25722504.d0
      disb(6,21) = -5191522.d0
      disb(6,22) = 850668.d0
      disb(6,23) = -111930.d0
      disb(6,24) = 11480.d0
      disb(6,25) = -861.d0
      disb(6,26) = 42.d0
      disb(6,27) = -1.d0
c
      disb(7,1) = 54264.d0
      disb(7,2) = -1159893.d0
      disb(7,3) = 11828754.d0
      disb(7,4) = -76571425.d0
      disb(7,5) = 353119200.d0
      disb(7,6) = -1234250682.d0
      disb(7,7) = 3396297164.d0
      disb(7,8) = -7544068602.d0
      disb(7,9) = 13761482280.d0
      disb(7,10) = -20863142755.d0
      disb(7,11) = 26505331398.d0
      disb(7,12) = -28372590351.d0
      disb(7,13) = 25674396608.d0
      disb(7,14) = -19669595340.d0
      disb(7,15) = 12759365640.d0
      disb(7,16) = -7001349292.d0
      disb(7,17) = 3244498824.d0
      disb(7,18) = -1267532427.d0
      disb(7,19) = 416905790.d0
      disb(7,20) = -115384815.d0
      disb(7,21) = 26862048.d0
      disb(7,22) = -5245786.d0
      disb(7,23) = 850668.d0
      disb(7,24) = -111930.d0
      disb(7,25) = 11480.d0
      disb(7,26) = -861.d0
      disb(7,27) = 42.d0
      disb(7,28) = -1.d0
c
      disb(8,1) = -116280.d0
      disb(8,2) = 2496144.d0
      disb(8,3) = -25578693.d0
      disb(8,4) = 166481154.d0
      disb(8,5) = -772507225.d0
      disb(8,6) = 2719300920.d0
      disb(8,7) = -7544068602.d0
      disb(8,8) = 16917335564.d0
      disb(8,9) = -31205885802.d0
      disb(8,10) = 47939662680.d0
      disb(8,11) = -61876959235.d0
      disb(8,12) = 67519147878.d0
      disb(8,13) = -62550770751.d0
      disb(8,14) = 49336213808.d0
      disb(8,15) = -33190633740.d0
      disb(8,16) = 19069183560.d0
      disb(8,17) = -9367531012.d0
      disb(8,18) = 3940434624.d0
      disb(8,19) = -1422184827.d0
      disb(8,20) = 441324590.d0
      disb(8,21) = -117826695.d0
      disb(8,22) = 26978328.d0
      disb(8,23) = -5245786.d0
      disb(8,24) = 850668.d0
      disb(8,25) = -111930.d0
      disb(8,26) = 11480.d0
      disb(8,27) = -861.d0
      disb(8,28) = 42.d0
      disb(8,29) = -1.d0
c
      disb(9,1) = 203490.d0
      disb(9,2) = -4389570.d0
      disb(9,3) = 45229044.d0
      disb(9,4) = -296220393.d0
      disb(9,5) = 1384368804.d0
      disb(9,6) = -4913325235.d0
      disb(9,7) = 13761482280.d0
      disb(9,8) = -31205885802.d0
      disb(9,9) = 58325515664.d0
      disb(9,10) = -91017701502.d0
      disb(9,11) = 119713841520.d0
      disb(9,12) = -133651138075.d0
      disb(9,13) = 127330963578.d0
      disb(9,14) = -103958950851.d0
      disb(9,15) = 72998031008.d0
      disb(9,16) = -44232815100.d0
      disb(9,17) = 23210001570.d0
      disb(9,18) = -10585418662.d0
      disb(9,19) = 4211076324.d0
      disb(9,20) = -1464917727.d0
      disb(9,21) = 445597880.d0
      disb(9,22) = -118030185.d0
      disb(9,23) = 26978328.d0
      disb(9,24) = -5245786.d0
      disb(9,25) = 850668.d0
      disb(9,26) = -111930.d0
      disb(9,27) = 11480.d0
      disb(9,28) = -861.d0
      disb(9,29) = 42.d0
      disb(9,30) = -1.d0
c
      disb(10,1) = -293930.d0
      disb(10,2) = 6376020.d0
      disb(10,3) = -66114870.d0
      disb(10,4) = 436155944.d0
      disb(10,5) = -2055391443.d0
      disb(10,6) = 7365550374.d0
      disb(10,7) = -20863142755.d0
      disb(10,8) = 47939662680.d0
      disb(10,9) = -91017701502.d0
      disb(10,10) = 144720360564.d0
      disb(10,11) = -194691515382.d0
      disb(10,12) = 223387655400.d0
      disb(10,13) = -220045982975.d0
      disb(10,14) = 187142779278.d0
      disb(10,15) = -138137131251.d0
      disb(10,16) = 88947848528.d0
      disb(10,17) = -50213996670.d0
      disb(10,18) = 24969172620.d0
      disb(10,19) = -10976345562.d0
      disb(10,20) = 4272801624.d0
      disb(10,21) = -1471090257.d0
      disb(10,22) = 445891810.d0
      disb(10,23) = -118030185.d0
      disb(10,24) = 26978328.d0
      disb(10,25) = -5245786.d0
      disb(10,26) = 850668.d0
      disb(10,27) = -111930.d0
      disb(10,28) = 11480.d0
      disb(10,29) = -861.d0
      disb(10,30) = 42.d0
      disb(10,31) = -1.d0
c
      disb(11,1) = 352716.d0
      disb(11,2) = -7700966.d0
      disb(11,3) = 80446380.d0
      disb(11,4) = -535227150.d0
      disb(11,5) = 2547161204.d0
      disb(11,6) = -9232809327.d0
      disb(11,7) = 26505331398.d0
      disb(11,8) = -61876959235.d0
      disb(11,9) = 119713841520.d0
      disb(11,10) = -194691515382.d0
      disb(11,11) = 269128937220.d0
      disb(11,12) = -319100092038.d0
      disb(11,13) = 327061469280.d0
      disb(11,14) = -291820161815.d0
      disb(11,15) = 228156595758.d0
      disb(11,16) = -157276912275.d0
      disb(11,17) = 96125266412.d0
      disb(11,18) = -52325001930.d0
      disb(11,19) = 25438284900.d0
      disb(11,20) = -11050415922.d0
      disb(11,21) = 4280208660.d0
      disb(11,22) = -1471442973.d0
      disb(11,23) = 445891810.d0
      disb(11,24) = -118030185.d0
      disb(11,25) = 26978328.d0
      disb(11,26) = -5245786.d0
      disb(11,27) = 850668.d0
      disb(11,28) = -111930.d0
      disb(11,29) = 11480.d0
      disb(11,30) = -861.d0
      disb(11,31) = 42.d0
      disb(11,32) = -1.d0
c
      disb(12,1) = -352716.d0
      disb(12,2) = 7759752.d0
      disb(12,3) = -81771326.d0
      disb(12,4) = 549558660.d0
      disb(12,5) = -2646232410.d0
      disb(12,6) = 9724579088.d0
      disb(12,7) = -28372590351.d0
      disb(12,8) = 67519147878.d0
      disb(12,9) = -133651138075.d0
      disb(12,10) = 223387655400.d0
      disb(12,11) = -319100092038.d0
      disb(12,12) = 393537513876.d0
      disb(12,13) = -422773905918.d0
      disb(12,14) = 398835648120.d0
      disb(12,15) = -332833978295.d0
      disb(12,16) = 247296376782.d0
      disb(12,17) = -164454330159.d0
      disb(12,18) = 98236271672.d0
      disb(12,19) = -52794114210.d0
      disb(12,20) = 25512355260.d0
      disb(12,21) = -11057822958.d0
      disb(12,22) = 4280561376.d0
      disb(12,23) = -1471442973.d0
      disb(12,24) = 445891810.d0
      disb(12,25) = -118030185.d0
      disb(12,26) = 26978328.d0
      disb(12,27) = -5245786.d0
      disb(12,28) = 850668.d0
      disb(12,29) = -111930.d0
      disb(12,30) = 11480.d0
      disb(12,31) = -861.d0
      disb(12,32) = 42.d0
      disb(12,33) = -1.d0
c
      disb(13,1) = 293930.d0
      disb(13,2) = -6525246.d0
      disb(13,3) = 69485052.d0
      disb(13,4) = -472698226.d0
      disb(13,5) = 2308729710.d0
      disb(13,6) = -8627413980.d0
      disb(13,7) = 25674396608.d0
      disb(13,8) = -62550770751.d0
      disb(13,9) = 127330963578.d0
      disb(13,10) = -220045982975.d0
      disb(13,11) = 327061469280.d0
      disb(13,12) = -422773905918.d0
      disb(13,13) = 479932358776.d0
      disb(13,14) = -482585721618.d0
      disb(13,15) = 433013828520.d0
      disb(13,16) = -348783795815.d0
      disb(13,17) = 253277558352.d0
      disb(13,18) = -166213501209.d0
      disb(13,19) = 98627198572.d0
      disb(13,20) = -52855839510.d0
      disb(13,21) = 25518527790.d0
      disb(13,22) = -11058116888.d0
      disb(13,23) = 4280561376.d0
      disb(13,24) = -1471442973.d0
      disb(13,25) = 445891810.d0
      disb(13,26) = -118030185.d0
      disb(13,27) = 26978328.d0
      disb(13,28) = -5245786.d0
      disb(13,29) = 850668.d0
      disb(13,30) = -111930.d0
      disb(13,31) = 11480.d0
      disb(13,32) = -861.d0
      disb(13,33) = 42.d0
      disb(13,34) = -1.d0
c
      disb(14,1) = -203490.d0
      disb(14,2) = 4567220.d0
      disb(14,3) = -49258146.d0
      disb(14,4) = 340126752.d0
      disb(14,5) = -1690585876.d0
      disb(14,6) = 6449547720.d0
      disb(14,7) = -19669595340.d0
      disb(14,8) = 49336213808.d0
      disb(14,9) = -103958950851.d0
      disb(14,10) = 187142779278.d0
      disb(14,11) = -291820161815.d0
      disb(14,12) = 398835648120.d0
      disb(14,13) = -482585721618.d0
      disb(14,14) = 521340538876.d0
      disb(14,15) = -506247538818.d0
      disb(14,16) = 444056009880.d0
      disb(14,17) = -352924613825.d0
      disb(14,18) = 254495446002.d0
      disb(14,19) = -166484142909.d0
      disb(14,20) = 98669931472.d0
      disb(14,21) = -52860112800.d0
      disb(14,22) = 25518731280.d0
      disb(14,23) = -11058116888.d0
      disb(14,24) = 4280561376.d0
      disb(14,25) = -1471442973.d0
      disb(14,26) = 445891810.d0
      disb(14,27) = -118030185.d0
      disb(14,28) = 26978328.d0
      disb(14,29) = -5245786.d0
      disb(14,30) = 850668.d0
      disb(14,31) = -111930.d0
      disb(14,32) = 11480.d0
      disb(14,33) = -861.d0
      disb(14,34) = 42.d0
      disb(14,35) = -1.d0
c
      disb(15,1) = 116280.d0
      disb(15,2) = -2645370.d0
      disb(15,3) = 28986020.d0
      disb(15,4) = -203910546.d0
      disb(15,5) = 1036062552.d0
      disb(15,6) = -4056767596.d0
      disb(15,7) = 12759365640.d0
      disb(15,8) = -33190633740.d0
      disb(15,9) = 72998031008.d0
      disb(15,10) = -138137131251.d0
      disb(15,11) = 228156595758.d0
      disb(15,12) = -332833978295.d0
      disb(15,13) = 433013828520.d0
      disb(15,14) = -506247538818.d0
      disb(15,15) = 534861577276.d0
      disb(15,16) = -512557356738.d0
      disb(15,17) = 446422191600.d0
      disb(15,18) = -353620549625.d0
      disb(15,19) = 254650098402.d0
      disb(15,20) = -166508561709.d0
      disb(15,21) = 98672373352.d0
      disb(15,22) = -52860229080.d0
      disb(15,23) = 25518731280.d0
      disb(15,24) = -11058116888.d0
      disb(15,25) = 4280561376.d0
      disb(15,26) = -1471442973.d0
      disb(15,27) = 445891810.d0
      disb(15,28) = -118030185.d0
      disb(15,29) = 26978328.d0
      disb(15,30) = -5245786.d0
      disb(15,31) = 850668.d0
      disb(15,32) = -111930.d0
      disb(15,33) = 11480.d0
      disb(15,34) = -861.d0
      disb(15,35) = 42.d0
      disb(15,36) = -1.d0
c
      disb(16,1) = -54264.d0
      disb(16,2) = 1255824.d0
      disb(16,3) = -14040810.d0
      disb(16,4) = 101157140.d0
      disb(16,5) = -528680586.d0
      disb(16,6) = 2140280688.d0
      disb(16,7) = -7001349292.d0
      disb(16,8) = 19069183560.d0
      disb(16,9) = -44232815100.d0
      disb(16,10) = 88947848528.d0
      disb(16,11) = -157276912275.d0
      disb(16,12) = 247296376782.d0
      disb(16,13) = -348783795815.d0
      disb(16,14) = 444056009880.d0
      disb(16,15) = -512557356738.d0
      disb(16,16) = 537806158972.d0
      disb(16,17) = -513661574874.d0
      disb(16,18) = 446746961640.d0
      disb(16,19) = -353692720745.d0
      disb(16,20) = 254661493842.d0
      disb(16,21) = -166509701253.d0
      disb(16,22) = 98672427616.d0
      disb(16,23) = -52860229080.d0
      disb(16,24) = 25518731280.d0
      disb(16,25) = -11058116888.d0
      disb(16,26) = 4280561376.d0
      disb(16,27) = -1471442973.d0
      disb(16,28) = 445891810.d0
      disb(16,29) = -118030185.d0
      disb(16,30) = 26978328.d0
      disb(16,31) = -5245786.d0
      disb(16,32) = 850668.d0
      disb(16,33) = -111930.d0
      disb(16,34) = 11480.d0
      disb(16,35) = -861.d0
      disb(16,36) = 42.d0
      disb(16,37) = -1.d0
c
      disb(17,1) = 20349.d0
      disb(17,2) = -481593.d0
      disb(17,3) = 5529114.d0
      disb(17,4) = -41104980.d0
      disb(17,5) = 222945905.d0
      disb(17,6) = -942762387.d0
      disb(17,7) = 3244498824.d0
      disb(17,8) = -9367531012.d0
      disb(17,9) = 23210001570.d0
      disb(17,10) = -50213996670.d0
      disb(17,11) = 96125266412.d0
      disb(17,12) = -164454330159.d0
      disb(17,13) = 253277558352.d0
      disb(17,14) = -352924613825.d0
      disb(17,15) = 446422191600.d0
      disb(17,16) = -513661574874.d0
      disb(17,17) = 538220240773.d0
      disb(17,18) = -513783363639.d0
      disb(17,19) = 446774025810.d0
      disb(17,20) = -353696994035.d0
      disb(17,21) = 254661921171.d0
      disb(17,22) = -166509721602.d0
      disb(17,23) = 98672427616.d0
      disb(17,24) = -52860229080.d0
      disb(17,25) = 25518731280.d0
      disb(17,26) = -11058116888.d0
      disb(17,27) = 4280561376.d0
      disb(17,28) = -1471442973.d0
      disb(17,29) = 445891810.d0
      disb(17,30) = -118030185.d0
      disb(17,31) = 26978328.d0
      disb(17,32) = -5245786.d0
      disb(17,33) = 850668.d0
      disb(17,34) = -111930.d0
      disb(17,35) = 11480.d0
      disb(17,36) = -861.d0
      disb(17,37) = 42.d0
      disb(17,38) = -1.d0
c
      disb(18,1) = -5985.d0
      disb(18,2) = 146034.d0
      disb(18,3) = -1738443.d0
      disb(18,4) = 13489164.d0
      disb(18,5) = -76925205.d0
      disb(18,6) = 344734670.d0
      disb(18,7) = -1267532427.d0
      disb(18,8) = 3940434624.d0
      disb(18,9) = -10585418662.d0
      disb(18,10) = 24969172620.d0
      disb(18,11) = -52325001930.d0
      disb(18,12) = 98236271672.d0
      disb(18,13) = -166213501209.d0
      disb(18,14) = 254495446002.d0
      disb(18,15) = -353620549625.d0
      disb(18,16) = 446746961640.d0
      disb(18,17) = -513783363639.d0
      disb(18,18) = 538256060998.d0
      disb(18,19) = -513791323689.d0
      disb(18,20) = 446775282660.d0
      disb(18,21) = -353697119720.d0
      disb(18,22) = 254661927156.d0
      disb(18,23) = -166509721602.d0
      disb(18,24) = 98672427616.d0
      disb(18,25) = -52860229080.d0
      disb(18,26) = 25518731280.d0
      disb(18,27) = -11058116888.d0
      disb(18,28) = 4280561376.d0
      disb(18,29) = -1471442973.d0
      disb(18,30) = 445891810.d0
      disb(18,31) = -118030185.d0
      disb(18,32) = 26978328.d0
      disb(18,33) = -5245786.d0
      disb(18,34) = 850668.d0
      disb(18,35) = -111930.d0
      disb(18,36) = 11480.d0
      disb(18,37) = -861.d0
      disb(18,38) = 42.d0
      disb(18,39) = -1.d0
c
      disb(19,1) = 1330.d0
      disb(19,2) = -33915.d0
      disb(19,3) = 425334.d0
      disb(19,4) = -3507343.d0
      disb(19,5) = 21449214.d0
      disb(19,6) = -103989375.d0
      disb(19,7) = 416905790.d0
      disb(19,8) = -1422184827.d0
      disb(19,9) = 4211076324.d0
      disb(19,10) = -10976345562.d0
      disb(19,11) = 25438284900.d0
      disb(19,12) = -52794114210.d0
      disb(19,13) = 98627198572.d0
      disb(19,14) = -166484142909.d0
      disb(19,15) = 254650098402.d0
      disb(19,16) = -353692720745.d0
      disb(19,17) = 446774025810.d0
      disb(19,18) = -513791323689.d0
      disb(19,19) = 538257829898.d0
      disb(19,20) = -513791602989.d0
      disb(19,21) = 446775310590.d0
      disb(19,22) = -353697121050.d0
      disb(19,23) = 254661927156.d0
      disb(19,24) = -166509721602.d0
      disb(19,25) = 98672427616.d0
      disb(19,26) = -52860229080.d0
      disb(19,27) = 25518731280.d0
      disb(19,28) = -11058116888.d0
      disb(19,29) = 4280561376.d0
      disb(19,30) = -1471442973.d0
      disb(19,31) = 445891810.d0
      disb(19,32) = -118030185.d0
      disb(19,33) = 26978328.d0
      disb(19,34) = -5245786.d0
      disb(19,35) = 850668.d0
      disb(19,36) = -111930.d0
      disb(19,37) = 11480.d0
      disb(19,38) = -861.d0
      disb(19,39) = 42.d0
      disb(19,40) = -1.d0
c
      disb(20,1) = -210.d0
      disb(20,2) = 5740.d0
      disb(20,3) = -78015.d0
      disb(20,4) = 704634.d0
      disb(20,5) = -4764193.d0
      disb(20,6) = 25722504.d0
      disb(20,7) = -115384815.d0
      disb(20,8) = 441324590.d0
      disb(20,9) = -1464917727.d0
      disb(20,10) = 4272801624.d0
      disb(20,11) = -11050415922.d0
      disb(20,12) = 25512355260.d0
      disb(20,13) = -52855839510.d0
      disb(20,14) = 98669931472.d0
      disb(20,15) = -166508561709.d0
      disb(20,16) = 254661493842.d0
      disb(20,17) = -353696994035.d0
      disb(20,18) = 446775282660.d0
      disb(20,19) = -513791602989.d0
      disb(20,20) = 538257873998.d0
      disb(20,21) = -513791607399.d0
      disb(20,22) = 446775310800.d0
      disb(20,23) = -353697121050.d0
      disb(20,24) = 254661927156.d0
      disb(20,25) = -166509721602.d0
      disb(20,26) = 98672427616.d0
      disb(20,27) = -52860229080.d0
      disb(20,28) = 25518731280.d0
      disb(20,29) = -11058116888.d0
      disb(20,30) = 4280561376.d0
      disb(20,31) = -1471442973.d0
      disb(20,32) = 445891810.d0
      disb(20,33) = -118030185.d0
      disb(20,34) = 26978328.d0
      disb(20,35) = -5245786.d0
      disb(20,36) = 850668.d0
      disb(20,37) = -111930.d0
      disb(20,38) = 11480.d0
      disb(20,39) = -861.d0
      disb(20,40) = 42.d0
      disb(20,41) = -1.d0
c
      disb(21,1) = 21.d0
      disb(21,2) = -651.d0
      disb(21,3) = 10150.d0
      disb(21,4) = -105945.d0
      disb(21,5) = 830319.d0
      disb(21,6) = -5191522.d0
      disb(21,7) = 26862048.d0
      disb(21,8) = -117826695.d0
      disb(21,9) = 445597880.d0
      disb(21,10) = -1471090257.d0
      disb(21,11) = 4280208660.d0
      disb(21,12) = -11057822958.d0
      disb(21,13) = 25518527790.d0
      disb(21,14) = -52860112800.d0
      disb(21,15) = 98672373352.d0
      disb(21,16) = -166509701253.d0
      disb(21,17) = 254661921171.d0
      disb(21,18) = -353697119720.d0
      disb(21,19) = 446775310590.d0
      disb(21,20) = -513791607399.d0
      disb(21,21) = 538257874439.d0
      disb(21,22) = -513791607420.d0
      disb(21,23) = 446775310800.d0
      disb(21,24) = -353697121050.d0
      disb(21,25) = 254661927156.d0
      disb(21,26) = -166509721602.d0
      disb(21,27) = 98672427616.d0
      disb(21,28) = -52860229080.d0
      disb(21,29) = 25518731280.d0
      disb(21,30) = -11058116888.d0
      disb(21,31) = 4280561376.d0
      disb(21,32) = -1471442973.d0
      disb(21,33) = 445891810.d0
      disb(21,34) = -118030185.d0
      disb(21,35) = 26978328.d0
      disb(21,36) = -5245786.d0
      disb(21,37) = 850668.d0
      disb(21,38) = -111930.d0
      disb(21,39) = 11480.d0
      disb(21,40) = -861.d0
      disb(21,41) = 42.d0
      disb(21,42) = -1.d0
c
      elseif(iord.eq.44)then
c 
c 44th order inner with 22nd order boundaries
c ------------------------------------------
c
      disi(1) = -2104098963720.d0
      disi(2) = 2012616400080.d0
      disi(3) = -1761039350070.d0
      disi(4) = 1408831480056.d0
      disi(5) = -1029530696964.d0
      disi(6) = 686353797976.d0
      disi(7) = -416714805914.d0
      disi(8) = 229911617056.d0
      disi(9) = -114955808528.d0
      disi(10) = 51915526432.d0
      disi(11) = -21090682613.d0
      disi(12) = 7669339132.d0
      disi(13) = -2481256778.d0
      disi(14) = 708930508.d0
      disi(15) = -177232627.d0
      disi(16) = 38320568.d0
      disi(17) = -7059052.d0
      disi(18) = 1086008.d0
      disi(19) = -135751.d0
      disi(20) = 13244.d0
      disi(21) = -946.d0
      disi(22) = 44.d0
      disi(23) = -1.d0
c
      disb(1,1) = -1.d0
      disb(1,2) = 22.d0
      disb(1,3) = -231.d0
      disb(1,4) = 1540.d0
      disb(1,5) = -7315.d0
      disb(1,6) = 26334.d0
      disb(1,7) = -74613.d0
      disb(1,8) = 170544.d0
      disb(1,9) = -319770.d0
      disb(1,10) = 497420.d0
      disb(1,11) = -646646.d0
      disb(1,12) = 705432.d0
      disb(1,13) = -646646.d0
      disb(1,14) = 497420.d0
      disb(1,15) = -319770.d0
      disb(1,16) = 170544.d0
      disb(1,17) = -74613.d0
      disb(1,18) = 26334.d0
      disb(1,19) = -7315.d0
      disb(1,20) = 1540.d0
      disb(1,21) = -231.d0
      disb(1,22) = 22.d0
      disb(1,23) = -1.d0
c
      disb(2,1) = 22.d0
      disb(2,2) = -485.d0
      disb(2,3) = 5104.d0
      disb(2,4) = -34111.d0
      disb(2,5) = 162470.d0
      disb(2,6) = -586663.d0
      disb(2,7) = 1667820.d0
      disb(2,8) = -3826581.d0
      disb(2,9) = 7205484.d0
      disb(2,10) = -11263010.d0
      disb(2,11) = 14723632.d0
      disb(2,12) = -16166150.d0
      disb(2,13) = 14931644.d0
      disb(2,14) = -11589886.d0
      disb(2,15) = 7532360.d0
      disb(2,16) = -4071738.d0
      disb(2,17) = 1812030.d0
      disb(2,18) = -653961.d0
      disb(2,19) = 187264.d0
      disb(2,20) = -41195.d0
      disb(2,21) = 6622.d0
      disb(2,22) = -715.d0
      disb(2,23) = 44.d0
      disb(2,24) = -1.d0
c
      disb(3,1) = -231.d0
      disb(3,2) = 5104.d0
      disb(3,3) = -53846.d0
      disb(3,4) = 360844.d0
      disb(3,5) = -1723876.d0
      disb(3,6) = 6245624.d0
      disb(3,7) = -17822266.d0
      disb(3,8) = 41063484.d0
      disb(3,9) = -77693451.d0
      disb(3,10) = 122109504.d0
      disb(3,11) = -160638236.d0
      disb(3,12) = 177678424.d0
      disb(3,13) = -165541376.d0
      disb(3,14) = 129835664.d0
      disb(3,15) = -85456756.d0
      disb(3,16) = 46928024.d0
      disb(3,17) = -21307341.d0
      disb(3,18) = 7895184.d0
      disb(3,19) = -2343726.d0
      disb(3,20) = 543004.d0
      disb(3,21) = -94556.d0
      disb(3,22) = 11704.d0
      disb(3,23) = -946.d0
      disb(3,24) = 44.d0
      disb(3,25) = -1.d0
c
      disb(4,1) = 1540.d0
      disb(4,2) = -34111.d0
      disb(4,3) = 360844.d0
      disb(4,4) = -2425446.d0
      disb(4,5) = 11625944.d0
      disb(4,6) = -42278236.d0
      disb(4,7) = 121149644.d0
      disb(4,8) = -280460026.d0
      disb(4,9) = 533509284.d0
      disb(4,10) = -843720251.d0
      disb(4,11) = 1117944344.d0
      disb(4,12) = -1247003516.d0
      disb(4,13) = 1173513264.d0
      disb(4,14) = -931568176.d0
      disb(4,15) = 622281464.d0
      disb(4,16) = -348094516.d0
      disb(4,17) = 161832044.d0
      disb(4,18) = -61861701.d0
      disb(4,19) = 19160284.d0
      disb(4,20) = -4715326.d0
      disb(4,21) = 898744.d0
      disb(4,22) = -128436.d0
      disb(4,23) = 13244.d0
      disb(4,24) = -946.d0
      disb(4,25) = 44.d0
      disb(4,26) = -1.d0
c
      disb(5,1) = -7315.d0
      disb(5,2) = 162470.d0
      disb(5,3) = -1723876.d0
      disb(5,4) = 11625944.d0
      disb(5,5) = -55934671.d0
      disb(5,6) = 204259154.d0
      disb(5,7) = -588072331.d0
      disb(5,8) = 1368679004.d0
      disb(5,9) = -2619577576.d0
      disb(5,10) = 4172136584.d0
      disb(5,11) = -5573935741.d0
      disb(5,12) = 6278179424.d0
      disb(5,13) = -5977219006.d0
      disb(5,14) = 4812140564.d0
      disb(5,15) = -3270685726.d0
      disb(5,16) = 1869810824.d0
      disb(5,17) = -893888611.d0
      disb(5,18) = 354465254.d0
      disb(5,19) = -115370926.d0
      disb(5,20) = 30425384.d0
      disb(5,21) = -6405091.d0
      disb(5,22) = 1059674.d0
      disb(5,23) = -135751.d0
      disb(5,24) = 13244.d0
      disb(5,25) = -946.d0
      disb(5,26) = 44.d0
      disb(5,27) = -1.d0
c
      disb(6,1) = 26334.d0
      disb(6,2) = -586663.d0
      disb(6,3) = 6245624.d0
      disb(6,4) = -42278236.d0
      disb(6,5) = 204259154.d0
      disb(6,6) = -749414227.d0
      disb(6,7) = 2169117896.d0
      disb(6,8) = -5079178027.d0
      disb(6,9) = 9789502184.d0
      disb(6,10) = -15718635856.d0
      disb(6,11) = 21200912348.d0
      disb(6,12) = -24150782029.d0
      disb(6,13) = 23306955188.d0
      disb(6,14) = -19076277286.d0
      disb(6,15) = 13232963744.d0
      disb(6,16) = -7761791422.d0
      disb(6,17) = 3834669566.d0
      disb(6,18) = -1587368167.d0
      disb(6,19) = 547098464.d0
      disb(6,20) = -155925286.d0
      disb(6,21) = 36508538.d0
      disb(6,22) = -6984439.d0
      disb(6,23) = 1086008.d0
      disb(6,24) = -135751.d0
      disb(6,25) = 13244.d0
      disb(6,26) = -946.d0
      disb(6,27) = 44.d0
      disb(6,28) = -1.d0
c
      disb(7,1) = -74613.d0
      disb(7,2) = 1667820.d0
      disb(7,3) = -17822266.d0
      disb(7,4) = 121149644.d0
      disb(7,5) = -588072331.d0
      disb(7,6) = 2169117896.d0
      disb(7,7) = -6316513996.d0
      disb(7,8) = 14893917368.d0
      disb(7,9) = -28938177037.d0
      disb(7,10) = 46903500644.d0
      disb(7,11) = -63966833854.d0
      disb(7,12) = 73835310164.d0
      disb(7,13) = -72398980027.d0
      disb(7,14) = 60420953648.d0
      disb(7,15) = -42935276296.d0
      disb(7,16) = 25957763216.d0
      disb(7,17) = -13328891191.d0
      disb(7,18) = 5799528308.d0
      disb(7,19) = -2133162262.d0
      disb(7,20) = 662002484.d0
      disb(7,21) = -173160889.d0
      disb(7,22) = 38150024.d0
      disb(7,23) = -7059052.d0
      disb(7,24) = 1086008.d0
      disb(7,25) = -135751.d0
      disb(7,26) = 13244.d0
      disb(7,27) = -946.d0
      disb(7,28) = 44.d0
      disb(7,29) = -1.d0
c
      disb(8,1) = 170544.d0
      disb(8,2) = -3826581.d0
      disb(8,3) = 41063484.d0
      disb(8,4) = -280460026.d0
      disb(8,5) = 1368679004.d0
      disb(8,6) = -5079178027.d0
      disb(8,7) = 14893917368.d0
      disb(8,8) = -35401769932.d0
      disb(8,9) = 69428772248.d0
      disb(8,10) = -113770173517.d0
      disb(8,11) = 157185096068.d0
      disb(8,12) = -184274028862.d0
      disb(8,13) = 184116905588.d0
      disb(8,14) = -157230976507.d0
      disb(8,15) = 114955808528.d0
      disb(8,16) = -72020532232.d0
      disb(8,17) = 38682562688.d0
      disb(8,18) = -17819996887.d0
      disb(8,19) = 7047057668.d0
      disb(8,20) = -2395800022.d0
      disb(8,21) = 701398148.d0
      disb(8,22) = -176912857.d0
      disb(8,23) = 38320568.d0
      disb(8,24) = -7059052.d0
      disb(8,25) = 1086008.d0
      disb(8,26) = -135751.d0
      disb(8,27) = 13244.d0
      disb(8,28) = -946.d0
      disb(8,29) = 44.d0
      disb(8,30) = -1.d0
c
      disb(9,1) = -319770.d0
      disb(9,2) = 7205484.d0
      disb(9,3) = -77693451.d0
      disb(9,4) = 533509284.d0
      disb(9,5) = -2619577576.d0
      disb(9,6) = 9789502184.d0
      disb(9,7) = -28938177037.d0
      disb(9,8) = 69428772248.d0
      disb(9,9) = -137654622832.d0
      disb(9,10) = 228488765648.d0
      disb(9,11) = -320548164937.d0
      disb(9,12) = 382761086708.d0
      disb(9,13) = -391052020282.d0
      disb(9,14) = 343176898988.d0
      disb(9,15) = -259483829407.d0
      disb(9,16) = 169490663408.d0
      disb(9,17) = -95879531242.d0
      disb(9,18) = 47103385868.d0
      disb(9,19) = -20159114437.d0
      disb(9,20) = 7539503468.d0
      disb(9,21) = -2469666892.d0
      disb(9,22) = 708433088.d0
      disb(9,23) = -177232627.d0
      disb(9,24) = 38320568.d0
      disb(9,25) = -7059052.d0
      disb(9,26) = 1086008.d0
      disb(9,27) = -135751.d0
      disb(9,28) = 13244.d0
      disb(9,29) = -946.d0
      disb(9,30) = 44.d0
      disb(9,31) = -1.d0
c
      disb(10,1) = 497420.d0
      disb(10,2) = -11263010.d0
      disb(10,3) = 122109504.d0
      disb(10,4) = -843720251.d0
      disb(10,5) = 4172136584.d0
      disb(10,6) = -15718635856.d0
      disb(10,7) = 46903500644.d0
      disb(10,8) = -113770173517.d0
      disb(10,9) = 228488765648.d0
      disb(10,10) = -385081279232.d0
      disb(10,11) = 550143418968.d0
      disb(10,12) = -671444150377.d0
      disb(10,13) = 704415740028.d0
      disb(10,14) = -638478676682.d0
      disb(10,15) = 502236892388.d0
      disb(10,16) = -344315825887.d0
      disb(10,17) = 206604661868.d0
      disb(10,18) = -108978589522.d0
      disb(10,19) = 50742013168.d0
      disb(10,20) = -20925141237.d0
      disb(10,21) = 7654407488.d0
      disb(10,22) = -2480610132.d0
      disb(10,23) = 708930508.d0
      disb(10,24) = -177232627.d0
      disb(10,25) = 38320568.d0
      disb(10,26) = -7059052.d0
      disb(10,27) = 1086008.d0
      disb(10,28) = -135751.d0
      disb(10,29) = 13244.d0
      disb(10,30) = -946.d0
      disb(10,31) = 44.d0
      disb(10,32) = -1.d0
c
      disb(11,1) = -646646.d0
      disb(11,2) = 14723632.d0
      disb(11,3) = -160638236.d0
      disb(11,4) = 1117944344.d0
      disb(11,5) = -5573935741.d0
      disb(11,6) = 21200912348.d0
      disb(11,7) = -63966833854.d0
      disb(11,8) = 157185096068.d0
      disb(11,9) = -320548164937.d0
      disb(11,10) = 550143418968.d0
      disb(11,11) = -803232328548.d0
      disb(11,12) = 1006308200040.d0
      disb(11,13) = -1089595199693.d0
      disb(11,14) = 1026070393348.d0
      disb(11,15) = -845256668102.d0
      disb(11,16) = 612518487812.d0
      disb(11,17) = -392564023885.d0
      disb(11,18) = 223633437632.d0
      disb(11,19) = -113708805012.d0
      disb(11,20) = 51737848008.d0
      disb(11,21) = -21074516463.d0
      disb(11,22) = 7668633700.d0
      disb(11,23) = -2481256778.d0
      disb(11,24) = 708930508.d0
      disb(11,25) = -177232627.d0
      disb(11,26) = 38320568.d0
      disb(11,27) = -7059052.d0
      disb(11,28) = 1086008.d0
      disb(11,29) = -135751.d0
      disb(11,30) = 13244.d0
      disb(11,31) = -946.d0
      disb(11,32) = 44.d0
      disb(11,33) = -1.d0
c
      disb(12,1) = 705432.d0
      disb(12,2) = -16166150.d0
      disb(12,3) = 177678424.d0
      disb(12,4) = -1247003516.d0
      disb(12,5) = 6278179424.d0
      disb(12,6) = -24150782029.d0
      disb(12,7) = 73835310164.d0
      disb(12,8) = -184274028862.d0
      disb(12,9) = 382761086708.d0
      disb(12,10) = -671444150377.d0
      disb(12,11) = 1006308200040.d0
      disb(12,12) = -1300866635172.d0
      disb(12,13) = 1462472981112.d0
      disb(12,14) = -1440491185133.d0
      disb(12,15) = 1251646383988.d0
      disb(12,16) = -965563863110.d0
      disb(12,17) = 665152885628.d0
      disb(12,18) = -411140870173.d0
      disb(12,19) = 228793672712.d0
      disb(12,20) = -114795170292.d0
      disb(12,21) = 51900802800.d0
      disb(12,22) = -21090035967.d0
      disb(12,23) = 7669339132.d0
      disb(12,24) = -2481256778.d0
      disb(12,25) = 708930508.d0
      disb(12,26) = -177232627.d0
      disb(12,27) = 38320568.d0
      disb(12,28) = -7059052.d0
      disb(12,29) = 1086008.d0
      disb(12,30) = -135751.d0
      disb(12,31) = 13244.d0
      disb(12,32) = -946.d0
      disb(12,33) = 44.d0
      disb(12,34) = -1.d0
c
      disb(13,1) = -646646.d0
      disb(13,2) = 14931644.d0
      disb(13,3) = -165541376.d0
      disb(13,4) = 1173513264.d0
      disb(13,5) = -5977219006.d0
      disb(13,6) = 23306955188.d0
      disb(13,7) = -72398980027.d0
      disb(13,8) = 184116905588.d0
      disb(13,9) = -391052020282.d0
      disb(13,10) = 704415740028.d0
      disb(13,11) = -1089595199693.d0
      disb(13,12) = 1462472981112.d0
      disb(13,13) = -1719017684488.d0
      disb(13,14) = 1784127634432.d0
      disb(13,15) = -1647269176553.d0
      disb(13,16) = 1361927979412.d0
      disb(13,17) = -1013812061108.d0
      disb(13,18) = 682181661392.d0
      disb(13,19) = -415871085663.d0
      disb(13,20) = 229789507552.d0
      disb(13,21) = -114944545518.d0
      disb(13,22) = 51915029012.d0
      disb(13,23) = -21090682613.d0
      disb(13,24) = 7669339132.d0
      disb(13,25) = -2481256778.d0
      disb(13,26) = 708930508.d0
      disb(13,27) = -177232627.d0
      disb(13,28) = 38320568.d0
      disb(13,29) = -7059052.d0
      disb(13,30) = 1086008.d0
      disb(13,31) = -135751.d0
      disb(13,32) = 13244.d0
      disb(13,33) = -946.d0
      disb(13,34) = 44.d0
      disb(13,35) = -1.d0
c
      disb(14,1) = 497420.d0
      disb(14,2) = -11589886.d0
      disb(14,3) = 129835664.d0
      disb(14,4) = -931568176.d0
      disb(14,5) = 4812140564.d0
      disb(14,6) = -19076277286.d0
      disb(14,7) = 60420953648.d0
      disb(14,8) = -157230976507.d0
      disb(14,9) = 343176898988.d0
      disb(14,10) = -638478676682.d0
      disb(14,11) = 1026070393348.d0
      disb(14,12) = -1440491185133.d0
      disb(14,13) = 1784127634432.d0
      disb(14,14) = -1966444340888.d0
      disb(14,15) = 1943187627832.d0
      disb(14,16) = -1732101173033.d0
      disb(14,17) = 1399041977872.d0
      disb(14,18) = -1026911119388.d0
      disb(14,19) = 685820288692.d0
      disb(14,20) = -416637112463.d0
      disb(14,21) = 229904411572.d0
      disb(14,22) = -114955488758.d0
      disb(14,23) = 51915526432.d0
      disb(14,24) = -21090682613.d0
      disb(14,25) = 7669339132.d0
      disb(14,26) = -2481256778.d0
      disb(14,27) = 708930508.d0
      disb(14,28) = -177232627.d0
      disb(14,29) = 38320568.d0
      disb(14,30) = -7059052.d0
      disb(14,31) = 1086008.d0
      disb(14,32) = -135751.d0
      disb(14,33) = 13244.d0
      disb(14,34) = -946.d0
      disb(14,35) = 44.d0
      disb(14,36) = -1.d0
c
      disb(15,1) = -319770.d0
      disb(15,2) = 7532360.d0
      disb(15,3) = -85456756.d0
      disb(15,4) = 622281464.d0
      disb(15,5) = -3270685726.d0
      disb(15,6) = 13232963744.d0
      disb(15,7) = -42935276296.d0
      disb(15,8) = 114955808528.d0
      disb(15,9) = -259483829407.d0
      disb(15,10) = 502236892388.d0
      disb(15,11) = -845256668102.d0
      disb(15,12) = 1251646383988.d0
      disb(15,13) = -1647269176553.d0
      disb(15,14) = 1943187627832.d0
      disb(15,15) = -2068697193788.d0
      disb(15,16) = 1997722482712.d0
      disb(15,17) = -1755960172043.d0
      disb(15,18) = 1407462801052.d0
      disb(15,19) = -1029250236938.d0
      disb(15,20) = 686312734492.d0
      disb(15,21) = -416710979333.d0
      disb(15,22) = 229911446512.d0
      disb(15,23) = -114955808528.d0
      disb(15,24) = 51915526432.d0
      disb(15,25) = -21090682613.d0
      disb(15,26) = 7669339132.d0
      disb(15,27) = -2481256778.d0
      disb(15,28) = 708930508.d0
      disb(15,29) = -177232627.d0
      disb(15,30) = 38320568.d0
      disb(15,31) = -7059052.d0
      disb(15,32) = 1086008.d0
      disb(15,33) = -135751.d0
      disb(15,34) = 13244.d0
      disb(15,35) = -946.d0
      disb(15,36) = 44.d0
      disb(15,37) = -1.d0
c
      disb(16,1) = 170544.d0
      disb(16,2) = -4071738.d0
      disb(16,3) = 46928024.d0
      disb(16,4) = -348094516.d0
      disb(16,5) = 1869810824.d0
      disb(16,6) = -7761791422.d0
      disb(16,7) = 25957763216.d0
      disb(16,8) = -72020532232.d0
      disb(16,9) = 169490663408.d0
      disb(16,10) = -344315825887.d0
      disb(16,11) = 612518487812.d0
      disb(16,12) = -965563863110.d0
      disb(16,13) = 1361927979412.d0
      disb(16,14) = -1732101173033.d0
      disb(16,15) = 1997722482712.d0
      disb(16,16) = -2097782449724.d0
      disb(16,17) = 2010447282184.d0
      disb(16,18) = -1760451277739.d0
      disb(16,19) = 1408710330412.d0
      disb(16,20) = -1029512874698.d0
      disb(16,21) = 686352130156.d0
      disb(16,22) = -416714731301.d0
      disb(16,23) = 229911617056.d0
      disb(16,24) = -114955808528.d0
      disb(16,25) = 51915526432.d0
      disb(16,26) = -21090682613.d0
      disb(16,27) = 7669339132.d0
      disb(16,28) = -2481256778.d0
      disb(16,29) = 708930508.d0
      disb(16,30) = -177232627.d0
      disb(16,31) = 38320568.d0
      disb(16,32) = -7059052.d0
      disb(16,33) = 1086008.d0
      disb(16,34) = -135751.d0
      disb(16,35) = 13244.d0
      disb(16,36) = -946.d0
      disb(16,37) = 44.d0
      disb(16,38) = -1.d0
c
      disb(17,1) = -74613.d0
      disb(17,2) = 1812030.d0
      disb(17,3) = -21307341.d0
      disb(17,4) = 161832044.d0
      disb(17,5) = -893888611.d0
      disb(17,6) = 3834669566.d0
      disb(17,7) = -13328891191.d0
      disb(17,8) = 38682562688.d0
      disb(17,9) = -95879531242.d0
      disb(17,10) = 206604661868.d0
      disb(17,11) = -392564023885.d0
      disb(17,12) = 665152885628.d0
      disb(17,13) = -1013812061108.d0
      disb(17,14) = 1399041977872.d0
      disb(17,15) = -1755960172043.d0
      disb(17,16) = 2010447282184.d0
      disb(17,17) = -2103349549493.d0
      disb(17,18) = 2012412140926.d0
      disb(17,19) = -1760997071834.d0
      disb(17,20) = 1408825234432.d0
      disb(17,21) = -1029530110301.d0
      disb(17,22) = 686353771642.d0
      disb(17,23) = -416714805914.d0
      disb(17,24) = 229911617056.d0
      disb(17,25) = -114955808528.d0
      disb(17,26) = 51915526432.d0
      disb(17,27) = -21090682613.d0
      disb(17,28) = 7669339132.d0
      disb(17,29) = -2481256778.d0
      disb(17,30) = 708930508.d0
      disb(17,31) = -177232627.d0
      disb(17,32) = 38320568.d0
      disb(17,33) = -7059052.d0
      disb(17,34) = 1086008.d0
      disb(17,35) = -135751.d0
      disb(17,36) = 13244.d0
      disb(17,37) = -946.d0
      disb(17,38) = 44.d0
      disb(17,39) = -1.d0
c
      disb(18,1) = 26334.d0
      disb(18,2) = -653961.d0
      disb(18,3) = 7895184.d0
      disb(18,4) = -61861701.d0
      disb(18,5) = 354465254.d0
      disb(18,6) = -1587368167.d0
      disb(18,7) = 5799528308.d0
      disb(18,8) = -17819996887.d0
      disb(18,9) = 47103385868.d0
      disb(18,10) = -108978589522.d0
      disb(18,11) = 223633437632.d0
      disb(18,12) = -411140870173.d0
      disb(18,13) = 682181661392.d0
      disb(18,14) = -1026911119388.d0
      disb(18,15) = 1407462801052.d0
      disb(18,16) = -1760451277739.d0
      disb(18,17) = 2012412140926.d0
      disb(18,18) = -2104043029049.d0
      disb(18,19) = 2012604774136.d0
      disb(18,20) = -1761037626194.d0
      disb(18,21) = 1408831317586.d0
      disb(18,22) = -1029530689649.d0
      disb(18,23) = 686353797976.d0
      disb(18,24) = -416714805914.d0
      disb(18,25) = 229911617056.d0
      disb(18,26) = -114955808528.d0
      disb(18,27) = 51915526432.d0
      disb(18,28) = -21090682613.d0
      disb(18,29) = 7669339132.d0
      disb(18,30) = -2481256778.d0
      disb(18,31) = 708930508.d0
      disb(18,32) = -177232627.d0
      disb(18,33) = 38320568.d0
      disb(18,34) = -7059052.d0
      disb(18,35) = 1086008.d0
      disb(18,36) = -135751.d0
      disb(18,37) = 13244.d0
      disb(18,38) = -946.d0
      disb(18,39) = 44.d0
      disb(18,40) = -1.d0
c
      disb(19,1) = -7315.d0
      disb(19,2) = 187264.d0
      disb(19,3) = -2343726.d0
      disb(19,4) = 19160284.d0
      disb(19,5) = -115370926.d0
      disb(19,6) = 547098464.d0
      disb(19,7) = -2133162262.d0
      disb(19,8) = 7047057668.d0
      disb(19,9) = -20159114437.d0
      disb(19,10) = 50742013168.d0
      disb(19,11) = -113708805012.d0
      disb(19,12) = 228793672712.d0
      disb(19,13) = -415871085663.d0
      disb(19,14) = 685820288692.d0
      disb(19,15) = -1029250236938.d0
      disb(19,16) = 1408710330412.d0
      disb(19,17) = -1760997071834.d0
      disb(19,18) = 2012604774136.d0
      disb(19,19) = -2104096538274.d0
      disb(19,20) = 2012616039236.d0
      disb(19,21) = -1761039315959.d0
      disb(19,22) = 1408831478516.d0
      disb(19,23) = -1029530696964.d0
      disb(19,24) = 686353797976.d0
      disb(19,25) = -416714805914.d0
      disb(19,26) = 229911617056.d0
      disb(19,27) = -114955808528.d0
      disb(19,28) = 51915526432.d0
      disb(19,29) = -21090682613.d0
      disb(19,30) = 7669339132.d0
      disb(19,31) = -2481256778.d0
      disb(19,32) = 708930508.d0
      disb(19,33) = -177232627.d0
      disb(19,34) = 38320568.d0
      disb(19,35) = -7059052.d0
      disb(19,36) = 1086008.d0
      disb(19,37) = -135751.d0
      disb(19,38) = 13244.d0
      disb(19,39) = -946.d0
      disb(19,40) = 44.d0
      disb(19,41) = -1.d0
c
      disb(20,1) = 1540.d0
      disb(20,2) = -41195.d0
      disb(20,3) = 543004.d0
      disb(20,4) = -4715326.d0
      disb(20,5) = 30425384.d0
      disb(20,6) = -155925286.d0
      disb(20,7) = 662002484.d0
      disb(20,8) = -2395800022.d0
      disb(20,9) = 7539503468.d0
      disb(20,10) = -20925141237.d0
      disb(20,11) = 51737848008.d0
      disb(20,12) = -114795170292.d0
      disb(20,13) = 229789507552.d0
      disb(20,14) = -416637112463.d0
      disb(20,15) = 686312734492.d0
      disb(20,16) = -1029512874698.d0
      disb(20,17) = 1408825234432.d0
      disb(20,18) = -1761037626194.d0
      disb(20,19) = 2012616039236.d0
      disb(20,20) = -2104098909874.d0
      disb(20,21) = 2012616394976.d0
      disb(20,22) = -1761039349839.d0
      disb(20,23) = 1408831480056.d0
      disb(20,24) = -1029530696964.d0
      disb(20,25) = 686353797976.d0
      disb(20,26) = -416714805914.d0
      disb(20,27) = 229911617056.d0
      disb(20,28) = -114955808528.d0
      disb(20,29) = 51915526432.d0
      disb(20,30) = -21090682613.d0
      disb(20,31) = 7669339132.d0
      disb(20,32) = -2481256778.d0
      disb(20,33) = 708930508.d0
      disb(20,34) = -177232627.d0
      disb(20,35) = 38320568.d0
      disb(20,36) = -7059052.d0
      disb(20,37) = 1086008.d0
      disb(20,38) = -135751.d0
      disb(20,39) = 13244.d0
      disb(20,40) = -946.d0
      disb(20,41) = 44.d0
      disb(20,42) = -1.d0
c
      disb(21,1) = -231.d0
      disb(21,2) = 6622.d0
      disb(21,3) = -94556.d0
      disb(21,4) = 898744.d0
      disb(21,5) = -6405091.d0
      disb(21,6) = 36508538.d0
      disb(21,7) = -173160889.d0
      disb(21,8) = 701398148.d0
      disb(21,9) = -2469666892.d0
      disb(21,10) = 7654407488.d0
      disb(21,11) = -21074516463.d0
      disb(21,12) = 51900802800.d0
      disb(21,13) = -114944545518.d0
      disb(21,14) = 229904411572.d0
      disb(21,15) = -416710979333.d0
      disb(21,16) = 686352130156.d0
      disb(21,17) = -1029530110301.d0
      disb(21,18) = 1408831317586.d0
      disb(21,19) = -1761039315959.d0
      disb(21,20) = 2012616394976.d0
      disb(21,21) = -2104098963235.d0
      disb(21,22) = 2012616400058.d0
      disb(21,23) = -1761039350070.d0
      disb(21,24) = 1408831480056.d0
      disb(21,25) = -1029530696964.d0
      disb(21,26) = 686353797976.d0
      disb(21,27) = -416714805914.d0
      disb(21,28) = 229911617056.d0
      disb(21,29) = -114955808528.d0
      disb(21,30) = 51915526432.d0
      disb(21,31) = -21090682613.d0
      disb(21,32) = 7669339132.d0
      disb(21,33) = -2481256778.d0
      disb(21,34) = 708930508.d0
      disb(21,35) = -177232627.d0
      disb(21,36) = 38320568.d0
      disb(21,37) = -7059052.d0
      disb(21,38) = 1086008.d0
      disb(21,39) = -135751.d0
      disb(21,40) = 13244.d0
      disb(21,41) = -946.d0
      disb(21,42) = 44.d0
      disb(21,43) = -1.d0
c
      disb(22,1) = 22.d0
      disb(22,2) = -715.d0
      disb(22,3) = 11704.d0
      disb(22,4) = -128436.d0
      disb(22,5) = 1059674.d0
      disb(22,6) = -6984439.d0
      disb(22,7) = 38150024.d0
      disb(22,8) = -176912857.d0
      disb(22,9) = 708433088.d0
      disb(22,10) = -2480610132.d0
      disb(22,11) = 7668633700.d0
      disb(22,12) = -21090035967.d0
      disb(22,13) = 51915029012.d0
      disb(22,14) = -114955488758.d0
      disb(22,15) = 229911446512.d0
      disb(22,16) = -416714731301.d0
      disb(22,17) = 686353771642.d0
      disb(22,18) = -1029530689649.d0
      disb(22,19) = 1408831478516.d0
      disb(22,20) = -1761039349839.d0
      disb(22,21) = 2012616400058.d0
      disb(22,22) = -2104098963719.d0
      disb(22,23) = 2012616400080.d0
      disb(22,24) = -1761039350070.d0
      disb(22,25) = 1408831480056.d0
      disb(22,26) = -1029530696964.d0
      disb(22,27) = 686353797976.d0
      disb(22,28) = -416714805914.d0
      disb(22,29) = 229911617056.d0
      disb(22,30) = -114955808528.d0
      disb(22,31) = 51915526432.d0
      disb(22,32) = -21090682613.d0
      disb(22,33) = 7669339132.d0
      disb(22,34) = -2481256778.d0
      disb(22,35) = 708930508.d0
      disb(22,36) = -177232627.d0
      disb(22,37) = 38320568.d0
      disb(22,38) = -7059052.d0
      disb(22,39) = 1086008.d0
      disb(22,40) = -135751.d0
      disb(22,41) = 13244.d0
      disb(22,42) = -946.d0
      disb(22,43) = 44.d0
      disb(22,44) = -1.d0
c
      elseif(iord.eq.46)then
c 
c 46th order inner with 23rd order boundaries
c ------------------------------------------
c
      disi(1) = 8233430727600.d0
      disi(2) = -7890371113950.d0
      disi(3) = 6943526580276.d0
      disi(4) = -5608233007146.d0
      disi(5) = 4154246671960.d0
      disi(6) = -2818953098830.d0
      disi(7) = 1749695026860.d0
      disi(8) = -991493848554.d0
      disi(9) = 511738760544.d0
      disi(10) = -239877544005.d0
      disi(11) = 101766230790.d0
      disi(12) = -38910617655.d0
      disi(13) = 13340783196.d0
      disi(14) = -4076350421.d0
      disi(15) = 1101716330.d0
      disi(16) = -260932815.d0
      disi(17) = 53524680.d0
      disi(18) = -9366819.d0
      disi(19) = 1370754.d0
      disi(20) = -163185.d0
      disi(21) = 15180.d0
      disi(22) = -1035.d0
      disi(23) = 46.d0
      disi(24) = -1.d0
c
      disb(1,1) = 1.d0
      disb(1,2) = -23.d0
      disb(1,3) = 253.d0
      disb(1,4) = -1771.d0
      disb(1,5) = 8855.d0
      disb(1,6) = -33649.d0
      disb(1,7) = 100947.d0
      disb(1,8) = -245157.d0
      disb(1,9) = 490314.d0
      disb(1,10) = -817190.d0
      disb(1,11) = 1144066.d0
      disb(1,12) = -1352078.d0
      disb(1,13) = 1352078.d0
      disb(1,14) = -1144066.d0
      disb(1,15) = 817190.d0
      disb(1,16) = -490314.d0
      disb(1,17) = 245157.d0
      disb(1,18) = -100947.d0
      disb(1,19) = 33649.d0
      disb(1,20) = -8855.d0
      disb(1,21) = 1771.d0
      disb(1,22) = -253.d0
      disb(1,23) = 23.d0
      disb(1,24) = -1.d0
c
      disb(2,1) = -23.d0
      disb(2,2) = 530.d0
      disb(2,3) = -5842.d0
      disb(2,4) = 40986.d0
      disb(2,5) = -205436.d0
      disb(2,6) = 782782.d0
      disb(2,7) = -2355430.d0
      disb(2,8) = 5739558.d0
      disb(2,9) = -11522379.d0
      disb(2,10) = 19285684.d0
      disb(2,11) = -27130708.d0
      disb(2,12) = 32241860.d0
      disb(2,13) = -32449872.d0
      disb(2,14) = 27665596.d0
      disb(2,15) = -19939436.d0
      disb(2,16) = 12094412.d0
      disb(2,17) = -6128925.d0
      disb(2,18) = 2566938.d0
      disb(2,19) = -874874.d0
      disb(2,20) = 237314.d0
      disb(2,21) = -49588.d0
      disb(2,22) = 7590.d0
      disb(2,23) = -782.d0
      disb(2,24) = 46.d0
      disb(2,25) = -1.d0
c
      disb(3,1) = 253.d0
      disb(3,2) = -5842.d0
      disb(3,3) = 64539.d0
      disb(3,4) = -453905.d0
      disb(3,5) = 2281301.d0
      disb(3,6) = -8718633.d0
      disb(3,7) = 26322373.d0
      disb(3,8) = -64380151.d0
      disb(3,9) = 129789000.d0
      disb(3,10) = -218271449.d0
      disb(3,11) = 308734382.d0
      disb(3,12) = -369206442.d0
      disb(3,13) = 374317594.d0
      disb(3,14) = -321898570.d0
      disb(3,15) = 234414666.d0
      disb(3,16) = -143988878.d0
      disb(3,17) = 74119133.d0
      disb(3,18) = -31668516.d0
      disb(3,19) = 11080135.d0
      disb(3,20) = -3115189.d0
      disb(3,21) = 685377.d0
      disb(3,22) = -113597.d0
      disb(3,23) = 13409.d0
      disb(3,24) = -1035.d0
      disb(3,25) = 46.d0
      disb(3,26) = -1.d0
c
      disb(4,1) = -1771.d0
      disb(4,2) = 40986.d0
      disb(4,3) = -453905.d0
      disb(4,4) = 3200980.d0
      disb(4,5) = -16136110.d0
      disb(4,6) = 61873680.d0
      disb(4,7) = -187495770.d0
      disb(4,8) = 460495420.d0
      disb(4,9) = -932726245.d0
      disb(4,10) = 1577032490.d0
      disb(4,11) = -2244412335.d0
      disb(4,12) = 2703264520.d0
      disb(4,13) = -2763736580.d0
      disb(4,14) = 2400458480.d0
      disb(4,15) = -1769142060.d0
      disb(4,16) = 1102760760.d0
      disb(4,17) = -578161925.d0
      disb(4,18) = 252896270.d0
      disb(4,19) = -91260895.d0
      disb(4,20) = 26762340.d0
      disb(4,21) = -6251630.d0
      disb(4,22) = 1133440.d0
      disb(4,23) = -154330.d0
      disb(4,24) = 15180.d0
      disb(4,25) = -1035.d0
      disb(4,26) = 46.d0
      disb(4,27) = -1.d0
c
      disb(5,1) = 8855.d0
      disb(5,2) = -205436.d0
      disb(5,3) = 2281301.d0
      disb(5,4) = -16136110.d0
      disb(5,5) = 81612005.d0
      disb(5,6) = -314098005.d0
      disb(5,7) = 955759365.d0
      disb(5,8) = -2358361005.d0
      disb(5,9) = 4802225890.d0
      disb(5,10) = -8168943695.d0
      disb(5,11) = 11707736920.d0
      disb(5,12) = -14217063025.d0
      disb(5,13) = 14675915210.d0
      disb(5,14) = -12894441010.d0
      disb(5,15) = 9636675930.d0
      disb(5,16) = -6110872530.d0
      disb(5,17) = 3273625995.d0
      disb(5,18) = -1472047610.d0
      disb(5,19) = 550858165.d0
      disb(5,20) = -169671920.d0
      disb(5,21) = 42444545.d0
      disb(5,22) = -8491945.d0
      disb(5,23) = 1337105.d0
      disb(5,24) = -163185.d0
      disb(5,25) = 15180.d0
      disb(5,26) = -1035.d0
      disb(5,27) = 46.d0
      disb(5,28) = -1.d0
c
      disb(6,1) = -33649.d0
      disb(6,2) = 782782.d0
      disb(6,3) = -8718633.d0
      disb(6,4) = 61873680.d0
      disb(6,5) = -314098005.d0
      disb(6,6) = 1213867206.d0
      disb(6,7) = -3710863608.d0
      disb(6,8) = 9205047258.d0
      disb(6,9) = -18856936791.d0
      disb(6,10) = 32299852200.d0
      disb(6,11) = -46665620529.d0
      disb(6,12) = 57203809542.d0
      disb(6,13) = -59713135647.d0
      disb(6,14) = 53172592044.d0
      disb(6,15) = -40392067320.d0
      disb(6,16) = 26135251716.d0
      disb(6,17) = -14360160423.d0
      disb(6,18) = 6670391598.d0
      disb(6,19) = -2604302811.d0
      disb(6,20) = 848820060.d0
      disb(6,21) = -229264299.d0
      disb(6,22) = 50957742.d0
      disb(6,23) = -9265872.d0
      disb(6,24) = 1370754.d0
      disb(6,25) = -163185.d0
      disb(6,26) = 15180.d0
      disb(6,27) = -1035.d0
      disb(6,28) = 46.d0
      disb(6,29) = -1.d0
c
      disb(7,1) = 100947.d0
      disb(7,2) = -2355430.d0
      disb(7,3) = 26322373.d0
      disb(7,4) = -187495770.d0
      disb(7,5) = 955759365.d0
      disb(7,6) = -3710863608.d0
      disb(7,7) = 11404164015.d0
      disb(7,8) = -28458727287.d0
      disb(7,9) = 58700774616.d0
      disb(7,10) = -101349815721.d0
      disb(7,11) = 147789882702.d0
      disb(7,12) = -183153838395.d0
      disb(7,13) = 193692027408.d0
      disb(7,14) = -175203166149.d0
      disb(7,15) = 135665470974.d0
      disb(7,16) = -89887794678.d0
      disb(7,17) = 50883115395.d0
      disb(7,18) = -24550457232.d0
      disb(7,19) = 10067157201.d0
      disb(7,20) = -3498188496.d0
      disb(7,21) = 1027597197.d0
      disb(7,22) = -254803890.d0
      disb(7,23) = 53279523.d0
      disb(7,24) = -9366819.d0
      disb(7,25) = 1370754.d0
      disb(7,26) = -163185.d0
      disb(7,27) = 15180.d0
      disb(7,28) = -1035.d0
      disb(7,29) = 46.d0
      disb(7,30) = -1.d0
c
      disb(8,1) = -245157.d0
      disb(8,2) = 5739558.d0
      disb(8,3) = -64380151.d0
      disb(8,4) = 460495420.d0
      disb(8,5) = -2358361005.d0
      disb(8,6) = 9205047258.d0
      disb(8,7) = -28458727287.d0
      disb(8,8) = 71506118664.d0
      disb(8,9) = -148662636585.d0
      disb(8,10) = 259040623446.d0
      disb(8,11) = -381825604083.d0
      disb(8,12) = 479261268948.d0
      disb(8,13) = -514625224641.d0
      disb(8,14) = 474167815770.d0
      disb(8,15) = -375543014979.d0
      disb(8,16) = 255869380272.d0
      disb(8,17) = -149989749327.d0
      disb(8,18) = 75630979074.d0
      disb(8,19) = -32799745125.d0
      disb(8,20) = 12238022436.d0
      disb(8,21) = -3932361543.d0
      disb(8,22) = 1089621918.d0
      disb(8,23) = -260442501.d0
      disb(8,24) = 53524680.d0
      disb(8,25) = -9366819.d0
      disb(8,26) = 1370754.d0
      disb(8,27) = -163185.d0
      disb(8,28) = 15180.d0
      disb(8,29) = -1035.d0
      disb(8,30) = 46.d0
      disb(8,31) = -1.d0
c
      disb(9,1) = 490314.d0
      disb(9,2) = -11522379.d0
      disb(9,3) = 129789000.d0
      disb(9,4) = -932726245.d0
      disb(9,5) = 4802225890.d0
      disb(9,6) = -18856936791.d0
      disb(9,7) = 58700774616.d0
      disb(9,8) = -148662636585.d0
      disb(9,9) = 311913937260.d0
      disb(9,10) = -549342334245.d0
      disb(9,11) = 819992200170.d0
      disb(9,12) = -1044768376575.d0
      disb(9,13) = 1142204041440.d0
      disb(9,14) = -1075576801365.d0
      disb(9,15) = 874847513430.d0
      disb(9,16) = -615950833575.d0
      disb(9,17) = 376073289570.d0
      disb(9,18) = -199485476685.d0
      disb(9,19) = 92129554860.d0
      disb(9,20) = -37141475595.d0
      disb(9,21) = 13106368530.d0
      disb(9,22) = -4056410985.d0
      disb(9,23) = 1100899140.d0
      disb(9,24) = -260932815.d0
      disb(9,25) = 53524680.d0
      disb(9,26) = -9366819.d0
      disb(9,27) = 1370754.d0
      disb(9,28) = -163185.d0
      disb(9,29) = 15180.d0
      disb(9,30) = -1035.d0
      disb(9,31) = 46.d0
      disb(9,32) = -1.d0
c
      disb(10,1) = -817190.d0
      disb(10,2) = 19285684.d0
      disb(10,3) = -218271449.d0
      disb(10,4) = 1577032490.d0
      disb(10,5) = -8168943695.d0
      disb(10,6) = 32299852200.d0
      disb(10,7) = -101349815721.d0
      disb(10,8) = 259040623446.d0
      disb(10,9) = -549342334245.d0
      disb(10,10) = 979713433360.d0
      disb(10,11) = -1484261628785.d0
      disb(10,12) = 1924896820990.d0
      disb(10,13) = -2149672997395.d0
      disb(10,14) = 2077123335980.d0
      disb(10,15) = -1743376297465.d0
      disb(10,16) = 1275527211090.d0
      disb(10,17) = -816290682405.d0
      disb(10,18) = 458566168500.d0
      disb(10,19) = -226983102995.d0
      disb(10,20) = 99365772310.d0
      disb(10,21) = -38588719085.d0
      disb(10,22) = 13313117600.d0
      disb(10,23) = -4075206355.d0
      disb(10,24) = 1101716330.d0
      disb(10,25) = -260932815.d0
      disb(10,26) = 53524680.d0
      disb(10,27) = -9366819.d0
      disb(10,28) = 1370754.d0
      disb(10,29) = -163185.d0
      disb(10,30) = 15180.d0
      disb(10,31) = -1035.d0
      disb(10,32) = 46.d0
      disb(10,33) = -1.d0
c
      disb(11,1) = 1144066.d0
      disb(11,2) = -27130708.d0
      disb(11,3) = 308734382.d0
      disb(11,4) = -2244412335.d0
      disb(11,5) = 11707736920.d0
      disb(11,6) = -46665620529.d0
      disb(11,7) = 147789882702.d0
      disb(11,8) = -381825604083.d0
      disb(11,9) = 819992200170.d0
      disb(11,10) = -1484261628785.d0
      disb(11,11) = 2288600445716.d0
      disb(11,12) = -3031128097933.d0
      disb(11,13) = 3471763290138.d0
      disb(11,14) = -3458560009751.d0
      disb(11,15) = 3012042630520.d0
      disb(11,16) = -2304327874189.d0
      disb(11,17) = 1556002999452.d0
      disb(11,18) = -931780712907.d0
      disb(11,19) = 497062845334.d0
      disb(11,20) = -237113807425.d0
      disb(11,21) = 101391913196.d0
      disb(11,22) = -38878167783.d0
      disb(11,23) = 13339431118.d0
      disb(11,24) = -4076350421.d0
      disb(11,25) = 1101716330.d0
      disb(11,26) = -260932815.d0
      disb(11,27) = 53524680.d0
      disb(11,28) = -9366819.d0
      disb(11,29) = 1370754.d0
      disb(11,30) = -163185.d0
      disb(11,31) = 15180.d0
      disb(11,32) = -1035.d0
      disb(11,33) = 46.d0
      disb(11,34) = -1.d0
c
      disb(12,1) = -1352078.d0
      disb(12,2) = 32241860.d0
      disb(12,3) = -369206442.d0
      disb(12,4) = 2703264520.d0
      disb(12,5) = -14217063025.d0
      disb(12,6) = 57203809542.d0
      disb(12,7) = -183153838395.d0
      disb(12,8) = 479261268948.d0
      disb(12,9) = -1044768376575.d0
      disb(12,10) = 1924896820990.d0
      disb(12,11) = -3031128097933.d0
      disb(12,12) = 4116715363800.d0
      disb(12,13) = -4859243016017.d0
      disb(12,14) = 5018629759286.d0
      disb(12,15) = -4563464630571.d0
      disb(12,16) = 3674985403012.d0
      disb(12,17) = -2635799260435.d0
      disb(12,18) = 1692491217318.d0
      disb(12,19) = -977276785529.d0
      disb(12,20) = 509035496024.d0
      disb(12,21) = -239508337563.d0
      disb(12,22) = 101733988930.d0
      disb(12,23) = -38909265577.d0
      disb(12,24) = 13340783196.d0
      disb(12,25) = -4076350421.d0
      disb(12,26) = 1101716330.d0
      disb(12,27) = -260932815.d0
      disb(12,28) = 53524680.d0
      disb(12,29) = -9366819.d0
      disb(12,30) = 1370754.d0
      disb(12,31) = -163185.d0
      disb(12,32) = 15180.d0
      disb(12,33) = -1035.d0
      disb(12,34) = 46.d0
      disb(12,35) = -1.d0
c
      disb(13,1) = 1352078.d0
      disb(13,2) = -32449872.d0
      disb(13,3) = 374317594.d0
      disb(13,4) = -2763736580.d0
      disb(13,5) = 14675915210.d0
      disb(13,6) = -59713135647.d0
      disb(13,7) = 193692027408.d0
      disb(13,8) = -514625224641.d0
      disb(13,9) = 1142204041440.d0
      disb(13,10) = -2149672997395.d0
      disb(13,11) = 3471763290138.d0
      disb(13,12) = -4859243016017.d0
      disb(13,13) = 5944830281884.d0
      disb(13,14) = -6406109485165.d0
      disb(13,15) = 6123534380106.d0
      disb(13,16) = -5226407403063.d0
      disb(13,17) = 4006456789258.d0
      disb(13,18) = -2772287478301.d0
      disb(13,19) = 1737987289940.d0
      disb(13,20) = -989249436219.d0
      disb(13,21) = 511430026162.d0
      disb(13,22) = -239850413297.d0
      disb(13,23) = 101765086724.d0
      disb(13,24) = -38910617655.d0
      disb(13,25) = 13340783196.d0
      disb(13,26) = -4076350421.d0
      disb(13,27) = 1101716330.d0
      disb(13,28) = -260932815.d0
      disb(13,29) = 53524680.d0
      disb(13,30) = -9366819.d0
      disb(13,31) = 1370754.d0
      disb(13,32) = -163185.d0
      disb(13,33) = 15180.d0
      disb(13,34) = -1035.d0
      disb(13,35) = 46.d0
      disb(13,36) = -1.d0
c
      disb(14,1) = -1144066.d0
      disb(14,2) = 27665596.d0
      disb(14,3) = -321898570.d0
      disb(14,4) = 2400458480.d0
      disb(14,5) = -12894441010.d0
      disb(14,6) = 53172592044.d0
      disb(14,7) = -175203166149.d0
      disb(14,8) = 474167815770.d0
      disb(14,9) = -1075576801365.d0
      disb(14,10) = 2077123335980.d0
      disb(14,11) = -3458560009751.d0
      disb(14,12) = 5018629759286.d0
      disb(14,13) = -6406109485165.d0
      disb(14,14) = 7253717294240.d0
      disb(14,15) = -7341028779705.d0
      disb(14,16) = 6684485956830.d0
      disb(14,17) = -5506883191425.d0
      disb(14,18) = 4121946819760.d0
      disb(14,19) = -2810784155135.d0
      disb(14,20) = 1748117994370.d0
      disb(14,21) = -991275577105.d0
      disb(14,22) = 511719474860.d0
      disb(14,23) = -239876726815.d0
      disb(14,24) = 101766230790.d0
      disb(14,25) = -38910617655.d0
      disb(14,26) = 13340783196.d0
      disb(14,27) = -4076350421.d0
      disb(14,28) = 1101716330.d0
      disb(14,29) = -260932815.d0
      disb(14,30) = 53524680.d0
      disb(14,31) = -9366819.d0
      disb(14,32) = 1370754.d0
      disb(14,33) = -163185.d0
      disb(14,34) = 15180.d0
      disb(14,35) = -1035.d0
      disb(14,36) = 46.d0
      disb(14,37) = -1.d0
c
      disb(15,1) = 817190.d0
      disb(15,2) = -19939436.d0
      disb(15,3) = 234414666.d0
      disb(15,4) = -1769142060.d0
      disb(15,5) = 9636675930.d0
      disb(15,6) = -40392067320.d0
      disb(15,7) = 135665470974.d0
      disb(15,8) = -375543014979.d0
      disb(15,9) = 874847513430.d0
      disb(15,10) = -1743376297465.d0
      disb(15,11) = 3012042630520.d0
      disb(15,12) = -4563464630571.d0
      disb(15,13) = 6123534380106.d0
      disb(15,14) = -7341028779705.d0
      disb(15,15) = 7921516790340.d0
      disb(15,16) = -7741708477365.d0
      disb(15,17) = 6884825805660.d0
      disb(15,18) = -5589376070355.d0
      disb(15,19) = 4149444446070.d0
      disb(15,20) = -2818020372585.d0
      disb(15,21) = 1749565237860.d0
      disb(15,22) = -991482326175.d0
      disb(15,23) = 511738270230.d0
      disb(15,24) = -239877544005.d0
      disb(15,25) = 101766230790.d0
      disb(15,26) = -38910617655.d0
      disb(15,27) = 13340783196.d0
      disb(15,28) = -4076350421.d0
      disb(15,29) = 1101716330.d0
      disb(15,30) = -260932815.d0
      disb(15,31) = 53524680.d0
      disb(15,32) = -9366819.d0
      disb(15,33) = 1370754.d0
      disb(15,34) = -163185.d0
      disb(15,35) = 15180.d0
      disb(15,36) = -1035.d0
      disb(15,37) = 46.d0
      disb(15,38) = -1.d0
c
      disb(16,1) = -490314.d0
      disb(16,2) = 12094412.d0
      disb(16,3) = -143988878.d0
      disb(16,4) = 1102760760.d0
      disb(16,5) = -6110872530.d0
      disb(16,6) = 26135251716.d0
      disb(16,7) = -89887794678.d0
      disb(16,8) = 255869380272.d0
      disb(16,9) = -615950833575.d0
      disb(16,10) = 1275527211090.d0
      disb(16,11) = -2304327874189.d0
      disb(16,12) = 3674985403012.d0
      disb(16,13) = -5226407403063.d0
      disb(16,14) = 6684485956830.d0
      disb(16,15) = -7741708477365.d0
      disb(16,16) = 8161924608936.d0
      disb(16,17) = -7861912386663.d0
      disb(16,18) = 6934321533018.d0
      disb(16,19) = -5605874646141.d0
      disb(16,20) = 4153786176540.d0
      disb(16,21) = -2818888718679.d0
      disb(16,22) = 1749689287302.d0
      disb(16,23) = -991493603397.d0
      disb(16,24) = 511738760544.d0
      disb(16,25) = -239877544005.d0
      disb(16,26) = 101766230790.d0
      disb(16,27) = -38910617655.d0
      disb(16,28) = 13340783196.d0
      disb(16,29) = -4076350421.d0
      disb(16,30) = 1101716330.d0
      disb(16,31) = -260932815.d0
      disb(16,32) = 53524680.d0
      disb(16,33) = -9366819.d0
      disb(16,34) = 1370754.d0
      disb(16,35) = -163185.d0
      disb(16,36) = 15180.d0
      disb(16,37) = -1035.d0
      disb(16,38) = 46.d0
      disb(16,39) = -1.d0
c
      disb(17,1) = 245157.d0
      disb(17,2) = -6128925.d0
      disb(17,3) = 74119133.d0
      disb(17,4) = -578161925.d0
      disb(17,5) = 3273625995.d0
      disb(17,6) = -14360160423.d0
      disb(17,7) = 50883115395.d0
      disb(17,8) = -149989749327.d0
      disb(17,9) = 376073289570.d0
      disb(17,10) = -816290682405.d0
      disb(17,11) = 1556002999452.d0
      disb(17,12) = -2635799260435.d0
      disb(17,13) = 4006456789258.d0
      disb(17,14) = -5506883191425.d0
      disb(17,15) = 6884825805660.d0
      disb(17,16) = -7861912386663.d0
      disb(17,17) = 8222026563585.d0
      disb(17,18) = -7886660250342.d0
      disb(17,19) = 6942570820911.d0
      disb(17,20) = -5608045511376.d0
      disb(17,21) = 4154220349587.d0
      disb(17,22) = -2818950743400.d0
      disb(17,23) = 1749694925913.d0
      disb(17,24) = -991493848554.d0
      disb(17,25) = 511738760544.d0
      disb(17,26) = -239877544005.d0
      disb(17,27) = 101766230790.d0
      disb(17,28) = -38910617655.d0
      disb(17,29) = 13340783196.d0
      disb(17,30) = -4076350421.d0
      disb(17,31) = 1101716330.d0
      disb(17,32) = -260932815.d0
      disb(17,33) = 53524680.d0
      disb(17,34) = -9366819.d0
      disb(17,35) = 1370754.d0
      disb(17,36) = -163185.d0
      disb(17,37) = 15180.d0
      disb(17,38) = -1035.d0
      disb(17,39) = 46.d0
      disb(17,40) = -1.d0
c
      disb(18,1) = -100947.d0
      disb(18,2) = 2566938.d0
      disb(18,3) = -31668516.d0
      disb(18,4) = 252896270.d0
      disb(18,5) = -1472047610.d0
      disb(18,6) = 6670391598.d0
      disb(18,7) = -24550457232.d0
      disb(18,8) = 75630979074.d0
      disb(18,9) = -199485476685.d0
      disb(18,10) = 458566168500.d0
      disb(18,11) = -931780712907.d0
      disb(18,12) = 1692491217318.d0
      disb(18,13) = -2772287478301.d0
      disb(18,14) = 4121946819760.d0
      disb(18,15) = -5589376070355.d0
      disb(18,16) = 6934321533018.d0
      disb(18,17) = -7886660250342.d0
      disb(18,18) = 8232216860394.d0
      disb(18,19) = -7890057015945.d0
      disb(18,20) = 6943464706596.d0
      disb(18,21) = -5608224288513.d0
      disb(18,22) = 4154245889178.d0
      disb(18,23) = -2818953065181.d0
      disb(18,24) = 1749695026860.d0
      disb(18,25) = -991493848554.d0
      disb(18,26) = 511738760544.d0
      disb(18,27) = -239877544005.d0
      disb(18,28) = 101766230790.d0
      disb(18,29) = -38910617655.d0
      disb(18,30) = 13340783196.d0
      disb(18,31) = -4076350421.d0
      disb(18,32) = 1101716330.d0
      disb(18,33) = -260932815.d0
      disb(18,34) = 53524680.d0
      disb(18,35) = -9366819.d0
      disb(18,36) = 1370754.d0
      disb(18,37) = -163185.d0
      disb(18,38) = 15180.d0
      disb(18,39) = -1035.d0
      disb(18,40) = 46.d0
      disb(18,41) = -1.d0
c
      disb(19,1) = 33649.d0
      disb(19,2) = -874874.d0
      disb(19,3) = 11080135.d0
      disb(19,4) = -91260895.d0
      disb(19,5) = 550858165.d0
      disb(19,6) = -2604302811.d0
      disb(19,7) = 10067157201.d0
      disb(19,8) = -32799745125.d0
      disb(19,9) = 92129554860.d0
      disb(19,10) = -226983102995.d0
      disb(19,11) = 497062845334.d0
      disb(19,12) = -977276785529.d0
      disb(19,13) = 1737987289940.d0
      disb(19,14) = -2810784155135.d0
      disb(19,15) = 4149444446070.d0
      disb(19,16) = -5605874646141.d0
      disb(19,17) = 6942570820911.d0
      disb(19,18) = -7890057015945.d0
      disb(19,19) = 8233349115595.d0
      disb(19,20) = -7890354977840.d0
      disb(19,21) = 6943524298975.d0
      disb(19,22) = -5608232801710.d0
      disb(19,23) = 4154246663105.d0
      disb(19,24) = -2818953098830.d0
      disb(19,25) = 1749695026860.d0
      disb(19,26) = -991493848554.d0
      disb(19,27) = 511738760544.d0
      disb(19,28) = -239877544005.d0
      disb(19,29) = 101766230790.d0
      disb(19,30) = -38910617655.d0
      disb(19,31) = 13340783196.d0
      disb(19,32) = -4076350421.d0
      disb(19,33) = 1101716330.d0
      disb(19,34) = -260932815.d0
      disb(19,35) = 53524680.d0
      disb(19,36) = -9366819.d0
      disb(19,37) = 1370754.d0
      disb(19,38) = -163185.d0
      disb(19,39) = 15180.d0
      disb(19,40) = -1035.d0
      disb(19,41) = 46.d0
      disb(19,42) = -1.d0
c
      disb(20,1) = -8855.d0
      disb(20,2) = 237314.d0
      disb(20,3) = -3115189.d0
      disb(20,4) = 26762340.d0
      disb(20,5) = -169671920.d0
      disb(20,6) = 848820060.d0
      disb(20,7) = -3498188496.d0
      disb(20,8) = 12238022436.d0
      disb(20,9) = -37141475595.d0
      disb(20,10) = 99365772310.d0
      disb(20,11) = -237113807425.d0
      disb(20,12) = 509035496024.d0
      disb(20,13) = -989249436219.d0
      disb(20,14) = 1748117994370.d0
      disb(20,15) = -2818020372585.d0
      disb(20,16) = 4153786176540.d0
      disb(20,17) = -5608045511376.d0
      disb(20,18) = 6943464706596.d0
      disb(20,19) = -7890354977840.d0
      disb(20,20) = 8233427526620.d0
      disb(20,21) = -7890370660045.d0
      disb(20,22) = 6943526539290.d0
      disb(20,23) = -5608233005375.d0
      disb(20,24) = 4154246671960.d0
      disb(20,25) = -2818953098830.d0
      disb(20,26) = 1749695026860.d0
      disb(20,27) = -991493848554.d0
      disb(20,28) = 511738760544.d0
      disb(20,29) = -239877544005.d0
      disb(20,30) = 101766230790.d0
      disb(20,31) = -38910617655.d0
      disb(20,32) = 13340783196.d0
      disb(20,33) = -4076350421.d0
      disb(20,34) = 1101716330.d0
      disb(20,35) = -260932815.d0
      disb(20,36) = 53524680.d0
      disb(20,37) = -9366819.d0
      disb(20,38) = 1370754.d0
      disb(20,39) = -163185.d0
      disb(20,40) = 15180.d0
      disb(20,41) = -1035.d0
      disb(20,42) = 46.d0
      disb(20,43) = -1.d0
c
      disb(21,1) = 1771.d0
      disb(21,2) = -49588.d0
      disb(21,3) = 685377.d0
      disb(21,4) = -6251630.d0
      disb(21,5) = 42444545.d0
      disb(21,6) = -229264299.d0
      disb(21,7) = 1027597197.d0
      disb(21,8) = -3932361543.d0
      disb(21,9) = 13106368530.d0
      disb(21,10) = -38588719085.d0
      disb(21,11) = 101391913196.d0
      disb(21,12) = -239508337563.d0
      disb(21,13) = 511430026162.d0
      disb(21,14) = -991275577105.d0
      disb(21,15) = 1749565237860.d0
      disb(21,16) = -2818888718679.d0
      disb(21,17) = 4154220349587.d0
      disb(21,18) = -5608224288513.d0
      disb(21,19) = 6943524298975.d0
      disb(21,20) = -7890370660045.d0
      disb(21,21) = 8233430663061.d0
      disb(21,22) = -7890371108108.d0
      disb(21,23) = 6943526580023.d0
      disb(21,24) = -5608233007146.d0
      disb(21,25) = 4154246671960.d0
      disb(21,26) = -2818953098830.d0
      disb(21,27) = 1749695026860.d0
      disb(21,28) = -991493848554.d0
      disb(21,29) = 511738760544.d0
      disb(21,30) = -239877544005.d0
      disb(21,31) = 101766230790.d0
      disb(21,32) = -38910617655.d0
      disb(21,33) = 13340783196.d0
      disb(21,34) = -4076350421.d0
      disb(21,35) = 1101716330.d0
      disb(21,36) = -260932815.d0
      disb(21,37) = 53524680.d0
      disb(21,38) = -9366819.d0
      disb(21,39) = 1370754.d0
      disb(21,40) = -163185.d0
      disb(21,41) = 15180.d0
      disb(21,42) = -1035.d0
      disb(21,43) = 46.d0
      disb(21,44) = -1.d0
c
      disb(22,1) = -253.d0
      disb(22,2) = 7590.d0
      disb(22,3) = -113597.d0
      disb(22,4) = 1133440.d0
      disb(22,5) = -8491945.d0
      disb(22,6) = 50957742.d0
      disb(22,7) = -254803890.d0
      disb(22,8) = 1089621918.d0
      disb(22,9) = -4056410985.d0
      disb(22,10) = 13313117600.d0
      disb(22,11) = -38878167783.d0
      disb(22,12) = 101733988930.d0
      disb(22,13) = -239850413297.d0
      disb(22,14) = 511719474860.d0
      disb(22,15) = -991482326175.d0
      disb(22,16) = 1749689287302.d0
      disb(22,17) = -2818950743400.d0
      disb(22,18) = 4154245889178.d0
      disb(22,19) = -5608232801710.d0
      disb(22,20) = 6943526539290.d0
      disb(22,21) = -7890371108108.d0
      disb(22,22) = 8233430727070.d0
      disb(22,23) = -7890371113927.d0
      disb(22,24) = 6943526580276.d0
      disb(22,25) = -5608233007146.d0
      disb(22,26) = 4154246671960.d0
      disb(22,27) = -2818953098830.d0
      disb(22,28) = 1749695026860.d0
      disb(22,29) = -991493848554.d0
      disb(22,30) = 511738760544.d0
      disb(22,31) = -239877544005.d0
      disb(22,32) = 101766230790.d0
      disb(22,33) = -38910617655.d0
      disb(22,34) = 13340783196.d0
      disb(22,35) = -4076350421.d0
      disb(22,36) = 1101716330.d0
      disb(22,37) = -260932815.d0
      disb(22,38) = 53524680.d0
      disb(22,39) = -9366819.d0
      disb(22,40) = 1370754.d0
      disb(22,41) = -163185.d0
      disb(22,42) = 15180.d0
      disb(22,43) = -1035.d0
      disb(22,44) = 46.d0
      disb(22,45) = -1.d0
c
      disb(23,1) = 23.d0
      disb(23,2) = -782.d0
      disb(23,3) = 13409.d0
      disb(23,4) = -154330.d0
      disb(23,5) = 1337105.d0
      disb(23,6) = -9265872.d0
      disb(23,7) = 53279523.d0
      disb(23,8) = -260442501.d0
      disb(23,9) = 1100899140.d0
      disb(23,10) = -4075206355.d0
      disb(23,11) = 13339431118.d0
      disb(23,12) = -38909265577.d0
      disb(23,13) = 101765086724.d0
      disb(23,14) = -239876726815.d0
      disb(23,15) = 511738270230.d0
      disb(23,16) = -991493603397.d0
      disb(23,17) = 1749694925913.d0
      disb(23,18) = -2818953065181.d0
      disb(23,19) = 4154246663105.d0
      disb(23,20) = -5608233005375.d0
      disb(23,21) = 6943526580023.d0
      disb(23,22) = -7890371113927.d0
      disb(23,23) = 8233430727599.d0
      disb(23,24) = -7890371113950.d0
      disb(23,25) = 6943526580276.d0
      disb(23,26) = -5608233007146.d0
      disb(23,27) = 4154246671960.d0
      disb(23,28) = -2818953098830.d0
      disb(23,29) = 1749695026860.d0
      disb(23,30) = -991493848554.d0
      disb(23,31) = 511738760544.d0
      disb(23,32) = -239877544005.d0
      disb(23,33) = 101766230790.d0
      disb(23,34) = -38910617655.d0
      disb(23,35) = 13340783196.d0
      disb(23,36) = -4076350421.d0
      disb(23,37) = 1101716330.d0
      disb(23,38) = -260932815.d0
      disb(23,39) = 53524680.d0
      disb(23,40) = -9366819.d0
      disb(23,41) = 1370754.d0
      disb(23,42) = -163185.d0
      disb(23,43) = 15180.d0
      disb(23,44) = -1035.d0
      disb(23,45) = 46.d0
      disb(23,46) = -1.d0
c
      elseif(iord.eq.48)then
c 
c 48th order inner with 24th order boundaries
c ------------------------------------------
c
      disi(1) = -32247603683100.d0
      disi(2) = 30957699535776.d0
      disi(3) = -27385657281648.d0
      disi(4) = 22314239266528.d0
      disi(5) = -16735679449896.d0
      disi(6) = 11541847896480.d0
      disi(7) = -7309837001104.d0
      disi(8) = 4244421484512.d0
      disi(9) = -2254848913647.d0
      disi(10) = 1093260079344.d0
      disi(11) = -482320623240.d0
      disi(12) = 192928249296.d0
      disi(13) = -69668534468.d0
      disi(14) = 22595200368.d0
      disi(15) = -6540715896.d0
      disi(16) = 1677106640.d0
      disi(17) = -377348994.d0
      disi(18) = 73629072.d0
      disi(19) = -12271512.d0
      disi(20) = 1712304.d0
      disi(21) = -194580.d0
      disi(22) = 17296.d0
      disi(23) = -1128.d0
      disi(24) = 48.d0
      disi(25) = -1.d0
c
      disb(1,1) = -1.d0
      disb(1,2) = 24.d0
      disb(1,3) = -276.d0
      disb(1,4) = 2024.d0
      disb(1,5) = -10626.d0
      disb(1,6) = 42504.d0
      disb(1,7) = -134596.d0
      disb(1,8) = 346104.d0
      disb(1,9) = -735471.d0
      disb(1,10) = 1307504.d0
      disb(1,11) = -1961256.d0
      disb(1,12) = 2496144.d0
      disb(1,13) = -2704156.d0
      disb(1,14) = 2496144.d0
      disb(1,15) = -1961256.d0
      disb(1,16) = 1307504.d0
      disb(1,17) = -735471.d0
      disb(1,18) = 346104.d0
      disb(1,19) = -134596.d0
      disb(1,20) = 42504.d0
      disb(1,21) = -10626.d0
      disb(1,22) = 2024.d0
      disb(1,23) = -276.d0
      disb(1,24) = 24.d0
      disb(1,25) = -1.d0
c
      disb(2,1) = 24.d0
      disb(2,2) = -577.d0
      disb(2,3) = 6648.d0
      disb(2,4) = -48852.d0
      disb(2,5) = 257048.d0
      disb(2,6) = -1030722.d0
      disb(2,7) = 3272808.d0
      disb(2,8) = -8441092.d0
      disb(2,9) = 17997408.d0
      disb(2,10) = -32115567.d0
      disb(2,11) = 48377648.d0
      disb(2,12) = -61868712.d0
      disb(2,13) = 67395888.d0
      disb(2,14) = -62611612.d0
      disb(2,15) = 49566288.d0
      disb(2,16) = -33341352.d0
      disb(2,17) = 18958808.d0
      disb(2,18) = -9041967.d0
      disb(2,19) = 3576408.d0
      disb(2,20) = -1154692.d0
      disb(2,21) = 297528.d0
      disb(2,22) = -59202.d0
      disb(2,23) = 8648.d0
      disb(2,24) = -852.d0
      disb(2,25) = 48.d0
      disb(2,26) = -1.d0
c
      disb(3,1) = -276.d0
      disb(3,2) = 6648.d0
      disb(3,3) = -76753.d0
      disb(3,4) = 565272.d0
      disb(3,5) = -2981628.d0
      disb(3,6) = 11988152.d0
      disb(3,7) = -38179218.d0
      disb(3,8) = 98797512.d0
      disb(3,9) = -211431088.d0
      disb(3,10) = 378868512.d0
      disb(3,11) = -573422223.d0
      disb(3,12) = 737313392.d0
      disb(3,13) = -808215768.d0
      disb(3,14) = 756331632.d0
      disb(3,15) = -603918268.d0
      disb(3,16) = 410437392.d0
      disb(3,17) = -236331348.d0
      disb(3,18) = 114483512.d0
      disb(3,19) = -46190463.d0
      disb(3,20) = 15307512.d0
      disb(3,21) = -4087468.d0
      disb(3,22) = 856152.d0
      disb(3,23) = -135378.d0
      disb(3,24) = 15272.d0
      disb(3,25) = -1128.d0
      disb(3,26) = 48.d0
      disb(3,27) = -1.d0
c
      disb(4,1) = 2024.d0
      disb(4,2) = -48852.d0
      disb(4,3) = 565272.d0
      disb(4,4) = -4173329.d0
      disb(4,5) = 22072296.d0
      disb(4,6) = -89009724.d0
      disb(4,7) = 284410456.d0
      disb(4,8) = -738693714.d0
      disb(4,9) = 1587390816.d0
      disb(4,10) = -2857819184.d0
      disb(4,11) = 4348450656.d0
      disb(4,12) = -5625617679.d0
      disb(4,13) = 6210525136.d0
      disb(4,14) = -5860411224.d0
      disb(4,15) = 4725913776.d0
      disb(4,16) = -3250306364.d0
      disb(4,17) = 1899030696.d0
      disb(4,18) = -936845844.d0
      disb(4,19) = 386905816.d0
      disb(4,20) = -132218559.d0
      disb(4,21) = 36814536.d0
      disb(4,22) = -8184044.d0
      disb(4,23) = 1414776.d0
      disb(4,24) = -183954.d0
      disb(4,25) = 17296.d0
      disb(4,26) = -1128.d0
      disb(4,27) = 48.d0
      disb(4,28) = -1.d0
c
      disb(5,1) = -10626.d0
      disb(5,2) = 257048.d0
      disb(5,3) = -2981628.d0
      disb(5,4) = 22072296.d0
      disb(5,5) = -117085205.d0
      disb(5,6) = 473719800.d0
      disb(5,7) = -1519226820.d0
      disb(5,8) = 3962111560.d0
      disb(5,9) = -8553808560.d0
      disb(5,10) = 15480928320.d0
      disb(5,11) = -23698125440.d0
      disb(5,12) = 30872476800.d0
      disb(5,13) = -34359979335.d0
      disb(5,14) = 32734551280.d0
      disb(5,15) = -26700717480.d0
      disb(5,16) = 18619451280.d0
      disb(5,17) = -11065421210.d0
      disb(5,18) = 5576731800.d0
      disb(5,19) = -2367062940.d0
      disb(5,20) = 838553320.d0
      disb(5,21) = -245130435.d0
      disb(5,22) = 58321560.d0
      disb(5,23) = -11116820.d0
      disb(5,24) = 1669800.d0
      disb(5,25) = -194580.d0
      disb(5,26) = 17296.d0
      disb(5,27) = -1128.d0
      disb(5,28) = 48.d0
      disb(5,29) = -1.d0
c
      disb(6,1) = 42504.d0
      disb(6,2) = -1030722.d0
      disb(6,3) = 11988152.d0
      disb(6,4) = -89009724.d0
      disb(6,5) = 473719800.d0
      disb(6,6) = -1923675221.d0
      disb(6,7) = 6194588184.d0
      disb(6,8) = -16230031236.d0
      disb(6,9) = 35222570944.d0
      disb(6,10) = -64127958576.d0
      disb(6,11) = 98842153344.d0
      disb(6,12) = -129794230016.d0
      disb(6,13) = 145809923424.d0
      disb(6,14) = -140456083911.d0
      disb(6,15) = 116095776304.d0
      disb(6,16) = -82274867496.d0
      disb(6,17) = 49879910664.d0
      disb(6,18) = -25776225626.d0
      disb(6,19) = 11297600184.d0
      disb(6,20) = -4173652956.d0
      disb(6,21) = 1290200824.d0
      disb(6,22) = -331158531.d0
      disb(6,23) = 70052664.d0
      disb(6,24) = -12136916.d0
      disb(6,25) = 1712304.d0
      disb(6,26) = -194580.d0
      disb(6,27) = 17296.d0
      disb(6,28) = -1128.d0
      disb(6,29) = 48.d0
      disb(6,30) = -1.d0
c
      disb(7,1) = -134596.d0
      disb(7,2) = 3272808.d0
      disb(7,3) = -38179218.d0
      disb(7,4) = 284410456.d0
      disb(7,5) = -1519226820.d0
      disb(7,6) = 6194588184.d0
      disb(7,7) = -20039758437.d0
      disb(7,8) = 52778802168.d0
      disb(7,9) = -115221485952.d0
      disb(7,10) = 211207379328.d0
      disb(7,11) = -328105171152.d0
      disb(7,12) = 434813151168.d0
      disb(7,13) = -493762810992.d0
      disb(7,14) = 481780921248.d0
      disb(7,15) = -404433296487.d0
      disb(7,16) = 292080584688.d0
      disb(7,17) = -181266322212.d0
      disb(7,18) = 96464124648.d0
      disb(7,19) = -43892308842.d0
      disb(7,20) = 17018468568.d0
      disb(7,21) = -5603870052.d0
      disb(7,22) = 1562623128.d0
      disb(7,23) = -368307027.d0
      disb(7,24) = 73282968.d0
      disb(7,25) = -12271512.d0
      disb(7,26) = 1712304.d0
      disb(7,27) = -194580.d0
      disb(7,28) = 17296.d0
      disb(7,29) = -1128.d0
      disb(7,30) = 48.d0
      disb(7,31) = -1.d0
c
      disb(8,1) = 346104.d0
      disb(8,2) = -8441092.d0
      disb(8,3) = 98797512.d0
      disb(8,4) = -738693714.d0
      disb(8,5) = 3962111560.d0
      disb(8,6) = -16230031236.d0
      disb(8,7) = 52778802168.d0
      disb(8,8) = -139827737253.d0
      disb(8,9) = 307328257152.d0
      disb(8,10) = -567753850368.d0
      disb(8,11) = 890005925952.d0
      disb(8,12) = -1192030594128.d0
      disb(8,13) = 1370732359392.d0
      disb(8,14) = -1357688233968.d0
      disb(8,15) = 1160579467872.d0
      disb(8,16) = -856965660903.d0
      disb(8,17) = 546630039672.d0
      disb(8,18) = -301054301028.d0
      disb(8,19) = 143048338632.d0
      disb(8,20) = -58603113258.d0
      disb(8,21) = 20696169672.d0
      disb(8,22) = -6304384548.d0
      disb(8,23) = 1658147832.d0
      disb(8,24) = -376613523.d0
      disb(8,25) = 73629072.d0
      disb(8,26) = -12271512.d0
      disb(8,27) = 1712304.d0
      disb(8,28) = -194580.d0
      disb(8,29) = 17296.d0
      disb(8,30) = -1128.d0
      disb(8,31) = 48.d0
      disb(8,32) = -1.d0
c
      disb(9,1) = -735471.d0
      disb(9,2) = 17997408.d0
      disb(9,3) = -211431088.d0
      disb(9,4) = 1587390816.d0
      disb(9,5) = -8553808560.d0
      disb(9,6) = 35222570944.d0
      disb(9,7) = -115221485952.d0
      disb(9,8) = 307328257152.d0
      disb(9,9) = -680745329094.d0
      disb(9,10) = 1268959531536.d0
      disb(9,11) = -2010200761944.d0
      disb(9,12) = 2725847449776.d0
      disb(9,13) = -3180858911604.d0
      disb(9,14) = 3206573883216.d0
      disb(9,15) = -2800135145544.d0
      disb(9,16) = 2122210742256.d0
      disb(9,17) = -1397883252744.d0
      disb(9,18) = 801179494656.d0
      disb(9,19) = -400045755744.d0
      disb(9,20) = 174308798016.d0
      disb(9,21) = -66418228104.d0
      disb(9,22) = 22184762976.d0
      disb(9,23) = -6507374544.d0
      disb(9,24) = 1675799136.d0
      disb(9,25) = -377348994.d0
      disb(9,26) = 73629072.d0
      disb(9,27) = -12271512.d0
      disb(9,28) = 1712304.d0
      disb(9,29) = -194580.d0
      disb(9,30) = 17296.d0
      disb(9,31) = -1128.d0
      disb(9,32) = 48.d0
      disb(9,33) = -1.d0
c
      disb(10,1) = 1307504.d0
      disb(10,2) = -32115567.d0
      disb(10,3) = 378868512.d0
      disb(10,4) = -2857819184.d0
      disb(10,5) = 15480928320.d0
      disb(10,6) = -64127958576.d0
      disb(10,7) = 211207379328.d0
      disb(10,8) = -567753850368.d0
      disb(10,9) = 1268959531536.d0
      disb(10,10) = -2390312039110.d0
      disb(10,11) = 3833309596560.d0
      disb(10,12) = -5273919026520.d0
      disb(10,13) = 6261542236400.d0
      disb(10,14) = -6444577176180.d0
      disb(10,15) = 5770923948240.d0
      disb(10,16) = -4509701855560.d0
      disb(10,17) = 3083842016640.d0
      disb(10,18) = -1850415617160.d0
      disb(10,19) = 977164303040.d0
      disb(10,20) = -455619905760.d0
      disb(10,21) = 188202335520.d0
      disb(10,22) = -69064616200.d0
      disb(10,23) = 22545634080.d0
      disb(10,24) = -6538754640.d0
      disb(10,25) = 1677106640.d0
      disb(10,26) = -377348994.d0
      disb(10,27) = 73629072.d0
      disb(10,28) = -12271512.d0
      disb(10,29) = 1712304.d0
      disb(10,30) = -194580.d0
      disb(10,31) = 17296.d0
      disb(10,32) = -1128.d0
      disb(10,33) = 48.d0
      disb(10,34) = -1.d0
c
      disb(11,1) = -1961256.d0
      disb(11,2) = 48377648.d0
      disb(11,3) = -573422223.d0
      disb(11,4) = 4348450656.d0
      disb(11,5) = -23698125440.d0
      disb(11,6) = 98842153344.d0
      disb(11,7) = -328105171152.d0
      disb(11,8) = 890005925952.d0
      disb(11,9) = -2010200761944.d0
      disb(11,10) = 3833309596560.d0
      disb(11,11) = -6236837136646.d0
      disb(11,12) = 8728886993424.d0
      disb(11,13) = -10577461206456.d0
      disb(11,14) = 11157119633264.d0
      disb(11,15) = -10291102273716.d0
      disb(11,16) = 8335274013264.d0
      disb(11,17) = -5952148767136.d0
      disb(11,18) = 3762640563264.d0
      disb(11,19) = -2114392829736.d0
      disb(11,20) = 1060525528064.d0
      disb(11,21) = -476460212016.d0
      disb(11,22) = 192171917664.d0
      disb(11,23) = -69605922856.d0
      disb(11,24) = 22592704224.d0
      disb(11,25) = -6540715896.d0
      disb(11,26) = 1677106640.d0
      disb(11,27) = -377348994.d0
      disb(11,28) = 73629072.d0
      disb(11,29) = -12271512.d0
      disb(11,30) = 1712304.d0
      disb(11,31) = -194580.d0
      disb(11,32) = 17296.d0
      disb(11,33) = -1128.d0
      disb(11,34) = 48.d0
      disb(11,35) = -1.d0
c
      disb(12,1) = 2496144.d0
      disb(12,2) = -61868712.d0
      disb(12,3) = 737313392.d0
      disb(12,4) = -5625617679.d0
      disb(12,5) = 30872476800.d0
      disb(12,6) = -129794230016.d0
      disb(12,7) = 434813151168.d0
      disb(12,8) = -1192030594128.d0
      disb(12,9) = 2725847449776.d0
      disb(12,10) = -5273919026520.d0
      disb(12,11) = 8728886993424.d0
      disb(12,12) = -12467572005382.d0
      disb(12,13) = 15478849767888.d0
      disb(12,14) = -16808196075192.d0
      disb(12,15) = 16052697030128.d0
      disb(12,16) = -13554820538292.d0
      disb(12,17) = 10171115537088.d0
      disb(12,18) = -6816074190112.d0
      disb(12,19) = 4098611561088.d0
      disb(12,20) = -2220488934312.d0
      disb(12,21) = 1087049554208.d0
      disb(12,22) = -481512407472.d0
      disb(12,23) = 192860853408.d0
      disb(12,24) = -69665830312.d0
      disb(12,25) = 22595200368.d0
      disb(12,26) = -6540715896.d0
      disb(12,27) = 1677106640.d0
      disb(12,28) = -377348994.d0
      disb(12,29) = 73629072.d0
      disb(12,30) = -12271512.d0
      disb(12,31) = 1712304.d0
      disb(12,32) = -194580.d0
      disb(12,33) = 17296.d0
      disb(12,34) = -1128.d0
      disb(12,35) = 48.d0
      disb(12,36) = -1.d0
c
      disb(13,1) = -2704156.d0
      disb(13,2) = 67395888.d0
      disb(13,3) = -808215768.d0
      disb(13,4) = 6210525136.d0
      disb(13,5) = -34359979335.d0
      disb(13,6) = 145809923424.d0
      disb(13,7) = -493762810992.d0
      disb(13,8) = 1370732359392.d0
      disb(13,9) = -3180858911604.d0
      disb(13,10) = 6261542236400.d0
      disb(13,11) = -10577461206456.d0
      disb(13,12) = 15478849767888.d0
      disb(13,13) = -19780031677718.d0
      disb(13,14) = 22228812542352.d0
      disb(13,15) = -22111738255128.d0
      disb(13,16) = 19588391816752.d0
      disb(13,17) = -15543648855768.d0
      disb(13,18) = 11107034745312.d0
      disb(13,19) = -7180042771088.d0
      disb(13,20) = 4213549007712.d0
      disb(13,21) = -2249223295968.d0
      disb(13,22) = 1092522765952.d0
      disb(13,23) = -482258754528.d0
      disb(13,24) = 192925753152.d0
      disb(13,25) = -69668534468.d0
      disb(13,26) = 22595200368.d0
      disb(13,27) = -6540715896.d0
      disb(13,28) = 1677106640.d0
      disb(13,29) = -377348994.d0
      disb(13,30) = 73629072.d0
      disb(13,31) = -12271512.d0
      disb(13,32) = 1712304.d0
      disb(13,33) = -194580.d0
      disb(13,34) = 17296.d0
      disb(13,35) = -1128.d0
      disb(13,36) = 48.d0
      disb(13,37) = -1.d0
c
      disb(14,1) = 2496144.d0
      disb(14,2) = -62611612.d0
      disb(14,3) = 756331632.d0
      disb(14,4) = -5860411224.d0
      disb(14,5) = 32734551280.d0
      disb(14,6) = -140456083911.d0
      disb(14,7) = 481780921248.d0
      disb(14,8) = -1357688233968.d0
      disb(14,9) = 3206573883216.d0
      disb(14,10) = -6444577176180.d0
      disb(14,11) = 11157119633264.d0
      disb(14,12) = -16808196075192.d0
      disb(14,13) = 22228812542352.d0
      disb(14,14) = -26010766546454.d0
      disb(14,15) = 27124389939216.d0
      disb(14,16) = -25375456519704.d0
      disb(14,17) = 21424233340576.d0
      disb(14,18) = -16407574278744.d0
      disb(14,19) = 11443005743136.d0
      disb(14,20) = -7286138875664.d0
      disb(14,21) = 4240073033856.d0
      disb(14,22) = -2254275491424.d0
      disb(14,23) = 1093211701696.d0
      disb(14,24) = -482318661984.d0
      disb(14,25) = 192928249296.d0
      disb(14,26) = -69668534468.d0
      disb(14,27) = 22595200368.d0
      disb(14,28) = -6540715896.d0
      disb(14,29) = 1677106640.d0
      disb(14,30) = -377348994.d0
      disb(14,31) = 73629072.d0
      disb(14,32) = -12271512.d0
      disb(14,33) = 1712304.d0
      disb(14,34) = -194580.d0
      disb(14,35) = 17296.d0
      disb(14,36) = -1128.d0
      disb(14,37) = 48.d0
      disb(14,38) = -1.d0
c
      disb(15,1) = -1961256.d0
      disb(15,2) = 49566288.d0
      disb(15,3) = -603918268.d0
      disb(15,4) = 4725913776.d0
      disb(15,5) = -26700717480.d0
      disb(15,6) = 116095776304.d0
      disb(15,7) = -404433296487.d0
      disb(15,8) = 1160579467872.d0
      disb(15,9) = -2800135145544.d0
      disb(15,10) = 5770923948240.d0
      disb(15,11) = -10291102273716.d0
      disb(15,12) = 16052697030128.d0
      disb(15,13) = -22111738255128.d0
      disb(15,14) = 27124389939216.d0
      disb(15,15) = -29857291643990.d0
      disb(15,16) = 29688740004240.d0
      disb(15,17) = -26817903431280.d0
      disb(15,18) = 22103031887200.d0
      disb(15,19) = -16671551491320.d0
      disb(15,20) = 11526366968160.d0
      disb(15,21) = -7306979181920.d0
      disb(15,22) = 4244042616000.d0
      disb(15,23) = -2254816798080.d0
      disb(15,24) = 1093258771840.d0
      disb(15,25) = -482320623240.d0
      disb(15,26) = 192928249296.d0
      disb(15,27) = -69668534468.d0
      disb(15,28) = 22595200368.d0
      disb(15,29) = -6540715896.d0
      disb(15,30) = 1677106640.d0
      disb(15,31) = -377348994.d0
      disb(15,32) = 73629072.d0
      disb(15,33) = -12271512.d0
      disb(15,34) = 1712304.d0
      disb(15,35) = -194580.d0
      disb(15,36) = 17296.d0
      disb(15,37) = -1128.d0
      disb(15,38) = 48.d0
      disb(15,39) = -1.d0
c
      disb(16,1) = 1307504.d0
      disb(16,2) = -33341352.d0
      disb(16,3) = 410437392.d0
      disb(16,4) = -3250306364.d0
      disb(16,5) = 18619451280.d0
      disb(16,6) = -82274867496.d0
      disb(16,7) = 292080584688.d0
      disb(16,8) = -856965660903.d0
      disb(16,9) = 2122210742256.d0
      disb(16,10) = -4509701855560.d0
      disb(16,11) = 8335274013264.d0
      disb(16,12) = -13554820538292.d0
      disb(16,13) = 19588391816752.d0
      disb(16,14) = -25375456519704.d0
      disb(16,15) = 29688740004240.d0
      disb(16,16) = -31566858354006.d0
      disb(16,17) = 30650371278624.d0
      disb(16,18) = -27270435795696.d0
      disb(16,19) = 22279016695584.d0
      disb(16,20) = -16727125641336.d0
      disb(16,21) = 11540260505664.d0
      disb(16,22) = -7309625570016.d0
      disb(16,23) = 4244403487104.d0
      disb(16,24) = -2254848178176.d0
      disb(16,25) = 1093260079344.d0
      disb(16,26) = -482320623240.d0
      disb(16,27) = 192928249296.d0
      disb(16,28) = -69668534468.d0
      disb(16,29) = 22595200368.d0
      disb(16,30) = -6540715896.d0
      disb(16,31) = 1677106640.d0
      disb(16,32) = -377348994.d0
      disb(16,33) = 73629072.d0
      disb(16,34) = -12271512.d0
      disb(16,35) = 1712304.d0
      disb(16,36) = -194580.d0
      disb(16,37) = 17296.d0
      disb(16,38) = -1128.d0
      disb(16,39) = 48.d0
      disb(16,40) = -1.d0
c
      disb(17,1) = -735471.d0
      disb(17,2) = 18958808.d0
      disb(17,3) = -236331348.d0
      disb(17,4) = 1899030696.d0
      disb(17,5) = -11065421210.d0
      disb(17,6) = 49879910664.d0
      disb(17,7) = -181266322212.d0
      disb(17,8) = 546630039672.d0
      disb(17,9) = -1397883252744.d0
      disb(17,10) = 3083842016640.d0
      disb(17,11) = -5952148767136.d0
      disb(17,12) = 10171115537088.d0
      disb(17,13) = -15543648855768.d0
      disb(17,14) = 21424233340576.d0
      disb(17,15) = -26817903431280.d0
      disb(17,16) = 30650371278624.d0
      disb(17,17) = -32107775945847.d0
      disb(17,18) = 30904920733608.d0
      disb(17,19) = -27369427250412.d0
      disb(17,20) = 22310277154968.d0
      disb(17,21) = -16734940756182.d0
      disb(17,22) = 11541749098968.d0
      disb(17,23) = -7309828560012.d0
      disb(17,24) = 4244421138408.d0
      disb(17,25) = -2254848913647.d0
      disb(17,26) = 1093260079344.d0
      disb(17,27) = -482320623240.d0
      disb(17,28) = 192928249296.d0
      disb(17,29) = -69668534468.d0
      disb(17,30) = 22595200368.d0
      disb(17,31) = -6540715896.d0
      disb(17,32) = 1677106640.d0
      disb(17,33) = -377348994.d0
      disb(17,34) = 73629072.d0
      disb(17,35) = -12271512.d0
      disb(17,36) = 1712304.d0
      disb(17,37) = -194580.d0
      disb(17,38) = 17296.d0
      disb(17,39) = -1128.d0
      disb(17,40) = 48.d0
      disb(17,41) = -1.d0
c
      disb(18,1) = 346104.d0
      disb(18,2) = -9041967.d0
      disb(18,3) = 114483512.d0
      disb(18,4) = -936845844.d0
      disb(18,5) = 5576731800.d0
      disb(18,6) = -25776225626.d0
      disb(18,7) = 96464124648.d0
      disb(18,8) = -301054301028.d0
      disb(18,9) = 801179494656.d0
      disb(18,10) = -1850415617160.d0
      disb(18,11) = 3762640563264.d0
      disb(18,12) = -6816074190112.d0
      disb(18,13) = 11107034745312.d0
      disb(18,14) = -16407574278744.d0
      disb(18,15) = 22103031887200.d0
      disb(18,16) = -27270435795696.d0
      disb(18,17) = 30904920733608.d0
      disb(18,18) = -32227563924663.d0
      disb(18,19) = 30951504947592.d0
      disb(18,20) = -27384138054828.d0
      disb(18,21) = 22313954856072.d0
      disb(18,22) = -16735641270678.d0
      disb(18,23) = 11541844623672.d0
      disb(18,24) = -7309836866508.d0
      disb(18,25) = 4244421484512.d0
      disb(18,26) = -2254848913647.d0
      disb(18,27) = 1093260079344.d0
      disb(18,28) = -482320623240.d0
      disb(18,29) = 192928249296.d0
      disb(18,30) = -69668534468.d0
      disb(18,31) = 22595200368.d0
      disb(18,32) = -6540715896.d0
      disb(18,33) = 1677106640.d0
      disb(18,34) = -377348994.d0
      disb(18,35) = 73629072.d0
      disb(18,36) = -12271512.d0
      disb(18,37) = 1712304.d0
      disb(18,38) = -194580.d0
      disb(18,39) = 17296.d0
      disb(18,40) = -1128.d0
      disb(18,41) = 48.d0
      disb(18,42) = -1.d0
c
      disb(19,1) = -134596.d0
      disb(19,2) = 3576408.d0
      disb(19,3) = -46190463.d0
      disb(19,4) = 386905816.d0
      disb(19,5) = -2367062940.d0
      disb(19,6) = 11297600184.d0
      disb(19,7) = -43892308842.d0
      disb(19,8) = 143048338632.d0
      disb(19,9) = -400045755744.d0
      disb(19,10) = 977164303040.d0
      disb(19,11) = -2114392829736.d0
      disb(19,12) = 4098611561088.d0
      disb(19,13) = -7180042771088.d0
      disb(19,14) = 11443005743136.d0
      disb(19,15) = -16671551491320.d0
      disb(19,16) = 22279016695584.d0
      disb(19,17) = -27369427250412.d0
      disb(19,18) = 30951504947592.d0
      disb(19,19) = -32245680007879.d0
      disb(19,20) = 30957225815976.d0
      disb(19,21) = -27385568271924.d0
      disb(19,22) = 22314227278376.d0
      disb(19,23) = -16735678419174.d0
      disb(19,24) = 11541847853976.d0
      disb(19,25) = -7309837001104.d0
      disb(19,26) = 4244421484512.d0
      disb(19,27) = -2254848913647.d0
      disb(19,28) = 1093260079344.d0
      disb(19,29) = -482320623240.d0
      disb(19,30) = 192928249296.d0
      disb(19,31) = -69668534468.d0
      disb(19,32) = 22595200368.d0
      disb(19,33) = -6540715896.d0
      disb(19,34) = 1677106640.d0
      disb(19,35) = -377348994.d0
      disb(19,36) = 73629072.d0
      disb(19,37) = -12271512.d0
      disb(19,38) = 1712304.d0
      disb(19,39) = -194580.d0
      disb(19,40) = 17296.d0
      disb(19,41) = -1128.d0
      disb(19,42) = 48.d0
      disb(19,43) = -1.d0
c
      disb(20,1) = 42504.d0
      disb(20,2) = -1154692.d0
      disb(20,3) = 15307512.d0
      disb(20,4) = -132218559.d0
      disb(20,5) = 838553320.d0
      disb(20,6) = -4173652956.d0
      disb(20,7) = 17018468568.d0
      disb(20,8) = -58603113258.d0
      disb(20,9) = 174308798016.d0
      disb(20,10) = -455619905760.d0
      disb(20,11) = 1060525528064.d0
      disb(20,12) = -2220488934312.d0
      disb(20,13) = 4213549007712.d0
      disb(20,14) = -7286138875664.d0
      disb(20,15) = 11526366968160.d0
      disb(20,16) = -16727125641336.d0
      disb(20,17) = 22310277154968.d0
      disb(20,18) = -27384138054828.d0
      disb(20,19) = 30957225815976.d0
      disb(20,20) = -32247486597895.d0
      disb(20,21) = 30957677463480.d0
      disb(20,22) = -27385654300020.d0
      disb(20,23) = 22314239009480.d0
      disb(20,24) = -16735679439270.d0
      disb(20,25) = 11541847896480.d0
      disb(20,26) = -7309837001104.d0
      disb(20,27) = 4244421484512.d0
      disb(20,28) = -2254848913647.d0
      disb(20,29) = 1093260079344.d0
      disb(20,30) = -482320623240.d0
      disb(20,31) = 192928249296.d0
      disb(20,32) = -69668534468.d0
      disb(20,33) = 22595200368.d0
      disb(20,34) = -6540715896.d0
      disb(20,35) = 1677106640.d0
      disb(20,36) = -377348994.d0
      disb(20,37) = 73629072.d0
      disb(20,38) = -12271512.d0
      disb(20,39) = 1712304.d0
      disb(20,40) = -194580.d0
      disb(20,41) = 17296.d0
      disb(20,42) = -1128.d0
      disb(20,43) = 48.d0
      disb(20,44) = -1.d0
c
      disb(21,1) = -10626.d0
      disb(21,2) = 297528.d0
      disb(21,3) = -4087468.d0
      disb(21,4) = 36814536.d0
      disb(21,5) = -245130435.d0
      disb(21,6) = 1290200824.d0
      disb(21,7) = -5603870052.d0
      disb(21,8) = 20696169672.d0
      disb(21,9) = -66418228104.d0
      disb(21,10) = 188202335520.d0
      disb(21,11) = -476460212016.d0
      disb(21,12) = 1087049554208.d0
      disb(21,13) = -2249223295968.d0
      disb(21,14) = 4240073033856.d0
      disb(21,15) = -7306979181920.d0
      disb(21,16) = 11540260505664.d0
      disb(21,17) = -16734940756182.d0
      disb(21,18) = 22313954856072.d0
      disb(21,19) = -27385568271924.d0
      disb(21,20) = 30957677463480.d0
      disb(21,21) = -32247599509771.d0
      disb(21,22) = 30957698970504.d0
      disb(21,23) = -27385657232796.d0
      disb(21,24) = 22314239264504.d0
      disb(21,25) = -16735679449896.d0
      disb(21,26) = 11541847896480.d0
      disb(21,27) = -7309837001104.d0
      disb(21,28) = 4244421484512.d0
      disb(21,29) = -2254848913647.d0
      disb(21,30) = 1093260079344.d0
      disb(21,31) = -482320623240.d0
      disb(21,32) = 192928249296.d0
      disb(21,33) = -69668534468.d0
      disb(21,34) = 22595200368.d0
      disb(21,35) = -6540715896.d0
      disb(21,36) = 1677106640.d0
      disb(21,37) = -377348994.d0
      disb(21,38) = 73629072.d0
      disb(21,39) = -12271512.d0
      disb(21,40) = 1712304.d0
      disb(21,41) = -194580.d0
      disb(21,42) = 17296.d0
      disb(21,43) = -1128.d0
      disb(21,44) = 48.d0
      disb(21,45) = -1.d0
c
      disb(22,1) = 2024.d0
      disb(22,2) = -59202.d0
      disb(22,3) = 856152.d0
      disb(22,4) = -8184044.d0
      disb(22,5) = 58321560.d0
      disb(22,6) = -331158531.d0
      disb(22,7) = 1562623128.d0
      disb(22,8) = -6304384548.d0
      disb(22,9) = 22184762976.d0
      disb(22,10) = -69064616200.d0
      disb(22,11) = 192171917664.d0
      disb(22,12) = -481512407472.d0
      disb(22,13) = 1092522765952.d0
      disb(22,14) = -2254275491424.d0
      disb(22,15) = 4244042616000.d0
      disb(22,16) = -7309625570016.d0
      disb(22,17) = 11541749098968.d0
      disb(22,18) = -16735641270678.d0
      disb(22,19) = 22314227278376.d0
      disb(22,20) = -27385654300020.d0
      disb(22,21) = 30957698970504.d0
      disb(22,22) = -32247603606347.d0
      disb(22,23) = 30957699529128.d0
      disb(22,24) = -27385657281372.d0
      disb(22,25) = 22314239266528.d0
      disb(22,26) = -16735679449896.d0
      disb(22,27) = 11541847896480.d0
      disb(22,28) = -7309837001104.d0
      disb(22,29) = 4244421484512.d0
      disb(22,30) = -2254848913647.d0
      disb(22,31) = 1093260079344.d0
      disb(22,32) = -482320623240.d0
      disb(22,33) = 192928249296.d0
      disb(22,34) = -69668534468.d0
      disb(22,35) = 22595200368.d0
      disb(22,36) = -6540715896.d0
      disb(22,37) = 1677106640.d0
      disb(22,38) = -377348994.d0
      disb(22,39) = 73629072.d0
      disb(22,40) = -12271512.d0
      disb(22,41) = 1712304.d0
      disb(22,42) = -194580.d0
      disb(22,43) = 17296.d0
      disb(22,44) = -1128.d0
      disb(22,45) = 48.d0
      disb(22,46) = -1.d0
c
      disb(23,1) = -276.d0
      disb(23,2) = 8648.d0
      disb(23,3) = -135378.d0
      disb(23,4) = 1414776.d0
      disb(23,5) = -11116820.d0
      disb(23,6) = 70052664.d0
      disb(23,7) = -368307027.d0
      disb(23,8) = 1658147832.d0
      disb(23,9) = -6507374544.d0
      disb(23,10) = 22545634080.d0
      disb(23,11) = -69605922856.d0
      disb(23,12) = 192860853408.d0
      disb(23,13) = -482258754528.d0
      disb(23,14) = 1093211701696.d0
      disb(23,15) = -2254816798080.d0
      disb(23,16) = 4244403487104.d0
      disb(23,17) = -7309828560012.d0
      disb(23,18) = 11541844623672.d0
      disb(23,19) = -16735678419174.d0
      disb(23,20) = 22314239009480.d0
      disb(23,21) = -27385657232796.d0
      disb(23,22) = 30957699529128.d0
      disb(23,23) = -32247603682523.d0
      disb(23,24) = 30957699535752.d0
      disb(23,25) = -27385657281648.d0
      disb(23,26) = 22314239266528.d0
      disb(23,27) = -16735679449896.d0
      disb(23,28) = 11541847896480.d0
      disb(23,29) = -7309837001104.d0
      disb(23,30) = 4244421484512.d0
      disb(23,31) = -2254848913647.d0
      disb(23,32) = 1093260079344.d0
      disb(23,33) = -482320623240.d0
      disb(23,34) = 192928249296.d0
      disb(23,35) = -69668534468.d0
      disb(23,36) = 22595200368.d0
      disb(23,37) = -6540715896.d0
      disb(23,38) = 1677106640.d0
      disb(23,39) = -377348994.d0
      disb(23,40) = 73629072.d0
      disb(23,41) = -12271512.d0
      disb(23,42) = 1712304.d0
      disb(23,43) = -194580.d0
      disb(23,44) = 17296.d0
      disb(23,45) = -1128.d0
      disb(23,46) = 48.d0
      disb(23,47) = -1.d0
c
      disb(24,1) = 24.d0
      disb(24,2) = -852.d0
      disb(24,3) = 15272.d0
      disb(24,4) = -183954.d0
      disb(24,5) = 1669800.d0
      disb(24,6) = -12136916.d0
      disb(24,7) = 73282968.d0
      disb(24,8) = -376613523.d0
      disb(24,9) = 1675799136.d0
      disb(24,10) = -6538754640.d0
      disb(24,11) = 22592704224.d0
      disb(24,12) = -69665830312.d0
      disb(24,13) = 192925753152.d0
      disb(24,14) = -482318661984.d0
      disb(24,15) = 1093258771840.d0
      disb(24,16) = -2254848178176.d0
      disb(24,17) = 4244421138408.d0
      disb(24,18) = -7309836866508.d0
      disb(24,19) = 11541847853976.d0
      disb(24,20) = -16735679439270.d0
      disb(24,21) = 22314239264504.d0
      disb(24,22) = -27385657281372.d0
      disb(24,23) = 30957699535752.d0
      disb(24,24) = -32247603683099.d0
      disb(24,25) = 30957699535776.d0
      disb(24,26) = -27385657281648.d0
      disb(24,27) = 22314239266528.d0
      disb(24,28) = -16735679449896.d0
      disb(24,29) = 11541847896480.d0
      disb(24,30) = -7309837001104.d0
      disb(24,31) = 4244421484512.d0
      disb(24,32) = -2254848913647.d0
      disb(24,33) = 1093260079344.d0
      disb(24,34) = -482320623240.d0
      disb(24,35) = 192928249296.d0
      disb(24,36) = -69668534468.d0
      disb(24,37) = 22595200368.d0
      disb(24,38) = -6540715896.d0
      disb(24,39) = 1677106640.d0
      disb(24,40) = -377348994.d0
      disb(24,41) = 73629072.d0
      disb(24,42) = -12271512.d0
      disb(24,43) = 1712304.d0
      disb(24,44) = -194580.d0
      disb(24,45) = 17296.d0
      disb(24,46) = -1128.d0
      disb(24,47) = 48.d0
      disb(24,48) = -1.d0
c
      elseif(iord.eq.50)then
c 
c 50th order inner with 25th order boundaries
c ------------------------------------------
c
      disi(1) = 126410606437752.d0
      disi(2) = -121548660036300.d0
      disi(3) = 108043253365600.d0
      disi(4) = -88749815264600.d0
      disi(5) = 67327446062800.d0
      disi(6) = -47129212243960.d0
      disi(7) = 30405943383200.d0
      disi(8) = -18053528883775.d0
      disi(9) = 9847379391150.d0
      disi(10) = -4923689695575.d0
      disi(11) = 2250829575120.d0
      disi(12) = -937845656300.d0
      disi(13) = 354860518600.d0
      disi(14) = -121399651100.d0
      disi(15) = 37353738800.d0
      disi(16) = -10272278170.d0
      disi(17) = 2505433700.d0
      disi(18) = -536878650.d0
      disi(19) = 99884400.d0
      disi(20) = -15890700.d0
      disi(21) = 2118760.d0
      disi(22) = -230300.d0
      disi(23) = 19600.d0
      disi(24) = -1225.d0
      disi(25) = 50.d0
      disi(26) = -1.d0
c
      disb(1,1) = 1.d0
      disb(1,2) = -25.d0
      disb(1,3) = 300.d0
      disb(1,4) = -2300.d0
      disb(1,5) = 12650.d0
      disb(1,6) = -53130.d0
      disb(1,7) = 177100.d0
      disb(1,8) = -480700.d0
      disb(1,9) = 1081575.d0
      disb(1,10) = -2042975.d0
      disb(1,11) = 3268760.d0
      disb(1,12) = -4457400.d0
      disb(1,13) = 5200300.d0
      disb(1,14) = -5200300.d0
      disb(1,15) = 4457400.d0
      disb(1,16) = -3268760.d0
      disb(1,17) = 2042975.d0
      disb(1,18) = -1081575.d0
      disb(1,19) = 480700.d0
      disb(1,20) = -177100.d0
      disb(1,21) = 53130.d0
      disb(1,22) = -12650.d0
      disb(1,23) = 2300.d0
      disb(1,24) = -300.d0
      disb(1,25) = 25.d0
      disb(1,26) = -1.d0
c
      disb(2,1) = -25.d0
      disb(2,2) = 626.d0
      disb(2,3) = -7525.d0
      disb(2,4) = 57800.d0
      disb(2,5) = -318550.d0
      disb(2,6) = 1340900.d0
      disb(2,7) = -4480630.d0
      disb(2,8) = 12194600.d0
      disb(2,9) = -27520075.d0
      disb(2,10) = 52155950.d0
      disb(2,11) = -83761975.d0
      disb(2,12) = 114703760.d0
      disb(2,13) = -134464900.d0
      disb(2,14) = 135207800.d0
      disb(2,15) = -116635300.d0
      disb(2,16) = 86176400.d0
      disb(2,17) = -54343135.d0
      disb(2,18) = 29082350.d0
      disb(2,19) = -13099075.d0
      disb(2,20) = 4908200.d0
      disb(2,21) = -1505350.d0
      disb(2,22) = 369380.d0
      disb(2,23) = -70150.d0
      disb(2,24) = 9800.d0
      disb(2,25) = -925.d0
      disb(2,26) = 50.d0
      disb(2,27) = -1.d0
c
      disb(3,1) = 300.d0
      disb(3,2) = -7525.d0
      disb(3,3) = 90626.d0
      disb(3,4) = -697525.d0
      disb(3,5) = 3852800.d0
      disb(3,6) = -16257550.d0
      disb(3,7) = 54470900.d0
      disb(3,8) = -148690630.d0
      disb(3,9) = 336667100.d0
      disb(3,10) = -640412575.d0
      disb(3,11) = 1032783950.d0
      disb(3,12) = -1420981975.d0
      disb(3,13) = 1674793760.d0
      disb(3,14) = -1694554900.d0
      disb(3,15) = 1472427800.d0
      disb(3,16) = -1097263300.d0
      disb(3,17) = 699068900.d0
      disb(3,18) = -378815635.d0
      disb(3,19) = 173292350.d0
      disb(3,20) = -66229075.d0
      disb(3,21) = 20847200.d0
      disb(3,22) = -5300350.d0
      disb(3,23) = 1059380.d0
      disb(3,24) = -160150.d0
      disb(3,25) = 17300.d0
      disb(3,26) = -1225.d0
      disb(3,27) = 50.d0
      disb(3,28) = -1.d0
c
      disb(4,1) = -2300.d0
      disb(4,2) = 57800.d0
      disb(4,3) = -697525.d0
      disb(4,4) = 5380626.d0
      disb(4,5) = -29792525.d0
      disb(4,6) = 126051800.d0
      disb(4,7) = -423587550.d0
      disb(4,8) = 1160080900.d0
      disb(4,9) = -2636313130.d0
      disb(4,10) = 5035509600.d0
      disb(4,11) = -8158560575.d0
      disb(4,12) = 11284803950.d0
      disb(4,13) = -13381671975.d0
      disb(4,14) = 13635483760.d0
      disb(4,15) = -11946574900.d0
      disb(4,16) = 8990575800.d0
      disb(4,17) = -5796105800.d0
      disb(4,18) = 3186691400.d0
      disb(4,19) = -1484425635.d0
      disb(4,20) = 580622350.d0
      disb(4,21) = -188428075.d0
      disb(4,22) = 49942200.d0
      disb(4,23) = -10590350.d0
      disb(4,24) = 1749380.d0
      disb(4,25) = -217650.d0
      disb(4,26) = 19600.d0
      disb(4,27) = -1225.d0
      disb(4,28) = 50.d0
      disb(4,29) = -1.d0
c
      disb(5,1) = 12650.d0
      disb(5,2) = -318550.d0
      disb(5,3) = 3852800.d0
      disb(5,4) = -29792525.d0
      disb(5,5) = 165403126.d0
      disb(5,6) = -701887025.d0
      disb(5,7) = 2366366800.d0
      disb(5,8) = -6504442550.d0
      disb(5,9) = 14842004650.d0
      disb(5,10) = -28479946880.d0
      disb(5,11) = 46385323600.d0
      disb(5,12) = -64544670575.d0
      disb(5,13) = 77068598950.d0
      disb(5,14) = -79165466975.d0
      disb(5,15) = 70021593760.d0
      disb(5,16) = -53296388900.d0
      disb(5,17) = 34834209550.d0
      disb(5,18) = -19478029550.d0
      disb(5,19) = 9267546400.d0
      disb(5,20) = -3724740635.d0
      disb(5,21) = 1252716850.d0
      disb(5,22) = -348450575.d0
      disb(5,23) = 79037200.d0
      disb(5,24) = -14385350.d0
      disb(5,25) = 2065630.d0
      disb(5,26) = -230300.d0
      disb(5,27) = 19600.d0
      disb(5,28) = -1225.d0
      disb(5,29) = 50.d0
      disb(5,30) = -1.d0
c
      disb(6,1) = -53130.d0
      disb(6,2) = 1340900.d0
      disb(6,3) = -16257550.d0
      disb(6,4) = 126051800.d0
      disb(6,5) = -701887025.d0
      disb(6,6) = 2988200026.d0
      disb(6,7) = -10111210025.d0
      disb(6,8) = 27905957800.d0
      disb(6,9) = -63968522300.d0
      disb(6,10) = 123385266400.d0
      disb(6,11) = -202149165680.d0
      disb(6,12) = 283206985600.d0
      disb(6,13) = -340836609575.d0
      disb(6,14) = 353360537950.d0
      disb(6,15) = -315987128975.d0
      disb(6,16) = 243690812560.d0
      disb(6,17) = -161839650650.d0
      disb(6,18) = 92298289300.d0
      disb(6,19) = -45017620550.d0
      disb(6,20) = 18676869400.d0
      disb(6,21) = -6547537535.d0
      disb(6,22) = 1924811350.d0
      disb(6,23) = -470649575.d0
      disb(6,24) = 94976200.d0
      disb(6,25) = -15713600.d0
      disb(6,26) = 2118760.d0
      disb(6,27) = -230300.d0
      disb(6,28) = 19600.d0
      disb(6,29) = -1225.d0
      disb(6,30) = 50.d0
      disb(6,31) = -1.d0
c
      disb(7,1) = 177100.d0
      disb(7,2) = -4480630.d0
      disb(7,3) = 54470900.d0
      disb(7,4) = -423587550.d0
      disb(7,5) = 2366366800.d0
      disb(7,6) = -10111210025.d0
      disb(7,7) = 34352610026.d0
      disb(7,8) = -95243180025.d0
      disb(7,9) = 219452890300.d0
      disb(7,10) = -425779394800.d0
      disb(7,11) = 702282662400.d0
      disb(7,12) = -991554705680.d0
      disb(7,13) = 1204180115600.d0
      disb(7,14) = -1261809739575.d0
      disb(7,15) = 1142766077950.d0
      disb(7,16) = -894884524975.d0
      disb(7,17) = 605501685060.d0
      disb(7,18) = -353386583150.d0
      disb(7,19) = 177430259300.d0
      disb(7,20) = -76382030550.d0
      disb(7,21) = 28086192400.d0
      disb(7,22) = -8787852535.d0
      disb(7,23) = 2332141350.d0
      disb(7,24) = -523779575.d0
      disb(7,25) = 99403700.d0
      disb(7,26) = -15890700.d0
      disb(7,27) = 2118760.d0
      disb(7,28) = -230300.d0
      disb(7,29) = 19600.d0
      disb(7,30) = -1225.d0
      disb(7,31) = 50.d0
      disb(7,32) = -1.d0
c
      disb(8,1) = -480700.d0
      disb(8,2) = 12194600.d0
      disb(8,3) = -148690630.d0
      disb(8,4) = 1160080900.d0
      disb(8,5) = -6504442550.d0
      disb(8,6) = 27905957800.d0
      disb(8,7) = -95243180025.d0
      disb(8,8) = 265425100026.d0
      disb(8,9) = -615156282525.d0
      disb(8,10) = 1201510972800.d0
      disb(8,11) = -1997072326800.d0
      disb(8,12) = 2844954842400.d0
      disb(8,13) = -3491338915680.d0
      disb(8,14) = 3703964325600.d0
      disb(8,15) = -3404481919575.d0
      disb(8,16) = 2714059009950.d0
      disb(8,17) = -1876942607475.d0
      disb(8,18) = 1125414787560.d0
      disb(8,19) = -584459073150.d0
      disb(8,20) = 262562229300.d0
      disb(8,21) = -101921621550.d0
      disb(8,22) = 34167047400.d0
      disb(8,23) = -9893462535.d0
      disb(8,24) = 2476351350.d0
      disb(8,25) = -535797075.d0
      disb(8,26) = 99884400.d0
      disb(8,27) = -15890700.d0
      disb(8,28) = 2118760.d0
      disb(8,29) = -230300.d0
      disb(8,30) = 19600.d0
      disb(8,31) = -1225.d0
      disb(8,32) = 50.d0
      disb(8,33) = -1.d0
c
      disb(9,1) = 1081575.d0
      disb(9,2) = -27520075.d0
      disb(9,3) = 336667100.d0
      disb(9,4) = -2636313130.d0
      disb(9,5) = 14842004650.d0
      disb(9,6) = -63968522300.d0
      disb(9,7) = 219452890300.d0
      disb(9,8) = -615156282525.d0
      disb(9,9) = 1435229580651.d0
      disb(9,10) = -2824786968150.d0
      disb(9,11) = 4736920069800.d0
      disb(9,12) = -6818084731800.d0
      disb(9,13) = 8469469314900.d0
      disb(9,14) = -9115853388180.d0
      disb(9,15) = 8524976730600.d0
      disb(9,16) = -6939891016575.d0
      disb(9,17) = 4923689695575.d0
      disb(9,18) = -3046747088100.d0
      disb(9,19) = 1645327890060.d0
      disb(9,20) = -776006005650.d0
      disb(9,21) = 320026309050.d0
      disb(9,22) = -115603545300.d0
      disb(9,23) = 36654669900.d0
      disb(9,24) = -10217935035.d0
      disb(9,25) = 2503390725.d0
      disb(9,26) = -536878650.d0
      disb(9,27) = 99884400.d0
      disb(9,28) = -15890700.d0
      disb(9,29) = 2118760.d0
      disb(9,30) = -230300.d0
      disb(9,31) = 19600.d0
      disb(9,32) = -1225.d0
      disb(9,33) = 50.d0
      disb(9,34) = -1.d0
c
      disb(10,1) = -2042975.d0
      disb(10,2) = 52155950.d0
      disb(10,3) = -640412575.d0
      disb(10,4) = 5035509600.d0
      disb(10,5) = -28479946880.d0
      disb(10,6) = 123385266400.d0
      disb(10,7) = -425779394800.d0
      disb(10,8) = 1201510972800.d0
      disb(10,9) = -2824786968150.d0
      disb(10,10) = 5608976431276.d0
      disb(10,11) = -9502781929150.d0
      disb(10,12) = 13843276834800.d0
      disb(10,13) = -17442167624300.d0
      disb(10,14) = 19093552207400.d0
      disb(10,15) = -18222210153180.d0
      disb(10,16) = 15202971691600.d0
      disb(10,17) = -11113637867200.d0
      disb(10,18) = 7133320381200.d0
      disb(10,19) = -4028805170600.d0
      disb(10,20) = 2007138762560.d0
      disb(10,21) = -884549267400.d0
      disb(10,22) = 345869942800.d0
      disb(10,23) = -120302387800.d0
      disb(10,24) = 37267562400.d0
      disb(10,25) = -10269009410.d0
      disb(10,26) = 2505433700.d0
      disb(10,27) = -536878650.d0
      disb(10,28) = 99884400.d0
      disb(10,29) = -15890700.d0
      disb(10,30) = 2118760.d0
      disb(10,31) = -230300.d0
      disb(10,32) = 19600.d0
      disb(10,33) = -1225.d0
      disb(10,34) = 50.d0
      disb(10,35) = -1.d0
c
      disb(11,1) = 3268760.d0
      disb(11,2) = -83761975.d0
      disb(11,3) = 1032783950.d0
      disb(11,4) = -8158560575.d0
      disb(11,5) = 46385323600.d0
      disb(11,6) = -202149165680.d0
      disb(11,7) = 702282662400.d0
      disb(11,8) = -1997072326800.d0
      disb(11,9) = 4736920069800.d0
      disb(11,10) = -9502781929150.d0
      disb(11,11) = 16293768368876.d0
      disb(11,12) = -24072952753150.d0
      disb(11,13) = 30841809462800.d0
      disb(11,14) = -34440700252300.d0
      disb(11,15) = 33663723031400.d0
      disb(11,16) = -28907002090780.d0
      disb(11,17) = 21880966652600.d0
      disb(11,18) = -14649046964200.d0
      disb(11,19) = 8704613313200.d0
      disb(11,20) = -4607702566600.d0
      disb(11,21) = 2180807981360.d0
      disb(11,22) = -925899081400.d0
      disb(11,23) = 353388090800.d0
      disb(11,24) = -121283015800.d0
      disb(11,25) = 37349281400.d0
      disb(11,26) = -10272278170.d0
      disb(11,27) = 2505433700.d0
      disb(11,28) = -536878650.d0
      disb(11,29) = 99884400.d0
      disb(11,30) = -15890700.d0
      disb(11,31) = 2118760.d0
      disb(11,32) = -230300.d0
      disb(11,33) = 19600.d0
      disb(11,34) = -1225.d0
      disb(11,35) = 50.d0
      disb(11,36) = -1.d0
c
      disb(12,1) = -4457400.d0
      disb(12,2) = 114703760.d0
      disb(12,3) = -1420981975.d0
      disb(12,4) = 11284803950.d0
      disb(12,5) = -64544670575.d0
      disb(12,6) = 283206985600.d0
      disb(12,7) = -991554705680.d0
      disb(12,8) = 2844954842400.d0
      disb(12,9) = -6818084731800.d0
      disb(12,10) = 13843276834800.d0
      disb(12,11) = -24072952753150.d0
      disb(12,12) = 36162183128876.d0
      disb(12,13) = -47252769973150.d0
      disb(12,14) = 54021626682800.d0
      disb(12,15) = -54309115012300.d0
      disb(12,16) = 48233893855400.d0
      disb(12,17) = -38013358855780.d0
      disb(12,18) = 26701979057600.d0
      disb(12,19) = -16791719144200.d0
      disb(12,20) = 9494018853200.d0
      disb(12,21) = -4844524228600.d0
      disb(12,22) = 2237194091360.d0
      disb(12,23) = -936151101400.d0
      disb(12,24) = 354725310800.d0
      disb(12,25) = -121394450800.d0
      disb(12,26) = 37353738800.d0
      disb(12,27) = -10272278170.d0
      disb(12,28) = 2505433700.d0
      disb(12,29) = -536878650.d0
      disb(12,30) = 99884400.d0
      disb(12,31) = -15890700.d0
      disb(12,32) = 2118760.d0
      disb(12,33) = -230300.d0
      disb(12,34) = 19600.d0
      disb(12,35) = -1225.d0
      disb(12,36) = 50.d0
      disb(12,37) = -1.d0
c
      disb(13,1) = 5200300.d0
      disb(13,2) = -134464900.d0
      disb(13,3) = 1674793760.d0
      disb(13,4) = -13381671975.d0
      disb(13,5) = 77068598950.d0
      disb(13,6) = -340836609575.d0
      disb(13,7) = 1204180115600.d0
      disb(13,8) = -3491338915680.d0
      disb(13,9) = 8469469314900.d0
      disb(13,10) = -17442167624300.d0
      disb(13,11) = 30841809462800.d0
      disb(13,12) = -47252769973150.d0
      disb(13,13) = 63205303218876.d0
      disb(13,14) = -74295890063150.d0
      disb(13,15) = 77201443902800.d0
      disb(13,16) = -71307647640300.d0
      disb(13,17) = 58857976747900.d0
      disb(13,18) = -43637873328280.d0
      disb(13,19) = 29201763267600.d0
      disb(13,20) = -17712692274200.d0
      disb(13,21) = 9770310792200.d0
      disb(13,22) = -4910308023600.d0
      disb(13,23) = 2249154781360.d0
      disb(13,24) = -937711191400.d0
      disb(13,25) = 354855318300.d0
      disb(13,26) = -121399651100.d0
      disb(13,27) = 37353738800.d0
      disb(13,28) = -10272278170.d0
      disb(13,29) = 2505433700.d0
      disb(13,30) = -536878650.d0
      disb(13,31) = 99884400.d0
      disb(13,32) = -15890700.d0
      disb(13,33) = 2118760.d0
      disb(13,34) = -230300.d0
      disb(13,35) = 19600.d0
      disb(13,36) = -1225.d0
      disb(13,37) = 50.d0
      disb(13,38) = -1.d0
c
      disb(14,1) = -5200300.d0
      disb(14,2) = 135207800.d0
      disb(14,3) = -1694554900.d0
      disb(14,4) = 13635483760.d0
      disb(14,5) = -79165466975.d0
      disb(14,6) = 353360537950.d0
      disb(14,7) = -1261809739575.d0
      disb(14,8) = 3703964325600.d0
      disb(14,9) = -9115853388180.d0
      disb(14,10) = 19093552207400.d0
      disb(14,11) = -34440700252300.d0
      disb(14,12) = 54021626682800.d0
      disb(14,13) = -74295890063150.d0
      disb(14,14) = 90248423308876.d0
      disb(14,15) = -97475707283150.d0
      disb(14,16) = 94199976530800.d0
      disb(14,17) = -81931730532800.d0
      disb(14,18) = 64482491220400.d0
      disb(14,19) = -46137657538280.d0
      disb(14,20) = 30122736397600.d0
      disb(14,21) = -17988984213200.d0
      disb(14,22) = 9836094587200.d0
      disb(14,23) = -4922268713600.d0
      disb(14,24) = 2250714871360.d0
      disb(14,25) = -937841198900.d0
      disb(14,26) = 354860518600.d0
      disb(14,27) = -121399651100.d0
      disb(14,28) = 37353738800.d0
      disb(14,29) = -10272278170.d0
      disb(14,30) = 2505433700.d0
      disb(14,31) = -536878650.d0
      disb(14,32) = 99884400.d0
      disb(14,33) = -15890700.d0
      disb(14,34) = 2118760.d0
      disb(14,35) = -230300.d0
      disb(14,36) = 19600.d0
      disb(14,37) = -1225.d0
      disb(14,38) = 50.d0
      disb(14,39) = -1.d0
c
      disb(15,1) = 4457400.d0
      disb(15,2) = -116635300.d0
      disb(15,3) = 1472427800.d0
      disb(15,4) = -11946574900.d0
      disb(15,5) = 70021593760.d0
      disb(15,6) = -315987128975.d0
      disb(15,7) = 1142766077950.d0
      disb(15,8) = -3404481919575.d0
      disb(15,9) = 8524976730600.d0
      disb(15,10) = -18222210153180.d0
      disb(15,11) = 33663723031400.d0
      disb(15,12) = -54309115012300.d0
      disb(15,13) = 77201443902800.d0
      disb(15,14) = -97475707283150.d0
      disb(15,15) = 110116838068876.d0
      disb(15,16) = -112045878107150.d0
      disb(15,17) = 103306333295800.d0
      disb(15,18) = -86752742937800.d0
      disb(15,19) = 66625163400400.d0
      disb(15,20) = -46927063078280.d0
      disb(15,21) = 30359558059600.d0
      disb(15,22) = -18045370323200.d0
      disb(15,23) = 9846346607200.d0
      disb(15,24) = -4923605933600.d0
      disb(15,25) = 2250826306360.d0
      disb(15,26) = -937845656300.d0
      disb(15,27) = 354860518600.d0
      disb(15,28) = -121399651100.d0
      disb(15,29) = 37353738800.d0
      disb(15,30) = -10272278170.d0
      disb(15,31) = 2505433700.d0
      disb(15,32) = -536878650.d0
      disb(15,33) = 99884400.d0
      disb(15,34) = -15890700.d0
      disb(15,35) = 2118760.d0
      disb(15,36) = -230300.d0
      disb(15,37) = 19600.d0
      disb(15,38) = -1225.d0
      disb(15,39) = 50.d0
      disb(15,40) = -1.d0
c
      disb(16,1) = -3268760.d0
      disb(16,2) = 86176400.d0
      disb(16,3) = -1097263300.d0
      disb(16,4) = 8990575800.d0
      disb(16,5) = -53296388900.d0
      disb(16,6) = 243690812560.d0
      disb(16,7) = -894884524975.d0
      disb(16,8) = 2714059009950.d0
      disb(16,9) = -6939891016575.d0
      disb(16,10) = 15202971691600.d0
      disb(16,11) = -28907002090780.d0
      disb(16,12) = 48233893855400.d0
      disb(16,13) = -71307647640300.d0
      disb(16,14) = 94199976530800.d0
      disb(16,15) = -112045878107150.d0
      disb(16,16) = 120801630006476.d0
      disb(16,17) = -118723873068150.d0
      disb(16,18) = 106841742392800.d0
      disb(16,19) = -88324035869800.d0
      disb(16,20) = 67204060796400.d0
      disb(16,21) = -47100732297080.d0
      disb(16,22) = 30400907873600.d0
      disb(16,23) = -18052888471200.d0
      disb(16,24) = 9847327235200.d0
      disb(16,25) = -4923687652600.d0
      disb(16,26) = 2250829575120.d0
      disb(16,27) = -937845656300.d0
      disb(16,28) = 354860518600.d0
      disb(16,29) = -121399651100.d0
      disb(16,30) = 37353738800.d0
      disb(16,31) = -10272278170.d0
      disb(16,32) = 2505433700.d0
      disb(16,33) = -536878650.d0
      disb(16,34) = 99884400.d0
      disb(16,35) = -15890700.d0
      disb(16,36) = 2118760.d0
      disb(16,37) = -230300.d0
      disb(16,38) = 19600.d0
      disb(16,39) = -1225.d0
      disb(16,40) = 50.d0
      disb(16,41) = -1.d0
c
      disb(17,1) = 2042975.d0
      disb(17,2) = -54343135.d0
      disb(17,3) = 699068900.d0
      disb(17,4) = -5796105800.d0
      disb(17,5) = 34834209550.d0
      disb(17,6) = -161839650650.d0
      disb(17,7) = 605501685060.d0
      disb(17,8) = -1876942607475.d0
      disb(17,9) = 4923689695575.d0
      disb(17,10) = -11113637867200.d0
      disb(17,11) = 21880966652600.d0
      disb(17,12) = -38013358855780.d0
      disb(17,13) = 58857976747900.d0
      disb(17,14) = -81931730532800.d0
      disb(17,15) = 103306333295800.d0
      disb(17,16) = -118723873068150.d0
      disb(17,17) = 124975376857101.d0
      disb(17,18) = -120933503753775.d0
      disb(17,19) = 107823800475300.d0
      disb(17,20) = -88685846742300.d0
      disb(17,21) = 67312604058150.d0
      disb(17,22) = -47126575930830.d0
      disb(17,23) = 30405606716100.d0
      disb(17,24) = -18053501363700.d0
      disb(17,25) = 9847378309575.d0
      disb(17,26) = -4923689695575.d0
      disb(17,27) = 2250829575120.d0
      disb(17,28) = -937845656300.d0
      disb(17,29) = 354860518600.d0
      disb(17,30) = -121399651100.d0
      disb(17,31) = 37353738800.d0
      disb(17,32) = -10272278170.d0
      disb(17,33) = 2505433700.d0
      disb(17,34) = -536878650.d0
      disb(17,35) = 99884400.d0
      disb(17,36) = -15890700.d0
      disb(17,37) = 2118760.d0
      disb(17,38) = -230300.d0
      disb(17,39) = 19600.d0
      disb(17,40) = -1225.d0
      disb(17,41) = 50.d0
      disb(17,42) = -1.d0
c
      disb(18,1) = -1081575.d0
      disb(18,2) = 29082350.d0
      disb(18,3) = -378815635.d0
      disb(18,4) = 3186691400.d0
      disb(18,5) = -19478029550.d0
      disb(18,6) = 92298289300.d0
      disb(18,7) = -353386583150.d0
      disb(18,8) = 1125414787560.d0
      disb(18,9) = -3046747088100.d0
      disb(18,10) = 7133320381200.d0
      disb(18,11) = -14649046964200.d0
      disb(18,12) = 26701979057600.d0
      disb(18,13) = -43637873328280.d0
      disb(18,14) = 64482491220400.d0
      disb(18,15) = -86752742937800.d0
      disb(18,16) = 106841742392800.d0
      disb(18,17) = -120933503753775.d0
      disb(18,18) = 126145181337726.d0
      disb(18,19) = -121453416856275.d0
      disb(18,20) = 108015347407800.d0
      disb(18,21) = -88743310822050.d0
      disb(18,22) = 67326285981900.d0
      disb(18,23) = -47129063553330.d0
      disb(18,24) = 30405931188600.d0
      disb(18,25) = -18053528403075.d0
      disb(18,26) = 9847379391150.d0
      disb(18,27) = -4923689695575.d0
      disb(18,28) = 2250829575120.d0
      disb(18,29) = -937845656300.d0
      disb(18,30) = 354860518600.d0
      disb(18,31) = -121399651100.d0
      disb(18,32) = 37353738800.d0
      disb(18,33) = -10272278170.d0
      disb(18,34) = 2505433700.d0
      disb(18,35) = -536878650.d0
      disb(18,36) = 99884400.d0
      disb(18,37) = -15890700.d0
      disb(18,38) = 2118760.d0
      disb(18,39) = -230300.d0
      disb(18,40) = 19600.d0
      disb(18,41) = -1225.d0
      disb(18,42) = 50.d0
      disb(18,43) = -1.d0
c
      disb(19,1) = 480700.d0
      disb(19,2) = -13099075.d0
      disb(19,3) = 173292350.d0
      disb(19,4) = -1484425635.d0
      disb(19,5) = 9267546400.d0
      disb(19,6) = -45017620550.d0
      disb(19,7) = 177430259300.d0
      disb(19,8) = -584459073150.d0
      disb(19,9) = 1645327890060.d0
      disb(19,10) = -4028805170600.d0
      disb(19,11) = 8704613313200.d0
      disb(19,12) = -16791719144200.d0
      disb(19,13) = 29201763267600.d0
      disb(19,14) = -46137657538280.d0
      disb(19,15) = 66625163400400.d0
      disb(19,16) = -88324035869800.d0
      disb(19,17) = 107823800475300.d0
      disb(19,18) = -121453416856275.d0
      disb(19,19) = 126376253827726.d0
      disb(19,20) = -121538548826275.d0
      disb(19,21) = 108040886998800.d0
      disb(19,22) = -88749391677050.d0
      disb(19,23) = 67327391591900.d0
      disb(19,24) = -47129207763330.d0
      disb(19,25) = 30405943206100.d0
      disb(19,26) = -18053528883775.d0
      disb(19,27) = 9847379391150.d0
      disb(19,28) = -4923689695575.d0
      disb(19,29) = 2250829575120.d0
      disb(19,30) = -937845656300.d0
      disb(19,31) = 354860518600.d0
      disb(19,32) = -121399651100.d0
      disb(19,33) = 37353738800.d0
      disb(19,34) = -10272278170.d0
      disb(19,35) = 2505433700.d0
      disb(19,36) = -536878650.d0
      disb(19,37) = 99884400.d0
      disb(19,38) = -15890700.d0
      disb(19,39) = 2118760.d0
      disb(19,40) = -230300.d0
      disb(19,41) = 19600.d0
      disb(19,42) = -1225.d0
      disb(19,43) = 50.d0
      disb(19,44) = -1.d0
c
      disb(20,1) = -177100.d0
      disb(20,2) = 4908200.d0
      disb(20,3) = -66229075.d0
      disb(20,4) = 580622350.d0
      disb(20,5) = -3724740635.d0
      disb(20,6) = 18676869400.d0
      disb(20,7) = -76382030550.d0
      disb(20,8) = 262562229300.d0
      disb(20,9) = -776006005650.d0
      disb(20,10) = 2007138762560.d0
      disb(20,11) = -4607702566600.d0
      disb(20,12) = 9494018853200.d0
      disb(20,13) = -17712692274200.d0
      disb(20,14) = 30122736397600.d0
      disb(20,15) = -46927063078280.d0
      disb(20,16) = 67204060796400.d0
      disb(20,17) = -88685846742300.d0
      disb(20,18) = 108015347407800.d0
      disb(20,19) = -121538548826275.d0
      disb(20,20) = 126407618237726.d0
      disb(20,21) = -121547958149275.d0
      disb(20,22) = 108043127313800.d0
      disb(20,23) = -88749799007050.d0
      disb(20,24) = 67327444721900.d0
      disb(20,25) = -47129212190830.d0
      disb(20,26) = 30405943383200.d0
      disb(20,27) = -18053528883775.d0
      disb(20,28) = 9847379391150.d0
      disb(20,29) = -4923689695575.d0
      disb(20,30) = 2250829575120.d0
      disb(20,31) = -937845656300.d0
      disb(20,32) = 354860518600.d0
      disb(20,33) = -121399651100.d0
      disb(20,34) = 37353738800.d0
      disb(20,35) = -10272278170.d0
      disb(20,36) = 2505433700.d0
      disb(20,37) = -536878650.d0
      disb(20,38) = 99884400.d0
      disb(20,39) = -15890700.d0
      disb(20,40) = 2118760.d0
      disb(20,41) = -230300.d0
      disb(20,42) = 19600.d0
      disb(20,43) = -1225.d0
      disb(20,44) = 50.d0
      disb(20,45) = -1.d0
c
      disb(21,1) = 53130.d0
      disb(21,2) = -1505350.d0
      disb(21,3) = 20847200.d0
      disb(21,4) = -188428075.d0
      disb(21,5) = 1252716850.d0
      disb(21,6) = -6547537535.d0
      disb(21,7) = 28086192400.d0
      disb(21,8) = -101921621550.d0
      disb(21,9) = 320026309050.d0
      disb(21,10) = -884549267400.d0
      disb(21,11) = 2180807981360.d0
      disb(21,12) = -4844524228600.d0
      disb(21,13) = 9770310792200.d0
      disb(21,14) = -17988984213200.d0
      disb(21,15) = 30359558059600.d0
      disb(21,16) = -47100732297080.d0
      disb(21,17) = 67312604058150.d0
      disb(21,18) = -88743310822050.d0
      disb(21,19) = 108040886998800.d0
      disb(21,20) = -121547958149275.d0
      disb(21,21) = 126410441034626.d0
      disb(21,22) = -121548630243775.d0
      disb(21,23) = 108043249512800.d0
      disb(21,24) = -88749814946050.d0
      disb(21,25) = 67327446050150.d0
      disb(21,26) = -47129212243960.d0
      disb(21,27) = 30405943383200.d0
      disb(21,28) = -18053528883775.d0
      disb(21,29) = 9847379391150.d0
      disb(21,30) = -4923689695575.d0
      disb(21,31) = 2250829575120.d0
      disb(21,32) = -937845656300.d0
      disb(21,33) = 354860518600.d0
      disb(21,34) = -121399651100.d0
      disb(21,35) = 37353738800.d0
      disb(21,36) = -10272278170.d0
      disb(21,37) = 2505433700.d0
      disb(21,38) = -536878650.d0
      disb(21,39) = 99884400.d0
      disb(21,40) = -15890700.d0
      disb(21,41) = 2118760.d0
      disb(21,42) = -230300.d0
      disb(21,43) = 19600.d0
      disb(21,44) = -1225.d0
      disb(21,45) = 50.d0
      disb(21,46) = -1.d0
c
      disb(22,1) = -12650.d0
      disb(22,2) = 369380.d0
      disb(22,3) = -5300350.d0
      disb(22,4) = 49942200.d0
      disb(22,5) = -348450575.d0
      disb(22,6) = 1924811350.d0
      disb(22,7) = -8787852535.d0
      disb(22,8) = 34167047400.d0
      disb(22,9) = -115603545300.d0
      disb(22,10) = 345869942800.d0
      disb(22,11) = -925899081400.d0
      disb(22,12) = 2237194091360.d0
      disb(22,13) = -4910308023600.d0
      disb(22,14) = 9836094587200.d0
      disb(22,15) = -18045370323200.d0
      disb(22,16) = 30400907873600.d0
      disb(22,17) = -47126575930830.d0
      disb(22,18) = 67326285981900.d0
      disb(22,19) = -88749391677050.d0
      disb(22,20) = 108043127313800.d0
      disb(22,21) = -121548630243775.d0
      disb(22,22) = 126410601057126.d0
      disb(22,23) = -121548659338775.d0
      disb(22,24) = 108043253307800.d0
      disb(22,25) = -88749815262300.d0
      disb(22,26) = 67327446062800.d0
      disb(22,27) = -47129212243960.d0
      disb(22,28) = 30405943383200.d0
      disb(22,29) = -18053528883775.d0
      disb(22,30) = 9847379391150.d0
      disb(22,31) = -4923689695575.d0
      disb(22,32) = 2250829575120.d0
      disb(22,33) = -937845656300.d0
      disb(22,34) = 354860518600.d0
      disb(22,35) = -121399651100.d0
      disb(22,36) = 37353738800.d0
      disb(22,37) = -10272278170.d0
      disb(22,38) = 2505433700.d0
      disb(22,39) = -536878650.d0
      disb(22,40) = 99884400.d0
      disb(22,41) = -15890700.d0
      disb(22,42) = 2118760.d0
      disb(22,43) = -230300.d0
      disb(22,44) = 19600.d0
      disb(22,45) = -1225.d0
      disb(22,46) = 50.d0
      disb(22,47) = -1.d0
c
      disb(23,1) = 2300.d0
      disb(23,2) = -70150.d0
      disb(23,3) = 1059380.d0
      disb(23,4) = -10590350.d0
      disb(23,5) = 79037200.d0
      disb(23,6) = -470649575.d0
      disb(23,7) = 2332141350.d0
      disb(23,8) = -9893462535.d0
      disb(23,9) = 36654669900.d0
      disb(23,10) = -120302387800.d0
      disb(23,11) = 353388090800.d0
      disb(23,12) = -936151101400.d0
      disb(23,13) = 2249154781360.d0
      disb(23,14) = -4922268713600.d0
      disb(23,15) = 9846346607200.d0
      disb(23,16) = -18052888471200.d0
      disb(23,17) = 30405606716100.d0
      disb(23,18) = -47129063553330.d0
      disb(23,19) = 67327391591900.d0
      disb(23,20) = -88749799007050.d0
      disb(23,21) = 108043249512800.d0
      disb(23,22) = -121548659338775.d0
      disb(23,23) = 126410606347126.d0
      disb(23,24) = -121548660028775.d0
      disb(23,25) = 108043253365300.d0
      disb(23,26) = -88749815264600.d0
      disb(23,27) = 67327446062800.d0
      disb(23,28) = -47129212243960.d0
      disb(23,29) = 30405943383200.d0
      disb(23,30) = -18053528883775.d0
      disb(23,31) = 9847379391150.d0
      disb(23,32) = -4923689695575.d0
      disb(23,33) = 2250829575120.d0
      disb(23,34) = -937845656300.d0
      disb(23,35) = 354860518600.d0
      disb(23,36) = -121399651100.d0
      disb(23,37) = 37353738800.d0
      disb(23,38) = -10272278170.d0
      disb(23,39) = 2505433700.d0
      disb(23,40) = -536878650.d0
      disb(23,41) = 99884400.d0
      disb(23,42) = -15890700.d0
      disb(23,43) = 2118760.d0
      disb(23,44) = -230300.d0
      disb(23,45) = 19600.d0
      disb(23,46) = -1225.d0
      disb(23,47) = 50.d0
      disb(23,48) = -1.d0
c
      disb(24,1) = -300.d0
      disb(24,2) = 9800.d0
      disb(24,3) = -160150.d0
      disb(24,4) = 1749380.d0
      disb(24,5) = -14385350.d0
      disb(24,6) = 94976200.d0
      disb(24,7) = -523779575.d0
      disb(24,8) = 2476351350.d0
      disb(24,9) = -10217935035.d0
      disb(24,10) = 37267562400.d0
      disb(24,11) = -121283015800.d0
      disb(24,12) = 354725310800.d0
      disb(24,13) = -937711191400.d0
      disb(24,14) = 2250714871360.d0
      disb(24,15) = -4923605933600.d0
      disb(24,16) = 9847327235200.d0
      disb(24,17) = -18053501363700.d0
      disb(24,18) = 30405931188600.d0
      disb(24,19) = -47129207763330.d0
      disb(24,20) = 67327444721900.d0
      disb(24,21) = -88749814946050.d0
      disb(24,22) = 108043253307800.d0
      disb(24,23) = -121548660028775.d0
      disb(24,24) = 126410606437126.d0
      disb(24,25) = -121548660036275.d0
      disb(24,26) = 108043253365600.d0
      disb(24,27) = -88749815264600.d0
      disb(24,28) = 67327446062800.d0
      disb(24,29) = -47129212243960.d0
      disb(24,30) = 30405943383200.d0
      disb(24,31) = -18053528883775.d0
      disb(24,32) = 9847379391150.d0
      disb(24,33) = -4923689695575.d0
      disb(24,34) = 2250829575120.d0
      disb(24,35) = -937845656300.d0
      disb(24,36) = 354860518600.d0
      disb(24,37) = -121399651100.d0
      disb(24,38) = 37353738800.d0
      disb(24,39) = -10272278170.d0
      disb(24,40) = 2505433700.d0
      disb(24,41) = -536878650.d0
      disb(24,42) = 99884400.d0
      disb(24,43) = -15890700.d0
      disb(24,44) = 2118760.d0
      disb(24,45) = -230300.d0
      disb(24,46) = 19600.d0
      disb(24,47) = -1225.d0
      disb(24,48) = 50.d0
      disb(24,49) = -1.d0
c
      disb(25,1) = 25.d0
      disb(25,2) = -925.d0
      disb(25,3) = 17300.d0
      disb(25,4) = -217650.d0
      disb(25,5) = 2065630.d0
      disb(25,6) = -15713600.d0
      disb(25,7) = 99403700.d0
      disb(25,8) = -535797075.d0
      disb(25,9) = 2503390725.d0
      disb(25,10) = -10269009410.d0
      disb(25,11) = 37349281400.d0
      disb(25,12) = -121394450800.d0
      disb(25,13) = 354855318300.d0
      disb(25,14) = -937841198900.d0
      disb(25,15) = 2250826306360.d0
      disb(25,16) = -4923687652600.d0
      disb(25,17) = 9847378309575.d0
      disb(25,18) = -18053528403075.d0
      disb(25,19) = 30405943206100.d0
      disb(25,20) = -47129212190830.d0
      disb(25,21) = 67327446050150.d0
      disb(25,22) = -88749815262300.d0
      disb(25,23) = 108043253365300.d0
      disb(25,24) = -121548660036275.d0
      disb(25,25) = 126410606437751.d0
      disb(25,26) = -121548660036300.d0
      disb(25,27) = 108043253365600.d0
      disb(25,28) = -88749815264600.d0
      disb(25,29) = 67327446062800.d0
      disb(25,30) = -47129212243960.d0
      disb(25,31) = 30405943383200.d0
      disb(25,32) = -18053528883775.d0
      disb(25,33) = 9847379391150.d0
      disb(25,34) = -4923689695575.d0
      disb(25,35) = 2250829575120.d0
      disb(25,36) = -937845656300.d0
      disb(25,37) = 354860518600.d0
      disb(25,38) = -121399651100.d0
      disb(25,39) = 37353738800.d0
      disb(25,40) = -10272278170.d0
      disb(25,41) = 2505433700.d0
      disb(25,42) = -536878650.d0
      disb(25,43) = 99884400.d0
      disb(25,44) = -15890700.d0
      disb(25,45) = 2118760.d0
      disb(25,46) = -230300.d0
      disb(25,47) = 19600.d0
      disb(25,48) = -1225.d0
      disb(25,49) = 50.d0
      disb(25,50) = -1.d0
c
      end if
!
!  Rescale ( Divide dissipation matrix by powers of 2 )
!
      do i = 1, iordd2
        do j = 1, iord
          disb(i,j)= alpha * disb(i,j)
        end do
      end do
!
      do i = 1, iordd2p1
        disi(i) = alpha * disi(i)
      end do
!
      return
      end
