c     Copyright (c) 2006, Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000
c     with Sandia Corporation, the U.S. Governement retains certain rights in this software.
c
c     All rights reserved.
c     
c     Redistribution and use in source and binary forms, with or without 
c     modification, are permitted provided that the following conditions are met:
c     
c     * Redistributions of source code must retain the above copyright notice, 
c     this list of conditions and the following disclaimer.
c     * Redistributions in binary form must reproduce the above copyright notice, 
c     this list of conditions and the following disclaimer in the documentation 
c     and/or other materials provided with the distribution.
c     * Neither the name of the Sandia National Laboratories nor the names of 
c     its contributors may be used to endorse or promote products derived 
c     from this software without specific prior written permission.
c     
c     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
c     ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
c     WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
c     IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
c     INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
c     BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
c     DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
c     LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
c     OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
c     OF THE POSSIBILITY OF SUCH DAMAGE.
c     

c
c     Version 0.1, June 2006.
c     Authors : Christopher Kennedy, M. H. Carpenter and Jaideep Ray.
c     Maintainer: Jaideep Ray, Advanced Software R. & D., Sandia National Laboratories, Livermore, CA, USA.
c                 jairay@ca.sandia.gov
c
c

c     This is an example code that shows how to use the interpolations' library.
c     It initializes a field on a vertex-centered mesh (NX cells x NY cells, i.e.
c     NX+1 vertices x NY+1 vertices) analytically and interpolates it onto a mesh 
c     twice as fine (2NX cells x 2NY cells i.e 2NX+1 x 2NY+1 vertices). Use 6th order
c     interpolations

c     Force definition of all variables.
      implicit none

c     Define a field size, a halo of 2
      integer NCELLS, CNX, CNY, CNZ, FNX, FNY, FNZ, NHALO
      parameter(NCELLS = 50, CNZ = 1, FNZ = 1, NHALO = 3)
      parameter(CNX = NCELLS, CNY = NCELLS)
      parameter(FNX = 2*NCELLS, FNY = 2*NCELLS)

c     Now define 2 arrays to hold the coarse and fine fields.All arrays have to
c     be 3D so that we can deal with 3D fields. 
      real*8 cfield(0-NHALO:CNX+NHALO, 0-NHALO:CNY+NHALO, 0:CNZ-1),
     &     ffield(0-NHALO:FNX+NHALO, 0-NHALO:FNY+NHALO, 0:FNZ-1)

c     Define a unit square and its dx, dy and dz
      real*8 unit, dx, dy, dz, error
      integer i, j, k

      unit = 1.D0
      dx = unit / CNX
      dy = unit / CNY
      dz = unit / CNZ 

c     Initialize the field
      call initialize_field(cfield, CNX, CNY, CNZ, NHALO, dx, dy, dz)

c     Interpolate onto a mesh twice as fine
      call interp_vc_field(cfield, CNX, CNY, CNZ, NHALO, ffield, FNX, 
     &     FNY, FNZ)

c     calculate error made
      call calc_rms_error(ffield, FNX, FNY, FNZ, NHALO, 0.5*dx, 0.5*dy, 
     &     0.5*dz, error)
      write(*,*)" Resolution : ", FNX, " interp error = ", error

      stop
      end

c     Subroutine to apply initial condition which is sin(2*pi*x) * sin(2*pi*y)

      subroutine initialize_field( field, mx, my, mz, mhalo, dx, dy, dz)
      
      implicit none
      integer mx, my, mz, mhalo
      integer i, j, k
      real*8 dx, dy, dz, x, y, z
      real*8 field(0-mhalo:mx+mhalo, 0-mhalo:my+mhalo, 0:mz-1)

      real*8 pi

      pi = 4 * atan(1.0)
      do k = 0, mz-1
         do j = 0-mhalo, my-1+mhalo
            do i = 0-mhalo, mx-1+mhalo
               x = i*dx
               y = j*dy
               z = k*dz
               field(i, j, k) = sin(2*pi*x) * sin(2*pi*y)
            enddo
         enddo
      enddo

      return
      end

c     Calculate first derivative
      subroutine interp_vc_field(cfield, cmx, cmy, cmz, mhalo, 
     &     ffield, fmx, fmy, fmz)
      
      implicit none
      integer cmx, cmy, cmz, fmx, fmy, fmz, mhalo
      real*8 cfield(0-mhalo:cmx+mhalo, 0-mhalo:cmy+mhalo, 0:cmz-1)
      real*8 ffield(0-mhalo:fmx+mhalo, 0-mhalo:fmy+mhalo, 0:fmz-1)

c     I need temp arrays to store x-interped, y-interped and xy-interped
c     data
      real*8 tempx(0-mhalo:cmx+mhalo, 0-mhalo:cmy+mhalo, 0:cmz-1)
      real*8 tempy(0-mhalo:cmx+mhalo, 0-mhalo:cmy+mhalo, 0:cmz-1)
      real*8 tempxy(0-mhalo:cmx+mhalo, 0-mhalo:cmy+mhalo, 0:cmz-1)

      integer orderi, orderb
      integer biL, biR, bjL, bjR, bkL, bkR
      integer fiL, fiR, fjL, fjR, fkL, fkR
      integer iiL, iiR, ijL, ijR, ikL, ikR
      integer dfiL, dfiR, dfjL, dfjR, dfkL, dfkR
      integer iperx, ipery, iperz
      integer ierror   ! Error return value
      character*256 error

      integer i, j, k
      logical corner

c     Make sure that the fine mesh is twice as fine as coarse
      if ( (fmx) .ne. (2*(cmx)) ) then
         write(*,*) "Fine mesh does not have exactly twice as"
         write(*,*) "points as the coarse mesh along x axis. ABORT"
         call abort()
      endif

      if ( (fmy) .ne. (2*(cmy)) ) then
         write(*,*) "Fine mesh does not have exactly twice as"
         write(*,*) "points as the coarse mesh along y axis. ABORT"
         call abort()
      endif

      
c     Order of interpolation in the interior of domain and near boundaries
      orderi = 6
      orderb = 6   ! this will be ignored; the subroutine will close as necesary

c     Is the domain periodic in x, y, or z? No !
      iperx = 0
      ipery = 0 
      iperz = 0 

c     How many halo cells do I have in the lower and upper ends of each axis?
      biL = 0
      biR = 0
      bjL = 0
      bjR = 0
      bkL = 0
      bkR = 0

c     what are the array limits of field?
      fiL = 0  - mhalo
      fiR = cmx  + mhalo
      fjL = 0 - mhalo
      fjR = cmy  + mhalo
      fkL = 0 
      fkR = cmz - 1

c     What are the indices of the valid points in the field
      iiL = fiL + biL
      iiR = fiR - biR
      ijL = fjL + bjL
      ijR = fjR - bjR
      ikL = fkL + bkL
      ikR = fkR - bkR
      
c     what are the array limits of deriv, the output array?
      dfiL = 0  - mhalo
      dfiR = cmx + mhalo
      dfjL = 0 - mhalo
      dfjR = cmy + mhalo
      dfkL = 0 
      dfkR = cmz - 1

c     Zero out temps before proceeding
      do k = dfkL, dfkR
         do j = dfjL, dfjR
            do i = dfiL, dfiR
               tempx(i, j, k) = 0.0
               tempy(i, j, k) = 0.0
               tempxy(i, j, k) = 0.0
            enddo
         enddo
      enddo

c     Do the x_intp - fill out u_{i+1/2,j} points
      call x_intp_cf_vc(cfield, tempx, orderi, orderb,
     &     biL, biR, bjL, bjR, bkL, bkR,
     &     fiL, fiR, fjL, fjR, fkL, fkR,
     &     iiL, iiR, ijL, ijR, ikL, ikR,
     &     dfiL,dfiR,dfjL,dfjR,dfkL,dfkR,
     &     iperx,ipery,iperz,ierror,error)

c     Do the y_intp - fill out u_{i,j+1/2} points
      call y_intp_cf_vc(cfield, tempy, orderi, orderb,
     &     biL, biR, bjL, bjR, bkL, bkR,
     &     fiL, fiR, fjL, fjR, fkL, fkR,
     &     iiL, iiR, ijL, ijR, ikL, ikR,
     &     dfiL,dfiR,dfjL,dfjR,dfkL,dfkR,
     &     iperx,ipery,iperz,ierror,error)

c     Do the xy_intp - fill out u_{i+1/2,j+1/2} points
      corner = .true.
      call xy_intp_cf_vc(cfield, tempxy, orderi, orderb,
     &     biL, biR, bjL, bjR, bkL, bkR,
     &     fiL, fiR, fjL, fjR, fkL, fkR,
     &     iiL, iiR, ijL, ijR, ikL, ikR,
     &     dfiL,dfiR,dfjL,dfjR,dfkL,dfkR,
     &     iperx,ipery,iperz,ierror,error, corner)
      
c     Now fill up the fine field. First, init it to zero
      do k = 0, fmz-1
         do j = -mhalo, fmy+mhalo
            do i = -mhalo, fmx+mhalo
               ffield(i, j, k) = 0.0
            enddo
         enddo
      enddo

c     Now fill it up with a combination of cfield, tempx, tempy, 
c     tempxy
      do k = 0, cmz-1
         do j = 0, cmy
            do i = 0, cmx
               ffield(2*i, 2*j, k) = cfield(i, j, k)
               ffield(2*i+1, 2*j, k) = tempx(i, j, k)
               ffield(2*i, 2*j+1, k) = tempy(i, j, k)
               ffield(2*i+1, 2*j+1,k) = tempxy(i, j, k)
            enddo
         enddo
      enddo

      return
      end

c     subroutine to calculate RMS errors

      subroutine calc_rms_error(ffield, mx, my, mz, mhalo, dx, dy, dz, 
     &     error)
 
      implicit none
      integer mx, my, mz, mhalo
      integer i, j, k, idir
      real*8 dx, dy, dz, x, y, z
      real*8 ffield(0-mhalo:mx+mhalo, 0-mhalo:my+mhalo, 0:mz-1)
      real*8 errSq, anal_ans

      real*8 pi, error
      integer count

      pi = 4 * atan(1.0)
      error = 0.0

c     loop over all interior points where you evaluated derivs and 
c     determine errors
      count = 0 
      do k = 0, mz-1
         do j = 0, my
            do i = 0, mx
               x = i*dx
               y = j*dy
               z = k*dz
               anal_ans = sin(2*pi*x)*sin(2*pi*y)
               errSq = (ffield(i, j, k) - anal_ans)**2 
c               write(*,20) i, j, x, y, anal_ans, ffield(i, j, k)
c 20            format( 2(I3, 1x), 2(F6.3, 1x), 2(E12.5, 1x) )
               error = error + errSq
               count = count + 1
            enddo
         enddo
      enddo
      
      error = sqrt( error / count )
      return
      end
