c     Copyright (c) 2006, Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000
c     with Sandia Corporation, the U.S. Governement retains certain rights in this software.
c
c     All rights reserved.
c     
c     Redistribution and use in source and binary forms, with or without 
c     modification, are permitted provided that the following conditions are met:
c     
c     * Redistributions of source code must retain the above copyright notice, 
c     this list of conditions and the following disclaimer.
c     * Redistributions in binary form must reproduce the above copyright notice, 
c     this list of conditions and the following disclaimer in the documentation 
c     and/or other materials provided with the distribution.
c     * Neither the name of the Sandia National Laboratories nor the names of 
c     its contributors may be used to endorse or promote products derived 
c     from this software without specific prior written permission.
c     
c     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
c     ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
c     WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
c     IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
c     INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
c     BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
c     DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
c     LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
c     OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
c     OF THE POSSIBILITY OF SUCH DAMAGE.
c     

c
c     Version 0.1, June 2006.
c     Authors : Christopher Kennedy, M. H. Carpenter and Jaideep Ray.
c     Maintainer: Jaideep Ray, Advanced Software R. & D., Sandia National Laboratories, Livermore, CA, USA.
c                 jairay@ca.sandia.gov
c
c

c     This is an example code that shows how to use the filters' library.
c     It initializes a field analytically and filter it using an 8th order 
c     filter. Calculate difference from original field

c     Force definition of all variables.
      implicit none

c     Define a field size, a halo of 2 
      integer N, NX, NY, NZ, NHALO
      parameter(N = 50, NZ = 1, NHALO = 2)
      parameter(NX = N, NY = N)

c     Now define the array to hold the field. All arrays have to be 3D so that we 
c     can deal with 3D fields. 
      real*8 field(0-NHALO:NX-1+NHALO, 0-NHALO:NY-1+NHALO, 0:NZ-1)

c     Define a unit square and its dx, dy and dz
      real*8 unit, dx, dy, dz, error
      integer i, j, k

      unit = 1.D0
      dx = unit / NX
      dy = unit / NY
      dz = unit / NZ 

c     Initialize the field
      call initialize_field(field, NX, NY, NZ, NHALO, dx, dy, dz)

c     Filter it
      call filter_field(field, NX, NY, NZ, NHALO, dx, dy, dz)

c     Figure out the error caused by filtering
      call calc_rms_error(field, NX, NY, NZ, NHALO, dx, dy, dz, error)
      write(*,*)" Resolution : ", N, " RMS error after filtering = ", 
     &     error

      stop
      end

c     Subroutine to apply initial condition which is sin(2*pi*x) * sin(2*pi*y)

      subroutine initialize_field( field, mx, my, mz, mhalo, dx, dy, dz)
      
      implicit none
      integer mx, my, mz, mhalo
      integer i, j, k
      real*8 dx, dy, dz, x, y, z
      real*8 field(0-mhalo:mx-1+mhalo, 0-mhalo:my-1+mhalo, 0:mz-1)

      real*8 pi

      pi = 4 * atan(1.0)
      do k = 0, mz-1
         do j = 0-mhalo, my-1+mhalo
            do i = 0-mhalo, mx-1+mhalo
               x = i*dx
               y = j*dy
               z = k*dz
               field(i, j, k) = sin(2*pi*x) * sin(2*pi*y)
            enddo
         enddo
      enddo

      return
      end

c     Filter field with 8th-order filter
      subroutine filter_field(field, mx, my, mz, mhalo, dx, dy, dz)
      
      implicit none
      integer mx, my, mz, mhalo, idir
      real*8 dx, dy, dz

      integer ifilt_x, ifilt_y, ifilt_z
      integer filter_order
      integer fiL, fiR, fjL, fjR, fkL, fkR
      integer wiL, wiR, wjL, wjR, wkL, wkR
      integer piL, piR, pjL, pjR, pkL, pkR
      integer ierror, ierr_coeff   ! Error return value
      character*256 error, error_coeff

      real*8 field(0-mhalo:mx-1+mhalo, 0-mhalo:my-1+mhalo, 0:mz-1)
      
c     Order of filter in the domain interior
      filter_order = 8 

c     Is filtering to be done in x (yes), y(yes) and z (no) ?
      ifilt_x = 1
      ifilt_y = 1
      ifilt_z = 0 

c     what are the array limits of field?
      fiL = 0  - mhalo
      fiR = mx - 1 + mhalo
      fjL = 0 - mhalo
      fjR = my - 1 + mhalo
      fkL = 0 
      fkR = mz - 1

c     What are the indices of the window where I intend to filter
      wiL = fiL + mhalo
      wiR = fiR - mhalo
      wjL = fjL + mhalo
      wjR = fjR - mhalo
      wkL = fkL 
      wkR = fkR 
      
c     what are the array limits of field where filterable data exists?
c     In our case, the whole array.
      piL = fiL 
      piR = fiR 
      pjL = fjL
      pjR = fjR
      pkL = fkL 
      pkR = fkR

      call expfil(field, ifilt_x, ifilt_y, ifilt_z, filter_order,
     &        fiL, fiR, fjL, fjR, fkL, fkR,
     &        piL, piR, pjL, pjR, pkL, pkR,
     &        wiL, wiR, wjL, wjR, wkL, wkR,
     &        ierror, error, ierr_coeff, error_coeff)

      return
      end

c     subroutine to calculate RMS errors

      subroutine calc_rms_error(field, mx, my, mz, mhalo, dx, dy, dz, 
     &     error)
 
      implicit none
      integer mx, my, mz, mhalo
      integer i, j, k
      real*8 dx, dy, dz, x, y, z
      real*8 field(0-mhalo:mx-1+mhalo, 0-mhalo:my-1+mhalo, 0:mz-1)
      real*8 errSq, anal_ans

      real*8 pi, error
      integer count

      pi = 4 * atan(1.0)
      error = 0.0

c     loop over all interior points where you evaluated derivs and 
c     determine errors
      count = 0 
      do k = 0, mz-1
         do j = 0, my-1
            do i = 0, mx-1
               x = i*dx
               y = j*dy
               z = k*dz
               anal_ans = sin(2*pi*x)*sin(2*pi*y)
               errSq = (field(i, j, k) - anal_ans)**2 
               error = error + errSq
               count = count + 1
            enddo
         enddo
      enddo

      error = sqrt( error / count ) 
      return
      end
