c     Copyright (c) 2006, Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000
c     with Sandia Corporation, the U.S. Governement retains certain rights in this software.
c
c     All rights reserved.
c     
c     Redistribution and use in source and binary forms, with or without 
c     modification, are permitted provided that the following conditions are met:
c     
c     * Redistributions of source code must retain the above copyright notice, 
c     this list of conditions and the following disclaimer.
c     * Redistributions in binary form must reproduce the above copyright notice, 
c     this list of conditions and the following disclaimer in the documentation 
c     and/or other materials provided with the distribution.
c     * Neither the name of the Sandia National Laboratories nor the names of 
c     its contributors may be used to endorse or promote products derived 
c     from this software without specific prior written permission.
c     
c     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
c     ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
c     WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
c     IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
c     INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
c     BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
c     DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
c     LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
c     OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
c     OF THE POSSIBILITY OF SUCH DAMAGE.
c     

c
c     Version 0.1, June 2006.
c     Authors : Christopher Kennedy, M. H. Carpenter and Jaideep Ray.
c     Maintainer: Jaideep Ray, Advanced Software R. & D., Sandia National Laboratories, Livermore, CA, USA.
c                 jairay@ca.sandia.gov
c
c

c     This is an example code that shows how to use the derivatives' library.
c     It initializes a field analytically and takes its first derivative using
c     a fourth order stencil. No upwinding is done.

c     Force definition of all variables.
      implicit none

c     Define a field size, a halo of 2
      integer N, NX, NY, NZ, NHALO
      parameter(N = 50, NZ = 1, NHALO = 2)
      parameter(NX = N, NY = N)

c     Now define the array to hold the field and its derivative. All arrays have to
c     be 3D so that we can deal with 3D fields. Also, allocate a velocity array that
c     will be set to zero - we will not do upwinding. Velocity is needed only if 
c     we upwind.
      real*8 field(0-NHALO:NX-1+NHALO, 0-NHALO:NY-1+NHALO, 0:NZ-1),
     &       deriv(0-NHALO:NX-1+NHALO, 0-NHALO:NY-1+NHALO, 0:NZ-1),
     &       vel  (0-NHALO:NX-1+NHALO, 0-NHALO:NY-1+NHALO, 0:NZ-1)

c     Define a unit square and its dx, dy and dz
      real*8 unit, dx, dy, dz, error
      integer idir, i, j, k

      unit = 1.D0
      dx = unit / NX
      dy = unit / NY
      dz = unit / NZ 

c     Initialize the field
      call initialize_field(field, NX, NY, NZ, NHALO, dx, dy, dz)

c     Initialize velocity to zero
      do k = 0, NZ-1
         do j = 0-NHALO, NY-1+NHALO
            do i = 0-NHALO, NX-1+NHALO
               vel(i, j, k) = 0.0 
            enddo
         enddo
      enddo

c     Calculate x-derivative, compare against analytical and print answer
      idir = 0  ! x-derivative
      call calc_first_deriv(field, NX, NY, NZ, NHALO, dx, dy, dz, vel,
     &                      deriv, idir)
      call calc_rms_error(deriv, NX, NY, NZ, NHALO, dx, dy, dz, idir, 
     &                    error)
      write(*,*)" Resolution : ", N, " X-derivative error = ", error

c     Calculate y-derivative, compare against analytical and print answer
      idir = 1  ! y-derivative
      call calc_first_deriv(field, NX, NY, NZ, NHALO, dx, dy, dz, vel,
     &                      deriv, idir)
      call calc_rms_error(deriv, NX, NY, NZ, NHALO, dx, dy, dz, idir, 
     &                    error)
      write(*,*)" Resolution : ", N, " Y-derivative error = ", error

      stop
      end

c     Subroutine to apply initial condition which is sin(2*pi*x) * sin(2*pi*y)

      subroutine initialize_field( field, mx, my, mz, mhalo, dx, dy, dz)
      
      implicit none
      integer mx, my, mz, mhalo
      integer i, j, k
      real*8 dx, dy, dz, x, y, z
      real*8 field(0-mhalo:mx-1+mhalo, 0-mhalo:my-1+mhalo, 0:mz-1)

      real*8 pi

      pi = 4 * atan(1.0)
      do k = 0, mz-1
         do j = 0-mhalo, my-1+mhalo
            do i = 0-mhalo, mx-1+mhalo
               x = i*dx
               y = j*dy
               z = k*dz
               field(i, j, k) = sin(2*pi*x) * sin(2*pi*y)
            enddo
         enddo
      enddo

      return
      end

c     Calculate first derivative
      subroutine calc_first_deriv(field, mx, my, mz, mhalo, dx, dy, dz, 
     &     vel, deriv, idir)  
      
      implicit none
      integer mx, my, mz, mhalo, idir
      integer i, j, k
      real*8 dx, dy, dz

      integer orderi, orderb, upwind
      integer biL, biR, bjL, bjR, bkL, bkR
      integer fiL, fiR, fjL, fjR, fkL, fkR
      integer iiL, iiR, ijL, ijR, ikL, ikR
      integer viL, viR, vjL, vjR, vkL, vkR
      integer dfiL, dfiR, dfjL, dfjR, dfkL, dfkR
      integer iperx, ipery, iperz
      integer ierror   ! Error return value
      character*256 error

      real*8 field(0-mhalo:mx-1+mhalo, 0-mhalo:my-1+mhalo, 0:mz-1),
     &       deriv(0-mhalo:mx-1+mhalo, 0-mhalo:my-1+mhalo, 0:mz-1),
     &       vel  (0-mhalo:mx-1+mhalo, 0-mhalo:my-1+mhalo, 0:mz-1)

      
c     Order of discretizations in the interior of domain and near boundaries
      orderi = 4
      orderb = 4

c     Is the domain periodic in x, y, or z? No !
      iperx = 0
      ipery = 0 
      iperz = 0 

c     Should I upwind? No !
      upwind = 0

c     How many halo cells do I have in the lower and upper ends of each axis/
      biL = mhalo
      biR = mhalo
      bjL = mhalo
      bjR = mhalo
      bkL = 0
      bkR = 0

c     what are the array limits of field?
      fiL = 0  - mhalo
      fiR = mx - 1 + mhalo
      fjL = 0 - mhalo
      fjR = my - 1 + mhalo
      fkL = 0 
      fkR = mz - 1

c     What are the indices of the valid points in the field
      iiL = fiL + biL
      iiR = fiR - biR
      ijL = fjL + bjL
      ijR = fjR - bjR
      ikL = fkL + bkL
      ikR = fkR - bkR
      
c     what are the array limits of velocity?
      viL = 0  - mhalo
      viR = mx - 1 + mhalo
      vjL = 0 - mhalo
      vjR = my - 1 + mhalo
      vkL = 0 
      vkR = mz - 1

c     what are the array limits of deriv, the output array?
      dfiL = 0  - mhalo
      dfiR = mx - 1 + mhalo
      dfjL = 0 - mhalo
      dfjR = my - 1 + mhalo
      dfkL = 0 
      dfkR = mz - 1

c     Zero out deriv before proceeding
      do k = dfkL, dfkR
         do j = dfjL, dfjR
            do i = dfiL, dfiR
               deriv(i, j, k) = 0.0
            enddo
         enddo
      enddo

      if ( idir .eq. 0 ) then
         call x_der1_co(field, deriv, dx, dy, dz, vel, upwind, 
     &        orderi, orderb,
     &        biL, biR, bjL, bjR, bkL, bkR,
     &        fiL, fiR, fjL, fjR, fkL, fkR,
     &        iiL, iiR, ijL, ijR, ikL, ikR,
     &        viL, viR, vjL, vjR, vkL, vkR,
     &        dfiL,dfiR,dfjL,dfjR,dfkL,dfkR,
     &        iperx,ipery,iperz,ierror,error)
      else
         call y_der1_co(field, deriv, dx, dy, dz, vel, upwind, 
     &        orderi, orderb,
     &        biL, biR, bjL, bjR, bkL, bkR,
     &        fiL, fiR, fjL, fjR, fkL, fkR,
     &        iiL, iiR, ijL, ijR, ikL, ikR,
     &        viL, viR, vjL, vjR, vkL, vkR,
     &        dfiL,dfiR,dfjL,dfjR,dfkL,dfkR,
     &        iperx,ipery,iperz,ierror,error)
      endif

      return
      end

c     subroutine to calculate RMS errors

      subroutine calc_rms_error(deriv, mx, my, mz, mhalo, dx, dy, dz, 
     &     idir, error)
 
      implicit none
      integer mx, my, mz, mhalo
      integer i, j, k, idir
      real*8 dx, dy, dz, x, y, z
      real*8 deriv(0-mhalo:mx-1+mhalo, 0-mhalo:my-1+mhalo, 0:mz-1)
      real*8 errSq, anal_ans

      real*8 pi, error
      integer count

      pi = 4 * atan(1.0)
      error = 0.0

c     loop over all interior points where you evaluated derivs and 
c     determine errors
      count = 0 
      do k = 0, mz-1
         do j = 0, my-1
            do i = 0, mx-1
               x = i*dx
               y = j*dy
               z = k*dz
               if ( idir .eq. 0 ) then
                  anal_ans = 2*pi*cos(2*pi*x)*sin(2*pi*y)
               else
                  anal_ans = 2*pi*sin(2*pi*x)*cos(2*pi*y)
               endif
               errSq = (deriv(i, j, k) - anal_ans)**2 
               error = error + errSq
               count = count + 1
            enddo
         enddo
      enddo

      error = sqrt( error / count ) 

      return
      end
