/*
  Copyright (c) 2006, Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000
  with Sandia Corporation, the U.S. Governement retains certain rights in this software.

  All rights reserved.
  
  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions are met:
  
  * Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, 
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.
  * Neither the name of the Sandia National Laboratories nor the names of 
    its contributors may be used to endorse or promote products derived 
    from this software without specific prior written permission.
    
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
  OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
  OF THE POSSIBILITY OF SUCH DAMAGE.
  

  Version 0.1, June 2006.
  Authors : Christopher Kennedy, M. H. Carpenter and Jaideep Ray.
  Maintainer: Jaideep Ray, Advanced Software R. & D., Sandia National Laboratories, Livermore, CA, USA.
              jairay@ca.sandia.gov
*/

/*
     This is an example code that shows how to use the interpolations' library.
     It initializes a field on a vertex-centered mesh (NX cells x NY cells, i.e.
     NX+1 vertices x NY+1 vertices) analytically and interpolates it onto a mesh 
     twice as fine (2NX cells x 2NY cells i.e 2NX+1 x 2NY+1 vertices). Use 6th order
     interpolations
*/

#include <stdio.h>
#include <stdlib.h>
#include "math.h"

/* The interpolation evaluation functions' signatures */
#include "interp.h"

#define NCELLS 50
#define NZ 1
#define NHALO 3

void initialize_field(double *field, int nx, int ny, int nz, int nhalo,
		      double dx, double dy, double dz) ;

void interp_vc_field(double *cfield, int cnx, int cny, int cnz, int nhalo, 
		     double *ffield, int fnx, int fny, int fnz ) ;

void calc_rms_error(double *field, int nx, int ny, int nz, int nhalo,
		    double dx, double dy, double dz, double *err) ;

int get_size(int start_i, int start_j, int start_k, int end_i, int end_j, 
	     int end_k) ;

int findx(int i, int j, int k, int start_i, int start_j, int start_k, 
	  int end_i, int end_j, int end_k) ;

int main()
{
  int cnx = NCELLS,     cny = NCELLS,     cnz = NZ;
  int fnx = 2*NCELLS, fny = 2*NCELLS, fnz = NZ ;
  int i, j, k, idir ;

  /*
    Allocate fields
  */
  int csize = get_size(-NHALO, -NHALO, 0, cnx+NHALO, cny+NHALO, cnz-1) ;
  int fsize = get_size(-NHALO, -NHALO, 0, fnx+NHALO, fny+NHALO, fnz-1) ;
  double *cfield = (double *) malloc( csize * sizeof(double) ) ;
  double *ffield = (double *) malloc( fsize * sizeof(double) ) ;

  /*
    Define a unit square
  */
  double unit = 1.0 ;
  double dx = unit / cnx, dy = unit / cny, dz = unit / cnz ;
  double error ;
  
  /* Initialize fields */
  initialize_field(cfield, cnx, cny, cnz, NHALO, dx, dy, dz) ;

  /* Interpolate them onto the fine mesh */
  interp_vc_field(cfield, cnx, cny, cnz, NHALO, ffield, fnx, 
		  fny, fnz) ;

  /* Check error versus analytical results */
  calc_rms_error(ffield, fnx, fny, fnz, NHALO, 0.5*dx, 0.5*dy, 
		 0.5*dz, &error) ;
  printf("Resolution : %d, Interpolation error = %e \n", fnx, error);

  free(cfield) ;
  free(ffield) ;

  return 0 ;
}

void initialize_field(double *field, int nx, int ny, int nz, int nhalo, 
		      double dx, double dy, double dz)
{

  int i, j, k ;

  double pi = 4.0 * atan(1.0) ;
  for( k = 0; k < nz; k++ )
    for ( j = -nhalo; j <= ny+nhalo ; j++)
      for ( i = -nhalo; i <= nx+nhalo ; i++ )
      {
	double x = i*dx ;
	double y = j*dy ;
	int index = findx(i, j, k, -nhalo, -nhalo, 0, nx+nhalo, ny+nhalo, nz-1) ;
	field[index] = sin(2*pi*x) * sin(2*pi*y) ;
      }

  return ;
}


void interp_vc_field(double *cfield, int cnx, int cny, int cnz, int nhalo, 
		     double *ffield, int fnx, int fny, int fnz )
{

  int csize = get_size(-nhalo, -nhalo, 0, cnx+nhalo, cny+nhalo, cnz-1) ;
  double *tempx = (double *) malloc( csize * sizeof(double) ) ;
  double *tempy = (double *) malloc( csize * sizeof(double) ) ;
  double *tempxy = (double *) malloc( csize * sizeof(double) ) ;

  int orderi, orderb ;
  int biL, biR, bjL, bjR, bkL, bkR ;
  int fiL, fiR, fjL, fjR, fkL, fkR ;
  int iiL, iiR, ijL, ijR, ikL, ikR ;
  int dfiL, dfiR, dfjL, dfjR, dfkL, dfkR ;
  int iperx, ipery, iperz ;
  int ierror  ;
  char error[256] ;

  int i, j, k, corner ;
  
  /* Make sure that the fine mesh is actually twice as fine as the coarse one */
  if ( ( (fnx) != 2*(cnx)) || ( (fny) != 2*(cny)) )
  {
    printf(" The valid region of the fine mesh %d X %d is NOT twice is \n", 
	   fnx, fny) ;
    printf(" not twice as fine as the valid region %d X %d of coarse mesh \n",
	   cnx, cny ) ;
    abort() ;
  }

  /* Order of interpolation in the interior of domain and near boundaries */
  /* orderb will be ignored; the subroutine will close as necesary */
  orderi = 6 ; orderb = 6 ;
  
  /* Is the domain periodic in x, y, or z? No ! */
  iperx = 0 ;     ipery = 0 ;     iperz = 0 ;

  /* How many halo cells do I have in the lower and upper ends of each axis? */
  biL = 0 ; biR = 0 ; bjL = 0 ; bjR = 0 ; bkL = 0 ; bkR = 0 ;

  /* what are the array limits of field? */
  fiL = 0 - nhalo  ;    fiR = cnx  + nhalo ;
  fjL = 0 - nhalo   ;    fjR = cny  + nhalo ;
  fkL = 0           ;     fkR = cnz - 1 ;

  /* What are the indices of the valid points in the field */
  iiL = fiL + biL ;     iiR = fiR - biR ;
  ijL = fjL + bjL ;     ijR = fjR - bjR ;
  ikL = fkL + bkL ;     ikR = fkR - bkR ;
      
  /* What are the array limits of deriv, the output array? */
  dfiL = 0  - nhalo ;       dfiR = cnx + nhalo ;
  dfjL = 0 - nhalo  ;       dfjR = cny + nhalo ;
  dfkL = 0          ;       dfkR = cnz - 1 ;

  /* Zero out the temps */
  for (i = 0; i < csize ; i++) { tempx[i] = tempy[i] = tempxy[i] = 0.0 ; }

  /* Do the x_interp - fill out u_{i+1/2,j} points */
  C_x_intp_cf_vc(cfield, tempx, &orderi, &orderb, 
		 &biL, &biR, &bjL, &bjR, &bkL, &bkR,
		 &fiL, &fiR, &fjL, &fjR, &fkL, &fkR, 
		 &iiL, &iiR, &ijL, &ijR, &ikL, &ikR,
		 &dfiL, &dfiR, &dfjL, &dfjR, &dfkL, &dfkR, 
		 &iperx, &ipery, &iperz, &ierror, error) ;
  
  /* Do the y_interp - fill out u_{i,j+1/2} points */
  C_y_intp_cf_vc(cfield, tempy, &orderi, &orderb, 
		 &biL, &biR, &bjL, &bjR, &bkL, &bkR,
		 &fiL, &fiR, &fjL, &fjR, &fkL, &fkR, 
		 &iiL, &iiR, &ijL, &ijR, &ikL, &ikR,
		 &dfiL, &dfiR, &dfjL, &dfjR, &dfkL, &dfkR, 
		 &iperx, &ipery, &iperz, &ierror, error) ;

  /* Do the xy_interp - fill out u_{i,j+1/2} points. I want corners */
  corner = 1 ;
  C_xy_intp_cf_vc(cfield, tempxy, &orderi, &orderb, 
		  &biL, &biR, &bjL, &bjR, &bkL, &bkR,
		  &fiL, &fiR, &fjL, &fjR, &fkL, &fkR, 
		  &iiL, &iiR, &ijL, &ijR, &ikL, &ikR,
		  &dfiL, &dfiR, &dfjL, &dfjR, &dfkL, &dfkR, 
		  &iperx, &ipery, &iperz, &ierror, error,
		  &corner) ;

  /* Now fill up the fine field. First init it to zero */
  {
    int fsize = get_size(-nhalo, -nhalo, 0, fnx+nhalo, fny+nhalo, fnz-1) ;
    for( i = 0; i < fsize; i++) ffield[i] = 0.0 ;
  }

  for( k = 0 ; k < cnz; k++ )
    for ( j = 0; j <= cny ; j++ )
      for ( i = 0; i <= cnx ; i++ )
	{
	  int cindex = findx(i, j, k, -nhalo, -nhalo, 0, cnx+nhalo, cny+nhalo,
			     fnz-1) ;
	  int f_ij = findx(2*i, 2*j, k, -nhalo, -nhalo, 0, fnx+nhalo, fny+nhalo,
			     fnz-1) ;
	  int f_iphj = findx(2*i+1, 2*j, k, -nhalo, -nhalo, 0, fnx+nhalo, fny+nhalo,
			     fnz-1) ;
	  int f_ijph = findx(2*i, 2*j+1, k, -nhalo, -nhalo, 0, fnx+nhalo, fny+nhalo,
			     fnz-1) ;
	  int f_iphjph = findx(2*i+1, 2*j+1, k, -nhalo, -nhalo, 0, fnx+nhalo, 
			       fny+nhalo, fnz-1) ;
	  ffield[f_ij]   = cfield[cindex] ;
	  ffield[f_iphj] = tempx[cindex] ;
	  ffield[f_ijph] = tempy[cindex] ;
	  ffield[f_iphjph]= tempxy[cindex] ;
	}

  free (tempx) ;
  free (tempy) ;
  free (tempxy) ;

  return ;
}


void calc_rms_error(double *field, int nx, int ny, int nz, int nhalo,
		    double dx, double dy, double dz, double *err)
{
  double pi = 4.0 * atan(1.0) ;
  int count = 0 ;
  int i, j, k ;

  for( k = 0; k < nz ; k++ )
    for ( j = 0; j <= ny; j++ )
      for ( i = 0; i <= nx ; i++ )
      {
	double x = i*dx ;
	double y = j*dy ;
	double anal_ans = sin(2*pi*x)*sin(2*pi*y) ;
	int index = findx(i, j, k, -nhalo, -nhalo, 0, nx+nhalo, ny+nhalo, nz-1) ;
	double errSq = pow( field[index] - anal_ans, 2) ; 
	*err += errSq ;
	count++ ;
      }
  *err = sqrt( *err / count ) ;

  return  ;
}

int get_size(int start_i, int start_j, int start_k, int end_i, int end_j, 
	     int end_k) 
{
  return ( (end_i-start_i+1) * (end_j-start_j+1) * (end_k-start_k+1) ) ;
}

int findx(int i, int j, int k, int start_i, int start_j, int start_k, 
	  int end_i, int end_j, int end_k)
{
  int nx = end_i-start_i+1 ;
  int ny = end_j-start_j+1 ;
  int nz = end_k-start_k+1 ;
  int ii = i - start_i  ;
  int jj = j - start_j  ;
  int kk = k - start_k  ;
  return ( ii + jj*nx + kk*nx*ny ) ;
}
