/*
  Copyright (c) 2006, Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000
  with Sandia Corporation, the U.S. Governement retains certain rights in this software.

  All rights reserved.
  
  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions are met:
  
  * Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, 
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.
  * Neither the name of the Sandia National Laboratories nor the names of 
    its contributors may be used to endorse or promote products derived 
    from this software without specific prior written permission.
    
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
  OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
  OF THE POSSIBILITY OF SUCH DAMAGE.
  

  Version 0.1, June 2006.
  Authors : Christopher Kennedy, M. H. Carpenter and Jaideep Ray.
  Maintainer: Jaideep Ray, Advanced Software R. & D., Sandia National Laboratories, Livermore, CA, USA.
              jairay@ca.sandia.gov
*/

/*
  This is an example code that shows how to use the filters' library.
  It initializes a field analytically and filter it using an 8th order 
  filter. Calculate difference from original field
*/

#include <stdio.h>
#include <stdlib.h>
#include "math.h"

/* The derivative evaluation functions' signatures */
#include "filter.h"

#define N 50
#define NZ 1
#define NHALO 2

void initialize_field(double *field, int nx, int ny, int nz, int nhalo,
		      double dx, double dy, double dz) ;

void filter_field(double *field, int nx, int ny, int nz, int nhalo,
		  double dx, double dy, double dz) ;

void calc_rms_error(double *field, int nx, int ny, int nz, int nhalo,
		    double dx, double dy, double dz, double *err) ;

int get_size(int start_i, int start_j, int start_k, int end_i, int end_j, 
	     int end_k) ;

int findx(int i, int j, int k, int start_i, int start_j, int start_k, 
	  int end_i, int end_j, int end_k) ;



int main()
{

  int nx = N, ny = N, nz = NZ;
  int i, j, k, idir ;

  /*
    Allocate fields
  */
  int size = get_size(-NHALO, -NHALO, 0, nx-1+NHALO, ny-1+NHALO, nz-1) ;
  double *field = (double *) malloc( size * sizeof(double) ) ;

  /*
    Define a unit square
  */
  double unit = 1.0 ;
  double dx = unit / nx, dy = unit / ny, dz = unit / nz ;
  double error ;
  
  /* Initialize fields */
  initialize_field(field, nx, ny, nz, NHALO, dx, dy, dz) ;

  /* Filter it */
  filter_field(field, nx, ny, nz, NHALO, dx, dy, dz) ;

  /* Check versus real results */
  calc_rms_error(field, nx, ny, nz, NHALO, dx, dy, dz, &error) ;

  /* Print difference */
   printf("Resolution : %d, RMS error after filtering = %e \n", N, error);
  free(field);

  return 0 ;
}


void initialize_field(double *field, int nx, int ny, int nz, int nhalo, 
		      double dx, double dy, double dz)
{

  int i, j, k ;

  double pi = 4.0 * atan(1.0) ;
  for( k = 0; k < nz; k++ )
    for ( j = -nhalo; j < ny+nhalo ; j++)
      for ( i = -nhalo; i < nx+nhalo ; i++ )
      {
	double x = i*dx ;
	double y = j*dy ;
	int index = findx(i, j, k, -nhalo, -nhalo, 0, nx+nhalo-1, 
			  ny+nhalo-1, nz-1) ;
	field[index] = sin(2*pi*x) * sin(2*pi*y) ;
      }

  return ;
}

void filter_field(double *field, int nx, int ny, int nz, int nhalo, 
		  double dx, double dy, double dz)
{

  int ifilt_x, ifilt_y, ifilt_z ;
  int filter_order ;
  int fiL, fiR, fjL, fjR, fkL, fkR ;
  int wiL, wiR, wjL, wjR, wkL, wkR ;
  int piL, piR, pjL, pjR, pkL, pkR ;
  int ierror, ierr_coeff   ;
  char error[256], error_coeff[256];

  /* Order of filter */
  filter_order = 8 ;

  /* Is filtering to be done in x [yup], y [yup] and z [nope] direction ?*/
  ifilt_x = ifilt_y = 1 ; ifilt_z = 0 ;

  /* What are the array limits of the field ? */
  fiL = 0 - nhalo ; fiR = nx - 1 + nhalo ;
  fjL = 0 - nhalo ; fjR = ny - 1 + nhalo ;
  fkL = 0 ;         fkR = nz - 1 ;

  /* what are the indices of the window where filtering will be done ?*/
  wiL = fiL + nhalo ;       wiR = fiR - nhalo ;
  wjL = fjL + nhalo ;       wjR = fjR - nhalo ;
  wkL = fkL  ;              wkR = fkR  ;

  /* what are the array limits of field where filterable data exists?
     In our case, the whole array. */
  piL = fiL  ;       piR = fiR  ;
  pjL = fjL  ;       pjR = fjR  ;
  pkL = fkL  ;       pkR = fkR  ;
  
  C_expfil(field, &ifilt_x, &ifilt_y, &ifilt_z, &filter_order,
	   &fiL, &fiR, &fjL, &fjR, &fkL, &fkR,
	   &piL, &piR, &pjL, &pjR, &pkL, &pkR,
	   &wiL, &wiR, &wjL, &wjR, &wkL, &wkR,
	   &ierror, error, &ierr_coeff, error_coeff) ;

  return ;
}

void calc_rms_error(double *field, int nx, int ny, int nz, int nhalo,
		    double dx, double dy, double dz, double *err)
{
  double pi = 4.0 * atan(1.0) ;
  int count = 0 ;
  int i, j, k ;

  for( k = 0; k < nz ; k++ )
    for ( j = 0; j < ny; j++ )
      for ( i = 0; i < nx ; i++ )
      {
	double x = i*dx ;
	double y = j*dy ;
	double anal_ans = sin(2*pi*x)*sin(2*pi*y) ;
	int index = findx(i, j, k, -nhalo, -nhalo, 0, nx-1+nhalo,
			  ny-1+nhalo, nz-1) ;
	double errSq = pow( field[index] - anal_ans, 2) ; 
	*err += errSq ;
	count++ ;
      }
  *err = sqrt( *err / count ) ;

  return  ;
}

int get_size(int start_i, int start_j, int start_k, int end_i, int end_j, 
	     int end_k) 
{
  return ( (end_i-start_i+1) * (end_j-start_j+1) * (end_k-start_k+1) ) ;
}

int findx(int i, int j, int k, int start_i, int start_j, int start_k, 
	  int end_i, int end_j, int end_k)
{
  int nx = end_i-start_i+1 ;
  int ny = end_j-start_j+1 ;
  int nz = end_k-start_k+1 ;
  int ii = i - start_i  ;
  int jj = j - start_j  ;
  int kk = k - start_k  ;
  return ( ii + jj*nx + kk*nx*ny ) ;
}
