/*
  Copyright (c) 2006, Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000
  with Sandia Corporation, the U.S. Governement retains certain rights in this software.

  All rights reserved.
  
  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions are met:
  
  * Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice, 
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.
  * Neither the name of the Sandia National Laboratories nor the names of 
    its contributors may be used to endorse or promote products derived 
    from this software without specific prior written permission.
    
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
  OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
  OF THE POSSIBILITY OF SUCH DAMAGE.
  

  Version 0.1, June 2006.
  Authors : Christopher Kennedy, M. H. Carpenter and Jaideep Ray.
  Maintainer: Jaideep Ray, Advanced Software R. & D., Sandia National Laboratories, Livermore, CA, USA.
              jairay@ca.sandia.gov
*/

/*
     This is an example code that shows how to use the derivatives' library.
     It initializes a field analytically and takes its first derivative using
     a fourth order stencil. No upwinding is done.
*/

#include <stdio.h>
#include <stdlib.h>
#include "math.h"

/* The derivative evaluation functions' signatures */
#include "deriv.h"

#define N 50
#define NZ 1
#define NHALO 2

void initialize_field(double *field, int nx, int ny, int nz, int nhalo,
		      double dx, double dy, double dz) ;

void calc_first_deriv(double *field, int nx, int ny, int nz, int nhalo,
		      double dx, double dy, double dz, double *vel, double *deriv,
		      int idir) ;

void calc_rms_error(double *field, int nx, int ny, int nz, int nhalo,
		    double dx, double dy, double dz, int idir, double *err) ;

int get_size(int start_i, int start_j, int start_k, int end_i, int end_j, 
	     int end_k) ;

int findx(int i, int j, int k, int start_i, int start_j, int start_k, 
	  int end_i, int end_j, int end_k) ;

int main()
{

  int nx = N, ny = N, nz = NZ;
  int i, j, k, idir ;

  /*
    Allocate fields
  */
  int size = get_size(-NHALO, -NHALO, 0, nx-1+NHALO, ny-1+NHALO, nz-1) ;
  double *field = (double *) malloc( size * sizeof(double) ) ;
  double *deriv = (double *) malloc( size * sizeof(double) ) ;
  double *vel   = (double *) malloc( size * sizeof(double) ) ;

  /*
    Define a unit square
  */
  double unit = 1.0 ;
  double dx = unit / nx, dy = unit / ny, dz = unit / nz ;
  double error ;
  
  /* Initialize fields */
  initialize_field(field, nx, ny, nz, NHALO, dx, dy, dz) ;

  /* Initialize velocity to zero */
  for( i = 0; i < size; i++) vel[i] = 0.0 ;

  /* Calculate x-deriv; compare against analytical answer */
  idir = 0 ;
  calc_first_deriv(field, nx, ny, nz, NHALO, dx, dy, dz, vel, 
		   deriv, idir) ;
  error = 0.0 ;
  calc_rms_error(deriv, nx, ny, nz, NHALO, dx, dy, dz, idir, &error) ;
  printf("Resolution : %d, X-derivative error = %e \n", N, error);

  /* Calculate y-deriv; compare against analytical answer */
  idir = 1 ;
  calc_first_deriv(field, nx, ny, nz, NHALO, dx, dy, dz, vel, 
		   deriv, idir) ;
  error = 0.0 ;
  calc_rms_error(deriv, nx, ny, nz, NHALO, dx, dy, dz, idir, &error) ;
  printf("Resolution : %d, Y-derivative error = %e \n", N, error);


  /*
    deallocate arrays
  */
  free(field) ;
  free(deriv) ;
  free(vel)  ;
    
  return 0 ;
}


void initialize_field(double *field, int nx, int ny, int nz, int nhalo, 
		      double dx, double dy, double dz)
{

  int i, j, k ;

  double pi = 4.0 * atan(1.0) ;
  for( k = 0; k < nz; k++ )
    for ( j = -nhalo; j < ny+nhalo ; j++)
      for ( i = -nhalo; i < nx+nhalo ; i++ )
      {
	double x = i*dx ;
	double y = j*dy ;
	int index = findx(i, j, k, -nhalo, -nhalo, 0, nx+nhalo-1, 
			  ny+nhalo-1, nz-1) ;
	field[index] = sin(2*pi*x) * sin(2*pi*y) ;
      }

  return ;
}

void calc_first_deriv(double *field, int nx, int ny, int nz, int nhalo,
		      double dx, double dy, double dz, double *vel, double *deriv,
		      int idir)
{
  int orderi, orderb, upwind ;
  int biL, biR, bjL, bjR, bkL, bkR ;
  int fiL, fiR, fjL, fjR, fkL, fkR ;
  int iiL, iiR, ijL, ijR, ikL, ikR ;
  int viL, viR, vjL, vjR, vkL, vkR ;
  int dfiL, dfiR, dfjL, dfjR, dfkL, dfkR ;
  int iperx, ipery, iperz ;
  int ierror  ;
  char error[256] ;

  int i, j, k ;
  int size ;

  /* Order of discretizations in the interior of the domain and near boundaries */
  orderi = 4 ; orderb = 4 ;

  /* Is the domain periodic in x-, y- and z-directions? No ! */
  iperx = ipery = iperz = 0 ;

  /* Should I upwind ? No ! */
  upwind = 0 ;

  /* How many halo cells do I have in the lower and upper ends of each axis ? */
  biL = biR = bjL = bjR = nhalo ; bkL = bkR = 0 ;

  /* What are the array limits of the field ? */
  fiL = 0 - nhalo ; fiR = nx - 1 + nhalo ;
  fjL = 0 - nhalo ; fjR = ny - 1 + nhalo ;
  fkL = 0 ;         fkR = nz - 1 ;

  /* What are the indices of the valid points in this field */
  iiL = fiL + biL ; iiR = fiR - biR ;
  ijL = fjL + bjL ; ijR = fjR - bjR ;
  ikL = fkL + bkL ; ikR = fkR - bkR ;

  /* What are the array limits of velocity ? */
  viL = 0 - nhalo ; viR = nx - 1 + nhalo ;
  vjL = 0 - nhalo ; vjR = ny - 1 + nhalo ;
  vkL = 0 ;         vkR = nz - 1 ;

  /* What are the array limits on deriv ? */
  dfiL = 0 - nhalo ; dfiR = nx - 1 + nhalo ;
  dfjL = 0 - nhalo ; dfjR = ny - 1 + nhalo ;
  dfkL = 0 ;         dfkR = nz - 1 ;
  
  /* Zero out deriv before proceeding */
  size = get_size(-nhalo, -nhalo, 0, nx-1+nhalo, ny-1+nhalo, nz-1) ;
  for(i = 0; i < size; i++) deriv[i] = 0.0 ;

  if ( idir == 0 ) /* X-derivative */
    C_x_der1_co(field, deriv, &dx, &dy, &dz, vel, &upwind, &orderi, &orderb,
		&biL, &biR, &bjL, &bjR, &bkL, &bkR, 
		&fiL, &fiR, &fjL, &fjR, &fkL, &fkR,
		&iiL, &iiR, &ijL, &ijR, &ikL, &ikR, 
		&viL, &viR, &vjL, &vjR, &vkL, &vkR,
		&dfiL, &dfiR, &dfjL, &dfjR, &dfkL, &dfkR, 
		&iperx, &ipery, &iperz, &ierror, error) ;
  else  /* Y-derivative */
    C_y_der1_co(field, deriv, &dx, &dy, &dz, vel, &upwind, &orderi, &orderb,
		&biL, &biR, &bjL, &bjR, &bkL, &bkR, 
		&fiL, &fiR, &fjL, &fjR, &fkL, &fkR,
		&iiL, &iiR, &ijL, &ijR, &ikL, &ikR, 
		&viL, &viR, &vjL, &vjR, &vkL, &vkR,
		&dfiL, &dfiR, &dfjL, &dfjR, &dfkL, &dfkR, 
		&iperx, &ipery, &iperz, &ierror, error) ;
  return ;
}

void calc_rms_error(double *field, int nx, int ny, int nz, int nhalo,
		    double dx, double dy, double dz, int idir, double *err)
{
  double pi = 4.0 * atan(1.0) ;
  int count = 0 ;
  int i, j, k ;

  for( k = 0; k < nz ; k++ )
    for ( j = 0; j < ny; j++ )
      for ( i = 0; i < nx ; i++ )
      {
	double x = i*dx ;
	double y = j*dy ;
	double anal_ans = 0 ;
	if ( idir == 0 )
	  anal_ans = 2*pi*cos(2*pi*x)*sin(2*pi*y) ;
	else
	  anal_ans = 2*pi*sin(2*pi*x)*cos(2*pi*y) ;

	int index = findx(i, j, k, -nhalo, -nhalo, 0, nx-1+nhalo,
			  ny-1+nhalo, nz-1) ;
	double errSq = pow( field[index] - anal_ans, 2) ; 
	*err += errSq ;
	count++ ;
      }
  *err = sqrt( *err / count ) ;

  return  ;
}

int get_size(int start_i, int start_j, int start_k, int end_i, int end_j, 
	     int end_k) 
{
  return ( (end_i-start_i+1) * (end_j-start_j+1) * (end_k-start_k+1) ) ;
}

int findx(int i, int j, int k, int start_i, int start_j, int start_k, 
	  int end_i, int end_j, int end_k)
{
  int nx = end_i-start_i+1 ;
  int ny = end_j-start_j+1 ;
  int nz = end_k-start_k+1 ;
  int ii = i - start_i  ;
  int jj = j - start_j  ;
  int kk = k - start_k  ;
  return ( ii + jj*nx + kk*nx*ny ) ;
}
