!--------------------------------------------------------------------------------
!	This routine calculates the retarded Green function
!--------------------------------------------------------------------------------
subroutine GetGR( Er, Ei, pot, N, SigLr, SigLi, SigRr, SigRi, GRr, GRi )

	implicit none
	
	!	INTERFACE PARAMETERS
	integer,  intent(in)   :: N        						! number of tube rings
	real(DP), intent(in)   :: Er, Ei	      			! real/imaginary value of energy
	real(DP), intent(in)   :: pot(:)							!	tube potential
	real(DP), intent(in)   :: SigLr, SigLi  			! real/imaginary self-energy
	real(DP), intent(in)   :: SigRr, SigRi  			! of left/right contact
	real(DP), intent(inout):: GRr(:,:), GRi(:,:)	! retarded Green function
	
	!	INTERNAL PARAMETERS
	integer  :: i
	real(DP) :: aa, bb                      ! effective 1D Hamiltonian
	real(DP) :: coupl                       ! coupling parameter of tube
	real(DP) :: gtLr(N), gtLi(N)
	real(DP) :: gtRr(N), gtRi(N)
	
	!==================================================================
	
	gtLr = 0.0d0 ; gtLi = 0.0d0 ; gtRr = 0.0d0 ; gtRi = 0.0d0
	GRr  = 0.0d0 ; GRi  = 0.0d0
	
	!	 move-in from left
	!	retarded Green function: 1st ring on left
  aa      =  Er + pot(1) - SigLr
  bb      =  Ei          - SigLi
  gtLr(1) =  aa/(aa**2 + bb**2)
  gtLi(1) = -bb/(aa**2 + bb**2)

	!	retarded Green function: inner rings
  do i = 2,N-1
  	coupl = vpi
  
  	if (mod(i,2) .eq. 0) then
  		coupl = me
		endif

  	aa      =  Er + pot(i) - coupl**2 * gtLr(i-1)
  	bb      =  Ei          - coupl**2 * gtLi(i-1)
  	gtLr(i) =  aa/(aa**2 + bb**2)
  	gtLi(i) = -bb/(aa**2 + bb**2)
  enddo
  
  !	retarded Green function: last ring on right
 	coupl   =  me
 	aa      =  Er + pot(N) - coupl**2 * gtLr(N-1) - SigRr
 	bb      =  Ei          - coupl**2 * gtLi(N-1) - SigRi
 	gtLr(N) =  aa/(aa**2 + bb**2)
 	gtLi(N) = -bb/(aa**2 + bb**2)

	!--------------------------------------------------------------
 	!	 move-in from right
 	!--------------------------------------------------------------
 	!	retarded Green function: last ring on right
 	aa      =  Er + pot(N) - SigRr 
 	bb      =  Ei          - SigRi
  gtRr(N) =  aa/(aa**2 + bb**2)
  gtRi(N) = -bb/(aa**2 + bb**2)

	!	retarded Green function: inner rings
 	do i = N-1,2,-1
 		coupl = me

 		if (mod(i,2) .eq. 0) then
 			coupl = vpi
 		endif
 		
 		aa      =  Er + pot(i) - coupl**2 * gtRr(i+1)
 		bb      =  Ei          - coupl**2 * gtRi(i+1)
 		gtRr(i) =  aa/(aa**2 + bb**2)
 		gtRi(i) = -bb/(aa**2 + bb**2)
 	enddo

	!	retarded Green function: 1st ring on left
  coupl   =  me
  aa      =  Er + pot(1) - coupl**2 * gtRr(2) - SigLr
  bb      =  Ei          - coupl**2 * gtRi(2) - SigLi
  gtRr(1) =  aa/(aa**2 + bb**2)
  gtRi(1) = -bb/(aa**2 + bb**2)

	!-------------------------------------------------------------
  !	 retarded Green Function: FULL connected
	!-------------------------------------------------------------
  GRr(N,N) = gtLr(N)
  GRi(N,N) = gtLi(N)
  GRr(1,1) = gtRr(1)
  GRi(1,1) = gtRi(1)

	! calculate diagonal elements of retarded Green function
!	DEBUG-DIEGO: changed on 05/08/2007
  do i = 2,N-1
  !do i = 1,N-1
  	coupl = me
  
  	if (mod(i,2) .eq. 0) then
  		coupl = vpi
		endif
	
  	aa       = 1.0d0 - coupl**2 * (gtLr(i)*gtRr(i+1) - gtLi(i)*gtRi(i+1))
  	bb       =       - coupl**2 * (gtLi(i)*gtRr(i+1) + gtLr(i)*gtRi(i+1))
		GRr(i,i) = ( gtLr(i)*aa + gtLi(i)*bb) / (aa**2 + bb**2)
  	GRi(i,i) = (-gtLr(i)*bb + gtLi(i)*aa) / (aa**2 + bb**2)
  enddo
	
	return
end subroutine GetGR


!-------------------------------------------------------------------------------------
!	This routine calculates G< over the NON-EQUILIBRIUM part of the contour
!-------------------------------------------------------------------------------------
subroutine GetGn( Er, Ei, U, N, SigLr, SigLi, SigRr, SigRi, Gni, Il, Ir )
                  
	implicit none

	!	INTERFACE PARAMETERS
	integer,  intent(in) 	:: N        			! number of tube rings
	real(DP), intent(in) 	:: Er, Ei	      	! real/imaginary value of energy
	real(DP), intent(in) 	:: U(:)
	real(DP), intent(in) 	:: SigLr, SigLi  	! real/imaginary self-energy: left
	real(DP), intent(in) 	:: SigRr, SigRi   ! real/imaginary self-energy: right
	real(DP), intent(out) :: Il, Ir       ! current through left/right contact
	real(DP), intent(out) :: Gni(:,:)     ! Green function G<
	
	!	INTERNAL PARAMETERS
	integer  :: i
	real(DP) :: aa, bb       				! effective 1D Hamiltonian
	real(DP) :: coupl               ! coupling parameter of tube
	real(DP) :: SignLi, SignRi      ! imaginary part of Sig<
	real(DP) :: GamL, GamR
	real(DP) :: gtLr(N), gtLi(N)
	real(DP) :: gtRr(N), gtRi(N)
	real(DP) :: GRr1(N), GRi1(N)
	real(DP) :: GRrN(N), GRiN(N)
	
	!=========================================================================

	gtLr = 0.0d0 ; gtLi = 0.0d0 ; gtRr = 0.0d0 ; gtRi = 0.0d0
	Il   = 0.0d0 ; Ir   = 0.0d0 ; Gni  = 0.0d0
 	GRr1 = 0.0d0 ; GRi1 = 0.0d0
 	GRrN = 0.0d0 ; GRiN = 0.0d0
 	
	GamL  = -2.0d0*SigLi
	GamR  = -2.0d0*SigRi
		   
 	!	calculate green's function
 	!	move-in from left
  aa      =  Er + U(1) - SigLr
  bb      =  Ei        - SigLi
  !	retarded Green function: 1st ring on left
  gtLr(1) =  aa/(aa**2 + bb**2)
  gtLi(1) = -bb/(aa**2 + bb**2)

	!	retarded Green function: inner rings
  do i = 2,N-1
    coupl = vpi
    
    if (mod(i,2) .eq. 0) then
    	coupl = me
		endif
			
    aa      =  Er + U(i) - coupl**2 * gtLr(i-1)
    bb      =  Ei        - coupl**2 * gtLi(i-1)
    gtLr(i) =  aa/(aa**2 + bb**2)
    gtLi(i) = -bb/(aa**2 + bb**2)
  enddo
  
	!	retarded Green function: last ring on right
	coupl   =  me
  aa      =  Er + U(N) - coupl**2 * gtLr(N-1) - SigRr
  bb      =  Ei        - coupl**2 * gtLi(N-1) - SigRi
  gtLr(N) =  aa/(aa**2 + bb**2)
  gtLi(N) = -bb/(aa**2 + bb**2)

	!----------------------------------------------
	!	move-in from right
	!----------------------------------------------
  aa           =  Er + U(N) - SigRr
  bb           =  Ei        - SigRi
  !	retarded Green function: last ring on right
  gtRr(N) =  aa/(aa**2 + bb**2)
  gtRi(N) = -bb/(aa**2 + bb**2)

	!	retarded Green function: inner rings
  do i = N-1,2,-1
  	coupl = me
  
  	if (mod(i,2) .eq. 0) then
  		coupl = vpi
		endif
	
  	aa      =  Er + U(i) - coupl**2 * gtRr(i+1)
  	bb      =  Ei        - coupl**2 * gtRi(i+1)
    gtRr(i) =  aa/(aa**2 + bb**2)
    gtRi(i) = -bb/(aa**2 + bb**2)
  enddo

	!	retarded Green function: 1st ring on left
  coupl   =  me
  aa      =  Er + U(1) - coupl**2 * gtRr(2) - SigLr
  bb      =  Ei        - coupl**2 * gtRi(2) - SigLi
  gtRr(1) =  aa/(aa**2 + bb**2)
  gtRi(1) = -bb/(aa**2 + bb**2)

	!	make Green function  
	GRrN(N) = gtLr(N)
	GRiN(N) = gtLi(N)
	GRr1(1) = gtRr(1)
	GRi1(1) = gtRi(1)

!	DEBUG-DIEGO: changed on 05/08/2007
!	do i = 1,1
  do i = 2,N-1
 		coupl    = me
 		
 		!	DEBUG-DIEGO: added on 05/04/2007
 		if (mod(i,2) .eq. 0) then
 			coupl = vpi
 		endif
 		
   	aa       = 1.0d0 - coupl**2*(gtLr(i)*gtRr(i+1) - gtLi(i)*gtRi(i+1))
 	  bb       =       - coupl**2*(gtLi(i)*gtRr(i+1) + gtLr(i)*gtRi(i+1))
   	!GRr(i,i) = ( gtLr(i)*aa + gtLi(i)*bb)/(aa**2 + bb**2)
   	!GRi(i,i) = (-gtLr(i)*bb + gtLi(i)*aa)/(aa**2 + bb**2)
  enddo

 	!	calculate needed off-diagonal elements of GR
 	!	(i>j)
  do i = 2,N
  	coupl = vpi
  
  	if (mod(i,2) .eq. 0) then
  		coupl = me
		endif
		
   	GRr1(i) = -gtRr(i)*coupl*GRr1(i-1) + gtRi(i)*coupl*GRi1(i-1)
  	GRi1(i) = -gtRi(i)*coupl*GRr1(i-1) - gtRr(i)*coupl*GRi1(i-1)
  enddo

	!	(i<j)
  do i = N-1,1,-1
  	coupl = me
  
  	if (mod(i,2) .eq. 0) then
  		coupl = vpi
		endif
	
  	GRrN(i) = -gtLr(i)*coupl*GRrN(i+1) + gtLi(i)*coupl*GRiN(i+1)
  	GRiN(i) = -gtLi(i)*coupl*GRrN(i+1) - gtLr(i)*coupl*GRiN(i+1)
  enddo

 	!	calculate Sigma<
  fermiL = 1.0d0/(1.0d0 + dexp((Er - (Ef-VS))/kT))
  fermiR = 1.0d0/(1.0d0 + dexp((Er - (Ef-VD))/kT))
	SignLi = fermiL*GamL
	SignRi = fermiR*GamR

 	!	calculate diagonal elements of G<
  do i = 1,N 
    Gni(i,i) = SignLi * (GRr1(i)**2 + GRi1(i)**2) &
     	       + SignRi * (GRrN(i)**2 + GRiN(i)**2)
  enddo

	!	calculate current through left/right contact
  Il = -GamL * (-2.0d0*fermiL*GRi1(1) - Gni(1,1))
  Ir = -GamR * (-2.0d0*fermiR*GRiN(N) - Gni(N,N))

	return
end subroutine GetGn
