!-------------------------------------------------------------------
!	This routine sets the electrostatic boundary conditions 
!	for Poisson's equation. Currently, we use only Dirichlet
!	(fixed=DBC) boundary conditions.    
!-------------------------------------------------------------------
subroutine SetBC( typeBC, valBC, il, ir, jl, jr, pot )

	implicit none
	
	!	INTERFACE PARAMETERS
	integer,          intent(in)    :: il, ir, jl, jr		! left/right indices specifying domain BC
	character(LEN=3), intent(in)    :: typeBC          	! type of boundary condition
	real(DP),         intent(in)    :: valBC           	! value at boundaries
	
	real(DP),         intent(inout) :: pot(:,:)		! potential
	
	!	INTERNAL PARAMETERS
	integer :: i, j
	
	!========================================================================
	
	if ( typeBC .eq. 'DBC' ) then
		write(*,*) "Type-BC: Dirichlet"
		
		! set boundary value
		do i = il,ir
			do j = jl,jr
				pot(i,j) = valBC
			enddo
		enddo

	elseif ( typeBC .eq. 'NBC' ) then
		write(*,*) "Type-BC: von Neumann"
	else
		write(*,*) "Error in subroutine SetBC. EXIT NOW!!!"
		stop
	endif

	return
end subroutine SetBC


!-----------------------------------------------------------------------------
!	This routine initializes the coefficients to solve Poisson
!	equation. These are input parameters for the subroutine sor.
!-----------------------------------------------------------------------------
subroutine SetPoissonCoeffs( c1, c2, c3, c4, c5 )

	implicit none
	
	!	INTERFACE PARAMETERS
	real(DP), intent(inout) :: c1(:,:), c2(:,:), c3(:,:), c4(:,:), c5(:,:)
	
	!	INTERNAL PARAMETERS
	integer  :: i, j
	real(DP) :: r, ratio, ratio2, r3
	
	!============================================================================

	do i = 2,zmax-1
		r = 0.0d0

   	do j = 2,rmax-1
   		r       = r + deltar
      ratio   = (deltar/deltaz)**2
      ratio2  = deltar/(2.0d0*r)
      r3      = (eps-1.0d0)*deltar/(kappa2*eps)
      
      !write(*,*) r, ratio, ratio2, r3
      !stop
      c1(i,j) =  ratio
      c2(i,j) =  ratio
      c3(i,j) =  1.0d0+ratio2
      c4(i,j) =  1.0d0-ratio2
      c5(i,j) = -2.0d0*(1.d0+ratio)
      
      if (j .eq. 2) then
      	c5(i,j) = c5(i,j) + 1.0d0 - ratio2
      	c4(i,j) = 0.0d0
      endif

      if (j .eq. NR+Ns) then
      	c3(i,j) = c3(i,j) + r3
      	c5(i,j) = c5(i,j) - r3
      endif

      if (i .eq. 2) then
      	c2(i,j) = 0.0d0
      	c1(i,j) = c1(i,j) + ratio
      endif
      
      if (i .eq. zmax-1) then
      	c1(i,j) = 0.0d0
      	c2(i,j) = c2(i,j) + ratio
      endif
      
      if ((j .ge. NR+Ns) .and. (i .le. NZ+1)) then
        c1(i,j) = 0.0d0
        c2(i,j) = 0.0d0
        c3(i,j) = 0.0d0
        c4(i,j) = 0.0d0
        c5(i,j) = 0.0d0
      endif

      if ((j .ge. NR+Ns) .and. (i .ge. zmax-NZ)) then
      	c1(i,j) = 0.0d0
        c2(i,j) = 0.0d0
        c3(i,j) = 0.0d0
        c4(i,j) = 0.0d0
        c5(i,j) = 0.0d0
      endif
		enddo
 	enddo

	return
end subroutine SetPoissonCoeffs


!-------------------------------------------------------------------------------
!	This routine calculates the smearing factors for charge smoothening
!	of the ring charge over the real-space grid.
!-------------------------------------------------------------------------------
subroutine SetSmearingFactors( smearing1, smearingl, smearingr, smearingz )

	implicit none
	
	!	INTERFACE PARAMETERS
	real(DP), intent(inout) :: smearing1(:,:), smearingl(:,:), &
	                           smearingr(:), smearingz(:,:)
	
	!	INTERNAL PARAMETERS
	integer  :: i, j
	real(DP) :: zz, sigma2r, sigma2z

	!============================================================================
	
 	smearing1 = 0.0d0 ; smearingl = 0.0d0 ; smearingr = 0.0d0 ; smearingz = 0.0d0

	factor  = float(m)*1.6e-1 / (4.0d0 * pi**2 * 0.66e-9 * sr * sz)
	sigma2r = 2.0d0 * sr**2
	sigma2z = 2.0d0 * sz**2 
 	r       = 0.0d0
   
  do j = 1,rmax
  	r            = r + deltar
    smearingr(j) = factor * exp(-(r-0.660d0)**2 / sigma2r)
  enddo

	!	set outermost LEFT tube boundary
  zz = position(1) - 3.0d0*deltaz

  do i = 1,zmax
  	zz = zz + deltaz
  	r  = 0.0d0
  
    do j = 1,rmax
      r = r + deltar

      smearing1(i,j) = smearing1(i,j)&
                     + factor*dexp( -(r-0.660d0)**2/sigma2r )&
                             *dexp( -(zz-position(1)+0.140d0)**2/sigma2z)

      smearing1(i,j) = smearing1(i,j)&
                     + factor*dexp( -(r-0.660d0)**2/sigma2r )&
                             *dexp( -(zz-position(1)+0.210d0)**2/sigma2z)

      smearing1(i,j) = smearing1(i,j)&
                     + factor*dexp( -(r-0.660d0)**2/sigma2r )&
                             *dexp(-(zz-position(1)+0.350d0)**2/sigma2z)

      smearing1(i,j) = smearing1(i,j)&
                     + factor*dexp( -(r-0.660d0)**2/sigma2r )&
                             *dexp(-(zz-position(1)+0.420d0)**2/sigma2z)

      smearingl(i,j) = smearingl(i,j)&
                     + factor*dexp( -(r-0.660d0)**2/sigma2r )&
                             *dexp(-(zz-position(l)-0.140d0)**2/sigma2z)

      smearingl(i,j) = smearingl(i,j)&
                     + factor*dexp( -(r-0.660d0)**2/sigma2r )&
                             *dexp(-(zz-position(l)-0.210d0)**2/sigma2z)

      smearingl(i,j) = smearingl(i,j)&
                     + factor*dexp( -(r-0.660d0)**2/sigma2r )&
                             *dexp(-(zz-position(l)-0.350d0)**2/sigma2z)

      smearingl(i,j) = smearingl(i,j)&
                     + factor*dexp( -(r-0.660d0)**2/sigma2r )&
                             *dexp(-(zz-position(l)-0.420d0)**2/sigma2z)
    enddo
  enddo

  zz = position(1) - 3.0d0*deltaz

  do i = 1,zmax
  	zz = zz + deltaz
  	
  	do n = 1,l
  		smearingz(i,n) = dexp(-(zz - position(n))**2 /sigma2z)
  	enddo
  enddo  
   
	return
end subroutine SetSmearingFactors


!-------------------------------------------------------------------
!	This routine initializes the charge (density) along the 
!	real-space grid (zmax,rmax). Further, an initial/guessed tube
!	charge --> input(n) can be included which is then smeared out
!	over the real-space grid.
!-------------------------------------------------------------------
subroutine SetChargeDensity( Lmax1, Lmax2, Nrings, rho_ext, rho )
   	
	implicit none
  
 	!	INTERFACE PARAMETERS
 	integer,  intent(in)    :: Lmax1, Lmax2 	!	max. domain length 
 	integer,  intent(in)    :: Nrings         ! number of rings
 	real(DP), intent(inout) :: rho_ext(:)     ! external/tube charge
 	real(DP), intent(inout) :: rho(:,:)       ! real-space charge
  
 	!	INTERNAL PARAMETERS
 	integer  :: i, j, n
 	real(DP) :: xjunk						! junk variable
 
 	!================================================================

	!rho_ext = 0.0d0
	rho     = 0.0d0
	
!	!	check whether potential and charge should be read
!	if ( present(mode) ) then
!		write(*,*) "Read potential and charge"
!  	do i = 1,Nrings
! 			read(11,*) xjunk, U0(i), rho_ext(i)
!  	enddo
!  endif
	
	! determine additional real-space charge density due to
	! tube charge on each ring n by SMEARING 
 	do i = 1,Lmax1
 		do j = 1,Lmax2
 			do n = 1,Nrings
 				! charge(i,j)=charge(i,j)+0.0040d0*smearingr(j)*smearingz(i,n)
   			rho(i,j) = rho(i,j) + rho_ext(n)*smearingr(j)*smearingz(i,n)
 			enddo

 			! charge(i,j)=charge(i,j)+0.0040d0*smearing1(i,j)
  		rho(i,j) = rho(i,j) + rho_ext(1)*smearing1(i,j)

 			! charge(i,j)=charge(i,j)+0.0040d0*smearingl(i,j)
   		rho(i,j) = rho(i,j) + rho_ext(Nrings)*smearingl(i,j)
 		enddo
 	enddo

	return	
end subroutine SetChargeDensity


!--------------------------------------------------------------
!	This routine updates the (total) charge at each energy
!--------------------------------------------------------------
subroutine UpdateCharge_Device( rho, Nrings, G, dE, mult )

	implicit none

	!	INTERFACE PARAMETERS
	integer,  intent(in)    :: Nrings		! number of tube rings
	real(DP), intent(in)    :: dE				! energy increment
	real(DP), intent(in)    :: mult			! multiplication factor
	real(DP), intent(in)    :: G(:,:)		! specific Green function
	real(DP), intent(inout) :: rho(:)		!	(total) charge density
	
	!	INTERNAL PARAMETERS
	integer :: i
	
	!==============================================================

	do i = 1,Nrings
 		rho(i) = rho(i) + mult*G(i,i)/pi * dE
 	enddo

	return
end subroutine UpdateCharge_Device


!----------------------------------------------------------------------
!	This routine maps the charge on the device grid onto the real-space
!	grid for use in Poisson's equation.
!----------------------------------------------------------------------
subroutine MapCharge( rho_dev, N, rho_RP, Lmax1, Lmax2, Zel )

	implicit none
	
	!	INTERFACE PARAMETERS
	integer, 	intent(in)  :: Lmax1, Lmax2		!	device dimensions in real-space
	integer, 	intent(in)  :: N			       	!	number of rings
	real(DP), intent(in)  :: Zel						!	number of background electrons per site
	real(DP), intent(in)  :: rho_dev(:)			! device charge
	real(DP), intent(out) :: rho_RP(:,:)	  ! real-space charge
	
	!	INTERNAL PARAMETERS
	integer :: i, j, k
	
	!==================================================================
	
	rho_RP = 0.0d0
	
	do i = 1,Lmax1
  	do j = 1,Lmax2
			!	smear (total) tube charge along the real-space grid
      do k = 1,N
       		rho_RP(i,j) = rho_RP(i,j) &
       		            + (-rho_dev(k) + Zel) * 2.0d0/m * smearingr(j)*smearingz(i,k)
 			enddo

			!	DEBUG-DIEGO: why is there are 2nd smearing of the charge ???
      rho_RP(i,j) = rho_RP(i,j) + (-rho_dev(1) + Zel) * 2.0d0/m * smearing1(i,j)
      rho_RP(i,j) = rho_RP(i,j) + (-rho_dev(N) + Zel) * 2.0d0/m * smearingl(i,j)
		enddo
	enddo

	return
end subroutine MapCharge


!------------------------------------------------------------------------
!	This routine checks whether the convergence criteria for the quantity 
!	Aold/Anew is met and if not performs an update of the "new" A.
!
!	A can be either --> potential or charge
!------------------------------------------------------------------------
subroutine CheckSCF( Aold, Anew, alpha, N, dAeps, status_scf )

	implicit none
	
	!	INTERFACE PARAMETERS
	integer,  intent(in)    :: N
	integer,  intent(inout) :: status_scf		!	default = 0
	real(DP), intent(in)    :: alpha, dAeps
	real(DP), intent(inout) :: Aold(:)
	real(DP), intent(in)    :: Anew(:)

	!	INTERNAL PARAMETERS
	integer  :: i, imax
	real(DP) :: dA, dAi, dAmax
	real(DP) :: sumAold
	
	!==============================================================
	
	!write(*,*) "alpha=",alpha
	
	dA      = 0.0d0
	dAi     = 0.0d0
	dAmax   = 0.0d0
	sumAold = 0.0d0
	
	!	sum up deviation in quantity A for all sites
	do i = 1,N
		sumAold = sumAold + abs(Aold(i))
		dAi     = abs(Anew(i) - Aold(i))
		dA      = dA + dAi
		
		if (dAi .gt. dAmax) then
			dAmax = dAi
			imax  = i
		endif
	enddo
	
	dA = dAmax
	!dA = dA / float(N)
	!dA = dA / sumAold
	
	!	compare deviation of A with allowed target deviation
	if (dA .gt. dAeps) then
		write(*,*) "dA  = ", dA, "  dAmax = ", dAmax, "  imax = ", imax
		
		!	mix old and new A's
		!Anew = alpha*Aold + (1.0d0-alpha)*Anew  --> OLD VERSION of MIXING
		!Aold = alpha*Aold + (1.0d0-alpha)*Anew
		call PulayMix( Aold, Anew, N, numPulay, it, alpha )
	else
		status_scf = 1
	endif
	
	return
end subroutine CheckSCF
