!Copyright (year first published) Sandia Corporation. 
!Under the terms of Contract DE-AC04-94AL85000, there 
!is a non-exclusive license for use of this work by or 
!on behalf of the U.S. Government. Export of this program 
!may require a license from the United States Government.
!-----------------------------------------------------------------------------------------
!	NOTE: This code is CNT-final-2.f90 and corresponds essentially to the
!				the CNT-v7.f90 version where the UNnecessary parts have been 
!				removed.
!
!		    DATE: 04/09/2007 - 04/11/2007
!
!	NOTATION: 
!	--> GRr, GRi : real/imaginary Retarded Green functions
!	--> Sai      : imaginary "less" Self-energy (inflow) == SigIn
!-----------------------------------------------------------------------------------------
program CNTFET

	implicit none

	include "const-LM.f90"

!integer           :: Ne		 			!	number of energy grid points along complex contour
!real(DP), pointer :: Egrid(:), dE(:)	!	energy grid used for contour integration 	

	! l is the number of rings, zmax = NR + 2(NR/4) + 1 
	! deltaz must be 0.07 nm
	
	!	tube parameters --> geometry
	integer, parameter :: zmax = 411, rmax = 300, l = 272
	integer, parameter :: NP = 400, Nxi = 61
	integer, parameter :: iemax = 100
	
  ! tube parameters
  integer,  parameter :: m = 17, index = 6
  real(DP), parameter :: acc = 0.14e-9, Vpi = 2.5d0, eg = 0.275d0
  
  ! Fermi level and temperature: [eV]
  real(DP), parameter :: kT = 0.0257d0, Ef = -1.0d0, VDS = 0.0d0
	
	! electrostatic parameters, doping, charge smoothening
	integer,  parameter :: iavg = 4
  real(DP), parameter :: dielectric = 1.0d0, epsilon = 8.85e-12, el = 1.6e-19
  real(DP), parameter :: doping = 0.0d0, s = 0.3d0
  real(DP), parameter :: sr = 0.06d0, sz = 0.14d0
	real(DP), parameter :: frac = 0.75d0, rjac = 0.99994d0, eps = 3.90d0
	real(DP), parameter :: deltar = 0.03d0, deltaz = 0.07d0, kappa2 = 0.0085d0
  real(DP), parameter :: dvmax = 5.0e-4, drmax = 1.0e-4
  !	back-ground total charge: per ring/per atom
  real(DP), parameter :: Zval = 1.0d0
	
!     parameter(zmax=159,rmax=100,l=104,NP=400,Nxi=61)
!     parameter(zmax=267,rmax=300,l=176,NP=400,Nxi=61)
!     parameter(zmax=291,rmax=300,l=192,NP=400,Nxi=61)
!     parameter(zmax=333,rmax=300,l=220,NP=400,Nxi=61)
!     parameter(zmax=363,rmax=300,l=240,NP=400,Nxi=61)
!     parameter(zmax=363,rmax=100,l=240,NP=400,Nxi=109)
!			parameter(zmax=411,rmax=300,l=272,NP=400,Nxi=61)
!     parameter(zmax=465,rmax=100,l=308,NP=400,Nxi=61)
!     parameter(zmax=543,rmax=100,l=360,NP=400,Nxi=61)
!     parameter(zmax=837,rmax=100,l=556,NP=400,Nxi=61)
     
  integer   :: i, j, k, n, it
  integer   :: site(l), imax, NVG, ivg
	!	coefficient matrices for Poisson's equation defined
	!	on real-space grid (zmax,rmax)
   real(DP) :: a(zmax,rmax), b(zmax,rmax), c(zmax,rmax), &
               d(zmax,rmax), e(zmax,rmax)
   real(DP) :: kappa
   real(DP) :: r
   integer  :: NR, NZ, Ns, NG
   real(DP) :: dx, LL, Lz, Lr, me
   real(DP) :: u(zmax,rmax), charge(zmax,rmax), vin(zmax), vreal(l), dz(zmax)
	 real(DP) :: diff, vtemp
   real(DP) :: VG, VS, VD, VGS ! applied bias:  gate, source, drain
   real(DP) :: lambda
   real(DP) :: tmp
   real(DP) :: table1(20000)
   real(DP) :: ring(l), position(l)
   real(DP) :: phi, si, co, coupl
   !	real/imaginary self-energy for left/right contacts
   real(DP) :: SigmarL, SigmaiL, GammaL, GammaR, Gr, Gi
   real(DP) :: SigmarR, SigmaiR
   !real(DP) :: df, da, P, cd(iemax)
   integer  :: status_scf
   real(DP) :: newcharge(zmax,rmax), trin(l)
   real(DP) :: pot(l), rin(l), input(l)
   real(DP) :: Uold(l), Unew(l)
   !real(DP) :: rold(l), rnew(l)
   real(DP) :: U0(l), xtmp(l)
   real(DP) :: Sigmar, Sigmai
   !	2D smearing function for charge smoothening
   real(DP) :: smearing1(zmax,rmax), smearingl(zmax,rmax)
   real(DP) :: smearingr(rmax), smearingz(zmax,l)
   !	real/imaginary Green/Self-Energy function: device
   real(DP) :: GRr(l,l), GRi(l,l)
   !real(DP) :: GHr(l,l), GHi(l,l)		! retarded/advanced
   real(DP) :: Gar(l,l), Gai(l,l)
   !real(DP) :: Sar(l,l), Sai(l,l)   ! lesser G<, Sig<
   real(DP) :: Er, Ei, dEr, dEi
   real(DP) :: factor, aa, bb
   integer  :: npoles
   real(DP) :: DVG, VGMIN, VGMAX
   !	variables for contour integration
   integer  :: iEr, iEi, nEr, nEi
   real(DP) :: Emin0, Emax0, Ei0, Ermin, Ermax, Eimin, Eimax
   real(DP) :: currentL(1000), currentR(1000), current, fermiL, fermiR
   !	real/imaginary left/right connected Green function: contacts
   !real(DP) :: gtLr(l), gtLi(l), gtRr(l), gtRi(l)
   ! left/right current
   real(DP) :: Il, Ir
   ! 
   character(len=5)             :: chVDS
   character(len=5)             :: chVG
   character(len=2)             :: chCH
   character(len=4)             :: chRG
   character(len=10), parameter :: string = '0123456789'
   character(len=3)             :: chD

	!	Pulay mixing
	integer, parameter :: numPulay = 5
	real(DP)           :: PulayOld(l,numPulay), PulayNew(l,numPulay)

	!====================================================================
	
	!	set terminal voltages --> source is used a GROUND
   VS = 0.0d0
   VD = VDS

   VGMIN = 0.90d0 + VS
   VGMAX = 0.90d0 + VS

   if (NVG .ne. 0) then 
			DVG = (VGMAX-VGMIN) / NVG
   else
   		DVG = 0.0d0
   endif
   
   if (mod(l,4) .ne.0 )         print*,'ohoh l'
   if (zmax .ne. (l+2*(l/4)+1)) print*,'ohoh zmax'
   if (deltaz .ne. 0.07)        print*,'ohoh deltaz'
   if (VS .ne. 0.0)             print*,'ohoh VS'

   print*, VG, VS, VD
	
   imax = 200

	 ! setup file name to store data
	 call SetFileName( chCH, chVG, chVDS, chD )
   open(40,file='cyl_Lg='//chCH//'nm_Rg='//chD//'nm_Vds='//chVDS//'_Vgs='//chVG//'')
   open(40,file='cyl_Lg='//chCH//'nm_Rg='//chD//'nm_Vds='//chVDS//'_Vgs='//chVG//'_current')

   Lz = 0.0d0
   NR = int( 0.660d0/deltar )
   NZ = int( Nxi )
   Ns = int( s/deltar )
   NG = zmax - 2*NZ - 1
 
   print*, NR, NZ, Ns, NG

	!	 calculate parameters
  me = 2.0d0 * Vpi * dcos( pi*index/m )

	!	 calculate positions of rings
	call SetRingPositions( position, site, l )
	
	!	set coefficients to solve Poisson's equation in
	!	cylindrical coordinates
	call SetPoissonCoeffs( a, b, c, d, e )

	!	 calculate smearing functions
	call SetSmearingFactors( smearing1, smearingl, smearingr, smearingz )
	
  !----------------------------------------------------------------
  !	GATE-BIAS LOOP
  !----------------------------------------------------------------
  VG = VGMIN - DVG
    
  Gate_Bias: do ivg = 0,NVG
 
    VG = VG + DVG
    print*, 'VG=',VG
 
 		!	intialize real-space charge density
  	call SetChargeDensity( zmax, rmax, l, input, charge )
 
  	!	set boundary conditions: electrostatic potential
  	write(*,*) "Set Boundary Conditions: Poisson"
		u = 0.0d0
  	call SetBC( 'DBC', VS, 1,       NZ+1,    NR+Ns, rmax, u )	!	source
  	call SetBC( 'DBC', VD, zmax-NZ, zmax,    NR+Ns, rmax, u )	!	drain
  	call SetBC( 'DBC', VG, NZ+2,    NZ+NG+1, rmax,  rmax, u )	!	gate
 
  	!	get suitable guess --> charge density: rold <==> charge
  	write(*,*) "Guess Initial Charge to obtain Guess for Potential."
		charge = 0.0d0
		call sor( a, b, c, d, e, -charge * deltar**2 * 1e-18/epsilon, &
							u, zmax, rmax, rjac )
    !	map potential: real-space grid --> device grid
    do i = 1,l
    	Uold(i) = u(site(i),NR)
    enddo
 
 		PulayOld = 0.0d0 ; PulayNew = 0.0d0
 
		!================================================================================
		!	                     S C F  -  C Y C L E :  S T A R T
		!================================================================================
		!	initialize convergence switch
		status_scf = 0
		
    scf: do it = 1,imax
			write(*,*) "#Iteration: ", it
			
			!--------------------------------------------------------------
			!					       T R A N S P O R T  -  N E G F
			!--------------------------------------------------------------   
			write(*,*) "Solve Transport: U --> Rho"
			!	reset tube charge
      rin = 0.0d0
      
      !-----------------------------------------------------------------------
 			!	             I.  E Q U I L I B R I U M    -->      GR
 			!-----------------------------------------------------------------------
 			!	Calculate (total) charge of the device (tube) by employing a 
 			!	contour integration in the complex energy plane.
 			!	The complex path is a rectangle.
 			!-----------------------------------------------------------------------
  		!	-->	set energy grid (Er, Ei) with Ei = const.
   		Emin0  = -10.0d0
   		Emax0  = Ef
   		Ei0    = 2.0d0
   		npoles = (Ei0 - pi*kT) / (2.0*pi*kT) + 1
 
  		!	-->	path 1: upper side, parallel to real axis --> going left
  		nEr   = 201
   		Ermin = Emin0             - 20.0*kT
   		Ermax = min(Ef-VS, Ef-VD) - 20.0*kT
    	dEr   = (Ermin-Ermax)/(nEr-1)
      Ei    = float(npoles)*2.0*pi*kT
 
  		iEr = 1
  		Er  = Ermax + (iEr-1)*dEr
  		call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
      call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
      call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
      call UpdateCharge_Device( rin, l, GRi, dEr, 1.0d0 )
      
      iEr = nEr
  		Er  = Ermax + (iEr-1)*dEr
  		call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
      call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
      call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
      call UpdateCharge_Device( rin, l, GRi, dEr, 1.0d0 )
      
   		do iEr = 2,nEr-1
  			Er = Ermax + (iEr-1)*dEr
  			call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
      	call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
  			call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
        call UpdateCharge_Device( rin, l, GRi, dEr, 2.0d0 )
    	enddo
  			
   		!	--> path 2: right side, parallel to imaginary axis --> going up
   		nEi   = 701
      Eimin = eta_contour
      Eimax = float(npoles)*2.0*pi*kT
      dEi   = (Eimax - Eimin)/(nEi-1)
      Er    = min(Ef-VS, Ef-VD) - 20.0*kT
 
  		iEi = 1
     	Ei  = Eimin + (iEi-1)*dEi
  		call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
      call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
  		call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
   		call UpdateCharge_Device( rin, l, GRr, dEi, 1.0d0 )
     		
     	iEi = nEi
  		Ei  = Eimin + (iEi-1)*dEi
  		call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
      call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
  		call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
  		call UpdateCharge_Device( rin, l, GRr, dEi, 1.0d0 )
  			
   		do iEi = 2,nEi-1
        Ei = Eimin + (iEi-1)*dEi
     		call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
        call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
        call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
  			call UpdateCharge_Device( rin, l, GRr, dEi, 2.0d0 )
   		enddo
 
      !	--> path 3: left side, parallel to imaginary axis --> going down
      nEi   = 101
      Eimin = eta_contour
      Eimax = npoles*2.0*pi*kT
      dEi   = (Eimin-Eimax)/(nEi-1)
      Er    = Emin0 - 20.0*kT
 
  		iEi = 1
  		Ei  = Eimax + (iEi-1)*dEi
  		call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
      call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
      call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
      call UpdateCharge_Device( rin, l, GRr, dEi, 1.0d0 )
      
      iEi = nEi
  		Ei  = Eimax + (iEi-1)*dEi
  		call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
      call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
  		call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
  		call UpdateCharge_Device( rin, l, GRr, dEi, 1.0d0 )
        
   		do iEi = 2,nEi-1
   			Ei = Eimax + (iEi-1)*dEi
      	call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
        call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
        call GetGR( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, GRr, GRi )
 			 	call UpdateCharge_Device( rin, l, GRr, dEi, 2.0d0 )
   		enddo
 
  		!-----------------------------------------------------------------------
 			!											II. NON-EQUILIBRIUM --> G<
 			!-----------------------------------------------------------------------
 			!	Calculate the charge due to the difference in the chemical potentials
 			!	through direct intergration along the real axis.
 			!-----------------------------------------------------------------------
 			nEr   = 401
      Ermin = min(Ef-VS,Ef-VD) - 20.0*kT
      Ermax = max(Ef-VS,Ef-VD) + 20.0*kT
      dEr   = (Ermax-Ermin)/(nEr-1)
      Ei    = eta_contour
      
      iEr = 1
      Er  = Ermin + (iEr-1)*dEr
      call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
  	  call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
      call GetGn( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, Gai, Il, Ir )
      currentL(iEr) = Il
      currentR(iEr) = Ir
      call UpdateCharge_Device( rin, l, Gai, dEr, 0.5d0 )
      
      iEr = nEr
      Er  = Ermin + (iEr-1)*dEr
      call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
  	  call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR )
      call GetGn( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, Gai, Il, Ir )
      currentL(iEr) = Il
      currentR(iEr) = Ir
      call UpdateCharge_Device( rin,  l, Gai, dEr, 0.5d0 )
      
  		do iEr = 2,nEr-1
 	      Er = Ermin + (iEr-1)*dEr
  			call leads( 'L', Er, Ei, Uold(1), SigmarL, SigmaiL, GammaL )
  	    call leads( 'R', Er, Ei, Uold(l), SigmarR, SigmaiR, GammaR ) 
  			call GetGn( Er, Ei, Uold, l, SigmarL, SigmaiL, SigmarR, SigmaiR, Gai, Il, Ir )		
        currentL(iEr) = Il
        currentR(iEr) = Ir
        call UpdateCharge_Device( rin,  l, Gai, dEr, 1.0d0 )
  		enddo      
      
      !	monitor current during simulation
      current = 0.0d0
      do iEr = 1,nEr
      	current = current + currentR(iEr)*dEr
      enddo
  
  		!	print current in microamps
      print*,'current=', current*1.55e-4/1E-6
      write(98,*) it, current
      
      !	get respective real-space charge density
			call MapCharge( rin, l, charge, zmax, rmax, Zval )
			
			!------------------------------------------------------------------
			!					             P O I S S O N
			!------------------------------------------------------------------
			write(*,*) "Solve Poisson: Rho --> U"
			!	solve Poisson
    	call sor( a, b, c, d, e, -charge * deltar**2 * 1e-18/epsilon, &
               	u, zmax, rmax, rjac )
      !	map potential: real-space grid --> device grid
      do i = 1,l
      	Unew(i) = u(site(i),NR)
      enddo
      
			!------------------------------------------------------------------
			!                C H E C K   C O N V E R G E N C E
			!------------------------------------------------------------------
			!	--> update Unew through linear mixing
			write(*,*) "Check Convergence & Mix Potentials."
			call CheckSCF( Uold, Unew, frac, l, dvmax, status_scf )
			
			if ( status_scf .eq. 1 ) then
				write(*,*) "Converged at Iteration ",it, " Exit SCF !!!"
				exit scf
			endif
			
		enddo scf
		!================================================================================
		!	                     S C F  -  C Y C L E :  E N D
		!================================================================================
    
    write(*,*) "Write Tube Data: Position, Potential, Charge"
    open(60, file="U0.dat", action="write")
    open(61, file="Rho0.dat", action="write")
   	do i = 1,l
   		write(60,*) real( position(i) ), real( Unew(i) )
   		write(61,*) real( position(i) ), real( (-rin(i)+Zval)*2.0d0/float(m) )
   	enddo
    close(60)
    close(61)
    
    do i = 1,l
  		!	write(50,*)position(i),real(vreal(i)),real(charge(i,NR))
     	write(40,*) real( position(i) ), real( Unew(i) ), real( (-rin(i)+Zval)*2.0d0/float(m) )
     	write(41,*) real( position(i) ), real( (-trin(i)+Zval)*2.0d0/float(m) )
  	enddo
    
    write(*,*) "Write Real Space Data: Charge"
    do i = 1,zmax
    	write(50,*) i, real( charge(i,NR) * deltar**2 * 1e-18/epsilon )
    enddo
  enddo Gate_Bias
	!--------------------------------------------------------------------------
  !	END-GATE-BIAS LOOP
  !--------------------------------------------------------------------------
	
	stop
      
	contains
	
	include "io-LM.f90"
	include "lattice-LM.f90"
	include "charge-LM.f90"
	include "solver-LM.f90"
	include "sigma-LM.f90"
	include "green-LM.f90"
	
	
	!--------------------------------------------------------------------------
	!	This routine performs a best-fit mixing (Pulay mix) of the old
	!	and new density matrices from the previous iterations to achieve
	!	a faster convergence.
	!
	!	NOTE: This version is an implementation for orthogonal TB, but it's 
	!				specific implementation is consistent with the non-orthogonal
	!				basis implementation.
	!--------------------------------------------------------------------------
	subroutine PulayMix( Pold, Pnew, N, numPul, iter, alpha )
		
		implicit none
		
		!	INTERFACE PARAMETERS
		integer, 	intent(in) 		:: N				!	number of rings/sites
		integer, 	intent(in) 		:: iter			!	iteration number
		integer, 	intent(in) 		:: numPul		!	number of DM's used for mixing
		real(DP), intent(in)    :: alpha		!	mixing factor
		real(DP), intent(in) 		:: Pnew(:)	!	new DM
		real(DP), intent(inout)	:: Pold(:)	!	old DM
		
		!	INTERNAL PARAMETERS
		integer               :: i, iP, jP, nP, idPulay
		real(DP)              :: PnewNext(N), PoldNext(N) 
		real(DP)              :: dPi(N), dPj(N)
		real(DP), allocatable :: D(:,:), B(:,:), beta(:), ones(:)
	
		!=========================================================================
	
		!	store new/old DM's to built up DM-history
		if ( mod(iter,numPul) .eq. 0 ) then
			idPulay = numPul
		else
			idPulay = mod(iter,numPul)
		endif
		
		write(*,*) "idPulay =",idPulay
		write(*,*) "Alpha:",alpha
		PulayOld(:,idPulay) = Pold
		PulayNew(:,idPulay) = Pnew
		write(*,*) "Pold(N/2):",Pold(N/2)
		write(*,*) "Pnew(N/2):",Pnew(N/2)
		write(*,*) "Pold(1)  :",Pold(1)
		write(*,*) "Pnew(1)  :",Pnew(1)
	
		!	get new density matrix for next iteration
		if (iter .lt. 2) then
			write(*,*) "iter =", iter
			Pold = alpha*Pold + (1.0d0-alpha)*Pnew	
		else
		
			!	get how many DM's are already stored and allocate buffer
			nP = min(iter,numPul)
			allocate( D(nP,nP), B(nP,nP), beta(nP), ones(nP) )
			D = 0.0d0 ; B = 0.0d0 ; beta = 0.0d0 ; ones = 1.0d0
			
			write(*,*) "nP =", nP
			
			do iP = 1,nP
				B(iP,iP) = 1.0d0	
			enddo
			
			! get deviations in old/new DM's from the #nP previous iterations
			do iP = 1,nP
				dPi(:) = PulayNew(:,iP) - PulayOld(:,iP) 
				
				do jP = iP,nP
					dPj(:) = PulayNew(:,jP) - PulayOld(:,jP)
					
					do i = 1,N
						D(iP,jP) = D(iP,jP) + dPi(i) * dPj(i)
					enddo
					
					D(jP,iP) = D(iP,jP)
				enddo
			enddo
		
			!	solve libear system: D * D^{-1} = I = B
			call SolveLGS( D, B, np )
			beta = matmul(B, ones)
			beta = beta / sum(beta) 
		
			!	get Pulay mixed old and new DM's and mix them linearly
			PoldNext = 0.0d0 ; PnewNext = 0.0d0
			do iP = 1,nP
				PoldNext(:) = PoldNext(:) + beta(iP) * PulayOld(:,iP)
				PnewNext(:) = PnewNext(:) + beta(iP) * PulayNew(:,iP)
			enddo
			
			!	mix best-fit DM's linearly
			Pold(:) = alpha*PoldNext(:) + (1.0d0-alpha)*PnewNext(:)
			
			!	deallocate buffer
			deallocate( D, B, beta, ones )
		endif
	
		return
	end subroutine PulayMix
	
	
	!----------------------------------------------------------------
	!	This routine calculates the inverse of matrix: AX=B, B=I
	!----------------------------------------------------------------
	subroutine SolveLGS( A, B, n )
	
		implicit none
		
		!	INTERFACE PARAMETERS
		integer,  intent(in)    :: n
		real(DP), intent(in)    :: A(:,:)
		real(DP), intent(inout) :: B(:,:)
	
		!	INTERNAL PARAMETERS
		integer :: info, ipiv(n)
		
		!==============================================================
		
		!	LU factorization of A
		!write(*,*) "dgetrf"
		call dgetrf( n, n, A, n, ipiv, info )
		if ( info .ne. 0 ) then
			write(*,*) "Error in Subroutine SolveLGS: DGETRF."
			stop
		endif
		
		!	solve linear system
		!write(*,*) "dgetrs"
		call dgetrs( 'N', n, n, A, n, ipiv, B, n, info )
		if ( info .ne. 0 ) then
			write(*,*) "Error in Subroutine SolveLGS: DGETRS."
			stop
		endif
		
		return
	end subroutine SolveLGS
	
end program CNTFET
