
///////////////////////////////////////////////////////////////////////////////
//
//  tIonization_ParticleStrike.cpp
//
///////////////////////////////////////////////////////////////////////////////

// Charon
#include "Charon_Ionization_ParticleStrike_impl.hpp"

// Teuchos
#include "Teuchos_UnitTestHarness.hpp"

#include "Panzer_CommonArrayFactories.hpp"


namespace charon
{
  /////////////////////////////////////////////////////////////////////////////
  // evaluator for storing the reference values
  // the reference values generated by the reference evaluator are precomputed
  // in general the user is responsible for providing this data
  /////////////////////////////////////////////////////////////////////////////
  template<typename EvalT, typename Traits>
  class RefFieldEvaluator
    :
    public PHX::EvaluatorWithBaseImpl<Traits>,
    public PHX::EvaluatorDerived<EvalT, Traits>
  {
  public:
    RefFieldEvaluator(const Teuchos::ParameterList& p);

    void evaluateFields(typename Traits::EvalData d);
    
  private:
    using ScalarT = typename EvalT::ScalarT;
    PHX::MDField<ScalarT,panzer::Cell,panzer::BASIS> fieldValue;
  }; 

  template<typename EvalT, typename Traits>
  RefFieldEvaluator<EvalT, Traits>::
  RefFieldEvaluator(const Teuchos::ParameterList& p)
  {
    // read from parameters
    const std::string name = p.get<std::string>("Name");
    Teuchos::RCP<panzer::PureBasis> basis 
      = p.get< Teuchos::RCP<panzer::PureBasis> >("Basis");

    fieldValue = PHX::MDField<ScalarT,panzer::Cell,panzer::BASIS>(name, basis->functional);
    this->addEvaluatedField(fieldValue);
  
    std::string n = "RefFieldEvaluator: " + name;
    this->setName(n);
  }

  template<typename EvalT, typename Traits>
  void RefFieldEvaluator<EvalT, Traits>::
  evaluateFields(typename Traits::EvalData  /* workset */)
  { 
    // generate reference values to compare to
    for(int cell=0;cell<fieldValue.extent_int(0);cell++) 
      for(int pt=0;pt<fieldValue.extent_int(1);pt++) 
	fieldValue(cell,pt) = -67.7571;
  }


  /////////////////////////////////////////////////////////////////////////////
  // evaluator for dependent fields
  // this evaluator initialize/set dynamically all dependent fields of the
  // evaluator under test; the structure and the names of the dependent fields 
  // have to match that of the dependent fields used in the test evaluator
  /////////////////////////////////////////////////////////////////////////////
  template<typename EvalT, typename Traits>
  class DepFieldsEvaluator
    :
    public PHX::EvaluatorWithBaseImpl<Traits>,
    public PHX::EvaluatorDerived<EvalT, Traits>
  {
  public:
    DepFieldsEvaluator(const Teuchos::ParameterList& p);

    void evaluateFields(typename Traits::EvalData d);

  private:
    using ScalarT = typename EvalT::ScalarT;
    std::vector<Teuchos::RCP<PHX::MDField<ScalarT,panzer::Cell,panzer::BASIS>>> fieldValues;
  }; 

  template<typename EvalT, typename Traits>
  DepFieldsEvaluator<EvalT, Traits>::
  DepFieldsEvaluator(const Teuchos::ParameterList& p)
  {
    // read from parameters
    Teuchos::RCP<panzer::PureBasis> basis = 
      p.get< Teuchos::RCP<panzer::PureBasis> >("Basis");
    Teuchos::RCP<std::vector<std::string>> dep_names =
      p.get< Teuchos::RCP<std::vector<std::string>> >("Names");

    // dynamically allocate an array with fields used as dependent fields 
    // in the evaluator under test; the names of the fields are passed as
    // input in the parameter list
    std::vector<std::string>::iterator it;
    for(it = (*dep_names).begin(); it != (*dep_names).end(); ++it) {
      fieldValues.push_back(Teuchos::rcp( 
        new PHX::MDField<ScalarT,panzer::Cell,panzer::BASIS>(*it, basis->functional)));
      this->addEvaluatedField(*fieldValues.back());
      
    }

    this->setName("DepFieldsEvaluator");
  }

  template<typename EvalT, typename Traits>
  void DepFieldsEvaluator<EvalT, Traits>::
  evaluateFields(typename Traits::EvalData  /* workset */) { 
    for(std::size_t fi=0; fi<fieldValues.size(); fi++) {
      // fill up the test dependent fields with values
      PHX::MDField<ScalarT,panzer::Cell,panzer::BASIS> val = *fieldValues[fi];
      for(int cell=0; cell<val.extent_int(0); cell++) 
	for(int pt=0; pt<val.extent_int(1); pt++) 
	  val(cell,pt) = 0.0;
    } 
  }
  


  /////////////////////////////////////////////////////
  // map test workset to coordinates 
  /////////////////////////////////////////////////////
  static void mapTestWorkset(int numCells, int numVerts, int dim,
                      Teuchos::RCP<panzer::Workset> workset) {
    panzer::MDFieldArrayFactory af("",true);
    
    workset->cell_vertex_coordinates = 
      af.buildStaticArray<double,panzer::Cell,panzer::NODE,panzer::Dim>(
          "coords",numCells,numVerts,dim);
    // describe the domain used for test evaluator
    panzer::Workset::CellCoordArray coords = workset->cell_vertex_coordinates;
    // cell 0
    coords(0,0,0) = 0.0; coords(0,0,1) = 0.00; coords(0,0,2) = 0.0;
    coords(0,1,0) = 1.0; coords(0,1,1) = 0.00; coords(0,1,2) = 0.0;
    coords(0,2,0) = 1.0; coords(0,2,1) = 0.00; coords(0,2,2) = -0.5;
    coords(0,3,0) = 1.0; coords(0,3,1) = 0.25; coords(0,3,2) = -0.5;
    coords(0,4,0) = 0.0; coords(0,4,1) = 0.25; coords(0,4,2) = -0.5;
    coords(0,5,0) = 0.0; coords(0,5,1) = 0.00; coords(0,5,2) = -0.5;
    coords(0,6,0) = 0.0; coords(0,6,1) = 0.25; coords(0,6,2) = 0.0;
    coords(0,7,0) = 1.0; coords(0,7,1) = 0.25; coords(0,7,2) = 0.0;
    // cell 1 
    coords(1,0,0) = 0.0; coords(1,0,1) = 0.25; coords(1,0,2) = 0.0;
    coords(1,1,0) = 1.0; coords(1,1,1) = 0.25; coords(1,1,2) = 0.0;
    coords(1,2,0) = 1.0; coords(1,2,1) = 0.25; coords(1,2,2) = -0.5;
    coords(1,3,0) = 1.0; coords(1,3,1) = 0.50; coords(1,3,2) = -0.5;
    coords(1,4,0) = 0.0; coords(1,4,1) = 0.50; coords(1,4,2) = -0.5;
    coords(1,5,0) = 0.0; coords(1,5,1) = 0.25; coords(1,5,2) = -0.5;
    coords(1,6,0) = 0.0; coords(1,6,1) = 0.50; coords(1,6,2) = 0.0;
    coords(1,7,0) = 1.0; coords(1,7,1) = 0.50; coords(1,7,2) = 0.0;
  }

 

  // You can create as many unit tests in this collection as you like.  For
  // any functionality of the class that needs to be tested, create another
  // unit test.  Feel free to delete this comment.

  /////////////////////////////////////////////////////////////////////////////
  //
  //  TEUCHOS_UNIT_TEST(Ionization_ParticleStrike, eval)
  //
  /////////////////////////////////////////////////////////////////////////////
  TEUCHOS_UNIT_TEST(Ionization_ParticleStrike, eval3D)
  {
    // The test creates a dummy 3D mesh with 2 cells used to test the  
    // 3D evaluation (computational part) of Ionization_ParticleStrike evaluator.
    // Size and values of the evaluated field under test are compared to
    // a reference evaluator/field with precomputed values. Also, the name of the 
    // tes evaluated field is checked.

    // Now insert whatever you need to build the unit test.  See
    // https://cee-gitlab.sandia.gov/tcad-charon/wikis/Creating-a-Unit-Test
    // for details on what Teuchos macros you can use here.
    
    using charon::Ionization_ParticleStrike; // evaluator under test

    using Teuchos::RCP;
    using Teuchos::rcp;
    using Teuchos::rcp_dynamic_cast;
    
    typedef panzer::Traits::Residual EvalType;
    typedef Sacado::ScalarValue<typename EvalType::ScalarT> SV;
   
    // create a test workset and map it to space 
    int numCells = 2, numVerts = 8, dim = 3;
    RCP<panzer::Workset> workset = rcp(new panzer::Workset);
    mapTestWorkset(numCells,numVerts,dim,workset);

    // build workset quadrature values
    RCP<shards::CellTopology> topo = rcp(
	new shards::CellTopology(shards::getCellTopologyData< shards::Hexahedron<8> >()));
    int cubatureDegree = 2;
    panzer::CellData cellData(numCells,topo);
    RCP<panzer::IntegrationRule> ir = 
      rcp(new panzer::IntegrationRule(cubatureDegree,cellData));
    RCP<panzer::IntegrationValues2<double> > ival = 
      rcp(new panzer::IntegrationValues2<double>("",true));
    ival->setupArrays(ir);
    ival->evaluateValues(workset->cell_vertex_coordinates);
  
    // build basis values
    std::string basisType = "HGrad";
    RCP<panzer::PureBasis> basis = 
      rcp(new panzer::PureBasis(basisType,1,cellData));
    RCP<panzer::BasisIRLayout> basisLayout = 
      rcp(new panzer::BasisIRLayout(basis,*ir));
    RCP<panzer::BasisValues2<double> > bval = 
      rcp(new panzer::BasisValues2<double>("",true,true));
    bval->setupArrays(basisLayout);
    bval->evaluateValues(ival->cub_points,ival->jac,ival->jac_det,ival->jac_inv,
			 ival->weighted_measure,workset->cell_vertex_coordinates);
		   
    // construct workset
    workset->cell_local_ids.push_back(0); workset->cell_local_ids.push_back(1);
    workset->num_cells = numCells;
    workset->block_id = "eblock-0_0";
    workset->ir_degrees = rcp(new std::vector<int>);
    workset->ir_degrees->push_back(ir->cubature_degree);
    workset->int_rules.push_back(ival);
    workset->basis_names = rcp(new std::vector<std::string>);
    workset->basis_names->push_back(basisLayout->name());
    workset->bases.push_back(bval);
    double t0 = 3.87597e-10; //  [s]
    workset->time = 9e-4/t0;

    // create the field manager
    RCP<PHX::FieldManager<panzer::Traits> > fm = 
      rcp(new PHX::FieldManager<panzer::Traits>); 

    // create reference evaluator and register it with the field manager
    Teuchos::ParameterList p_ref;
    p_ref.set("Name","Ref_Ionization_ParticleStrike");
    p_ref.set("Basis",basis);
    RCP<PHX::Evaluator<panzer::Traits> > ref_eval =
      rcp(new RefFieldEvaluator<EvalType,panzer::Traits>(p_ref));
    fm->registerEvaluator<EvalType>(ref_eval);
    fm->requireField<EvalType>(*ref_eval->evaluatedFields()[0]);
    
    // create dependencies evaluator and register it with the field manager
    const RCP<const charon::Names>& names = rcp(new charon::Names(3,"","",""));
    Teuchos::ParameterList p_deps;
    p_deps.set("Basis",basis);
    RCP<std::vector<std::string>> dep_names = rcp(new std::vector<std::string>);
    // list the names of the dependent fields used by the test evaluator
    /*
    dep_names->push_back(names->field.intrin_conc);
    dep_names->push_back(names->dof.edensity);
    dep_names->push_back(names->dof.hdensity);
    */
    p_deps.set("Names",dep_names);
    RCP<PHX::Evaluator<panzer::Traits> > deps_eval =
      rcp(new DepFieldsEvaluator<EvalType,panzer::Traits>(p_deps));
    fm->registerEvaluator<EvalType>(deps_eval);

    // create test evaluator and register it with the field manager
    // input parameter list for the test evaluator (see closure
    // factory example for this particular evaluator)
    Teuchos::ParameterList p;
    p.set("Names", names);
    p.set("IR", ir);
    p.set("Basis", basisLayout);
    p.set("Start Point X", 0.5);
    p.set("Start Point Y", 0.5);
    p.set("Start Point Z", -0.25);
    p.set("End Point X", 0.5);
    p.set("End Point Y", 0.0);
    p.set("End Point Z", -0.25);
    p.set("Data Layout", basisLayout->functional);
    double C0 = 1e16; // [cm^-3]
    double D0 = 25.8; //  [cm^2/s]
    double T0 = 300.0; // [K]
    RCP<charon::Scaling_Parameters> scale_param
      = rcp(new charon::Scaling_Parameters(T0,C0,D0,false,0,false));
    p.set("Scaling Parameters", scale_param);
    p.set("Strike Radius", 1.0);
    p.set("Start Time", 1e-4);
    p.set("End Time", 1e-3);
    p.set("Temporal Waveform", "Gaussian");
    p.set("Generation Rate", 1e25);
    RCP<PHX::Evaluator<panzer::Traits> > eval =
      rcp(new Ionization_ParticleStrike<EvalType,panzer::Traits>(p));
    fm->registerEvaluator<EvalType>(eval);
    fm->requireField<EvalType>(*eval->evaluatedFields()[0]);

    // evaluate fields associated with the ref and test evaluators
    panzer::Traits::SD setupData;
    auto worksets = rcp(new std::vector<panzer::Workset>);
    worksets->push_back(*workset);
    setupData.worksets_ = worksets;
    fm->postRegistrationSetup(setupData);
    fm->writeGraphvizFile();
    panzer::Traits::PED preEvalData;
    fm->preEvaluate<EvalType>(preEvalData);
    fm->evaluateFields<EvalType>(*workset);
    fm->postEvaluate<EvalType>(0);

    // retrieve the computed fields
    PHX::MDField<typename EvalType::ScalarT,panzer::Cell,panzer::BASIS> ips_rate = 
      PHX::MDField<typename EvalType::ScalarT,panzer::Cell,panzer::BASIS> (
		   names->field.ionization_particle_strike_rate,basisLayout->functional);
    PHX::MDField<typename EvalType::ScalarT,panzer::Cell,panzer::BASIS> ref_ips_rate = 
      PHX::MDField<typename EvalType::ScalarT,panzer::Cell,panzer::BASIS> (
                   "Ref_Ionization_ParticleStrike",basisLayout->functional);
    fm->getFieldData<EvalType>(ips_rate);
    fm->getFieldData<EvalType>(ref_ips_rate);

    // check names to make sure they are still correct
    TEST_EQUALITY(ref_ips_rate.fieldTag().name(),"Ref_Ionization_ParticleStrike");
    TEST_EQUALITY(ips_rate.fieldTag().name(),names->field.ionization_particle_strike_rate);

    // check sizes
    TEST_EQUALITY(ref_ips_rate.size(),ips_rate.size());

    // check evaluator values
    double tol = 6e-7;
    for(int cell=0;cell<ips_rate.extent_int(0);cell++) 
      for(int pt=0;pt<ips_rate.extent_int(1);pt++) 
	TEST_FLOATING_EQUALITY(SV::eval(ips_rate(cell,pt)),
			       SV::eval(ref_ips_rate(cell,pt)),tol);
    
    TEST_ASSERT(true);
  } // end of TEUCHOS_UNIT_TEST(Ionization_ParticleStrike, eval3D)



} // end of namespace charon

// end of tIonization_ParticleStrike.cpp
