
///////////////////////////////////////////////////////////////////////////////
//
//  tAvalanche_CrowellSze.cpp
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//
//  Include Files
//
///////////////////////////////////////////////////////////////////////////////

// Include the header file for the class under testing below.  Also include any
// other Trilinos header files that are needed.

// Charon
#include "Charon_Avalanche_CrowellSze_decl.hpp"
#include "Charon_Names.hpp"

// Teuchos
#include "Teuchos_UnitTestHarness.hpp"

// Panzer
#include "Panzer_CommonArrayFactories.hpp"
#include "Panzer_Workset_Utilities.hpp"


using Teuchos::RCP;
using Teuchos::rcp;
using panzer::Cell;
using panzer::Dim;
using panzer::IntegrationRule;
using PHX::MDField;
using PHX::DataLayout;

using std::vector;
using std::pair;
using std::string;

// A set of unit tests for a class has the following general form.  Feel free
// to delete this comment.
namespace charon
{
  /////////////////////////////////////////////////////////////////////////////
  // evaluator for storing the reference values
  // the reference values generated by the reference evaluator are precomputed
  // in general the user is responsible for providing this data
  /////////////////////////////////////////////////////////////////////////////
  template<typename EvalT, typename Traits>
  class RefFieldEvaluator
    :
    public PHX::EvaluatorWithBaseImpl<Traits>,
    public PHX::EvaluatorDerived<EvalT, Traits>
  {
  public:
    RefFieldEvaluator(const Teuchos::ParameterList& p);

    void evaluateFields(typename Traits::EvalData d);
    
  private:
    using ScalarT = typename EvalT::ScalarT;
    MDField<ScalarT,Cell,panzer::IP> fieldValue;
  }; 

  template<typename EvalT, typename Traits>
  RefFieldEvaluator<EvalT, Traits>::
  RefFieldEvaluator(const Teuchos::ParameterList& p)
  {
    // read from parameters
    const string name = p.get<std::string>("Name");
    RCP<IntegrationRule> ir = p.get< RCP<IntegrationRule> >("IR");
    RCP<DataLayout> ip_scalar = ir->dl_scalar;

    fieldValue = MDField<ScalarT,Cell,panzer::IP>(name, ip_scalar);
    this->addEvaluatedField(fieldValue);
  
    std::string n = "RefFieldEvaluator: " + name;
    this->setName(n);
  }

  template<typename EvalT, typename Traits>
  void RefFieldEvaluator<EvalT, Traits>::
  evaluateFields(typename Traits::EvalData workset)
  { 
    // generate reference values to compare to
    // cell 0 
    fieldValue(0,0) = 5.30422e-06; 
    fieldValue(0,1) = 5.30422e-06;
    fieldValue(0,2) = 9.24813e-08; 
    fieldValue(0,3) = 9.24813e-08;
    // cell 1
    fieldValue(1,0) = 8.23785e-06; 
    fieldValue(1,1) = 8.23785e-06;
    fieldValue(1,2) = 0.0; 
    fieldValue(1,3) = 0.0;
  }  


  /////////////////////////////////////////////////////////////////////////////
  // evaluator for dependent fields
  // this evaluator initialize/set dynamically all dependent fields of the
  // evaluator under test; the structure and the names of the dependent fields 
  // have to match that of the dependent fields used in the test evaluator
  /////////////////////////////////////////////////////////////////////////////
  template<typename EvalT, typename Traits>
  class DepFieldsEvaluator
    :
    public PHX::EvaluatorWithBaseImpl<Traits>,
    public PHX::EvaluatorDerived<EvalT, Traits>
  {
  public:
    DepFieldsEvaluator(const Teuchos::ParameterList& p);

    void evaluateFields(typename Traits::EvalData d);

  private:
    using ScalarT = typename EvalT::ScalarT;
    vector<RCP<MDField<ScalarT,Cell,panzer::IP,Dim> > > fieldValues1;
    vector<RCP<MDField<ScalarT,Cell,panzer::BASIS> > > fieldValues2;
    vector<RCP<MDField<ScalarT,Cell,panzer::IP> > > fieldValues3;
    vector<string> field_names1;
    vector<string> field_names2;
    vector<string> field_names3;
    RCP<const charon::Names> names;
  }; 

  template<typename EvalT, typename Traits>
  DepFieldsEvaluator<EvalT, Traits>::
  DepFieldsEvaluator(const Teuchos::ParameterList& p)
  {
    // read from parameters
    names = p.get< RCP <const charon::Names>>("Names");
    RCP<vector<RCP<pair<string,RCP<DataLayout> > > > > deps1 =    
     p.get<RCP<vector<RCP<pair<string,RCP<DataLayout> > > > > >("Deps1");
    RCP<vector<RCP<pair<string,RCP<DataLayout> > > > > deps2 =    
     p.get<RCP<vector<RCP<pair<string,RCP<DataLayout> > > > > >("Deps2");
    RCP<vector<RCP<pair<string,RCP<DataLayout> > > > > deps3 =    
     p.get<RCP<vector<RCP<pair<string,RCP<DataLayout> > > > > >("Deps3");
    
    // dynamically allocate an array with fields used as dependent fields 
    // in the evaluator under test; the names and the layouts of the fields 
    // are passed as input in the parameter list
    vector<RCP<pair<string,RCP<DataLayout> > > >::iterator it;
    for(it=(*deps1).begin(); it!=(*deps1).end(); ++it) {
      std::string name = (*it)->first;
      Teuchos::RCP<DataLayout> dl = (*it)->second;
      fieldValues1.push_back(rcp(
	 new MDField<ScalarT,Cell,panzer::IP,Dim>(name,dl)));	
      field_names1.push_back(name);
      this->addEvaluatedField(*fieldValues1.back()); 
    }
    for(it=(*deps2).begin(); it!=(*deps2).end(); ++it) {
      std::string name = (*it)->first;
      Teuchos::RCP<DataLayout> dl = (*it)->second;
      fieldValues2.push_back(rcp(
	 new MDField<ScalarT,Cell,panzer::BASIS>(name,dl)));	
      field_names2.push_back(name);
      this->addEvaluatedField(*fieldValues2.back()); 
    }
    for(it=(*deps3).begin(); it!=(*deps3).end(); ++it) {
      std::string name = (*it)->first;
      Teuchos::RCP<DataLayout> dl = (*it)->second;
      fieldValues3.push_back(rcp(
	 new MDField<ScalarT,Cell,panzer::IP>(name,dl)));	
      field_names3.push_back(name);
      this->addEvaluatedField(*fieldValues3.back()); 
    }
    
    this->setName("DepFieldsEvaluator");
  }
    
  template<typename EvalT, typename Traits>
  void DepFieldsEvaluator<EvalT, Traits>::
  evaluateFields(typename Traits::EvalData workset) 
  {     
    ScalarT J_sc = 413361.5336;
    ScalarT E_sc = 258.52029;
    for(size_t fi=0; fi<fieldValues1.size(); fi++) {
      // fill up the test dependent fields with values
      MDField<ScalarT,Cell,panzer::IP,Dim> val = *fieldValues1[fi];
      if(field_names1[fi] == names->field.elec_efield) {
	// electron effective field
	// cell 0 
	val(0,0,0) = 0.0; 
	val(0,0,1) = 5.95e5/E_sc;
	val(0,1,0) = 0.0;
	val(0,1,1) = 5.95e5/E_sc;
	val(0,2,0) = 0.0; 
	val(0,2,1) = 7.58e5/E_sc;
	val(0,3,0) = 0.0; 
	val(0,3,1) = 7.58e5/E_sc;
	// cell 1
	val(1,0,0) = 0.0; 
	val(1,0,1) = 7.58e5/E_sc;
	val(1,1,0) = 0.0;
	val(1,1,1) = 7.58e5/E_sc;
	val(1,2,0) = 0.0; 
	val(1,2,1) = 0.0038/E_sc;
	val(1,3,0) = 0.0; 
	val(1,3,1) = 0.0038/E_sc;	
      } else if(field_names1[fi] == names->field.hole_efield) {
	// hole effective field
	// cell 0 
	val(0,0,0) = 0.0; 
	val(0,0,1) = 5.95e5/E_sc;
	val(0,1,0) = 0.0;
	val(0,1,1) = 5.95e5/E_sc;
	val(0,2,0) = 0.0; 
	val(0,2,1) = 7.58e5/E_sc;
	val(0,3,0) = 0.0; 
	val(0,3,1) = 7.58e5/E_sc;
	// cell 1
	val(1,0,0) = 0.0; 
	val(1,0,1) = 7.58e5/E_sc;
	val(1,1,0) = 0.0;
	val(1,1,1) = 7.58e5/E_sc;
	val(1,2,0) = 0.0; 
	val(1,2,1) = 0.0038/E_sc;
	val(1,3,0) = 0.0; 
	val(1,3,1) = 0.0038/E_sc;
      } else if(field_names1[fi] == names->field.elec_curr_density) {
	// electron current density
	// cell 0 
	val(0,0,0) = 0.0; 
	val(0,0,1) = 0.088/J_sc;
	val(0,1,0) = 0.0;
	val(0,1,1) = 0.088/J_sc;
	val(0,2,0) = 0.0; 
	val(0,2,1) = 0.001/J_sc;
	val(0,3,0) = 0.0; 
	val(0,3,1) = 0.001/J_sc;
	// cell 1
	val(1,0,0) = 0.0; 
	val(1,0,1) = 0.001/J_sc;
	val(1,1,0) = 0.0;
	val(1,1,1) = 0.001/J_sc;
	val(1,2,0) = 0.0; 
	val(1,2,1) = -0.0003/J_sc;
	val(1,3,0) = 0.0; 
	val(1,3,1) = -0.0003/J_sc;
      } else if(field_names1[fi] == names->field.hole_curr_density) {
	// hole current density
	// cell 0 
	val(0,0,0) = 0.0; 
	val(0,0,1) = 0.0;
	val(0,1,0) = 0.0;
	val(0,1,1) = 0.0/J_sc;
	val(0,2,0) = 0.0; 
	val(0,2,1) = 0.087/J_sc;
	val(0,3,0) = 0.0; 
	val(0,3,1) = 0.087/J_sc;
	// cell 1
	val(1,0,0) = 0.0; 
	val(1,0,1) = 0.087/J_sc;
	val(1,1,0) = 0.0;
	val(1,1,1) = 0.087/J_sc;
	val(1,2,0) = 0.0; 
	val(1,2,1) = 0.088/J_sc;
	val(1,3,0) = 0.0; 
	val(1,3,1) = 0.088/J_sc;
      }
    }

    ScalarT dens_sc = 1e19;
    ScalarT T_sc = 300.0;
    for(size_t fi=0; fi<fieldValues2.size(); fi++) {
      // fill up the test dependent fields with values
      MDField<ScalarT,Cell,panzer::BASIS> val = *fieldValues2[fi];
      if(field_names2[fi] == names->dof.edensity) {
	// electron density
	// cell 0 
	val(0,0) = 2.5e16/dens_sc; 
	val(0,1) = 2.5e16/dens_sc;
	val(0,2) = 6.38e6/dens_sc; 
	val(0,3) = 6.38e6/dens_sc;
	// cell 1
	val(1,0) = 6.38e6/dens_sc; 
	val(1,1) = 6.38e6/dens_sc;
	val(1,2) = 202.4/dens_sc; 
	val(1,3) = 202.4/dens_sc;
      } else if(field_names2[fi] == names->dof.hdensity) {
	// hole density
	// cell 0
	val(0,0) = 8374.01/dens_sc; 
	val(0,1) = 8374.01/dens_sc;
	val(0,2) = 2.49e8/dens_sc; 
	val(0,3) = 2.49e8/dens_sc;
	// cell 1
	val(1,0) = 2.49e8/dens_sc; 
	val(1,1) = 2.49e8/dens_sc;
	val(1,2) = 1.0e18/dens_sc; 
	val(1,3) = 1.0e18/dens_sc;
      } else if(field_names2[fi] == names->field.latt_temp) {
	// lattice temperature
	// cell 0
	val(0,0) = 300.0/T_sc; 
	val(0,1) = 300.0/T_sc;
	val(0,2) = 300.0/T_sc; 
	val(0,3) = 300.0/T_sc;
	// cell 1
	val(1,0) = 300.0/T_sc; 
	val(1,1) = 300.0/T_sc;
	val(1,2) = 300.0/T_sc; 
	val(1,3) = 300.0/T_sc;
      } 
    }
   
    for(size_t fi=0; fi<fieldValues3.size(); fi++) {
      // fill up the test dependent fields with values
      MDField<ScalarT,Cell,panzer::IP> val = *fieldValues3[fi];
      if(field_names3[fi] == names->dof.edensity) {
	// electron density
	// cell 0 
	val(0,0) = 2.5e16/dens_sc; 
	val(0,1) = 2.5e16/dens_sc;
	val(0,2) = 6.38e6/dens_sc; 
	val(0,3) = 6.38e6/dens_sc;
	// cell 1
	val(1,0) = 6.38e6/dens_sc; 
	val(1,1) = 6.38e6/dens_sc;
	val(1,2) = 202.4/dens_sc; 
	val(1,3) = 202.4/dens_sc;
      } else if(field_names3[fi] == names->dof.hdensity) {
	// hole density
	// cell 0
	val(0,0) = 8374.01/dens_sc; 
	val(0,1) = 8374.01/dens_sc;
	val(0,2) = 2.49e8/dens_sc; 
	val(0,3) = 2.49e8/dens_sc;
	// cell 1
	val(1,0) = 2.49e8/dens_sc; 
	val(1,1) = 2.49e8/dens_sc;
	val(1,2) = 1.0e18/dens_sc; 
	val(1,3) = 1.0e18/dens_sc;
      } else if(field_names3[fi] == names->field.latt_temp) {
	// lattice temperature
	// cell 0
	val(0,0) = 300.0/T_sc; 
	val(0,1) = 300.0/T_sc;
	val(0,2) = 300.0/T_sc; 
	val(0,3) = 300.0/T_sc;
	// cell 1
	val(1,0) = 300.0/T_sc; 
	val(1,1) = 300.0/T_sc;
	val(1,2) = 300.0/T_sc; 
	val(1,3) = 300.0/T_sc;
      } 
    }
  }

  /////////////////////////////////////////////////////
  // map test workset to coordinates 
  /////////////////////////////////////////////////////
  static void mapTestWorkset(int numCells, int numVerts, int dim,
                      Teuchos::RCP<panzer::Workset> workset) {
    panzer::MDFieldArrayFactory af("",true);
    
    workset->cell_vertex_coordinates = 
      af.buildStaticArray<double,Cell,panzer::NODE,Dim>(
          "coords",numCells,numVerts,dim);
    // describe the domain used for test evaluator
    // coords(cell,node,dim)
    panzer::Workset::CellCoordArray coords = workset->cell_vertex_coordinates;
    // cell 0
    coords(0,0,0) = 0.0; coords(0,0,1) = 0.0; 
    coords(0,1,0) = 1.0; coords(0,1,1) = 0.0;
    coords(0,2,0) = 1.0; coords(0,2,1) = 0.5;
    coords(0,3,0) = 0.0; coords(0,3,1) = 0.5;
    // cell 1 
    coords(1,0,0) = 0.0; coords(1,0,1) = 0.5; 
    coords(1,1,0) = 1.0; coords(1,1,1) = 0.5;
    coords(1,2,0) = 1.0; coords(1,2,1) = 1.0;
    coords(1,3,0) = 0.0; coords(1,3,1) = 1.0;
  }

  
  // You can create as many unit tests in this collection as you like.  For
  // any functionality of the class that needs to be tested, create another
  // unit test.  Feel free to delete this comment.

  /////////////////////////////////////////////////////////////////////////////
  //
  //  TEUCHOS_UNIT_TEST(Avalanche_CrowellSze, specificTestName)
  //
  /////////////////////////////////////////////////////////////////////////////
  TEUCHOS_UNIT_TEST(Avalanche_CrowellSze, eval1)
  {
    using charon::Avalanche_CrowellSze; // evaluator under test
    
    typedef panzer::Traits::Residual EvalType;
    typedef Sacado::ScalarValue<typename EvalType::ScalarT> SV;

    int numCells = 2, numVerts = 4, dim = 2;
    RCP<shards::CellTopology> topo = rcp(
	new shards::CellTopology(shards::getCellTopologyData< shards::Quadrilateral<4> >()));
    panzer::CellData cellData(numCells,topo);
    RCP<panzer::PureBasis> Hgradbasis = 
      rcp(new panzer::PureBasis("HGrad",1,cellData));
    string cvfem_type = "volume";
    RCP<IntegrationRule> cvfem_vol_ir = rcp(new IntegrationRule(cellData,cvfem_type));
    RCP<panzer::BasisIRLayout> hgrad_vol_cvfem =
      rcp(new panzer::BasisIRLayout(Hgradbasis,*cvfem_vol_ir));
   
    // construct the workset 
    RCP<panzer::Workset> workset = rcp(new panzer::Workset);
    mapTestWorkset(numCells,numVerts,dim,workset);
    RCP<panzer::IntegrationValues2<double> > ival = 
      rcp(new panzer::IntegrationValues2<double>("",true));
    ival->setupArrays(cvfem_vol_ir);
    ival->evaluateValues(workset->cell_vertex_coordinates);
    RCP<panzer::BasisValues2<double> > bval = 
      rcp(new panzer::BasisValues2<double>("",true,true));
    bval->setupArrays(hgrad_vol_cvfem);
    bval->evaluateValues(ival->cub_points,ival->jac,ival->jac_det,ival->jac_inv,
			 ival->weighted_measure,workset->cell_vertex_coordinates);

    workset->cell_local_ids.push_back(0); workset->cell_local_ids.push_back(1);
    workset->num_cells = numCells;
    workset->block_id = "eblock-0_0";
    workset->ir_degrees = rcp(new std::vector<int>);
    workset->ir_degrees->push_back(cvfem_vol_ir->cubature_degree);
    workset->int_rules.push_back(ival);
    workset->basis_names = rcp(new std::vector<std::string>);
    workset->basis_names->push_back(hgrad_vol_cvfem->name());
    workset->bases.push_back(bval);

    // create the field manager
    RCP<PHX::FieldManager<panzer::Traits> > fm = 
      rcp(new PHX::FieldManager<panzer::Traits>);

    // create dependencies evaluator and register it with the field manager
    Teuchos::ParameterList p_deps;
    const RCP<const charon::Names>& names = rcp(new charon::Names(7,"","",""));
    p_deps.set("Names", names);
    RCP<DataLayout> dl_scalar = hgrad_vol_cvfem->functional;
    RCP<DataLayout> dl_scalar1 = cvfem_vol_ir->dl_scalar;
    RCP<DataLayout> dl_vector = cvfem_vol_ir->dl_vector;
    RCP<vector<RCP<std::pair<std::string,RCP<PHX::DataLayout> > > > > deps1 = 
      rcp(new vector<RCP<std::pair<std::string,RCP<PHX::DataLayout> > > >); 
    RCP<vector<RCP<std::pair<std::string,RCP<PHX::DataLayout> > > > > deps2 = 
      rcp(new vector<RCP<std::pair<std::string,RCP<PHX::DataLayout> > > >); 
    RCP<vector<RCP<std::pair<std::string,RCP<PHX::DataLayout> > > > > deps3 = 
      rcp(new vector<RCP<std::pair<std::string,RCP<PHX::DataLayout> > > >);
    // list with names of the dependent fields and layouts used by the test evaluator
    deps1->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->field.elec_efield,dl_vector)));
    deps1->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->field.hole_efield,dl_vector)));
    deps1->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->field.elec_curr_density,dl_vector)));
    deps1->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->field.hole_curr_density,dl_vector)));
    
    deps2->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->dof.edensity,dl_scalar)));
    deps2->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->dof.hdensity,dl_scalar)));
    deps2->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->field.latt_temp,dl_scalar)));

    deps3->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->dof.edensity,dl_scalar1)));
    deps3->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->dof.hdensity,dl_scalar1)));
    deps3->push_back(rcp(
      new pair<string,RCP<DataLayout> >(names->field.latt_temp,dl_scalar1)));

    p_deps.set("Deps1",deps1);
    p_deps.set("Deps2",deps2);
    p_deps.set("Deps3",deps3);
    RCP<PHX::Evaluator<panzer::Traits> > deps_eval =
      rcp(new DepFieldsEvaluator<EvalType,panzer::Traits>(p_deps));
    fm->registerEvaluator<EvalType>(deps_eval);

    // create test evaluator and register it with the field manager
    // input parameter list for the test evaluator (see closure
    // factory example for this particular evaluator)
    Teuchos::ParameterList p;
    p.set("Names", names);
    p.set("Material Name", "Silicon");
    p.set("Equation Set Type", "SGCVFEM Drift Diffusion");
    double C0 = 1e16; // [cm^-3]
    double D0 = 25.8; //  [cm^2/s]
    double T0 = 300.0; // [K]
    RCP<charon::Scaling_Parameters> scale_param
      = rcp(new charon::Scaling_Parameters(T0,C0,D0,false,0,false));
    p.set("Scaling Parameters", scale_param);
    p.set("IR", cvfem_vol_ir);
    p.set("Basis", hgrad_vol_cvfem);
    Teuchos::ParameterList psub;
    psub.set("Value", "Crowell-Sze");
    psub.set("Driving Force", "EffectiveFieldParallelJ");
    psub.set("eDrForceRefDens", 1e10);
    psub.set("hDrForceRefDens", 1e10);
    psub.set("Minimum Field", 1e4);
    psub.set("E_opt_ph", 0.063);
    psub.set("lambda300_e", 6.2e-7);
    psub.set("lambda300_h", 3.8e-7);
    psub.set("Ei_e", 1.1);
    psub.set("Ei_h", 1.8);
    p.sublist("Avalanche ParameterList") = psub;
    RCP<PHX::Evaluator<panzer::Traits> > eval =
      rcp(new Avalanche_CrowellSze<EvalType,panzer::Traits>(p));
    fm->registerEvaluator<EvalType>(eval);
    fm->requireField<EvalType>(*eval->evaluatedFields()[0]);

    // create reference evaluator and register it with the field manager
    Teuchos::ParameterList p_ref;
    p_ref.set("Name","Ref_CrowellSze_Avalanche");
    p_ref.set("IR", cvfem_vol_ir);
    RCP<PHX::Evaluator<panzer::Traits> > ref_eval =
      rcp(new RefFieldEvaluator<EvalType,panzer::Traits>(p_ref));
    fm->registerEvaluator<EvalType>(ref_eval);
    fm->requireField<EvalType>(*ref_eval->evaluatedFields()[0]);

    // evaluate fields associated with the ref and test evaluators
    panzer::Traits::SD setupData;
    auto worksets = rcp(new vector<panzer::Workset>);
    worksets->push_back(*workset);
    setupData.worksets_ = worksets;
    fm->postRegistrationSetup(setupData);
    fm->writeGraphvizFile();
    panzer::Traits::PED preEvalData;
    fm->preEvaluate<EvalType>(preEvalData);
    fm->evaluateFields<EvalType>(*workset);
    fm->postEvaluate<EvalType>(0);

    // retrieve the computed fields
    PHX::MDField<typename EvalType::ScalarT,panzer::Cell,panzer::IP> ava_rate = 
      PHX::MDField<typename EvalType::ScalarT,panzer::Cell,panzer::IP> (
		   names->field.avalanche_rate,dl_scalar1);
    PHX::MDField<typename EvalType::ScalarT,panzer::Cell,panzer::IP> ref_ava_rate = 
      PHX::MDField<typename EvalType::ScalarT,panzer::Cell,panzer::IP> (
		   "Ref_CrowellSze_Avalanche",dl_scalar1);
    fm->getFieldData<EvalType>(ava_rate);
    fm->getFieldData<EvalType>(ref_ava_rate);

    // check names to make sure they are still correct
    TEST_EQUALITY(ref_ava_rate.fieldTag().name(),"Ref_CrowellSze_Avalanche");
    TEST_EQUALITY(ava_rate.fieldTag().name(),names->field.avalanche_rate);

    // check sizes
    TEST_EQUALITY(ref_ava_rate.size(), ava_rate.size());

    // check evaluator values
    double tol = 1e-6;
    for(int cell=0;cell<ava_rate.extent_int(0);cell++) {
      for(int pt=0;pt<ava_rate.extent_int(1);pt++) {
	std::cout << SV::eval(ava_rate(cell,pt)) << std::endl;
	TEST_FLOATING_EQUALITY(SV::eval(ava_rate(cell,pt)),
			       SV::eval(ref_ava_rate(cell,pt)),tol);
      }
    }

    TEST_ASSERT(true);  
  } // end of TEUCHOS_UNIT_TEST(Avalanche_CrowellSze, specificTestName)

  

} // end of namespace charon

// end of tAvalanche_CrowellSze.cpp
