
#ifndef CHARON_AVALANCHE_UNIBONEW_IMPL_HPP
#define CHARON_AVALANCHE_UNIBONEW_IMPL_HPP

#include <cmath>
#include "Panzer_Workset.hpp"
#include "Panzer_Workset_Utilities.hpp"
#include "Panzer_IntegrationRule.hpp"

#include "Charon_Material_Properties.hpp"
#include "Charon_Physical_Constants.hpp"
#include "Charon_Names.hpp"


/*
The New University of Bologna impact ionization model (UniBoNew) is a further
developed version of the UniBo model by [1][2][3] to cover an extended temperature
range between 25 celcius degree and 500 celcius degree (773 K). It is based on
impact ionization data generated by the Boltzmann solver HARM [4] and is calibrated
against specially designed impact ionization measurements [5][6]. It covers
a wide range of electric fields.

[1] E. Gnani et al., “Extraction method for the impact-ionization multiplication factor in
silicon at large operating temperatures,” in Proceedings of the 32nd European Solid-
State Device Research Conference (ESSDERC), Florence, Italy, pp. 227–230,
September 2002.
[2] S. Reggiani et al., “Investigation about the High-Temperature Impact-Ionization
Coefficient in Silicon,” in Proceedings of the 34th European Solid-State Device
Research Conference (ESSDERC), Leuven, Belgium, pp. 245–248, September 2004.
[3] S. Reggiani et al., “Experimental extraction of the electron impact-ionization coefficient
at large operating temperatures,” in IEDM Technical Digest, San Francisco, CA, USA,
pp. 407–410, December 2004.
[4] M. C. Vecchi and M. Rudan, “Modeling Electron and Hole Transport with Full-Band
Structure Effects by Means of the Spherical-Harmonics Expansion of the BTE,” IEEE
Transactions on Electron Devices, vol. 45, no. 1, pp. 230–238, 1998.
[5] S. Reggiani et al., “Electron and Hole Mobility in Silicon at Large Operating
Temperatures—Part I: Bulk Mobility,” IEEE Transactions on Electron Devices, vol. 49,
no. 3, pp. 490–499, 2002.
[6] M. Valdinoci et al., “Impact-ionization in silicon at large operating temperature,” in
International Conference on Simulation of Semiconductor Processes and Devices
(SISPAD), Kyoto, Japan, pp. 27–30, September 1999.

The ionization coefficient takes the following form:

α(F, T) = F / [a(T) + b(T) * exp( d(T)/(F + c(T)) ) ], where

a(T)=sum_k(a_k*(T/1K)^k) (k=0 to 3), b(T)=sum_k(b_k*(T/1K)^k) (k=0 to 10),
c(T)=sum_k(c_k*(T/1K)^k) (k=0 to 3), d(T)=sum_k(d_k*(T/1K)^k) (k=0 to 3).

a_k, b_k, c_k, d_k are model parameters whose default values are given in
charon::Material_Properties. F is the driving field, i.e., GradQuasiFermi,
EParallelJ, EParallelJtot, EeffParallelJ, or EeffParallelJtot.
Eeff is computed by charon::FEM_ElectricField.

For numerical reasons, the field F is computed as n*Fn/(n+n0) for electrons,
and p*Fp/(p+p0) for holes. n0 and p0 are damping parameters and default to 0 and
can be set in the Avalanche Generation section through eDrForceRefDens and
hDrForceRefDens respectively. Using positive values for n0 and p0 can improve
convergence for problems where strong generation–recombination occurs in regions
with small density. 
*/


namespace charon {

///////////////////////////////////////////////////////////////////////////////
//
//  Constructor
//
///////////////////////////////////////////////////////////////////////////////
template<typename EvalT, typename Traits>
Avalanche_UniBoNew<EvalT, Traits>::
Avalanche_UniBoNew(
  const Teuchos::ParameterList& p)
{
  using std::string;
  using Teuchos::RCP;
  using PHX::DataLayout;
  using PHX::MDField;
  using panzer::IntegrationRule;
  using Teuchos::ParameterList;

  RCP<ParameterList> valid_params = this->getValidParameters();
  p.validateParameters(*valid_params);

  const charon::Names& n = *(p.get< RCP<const charon::Names> >("Names"));

  // Retrieve data layouts
  RCP<DataLayout> scalar = p.get< RCP<DataLayout> >("Scalar Data Layout");
  RCP<DataLayout> vector = p.get< RCP<DataLayout> >("Vector Data Layout");
  num_points = vector->dimension(1);
  num_dims = vector->dimension(2);

  // Material name
  const string& materialName = p.get<string>("Material Name");

  // Obtain the instance of charon::Material_Properties.
  charon::Material_Properties& matProperty = charon::Material_Properties::getInstance();

  // Retrieve UniBoNew model parameters
  a0_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron a0");
  a1_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron a1");
  a2_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron a2");
  a3_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron a3");
  b0_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b0");
  b1_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b1");
  b2_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b2");
  b3_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b3");
  b4_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b4");
  b5_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b5");
  b6_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b6");
  b7_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b7");
  b8_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b8");
  b9_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b9");
  b10_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron b10");
  c0_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron c0");
  c1_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron c1");
  c2_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron c2");
  c3_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron c3");
  d0_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron d0");
  d1_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron d1");
  d2_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron d2");
  d3_e = matProperty.getPropertyValue(materialName, "UniBoNew Electron d3");

  a0_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole a0");
  a1_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole a1");
  a2_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole a2");
  a3_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole a3");
  b0_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b0");
  b1_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b1");
  b2_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b2");
  b3_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b3");
  b4_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b4");
  b5_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b5");
  b6_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b6");
  b7_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b7");
  b8_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b8");
  b9_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b9");
  b10_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole b10");
  c0_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole c0");
  c1_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole c1");
  c2_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole c2");
  c3_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole c3");
  d0_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole d0");
  d1_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole d1");
  d2_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole d2");
  d3_h = matProperty.getPropertyValue(materialName, "UniBoNew Hole d3");

  // Avalanche ParameterList
  const ParameterList& avaParamList = p.sublist("Avalanche ParameterList");

  // Overwrite parameters when specified by users
  if (avaParamList.isParameter("a0_e"))
    a0_e = avaParamList.get<double>("a0_e");
  if (avaParamList.isParameter("a1_e"))
    a1_e = avaParamList.get<double>("a1_e");
  if (avaParamList.isParameter("a2_e"))
    a2_e = avaParamList.get<double>("a2_e");
  if (avaParamList.isParameter("a3_e"))
    a3_e = avaParamList.get<double>("a3_e");
  if (avaParamList.isParameter("b0_e"))
    b0_e = avaParamList.get<double>("b0_e");
  if (avaParamList.isParameter("b1_e"))
    b1_e = avaParamList.get<double>("b1_e");
  if (avaParamList.isParameter("b2_e"))
    b2_e = avaParamList.get<double>("b2_e");
  if (avaParamList.isParameter("b3_e"))
    b3_e = avaParamList.get<double>("b3_e");
  if (avaParamList.isParameter("b4_e"))
    b4_e = avaParamList.get<double>("b4_e");
  if (avaParamList.isParameter("b5_e"))
    b5_e = avaParamList.get<double>("b5_e");
  if (avaParamList.isParameter("b6_e"))
    b6_e = avaParamList.get<double>("b6_e");
  if (avaParamList.isParameter("b7_e"))
    b7_e = avaParamList.get<double>("b7_e");
  if (avaParamList.isParameter("b8_e"))
    b8_e = avaParamList.get<double>("b8_e");
  if (avaParamList.isParameter("b9_e"))
    b9_e = avaParamList.get<double>("b9_e");
  if (avaParamList.isParameter("b10_e"))
    b10_e = avaParamList.get<double>("b10_e");
  if (avaParamList.isParameter("c0_e"))
    c0_e = avaParamList.get<double>("c0_e");
  if (avaParamList.isParameter("c1_e"))
    c1_e = avaParamList.get<double>("c1_e");
  if (avaParamList.isParameter("c2_e"))
    c2_e = avaParamList.get<double>("c2_e");
  if (avaParamList.isParameter("c3_e"))
    c3_e = avaParamList.get<double>("c3_e");
  if (avaParamList.isParameter("d0_e"))
    d0_e = avaParamList.get<double>("d0_e");
  if (avaParamList.isParameter("d1_e"))
    d1_e = avaParamList.get<double>("d1_e");
  if (avaParamList.isParameter("d2_e"))
    d2_e = avaParamList.get<double>("d2_e");
  if (avaParamList.isParameter("d3_e"))
    d3_e = avaParamList.get<double>("d3_e");

  if (avaParamList.isParameter("a0_h"))
    a0_h = avaParamList.get<double>("a0_h");
  if (avaParamList.isParameter("a1_h"))
    a1_h = avaParamList.get<double>("a1_h");
  if (avaParamList.isParameter("a2_h"))
    a2_h = avaParamList.get<double>("a2_h");
  if (avaParamList.isParameter("a3_h"))
    a3_h = avaParamList.get<double>("a3_h");
  if (avaParamList.isParameter("b0_h"))
    b0_h = avaParamList.get<double>("b0_h");
  if (avaParamList.isParameter("b1_h"))
    b1_h = avaParamList.get<double>("b1_h");
  if (avaParamList.isParameter("b2_h"))
    b2_h = avaParamList.get<double>("b2_h");
  if (avaParamList.isParameter("b3_h"))
    b3_h = avaParamList.get<double>("b3_h");
  if (avaParamList.isParameter("b4_h"))
    b4_h = avaParamList.get<double>("b4_h");
  if (avaParamList.isParameter("b5_h"))
    b5_h = avaParamList.get<double>("b5_h");
  if (avaParamList.isParameter("b6_h"))
    b6_h = avaParamList.get<double>("b6_h");
  if (avaParamList.isParameter("b7_h"))
    b7_h = avaParamList.get<double>("b7_h");
  if (avaParamList.isParameter("b8_h"))
    b8_h = avaParamList.get<double>("b8_h");
  if (avaParamList.isParameter("b9_h"))
    b9_h = avaParamList.get<double>("b9_h");
  if (avaParamList.isParameter("b10_h"))
    b10_h = avaParamList.get<double>("b10_h");
  if (avaParamList.isParameter("c0_h"))
    c0_h = avaParamList.get<double>("c0_h");
  if (avaParamList.isParameter("c1_h"))
    c1_h = avaParamList.get<double>("c1_h");
  if (avaParamList.isParameter("c2_h"))
    c2_h = avaParamList.get<double>("c2_h");
  if (avaParamList.isParameter("c3_h"))
    c3_h = avaParamList.get<double>("c3_h");
  if (avaParamList.isParameter("d0_h"))
    d0_h = avaParamList.get<double>("d0_h");
  if (avaParamList.isParameter("d1_h"))
    d1_h = avaParamList.get<double>("d1_h");
  if (avaParamList.isParameter("d2_h"))
    d2_h = avaParamList.get<double>("d2_h");
  if (avaParamList.isParameter("d3_h"))
    d3_h = avaParamList.get<double>("d3_h");

  // Set minimum field below which avalanche generation = 0
  minField = 5e4;  // default, [V/cm]
  if (avaParamList.isParameter("Minimum Field"))
    minField = avaParamList.get<double>("Minimum Field");

  // Driving Force = GradQuasiFermi, GradPotentialParallelJ, GradPotentialParallelJtot,
  // EffectiveFieldParallelJ, or EffectiveFieldParallelJtot.
  driveForce = "GradQuasiFermi" ;   // default
  if (avaParamList.isParameter("Driving Force"))
    driveForce = avaParamList.get<string>("Driving Force");

  // Set driving force damping parameters
  eDrForceRefDens = 0.0;  // default, using positive value might improve convergence
  hDrForceRefDens = 0.0;
  if (avaParamList.isParameter("eDrForceRefDens"))
    eDrForceRefDens = avaParamList.get<double>("eDrForceRefDens");
  if (avaParamList.isParameter("hDrForceRefDens"))
    hDrForceRefDens = avaParamList.get<double>("hDrForceRefDens");

  // Define and add dependent fields according to driveForce
  if (driveForce == "GradQuasiFermi")
  {
    grad_qfp_e = MDField<const ScalarT,Cell,Point,Dim>(n.field.elec_grad_qfp,vector);
    grad_qfp_h = MDField<const ScalarT,Cell,Point,Dim>(n.field.hole_grad_qfp,vector);
    this->addDependentField(grad_qfp_e);
    this->addDependentField(grad_qfp_h);
  }
  else if ( (driveForce == "GradPotentialParallelJ") || (driveForce == "GradPotentialParallelJtot") )
  {
    grad_pot = MDField<const ScalarT,Cell,Point,Dim>(n.grad_dof.phi,vector);
    this->addDependentField(grad_pot);
  }
  else if ( (driveForce == "EffectiveFieldParallelJ") || (driveForce == "EffectiveFieldParallelJtot") )
  {
    // these fields are always used (see below) because the derivates need to
    // be computed

//    eff_field_e = MDField<const ScalarT,Cell,Point,Dim>(n.field.elec_efield,vector);
//    eff_field_h = MDField<const ScalarT,Cell,Point,Dim>(n.field.hole_efield,vector);
//    this->addDependentField(eff_field_e);
//    this->addDependentField(eff_field_h);
  }
  else
    TEUCHOS_TEST_FOR_EXCEPTION(true, std::logic_error, "Error: Invalid Driving Force !");

  // Always need current densities and carrier densities
  curr_dens_e = MDField<const ScalarT,Cell,Point,Dim>(n.field.elec_curr_density,vector);
  curr_dens_h = MDField<const ScalarT,Cell,Point,Dim>(n.field.hole_curr_density,vector);
  dens_e = MDField<const ScalarT,Cell,Point>(n.dof.edensity,scalar);
  dens_h = MDField<const ScalarT,Cell,Point>(n.dof.hdensity,scalar);

  this->addDependentField(curr_dens_e);
  this->addDependentField(curr_dens_h);
  this->addDependentField(dens_e);
  this->addDependentField(dens_h);

  // Scaling parameters
  scaleParams = p.get< RCP<charon::Scaling_Parameters> >("Scaling Parameters");
  J0 = scaleParams->scale_params.J0;
  R0 = scaleParams->scale_params.R0;
  E0 = scaleParams->scale_params.E0;
  C0 = scaleParams->scale_params.C0;
  T0 = scaleParams->scale_params.T0;

  // Add other dependent fields
  latt_temp = MDField<const ScalarT,Cell,Point>(n.field.latt_temp,scalar);
  this->addDependentField(latt_temp);

  // Always need the following additional fields for computing derivatives
  eff_field_e = MDField<const ScalarT,Cell,Point,Dim>(n.field.elec_efield,vector);
  eff_field_h = MDField<const ScalarT,Cell,Point,Dim>(n.field.hole_efield,vector);
  grad_dens_e = MDField<const ScalarT,Cell,Point,Dim>(n.grad_dof.edensity,vector);
  grad_dens_h = MDField<const ScalarT,Cell,Point,Dim>(n.grad_dof.hdensity,vector);

  diff_coeff_e = MDField<const ScalarT,Cell,Point>(n.field.elec_diff_coeff,scalar);
  diff_coeff_h = MDField<const ScalarT,Cell,Point>(n.field.hole_diff_coeff,scalar);
  mobility_e = MDField<const ScalarT,Cell,Point>(n.field.elec_mobility,scalar);
  mobility_h = MDField<const ScalarT,Cell,Point>(n.field.hole_mobility,scalar);

  this->addDependentField(eff_field_e);
  this->addDependentField(eff_field_h);
  this->addDependentField(grad_dens_e);
  this->addDependentField(grad_dens_h);
  this->addDependentField(diff_coeff_e);
  this->addDependentField(diff_coeff_h);
  this->addDependentField(mobility_e);
  this->addDependentField(mobility_h);

  // Add evaluated field
  avalanche_rate = MDField<ScalarT,Cell,Point>(n.field.avalanche_rate,scalar);
  ava_deriv_e = MDField<ScalarT,Cell,Point>(n.field.ava_deriv_e,scalar);
  ava_deriv_h = MDField<ScalarT,Cell,Point>(n.field.ava_deriv_h,scalar);

  this->addEvaluatedField(avalanche_rate);
  this->addEvaluatedField(ava_deriv_e);
  this->addEvaluatedField(ava_deriv_h);

  std::string name = "UniBoNew_Avalanche";
  this->setName(name);
}


///////////////////////////////////////////////////////////////////////////////
//
//  evaluateFields()
//
///////////////////////////////////////////////////////////////////////////////
template<typename EvalT, typename Traits>
void
Avalanche_UniBoNew<EvalT, Traits>::
evaluateFields(
  typename Traits::EvalData workset)
{
  using panzer::index_t;

  // Scale the damping parameters
  ScalarT scaled_eRefDens = eDrForceRefDens / C0;
  ScalarT scaled_hRefDens = hDrForceRefDens / C0;

  // Retrieve electron charge in [C]
  charon::PhysicalConstants const& cpc = charon::PhysicalConstants::Instance();
  double eleQ = cpc.q;

  // Avalanche generate rate scaling
  ScalarT scaling = J0 / (eleQ * R0);

  // Loop over cells
  for (index_t cell = 0; cell < workset.num_cells; ++cell)
  {
    for (int point = 0; point < num_points; ++point)
    {
      // Obtain temperature
      ScalarT lattT = latt_temp(cell,point)*T0;  // lattice temperature [K]
      ScalarT normT = lattT / 1.0;   // lattT/1K, [1]

      // Compute temperature-related parameters in the model
      ScalarT ae, be, ce, de;
      ScalarT ah, bh, ch, dh;

      ae = a0_e + a1_e*normT + a2_e*pow(normT,2.) + a3_e*pow(normT,3.);
      be = b0_e + b1_e*normT + b2_e*pow(normT,2.) + b3_e*pow(normT,3.) + b4_e*pow(normT,4.)
         + b5_e*pow(normT,5.)+ b6_e*pow(normT,6.) + b7_e*pow(normT,7.) + b8_e*pow(normT,8.)
         + b9_e*pow(normT,9.)+ b10_e*pow(normT,10.);
      ce = c0_e + c1_e*normT + c2_e*pow(normT,2.) + c3_e*pow(normT,3.);
      de = d0_e + d1_e*normT + d2_e*pow(normT,2.) + d3_e*pow(normT,3.);

      ah = a0_h + a1_h*normT + a2_h*pow(normT,2.) + a3_h*pow(normT,3.);
      bh = b0_h + b1_h*normT + b2_h*pow(normT,2.) + b3_h*pow(normT,3.) + b4_h*pow(normT,4.)
         + b5_h*pow(normT,5.)+ b6_h*pow(normT,6.) + b7_h*pow(normT,7.) + b8_h*pow(normT,8.)
         + b9_h*pow(normT,9.)+ b10_h*pow(normT,10.);
      ch = c0_h + c1_h*normT + c2_h*pow(normT,2.) + c3_h*pow(normT,3.);
      dh = d0_h + d1_h*normT + d2_h*pow(normT,2.) + d3_h*pow(normT,3.);

      // Obtain carrier density
      const ScalarT& eden = dens_e(cell,point);
      const ScalarT& hden = dens_h(cell,point);

      ScalarT Fe = 0.0;        // driving field
      ScalarT Fh = 0.0;

      // ----------------------------------------------------------------------
      // --- Driving field calculation is the same for all avalanche models ---
      // ----------------------------------------------------------------------

      // Compute absolute values of Electron and Hole GradQuasiFermiPotential
      if (driveForce == "GradQuasiFermi")
      {
        ScalarT egradqfp = 0.0;
        ScalarT hgradqfp = 0.0;
        for (int dim = 0; dim < num_dims; ++dim)
        {
          const ScalarT& gqfpe = grad_qfp_e(cell,point,dim);
          const ScalarT& gqfph = grad_qfp_h(cell,point,dim);

          egradqfp += gqfpe * gqfpe;
          hgradqfp += gqfph * gqfph;
        }
        // The density term is for damping to obtain better convergence in some cases
        Fe = std::sqrt(egradqfp) * E0 * eden/(eden + scaled_eRefDens);  // in [V/cm]
        Fh = std::sqrt(hgradqfp) * E0 * hden/(hden + scaled_hRefDens);
      }

      // Compute projection of negative gradient of potential along carrier current density
      else if (driveForce == "GradPotentialParallelJ")
      {
        ScalarT normJe = 0.0;    // absolute value of Je
        ScalarT normJh = 0.0;
        ScalarT ngpdotJe = 0.0;  // -grad(phi) \dot Je
        ScalarT ngpdotJh = 0.0;
        for (int dim = 0; dim < num_dims; ++dim)
        {
          const ScalarT& Je = curr_dens_e(cell,point,dim);
          const ScalarT& Jh = curr_dens_h(cell,point,dim);
          const ScalarT& gp = grad_pot(cell,point,dim);

          normJe += Je * Je;
          normJh += Jh * Jh;
          ngpdotJe += -gp * Je;
          ngpdotJh += -gp * Jh;
        }
        normJe = std::sqrt(normJe);      // note: Je/normJe has no unit.
        normJh = std::sqrt(normJh);
        Fe = ngpdotJe / normJe * E0 * eden/(eden + scaled_eRefDens);
        Fh = ngpdotJh / normJh * E0 * hden/(hden + scaled_hRefDens);
      }

      // Compute projection of negative gradient of potential along total current density
      else if (driveForce == "GradPotentialParallelJtot")
      {
        ScalarT normJtot = 0.0;
        ScalarT ngpdotJtot = 0.0;
        for (int dim = 0; dim < num_dims; ++dim)
        {
          const ScalarT& Je = curr_dens_e(cell,point,dim);
          const ScalarT& Jh = curr_dens_h(cell,point,dim);
          const ScalarT& gp = grad_pot(cell,point,dim);

          ScalarT Jtot = Je + Jh;
          normJtot += Jtot * Jtot;
          ngpdotJtot += -gp * Jtot;
        }
        normJtot = std::sqrt(normJtot);
        Fe = ngpdotJtot / normJtot * E0 * eden/(eden + scaled_eRefDens);
        Fh = ngpdotJtot / normJtot * E0 * hden/(hden + scaled_hRefDens);
      }

      // Compute projection of effective electric field along carrier current density
      else if (driveForce == "EffectiveFieldParallelJ")
      {
        ScalarT normJe = 0.0;      // absolute value of Je
        ScalarT normJh = 0.0;
        ScalarT eEeffdotJe = 0.0;  // eEeff \dot Je
        ScalarT hEeffdotJh = 0.0;
        for (int dim = 0; dim < num_dims; ++dim)
        {
          const ScalarT& Je = curr_dens_e(cell,point,dim);
          const ScalarT& Jh = curr_dens_h(cell,point,dim);
          const ScalarT& eEeff = eff_field_e(cell,point,dim);
          const ScalarT& hEeff = eff_field_h(cell,point,dim);

          normJe += Je * Je;
          normJh += Jh * Jh;
          eEeffdotJe += eEeff * Je;
          hEeffdotJh += hEeff * Jh;
        }
        normJe = std::sqrt(normJe);
        normJh = std::sqrt(normJh);
        Fe = eEeffdotJe / normJe * E0 * eden/(eden + scaled_eRefDens);
        Fh = hEeffdotJh / normJh * E0 * hden/(hden + scaled_hRefDens);
      }

      // Compute projection of effective electric field along total current density
      else if (driveForce == "EffectiveFieldParallelJtot")
      {
        ScalarT normJtot = 0.0;
        ScalarT eEeffdotJtot = 0.0;  // eEeff \dot Jtot
        ScalarT hEeffdotJtot = 0.0;
        for (int dim = 0; dim < num_dims; ++dim)
        {
          const ScalarT& Je = curr_dens_e(cell,point,dim);
          const ScalarT& Jh = curr_dens_h(cell,point,dim);
          const ScalarT& eEeff = eff_field_e(cell,point,dim);
          const ScalarT& hEeff = eff_field_h(cell,point,dim);

          ScalarT Jtot = Je + Jh;
          normJtot += Jtot * Jtot;
          eEeffdotJtot += eEeff * Jtot;
          hEeffdotJtot += hEeff * Jtot;
        }
        normJtot = std::sqrt(normJtot);
        Fe = eEeffdotJtot / normJtot * E0 * eden/(eden + scaled_eRefDens);
        Fh = hEeffdotJtot / normJtot * E0 * hden/(hden + scaled_hRefDens);
      }

      // Take the absolute values
      Fe = std::abs(Fe);
      Fh = std::abs(Fh);

      // end of driving field calculation -------------------------------------

      // Compute absolute values of current densities
      ScalarT normJe = 0.0;
      ScalarT normJh = 0.0;
      for (int dim = 0; dim < num_dims; ++dim)
      {
        const ScalarT& Je = curr_dens_e(cell,point,dim);
        const ScalarT& Jh = curr_dens_h(cell,point,dim);
        normJe += Je * Je;
        normJh += Jh * Jh;
      }
      normJe = std::sqrt(normJe);
      normJh = std::sqrt(normJh);

      // -----------------------------------------------------------------------
      // --- Ionization coefficients calculation depends on avalanche models ---
      // -----------------------------------------------------------------------

      // Compute impact ionization coefficients [1/cm]
      ScalarT iicoeffe = 0.0;
      ScalarT iicoeffh = 0.0;
      if (Fe > minField)
        iicoeffe = Fe / (ae + be * exp(de/(Fe + ce)) );
      if (Fh > minField)
        iicoeffh = Fh / (ah + bh * exp(dh/(Fh + ch)) );

      // Compute avalanche generate rate (scaled)
      ScalarT eAvaRate = 0.0;
      ScalarT hAvaRate = 0.0;
      if (normJe > 0.)  eAvaRate = iicoeffe*normJe;
      if (normJh > 0.)  hAvaRate = iicoeffh*normJh;
      avalanche_rate(cell,point) = (eAvaRate + hAvaRate)*scaling;

      // -----------------------------------------------------------------------
      // --- Avalanche gen. derivative w.r.t. eden and hden respectively -------
      // -----------------------------------------------------------------------

      // Calculate grad(n) \dot En, |En|^2, grad(p) \dot Ep, and |Ep|^2
      ScalarT gcdotEeffe = 0.0;
      ScalarT gcdotEeffh = 0.0;
      ScalarT normEeffe2 = 0.0;
      ScalarT normEeffh2 = 0.0;
      for (int dim = 0; dim < num_dims; ++dim)
      {
        const ScalarT& grad_eden = grad_dens_e(cell,point,dim);
        const ScalarT& grad_hden = grad_dens_h(cell,point,dim);
        const ScalarT& eEeff = eff_field_e(cell,point,dim);
        const ScalarT& hEeff = eff_field_h(cell,point,dim);
        gcdotEeffe += grad_eden * eEeff;
        gcdotEeffh += grad_hden * hEeff;
        normEeffe2 += eEeff * eEeff;
        normEeffh2 += hEeff * hEeff;
      }

      const ScalarT& ediff = diff_coeff_e(cell,point);
      const ScalarT& hdiff = diff_coeff_h(cell,point);
      const ScalarT& emob = mobility_e(cell,point);
      const ScalarT& hmob = mobility_h(cell,point);

      // Compute avalanche derivative
      if (normJe > 0.)
        ava_deriv_e(cell,point) = iicoeffe / normJe * (ediff*emob*gcdotEeffe
                                + eden*emob*emob*normEeffe2) * scaling;
      else
        ava_deriv_e(cell,point) = 0.0;
      if (normJh > 0.)
        ava_deriv_h(cell,point) = iicoeffh / normJh * (-hdiff*hmob*gcdotEeffh
                                + hden*hmob*hmob*normEeffh2) * scaling;
      else
        ava_deriv_h(cell,point) = 0.0;

    }  // end of loop over point
  }  // end of loop over cell

}


///////////////////////////////////////////////////////////////////////////////
//
//  getValidParameters()
//
///////////////////////////////////////////////////////////////////////////////
template<typename EvalT, typename Traits>
Teuchos::RCP<Teuchos::ParameterList>
Avalanche_UniBoNew<EvalT, Traits>::getValidParameters() const
{
  Teuchos::RCP<Teuchos::ParameterList> p = Teuchos::rcp(new Teuchos::ParameterList);
  p->set<std::string>("Material Name", "?");

  Teuchos::RCP<const charon::Names> n;
  p->set("Names", n);

  Teuchos::RCP<PHX::DataLayout> scalar_dl;
  p->set("Scalar Data Layout", scalar_dl);

  Teuchos::RCP<PHX::DataLayout> vector_dl;
  p->set("Vector Data Layout", vector_dl);

  p->sublist("Avalanche ParameterList", false, "");
  p->sublist("Avalanche ParameterList").set<std::string>("Value", "UniBoNew", "Use the UniBoNew avalanche model");
  p->sublist("Avalanche ParameterList").set<std::string>("Driving Force", "GradQuasiFermi", "Specify the driving force");
  p->sublist("Avalanche ParameterList").set<double>("eDrForceRefDens", 0., "[cm^-3]");
  p->sublist("Avalanche ParameterList").set<double>("hDrForceRefDens", 0., "[cm^-3]");
  p->sublist("Avalanche ParameterList").set<double>("Minimum Field", 0., "[V/cm]");

  p->sublist("Avalanche ParameterList").set<double>("a0_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("a1_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("a2_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("a3_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b0_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b1_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b2_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b3_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b4_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b5_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b6_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b7_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b8_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b9_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b10_e", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("c0_e", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("c1_e", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("c2_e", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("c3_e", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("d0_e", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("d1_e", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("d2_e", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("d3_e", 0., "[V/cm]");

  p->sublist("Avalanche ParameterList").set<double>("a0_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("a1_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("a2_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("a3_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b0_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b1_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b2_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b3_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b4_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b5_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b6_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b7_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b8_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b9_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("b10_h", 0., "[V]");
  p->sublist("Avalanche ParameterList").set<double>("c0_h", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("c1_h", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("c2_h", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("c3_h", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("d0_h", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("d1_h", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("d2_h", 0., "[V/cm]");
  p->sublist("Avalanche ParameterList").set<double>("d3_h", 0., "[V/cm]");

  Teuchos::RCP<charon::Scaling_Parameters> sp;
  p->set("Scaling Parameters", sp);

  return p;
}

}

#endif
