
#ifndef TABLE_H
#define TABLE_H

/* This table is based on a nomenclature for the points that is different
from Lorenson and Cline.  I take

0: (0,0,0)
1: (1,0,0)
2: (0,1,0)
3: (1,1,0)
4: (0,0,1)
5: (1,0,1)
6: (0,1,1)
7: (1,1,1)
as the vertices of the unit cube (ist binary digit for x, 2nd for y, 3rd
for z)

 How to relate the zero based edge number to the zero based vertex number

 GDK's order is based on assigning a ternary number to each vertex,
 edge center, face center, centroid.  (GDK 12-5-88)

    Edge from to  zyx ternary index

            0  1  001       1     0
            0  2  010       3     1
            1  3  012       5     2
            2  3  021       7     3

            0  4  100       9     4
            1  5  102      11     5
            2  6  120      15     6
            3  7  122      17     7

            4  5  201      19     8
            4  6  210      21     9
            5  7  212      23    10
            6  7  221      25    11

 */
#define NTRIANGLES	 	4
#define NSIDES 			3
#define NUM_HEX_EDGES 	12

typedef struct {
  char n_triangles;
  char triangle[NTRIANGLES][NSIDES];
} triangulated_cubes;

typedef struct {
  char n_triangles;
  char triangle[5][NSIDES];
} triangulated_cubes5;

typedef struct {
  char n_triangles;
  char triangle[6][NSIDES];
} triangulated_cubes6;

typedef struct {
  char n_triangles;
  char triangle[8][NSIDES];
} triangulated_cubes8;

typedef struct {
  char n_triangles;
  char triangle[9][NSIDES];
} triangulated_cubes9;

typedef struct {
  char n_triangles;
  char triangle[12][NSIDES];
} triangulated_cubes12;

/* for each edge gives the corresponding face and the corresponding
   edge on the other side of the face. */
char edge_face_edge [12][2][2] = {
   {{0,  3}, {1,  8}},
   {{0,  2}, {2,  9}},
   {{0,  1}, {3, 10}},
   {{0,  0}, {4, 11}},
   {{1,  5}, {2,  6}},
   {{1,  4}, {3,  7}},
   {{2,  4}, {4,  7}},
   {{3,  5}, {4,  6}},
   {{1,  0}, {5, 11}},
   {{2,  1}, {5, 10}},
   {{3,  2}, {5,  9}},
   {{4,  3}, {5,  8}}
};

/* for each of the 12 edges, the vertex numbers of the endpts */
char edges[12][2] = {
  {0, 1}, {0, 2}, {1, 3}, {2, 3},
  {0, 4}, {1, 5}, {2, 6}, {3, 7},
  {4, 5}, {4, 6}, {5, 7}, {6, 7}
};

/* for each of the 8 vertices, the edges that use that vertex as an endpt */
char vertex_edges [8][3] = {
   {0, 1, 4}, {0, 2, 5}, {1, 3, 6}, {2, 3, 7},
   {4, 8, 9}, {5, 8, 10}, {6, 9, 11}, {7, 10, 11}
};

/* for each of the 12 edges, what other edges share its two endpts */
char adj_edges [12][4] = {
   {1, 4, 2, 5}, {0, 4, 3, 6}, {0, 5, 3, 7}, {1, 6, 2, 7},
   {0, 1, 8, 9}, {0, 2, 8, 10}, {1, 3, 9, 11}, {2, 3, 10, 11},
   {4, 9, 5, 10}, {4, 8, 6, 11}, {5, 8, 7, 11}, {6, 9, 7, 10}
};

/* for each of the 12 edges, all the edges that are parallel */
char parallel_edges [12][4] = {
   {0, 3, 8, 11}, {1, 2, 9, 10}, {1, 2, 9, 10}, {0, 3, 8, 11},
   {4, 5, 6, 7}, {4, 5, 6, 7}, {4, 5, 6, 7}, {4, 5, 6, 7},
   {0, 3, 8, 11}, {1, 2, 9, 10}, {1, 2, 9, 10}, {0, 3, 8, 11}
};

char edge_x [12] = {1, 0, 2, 1, 0, 2, 0, 2, 1, 0, 2, 1};
char edge_y [12] = {0, 1, 1, 2, 0, 0, 2, 2, 0, 1, 1, 2};
char edge_z [12] = {0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2};

char face_vertices [6][4] = { /* vertices in each face */
   {1, 0, 3, 2}, {0, 1, 4, 5}, {0, 4, 2, 6},
   {5, 1, 7, 3}, {6, 7, 2, 3}, { 4, 5, 6, 7}
};

char face_components [6][3] = { /* 0 = x, 1 = y, 2 = z */
   {0, 1, 2}, { 0, 2, 1}, {1, 2, 0}, {1, 2, 0}, {0, 2, 1}, {0, 1, 2}
};

/*
 * HEXAHEDRON SECTION
 *
 * January 4, 1989
 * This file allows one to do direct bitmaps
 * with the triangles always in right-hand-rule order pointing in the 
 * direction of increasing scalar function, instead of pattern and 
 * permutation tables.
 *
 */

/*
 * Pat  0  1  2  3  4  5  6  7  8  9 10 11 12 13 14
 * rep  2 16 24 24  8 48 48 16  6  8  6 12 24  2 12
 * ntri 0  1  2  2  2  3  3  3  2  4  4  4  4  4  4
 * nhit 0  3  4  6  6  5  7  9  4  6  8  6  8 12  6
 *
 *      0 triangles: 1 pattern (0          )  2 occurrences
 *      1 triangle : 1 pattern (1          ) 16 occurrences
 *      2 triangles: 4 patterns(2 3 4 8    ) 62 occurrences
 *      3 triangles: 3 patterns(5 6 7      )112 occurrences
 *      4 triangles  6 patterns(9 thru 14  ) 64 occurrences
 *
 *       total number of triangles in all bitmaps = 732 
 *
*/
triangulated_cubes subtable3 [] = {
/*(0)    6- 3A*/ {2, {{ 1, 6, 3}, { 0, 2, 5}, { 0, 0, 0}, { 0, 0, 0}}},
/*(1)    6- 3B*/ {4, {{ 2, 5, 3}, { 3, 5, 6}, { 5, 1, 6}, { 0, 1, 5}}},
/*(2)    9- 3A*/ {2, {{ 3, 7, 2}, { 4, 1, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(3)    9- 3B*/ {4, {{ 3, 4, 1}, { 7, 4, 3}, { 7, 2, 4}, { 4, 2, 0}}},
/*(4)   18- 3A*/ {2, {{ 0, 2, 5}, { 9, 4, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(5)   18- 3B*/ {4, {{ 0, 9, 4}, { 2, 9, 0}, { 2, 5, 9}, { 9, 5, 8}}},
/*(6)   20- 3A*/ {2, {{ 4, 8, 9}, { 3, 1, 6}, { 0, 0, 0}, { 0, 0, 0}}},
/*(7)   20- 3B*/ {4, {{ 4, 3, 1}, { 8, 3, 4}, { 8, 9, 3}, { 3, 9, 6}}},
/*(8)   33- 3A*/ {2, {{ 4, 1, 0}, { 8, 5,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(9)   33- 3B*/ {4, {{ 5,10, 0}, { 0,10, 1}, {10, 4, 1}, { 8, 4,10}}},
/*(10)  40- 3A*/ {2, {{ 2, 3, 7}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(11)  40- 3B*/ {4, {{10, 8, 7}, { 7, 8, 3}, { 8, 2, 3}, { 5, 2, 8}}},
/*(12)  65- 3A*/ {2, {{ 9,11, 6}, { 4, 1, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(13)  65- 3B*/ {4, {{ 1, 0, 6}, { 6, 0,11}, { 0, 9,11}, { 4, 9, 0}}},
/*(14)  72- 3A*/ {2, {{ 7, 2, 3}, {11, 6, 9}, { 0, 0, 0}, { 0, 0, 0}}},
/*(15)  72- 3B*/ {4, {{ 6, 9, 3}, { 3, 9, 2}, { 9, 7, 2}, {11, 7, 9}}},
/*(16)  96- 3A*/ {2, {{11, 6, 9}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(17)  96- 3B*/ {4, {{11, 5,10}, { 6, 5,11}, { 6, 9, 5}, { 5, 9, 8}}},
/*(18) 111- 3B*/ {4, {{ 4,10, 8}, { 7,10, 4}, { 7, 4,11}, {11, 4, 9}}},
/*(19) 111- 3A*/ {2, {{ 4, 9, 8}, {11, 7,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(20) 123- 3B*/ {4, {{11, 6,10}, {10, 6, 1}, { 3,10, 1}, { 7,10, 3}}},
/*(21) 123- 3A*/ {2, {{11, 7,10}, { 6, 1, 3}, { 0, 0, 0}, { 0, 0, 0}}},
/*(22) 125- 3B*/ {4, {{ 5,10, 0}, { 0,10,11}, { 7, 0,11}, { 2, 0, 7}}},
/*(23) 125- 3A*/ {2, {{ 5, 2, 0}, {10,11, 7}, { 0, 0, 0}, { 0, 0, 0}}}
}; /* subtable3 end */

triangulated_cubes5 subtable6 [] = {
/*(0)   25- 6A*/ {3, {{ 9, 1, 8}, { 1, 0, 8}, { 7, 2, 3}, { 0, 0, 0}, { 0, 0, 0}}},
/*(1)   25- 6B*/ {5, {{ 3, 9, 1}, { 7, 9, 3}, { 7, 8, 9}, { 7, 2, 8}, { 2, 0, 8}}},
/*(2)   26- 6A*/ {3, {{ 7, 0, 3}, { 7, 5, 0}, { 4, 8, 9}, { 0, 0, 0}, { 0, 0, 0}}},
/*(3)   26- 6B*/ {5, {{ 0, 3, 4}, { 4, 3, 9}, { 3, 7, 9}, { 7, 8, 9}, { 7, 5, 8}}},
/*(4)   28- 6A*/ {3, {{ 2, 1, 7}, { 1, 6, 7}, { 8, 9, 4}, { 0, 0, 0}, { 0, 0, 0}}},
/*(5)   28- 6B*/ {5, {{ 4, 2, 1}, { 8, 2, 4}, { 8, 7, 2}, { 8, 9, 7}, { 9, 6, 7}}},
/*(6)   37- 6A*/ {3, {{ 3, 0, 6}, { 0, 4, 6}, {10, 8, 5}, { 0, 0, 0}, { 0, 0, 0}}},
/*(7)   37- 6B*/ {5, {{ 5, 3, 0}, {10, 3, 5}, {10, 6, 3}, {10, 8, 6}, { 8, 4, 6}}},
/*(8)   38- 6A*/ {3, {{ 8, 0,10}, { 0, 2,10}, { 6, 3, 1}, { 0, 0, 0}, { 0, 0, 0}}},
/*(9)   38- 6B*/ {5, {{ 1, 8, 0}, { 6, 8, 1}, { 6,10, 8}, { 6, 3,10}, { 3, 2,10}}},
/*(10)  44- 6A*/ {3, {{ 6, 7, 1}, { 7, 2, 1}, { 8, 5,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(11)  44- 6B*/ {5, {{10, 6, 7}, { 8, 6,10}, { 8, 1, 6}, { 8, 5, 1}, { 5, 2, 1}}},
/*(12)  52- 6A*/ {3, {{10, 9, 5}, { 9, 4, 5}, { 3, 1, 6}, { 0, 0, 0}, { 0, 0, 0}}},
/*(13)  52- 6B*/ {5, {{ 6,10, 9}, { 3,10, 6}, { 3, 5,10}, { 3, 1, 5}, { 1, 4, 5}}},
/*(14)  56- 6A*/ {3, {{ 4,10, 9}, { 4, 5,10}, { 7, 2, 3}, { 0, 0, 0}, { 0, 0, 0}}},
/*(15)  56- 6B*/ {5, {{10, 9, 7}, { 7, 9, 3}, { 9, 4, 3}, { 4, 2, 3}, { 4, 5, 2}}},
/*(16)  61- 6B*/ {5, {{ 6, 7, 2}, { 6, 2, 0}, { 9, 6, 0}, { 5, 9, 0}, {10, 9, 5}}},
/*(17)  61- 6A*/ {3, {{ 5, 2, 0}, { 6, 7,10}, { 6,10, 9}, { 0, 0, 0}, { 0, 0, 0}}},
/*(18)  62- 6B*/ {5, {{10, 9, 4}, {10, 4, 0}, { 7,10, 0}, { 1, 7, 0}, { 6, 7, 1}}},
/*(19)  62- 6A*/ {3, {{ 1, 4, 0}, {10, 9, 6}, {10, 6, 7}, { 0, 0, 0}, { 0, 0, 0}}},
/*(20)  67- 6A*/ {3, {{ 5, 4, 2}, { 4, 1, 2}, {11, 6, 9}, { 0, 0, 0}, { 0, 0, 0}}},
/*(21)  67- 6B*/ {5, {{ 9, 5, 4}, {11, 5, 9}, {11, 2, 5}, {11, 6, 2}, { 6, 1, 2}}},
/*(22)  70- 6A*/ {3, {{11, 3, 9}, { 3, 1, 9}, { 5, 0, 2}, { 0, 0, 0}, { 0, 0, 0}}},
/*(23)  70- 6B*/ {5, {{ 2,11, 3}, { 5,11, 2}, { 5, 9,11}, { 5, 0, 9}, { 0, 1, 9}}},
/*(24)  74- 6A*/ {3, {{ 0, 3, 5}, { 3, 7, 5}, { 9,11, 6}, { 0, 0, 0}, { 0, 0, 0}}},
/*(25)  74- 6B*/ {5, {{ 6, 0, 3}, { 9, 0, 6}, { 9, 5, 0}, { 9,11, 5}, {11, 7, 5}}},
/*(26)  82- 6A*/ {3, {{ 6, 4,11}, { 4, 8,11}, { 2, 5, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(27)  82- 6B*/ {5, {{ 0, 6, 4}, { 2, 6, 0}, { 2,11, 6}, { 2, 5,11}, { 5, 8,11}}},
/*(28)  88- 6A*/ {3, {{ 8,11, 4}, {11, 6, 4}, { 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}}},
/*(29)  88- 6B*/ {5, {{ 7, 8,11}, { 2, 8, 7}, { 2, 4, 8}, { 2, 3, 4}, { 3, 6, 4}}},
/*(30)  91- 6B*/ {5, {{ 3, 7, 5}, { 1, 3, 5}, { 1, 5, 8}, { 1, 8, 6}, { 6, 8,11}}},
/*(31)  91- 6A*/ {3, {{ 1, 3, 6}, {11, 7, 5}, { 8,11, 5}, { 0, 0, 0}, { 0, 0, 0}}},
/*(32)  94- 6B*/ {5, {{ 7, 5, 0}, { 7, 0, 1}, {11, 7, 1}, { 4,11, 1}, { 8,11, 4}}},
/*(33)  94- 6A*/ {3, {{ 4, 0, 1}, { 7, 5, 8}, { 7, 8,11}, { 0, 0, 0}, { 0, 0, 0}}},
/*(34)  98- 6A*/ {3, {{ 2,10, 0}, {10, 8, 0}, { 6, 9,11}, { 0, 0, 0}, { 0, 0, 0}}},
/*(35)  98- 6B*/ {5, {{11, 2,10}, { 6, 2,11}, { 6, 0, 2}, { 6, 9, 0}, { 9, 8, 0}}},
/*(36) 100- 6A*/ {3, {{ 1, 9, 3}, { 9,11, 3}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(37) 100- 6B*/ {5, {{ 8, 1, 9}, { 5, 1, 8}, { 5, 3, 1}, { 5,10, 3}, {10,11, 3}}},
/*(38) 103- 6B*/ {5, {{ 2,10, 8}, { 2, 8, 4}, { 3, 2, 4}, { 9, 3, 4}, {11, 3, 9}}},
/*(39) 103- 6A*/ {3, {{ 9, 8, 4}, { 2,10,11}, { 2,11, 3}, { 0, 0, 0}, { 0, 0, 0}}},
/*(40) 110- 6B*/ {5, {{10, 8, 0}, { 7,10, 0}, { 7, 0, 1}, { 7, 1,11}, {11, 1, 9}}},
/*(41) 110- 6A*/ {3, {{ 7,10,11}, { 9, 8, 0}, { 1, 9, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(42) 118- 6B*/ {5, {{11, 3, 1}, {11, 1, 4}, {10,11, 4}, { 0,10, 4}, { 2,10, 0}}},
/*(43) 118- 6A*/ {3, {{ 0, 1, 4}, {11, 3, 2}, {11, 2,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(44) 122- 6B*/ {5, {{ 0, 3, 7}, { 0, 7,10}, { 4, 0,10}, {11, 4,10}, { 6, 4,11}}},
/*(45) 122- 6A*/ {3, {{11, 7,10}, { 0, 3, 6}, { 0, 6, 4}, { 0, 0, 0}, { 0, 0, 0}}},
/*(46) 124- 6B*/ {5, {{ 4, 5,10}, { 4,10,11}, { 1, 4,11}, { 7, 1,11}, { 2, 1, 7}}},
/*(47) 124- 6A*/ {3, {{ 7,10,11}, { 4, 5, 2}, { 4, 2, 1}, { 0, 0, 0}, { 0, 0, 0}}}
}; /* subtable6 end */

triangulated_cubes9 subtable7 [] = {
/*(0)   22- 7A*/ {3, {{ 6, 3, 1}, { 0, 2, 5}, { 4, 8, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(1)   22- 7B*/ {5, {{ 1, 6, 3}, { 9, 4, 2}, { 4, 0, 2}, { 5, 9, 2}, { 8, 9, 5}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(2)   22- 7C*/ {5, {{ 4, 8, 9}, { 5, 0, 6}, { 0, 1, 6}, { 3, 5, 6}, { 2, 5, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(3)   22- 7D*/ {9, {{12, 2, 5}, {12, 3, 2}, {12, 6, 3}, {12, 1, 6}, {12, 0, 1}, {12, 4, 0}, {12, 9, 4}, {12, 8, 9}, {12, 5, 8}}},
/*(4)   22- 7E*/ {5, {{ 0, 2, 5}, { 3, 1, 8}, { 1, 4, 8}, { 9, 3, 8}, { 6, 3, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(5)   22- 7F*/ {9, {{12, 8, 9}, {12, 5, 8}, {12, 2, 5}, {12, 0, 2}, {12, 4, 0}, {12, 1, 4}, {12, 3, 1}, {12, 6, 3}, {12, 9, 6}}},
/*(6)   22- 7G*/ {9, {{12, 6, 3}, {12, 9, 6}, {12, 8, 9}, {12, 4, 8}, {12, 1, 4}, {12, 0, 1}, {12, 5, 0}, {12, 2, 5}, {12, 3, 2}}},
/*(7)   22- 7H*/ {5, {{ 0, 1, 4}, { 2, 5, 8}, { 9, 2, 8}, { 3, 2, 9}, { 6, 3, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(8)   41- 7A*/ {3, {{ 4, 1, 0}, { 2, 3, 7}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(9)   41- 7B*/ {5, {{ 0, 4, 1}, { 8, 5, 3}, { 5, 2, 3}, { 7, 8, 3}, {10, 8, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(10)  41- 7C*/ {5, {{ 5,10, 8}, { 7, 2, 4}, { 2, 0, 4}, { 1, 7, 4}, { 3, 7, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(11)  41- 7D*/ {9, {{12, 3, 7}, {12, 1, 3}, {12, 4, 1}, {12, 0, 4}, {12, 2, 0}, {12, 5, 2}, {12, 8, 5}, {12,10, 8}, {12, 7,10}}},
/*(12)  41- 7E*/ {5, {{ 2, 3, 7}, { 1, 0,10}, { 0, 5,10}, { 8, 1,10}, { 4, 1, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(13)  41- 7F*/ {9, {{12,10, 8}, {12, 7,10}, {12, 3, 7}, {12, 2, 3}, {12, 5, 2}, {12, 0, 5}, {12, 1, 0}, {12, 4, 1}, {12, 8, 4}}},
/*(14)  41- 7G*/ {9, {{12, 4, 1}, {12, 8, 4}, {12,10, 8}, {12, 5,10}, {12, 0, 5}, {12, 2, 0}, {12, 7, 2}, {12, 3, 7}, {12, 1, 3}}},
/*(15)  41- 7H*/ {5, {{ 2, 0, 5}, { 3, 7,10}, { 8, 3,10}, { 1, 3, 8}, { 4, 1, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(16)  73- 7A*/ {3, {{ 9,11, 6}, { 3, 7, 2}, { 1, 0, 4}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(17)  73- 7B*/ {5, {{ 6, 9,11}, { 4, 1, 7}, { 1, 3, 7}, { 2, 4, 7}, { 0, 4, 2}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(18)  73- 7C*/ {5, {{ 1, 0, 4}, { 2, 3, 9}, { 3, 6, 9}, {11, 2, 9}, { 7, 2,11}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(19)  73- 7D*/ {9, {{12, 7, 2}, {12,11, 7}, {12, 9,11}, {12, 6, 9}, {12, 3, 6}, {12, 1, 3}, {12, 4, 1}, {12, 0, 4}, {12, 2, 0}}},
/*(20)  73- 7E*/ {5, {{ 3, 7, 2}, {11, 6, 0}, { 6, 1, 0}, { 4,11, 0}, { 9,11, 4}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(21)  73- 7F*/ {9, {{12, 0, 4}, {12, 2, 0}, {12, 7, 2}, {12, 3, 7}, {12, 1, 3}, {12, 6, 1}, {12,11, 6}, {12, 9,11}, {12, 4, 9}}},
/*(22)  73- 7G*/ {9, {{12, 9,11}, {12, 4, 9}, {12, 0, 4}, {12, 1, 0}, {12, 6, 1}, {12, 3, 6}, {12, 2, 3}, {12, 7, 2}, {12,11, 7}}},
/*(23)  73- 7H*/ {5, {{ 3, 6, 1}, { 7, 2, 0}, { 4, 7, 0}, {11, 7, 4}, { 9,11, 4}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(24)  97- 7A*/ {3, {{11, 6, 9}, { 4, 1, 0}, { 8, 5,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(25)  97- 7B*/ {5, {{ 9,11, 6}, {10, 8, 1}, { 8, 4, 1}, { 0,10, 1}, { 5,10, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(26)  97- 7C*/ {5, {{ 8, 5,10}, { 0, 4,11}, { 4, 9,11}, { 6, 0,11}, { 1, 0, 6}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(27)  97- 7D*/ {9, {{12, 1, 0}, {12, 6, 1}, {12,11, 6}, {12, 9,11}, {12, 4, 9}, {12, 8, 4}, {12,10, 8}, {12, 5,10}, {12, 0, 5}}},
/*(28)  97- 7E*/ {5, {{ 4, 1, 0}, { 6, 9, 5}, { 9, 8, 5}, {10, 6, 5}, {11, 6,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(29)  97- 7F*/ {9, {{12, 5,10}, {12, 0, 5}, {12, 1, 0}, {12, 4, 1}, {12, 8, 4}, {12, 9, 8}, {12, 6, 9}, {12,11, 6}, {12,10,11}}},
/*(30)  97- 7G*/ {9, {{12,11, 6}, {12,10,11}, {12, 5,10}, {12, 8, 5}, {12, 9, 8}, {12, 4, 9}, {12, 0, 4}, {12, 1, 0}, {12, 6, 1}}},
/*(31)  97- 7H*/ {5, {{ 4, 9, 8}, { 1, 0, 5}, {10, 1, 5}, { 6, 1,10}, {11, 6,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(32) 104- 7A*/ {3, {{11, 6, 9}, { 2, 3, 7}, { 8, 5,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(33) 104- 7B*/ {5, {{ 6, 9,11}, { 3,10, 8}, { 3, 7,10}, { 3, 8, 2}, { 2, 8, 5}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(34) 104- 7C*/ {5, {{ 8, 5,10}, { 9, 7, 2}, { 9,11, 7}, { 9, 2, 6}, { 6, 2, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(35) 104- 7D*/ {9, {{ 2, 3,12}, { 3, 6,12}, { 6, 9,12}, { 9,11,12}, {11, 7,12}, { 7,10,12}, {10, 8,12}, { 8, 5,12}, { 5, 2,12}}},
/*(36) 104- 7E*/ {5, {{ 2, 3, 7}, { 5,11, 6}, { 5,10,11}, { 5, 6, 8}, { 8, 6, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(37) 104- 7F*/ {9, {{ 8, 5,12}, { 5, 2,12}, { 2, 3,12}, { 3, 7,12}, { 7,10,12}, {10,11,12}, {11, 6,12}, { 6, 9,12}, { 9, 8,12}}},
/*(38) 104- 7G*/ {9, {{ 6, 9,12}, { 9, 8,12}, { 8, 5,12}, { 5,10,12}, {10,11,12}, {11, 7,12}, { 7, 2,12}, { 2, 3,12}, { 3, 6,12}}},
/*(39) 104- 7H*/ {5, {{10,11, 7}, { 5, 2, 3}, { 5, 3, 8}, { 8, 3, 6}, { 8, 6, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(40) 107- 7H*/ {5, {{ 8, 7,10}, { 8, 3, 7}, { 4, 3, 8}, { 4, 1, 3}, { 9,11, 6}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(41) 107- 7G*/ {9, {{ 3, 7,12}, { 1, 3,12}, { 6, 1,12}, {11, 6,12}, { 9,11,12}, { 4, 9,12}, { 8, 4,12}, {10, 8,12}, { 7,10,12}}},
/*(42) 107- 7F*/ {9, {{10, 8,12}, { 7,10,12}, {11, 7,12}, { 9,11,12}, { 6, 9,12}, { 3, 6,12}, { 1, 3,12}, { 4, 1,12}, { 8, 4,12}}},
/*(43) 107- 7E*/ {5, {{ 8, 7,10}, { 4, 7, 8}, { 4, 9,11}, { 4,11, 7}, { 1, 3, 6}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(44) 107- 7D*/ {9, {{ 4, 1,12}, { 8, 4,12}, { 9, 8,12}, { 6, 9,12}, {11, 6,12}, {10,11,12}, { 7,10,12}, { 3, 7,12}, { 1, 3,12}}},
/*(45) 107- 7C*/ {5, {{ 7, 1, 3}, {10, 1, 7}, {10,11, 6}, {10, 6, 1}, { 8, 4, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(46) 107- 7B*/ {5, {{ 1, 8, 4}, { 3, 8, 1}, { 3, 6, 9}, { 3, 9, 8}, { 7,10,11}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(47) 107- 7A*/ {3, {{ 8, 4, 9}, { 1, 3, 6}, {11, 7,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(48) 109- 7H*/ {5, {{11, 4, 9}, {11, 0, 4}, { 7, 0,11}, { 7, 2, 0}, {10, 8, 5}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(49) 109- 7G*/ {9, {{ 0, 4,12}, { 2, 0,12}, { 5, 2,12}, { 8, 5,12}, {10, 8,12}, { 7,10,12}, {11, 7,12}, { 9,11,12}, { 4, 9,12}}},
/*(50) 109- 7F*/ {9, {{ 9,11,12}, { 4, 9,12}, { 8, 4,12}, {10, 8,12}, { 5,10,12}, { 0, 5,12}, { 2, 0,12}, { 7, 2,12}, {11, 7,12}}},
/*(51) 109- 7E*/ {5, {{11, 4, 9}, { 7, 4,11}, { 7,10, 8}, { 7, 8, 4}, { 2, 0, 5}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(52) 109- 7D*/ {9, {{ 7, 2,12}, {11, 7,12}, {10,11,12}, { 5,10,12}, { 8, 5,12}, { 9, 8,12}, { 4, 9,12}, { 0, 4,12}, { 2, 0,12}}},
/*(53) 109- 7C*/ {5, {{ 4, 2, 0}, { 9, 2, 4}, { 9, 8, 5}, { 9, 5, 2}, {11, 7,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(54) 109- 7B*/ {5, {{ 2,11, 7}, { 0,11, 2}, { 0, 5,10}, { 0,10,11}, { 4, 9, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(55) 109- 7A*/ {3, {{11, 7,10}, { 2, 0, 5}, { 8, 4, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(56) 121- 7H*/ {5, {{ 0, 6, 1}, { 0,11, 6}, { 5,11, 0}, { 5,10,11}, { 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(57) 121- 7G*/ {9, {{11, 6,12}, {10,11,12}, { 7,10,12}, { 3, 7,12}, { 2, 3,12}, { 5, 2,12}, { 0, 5,12}, { 1, 0,12}, { 6, 1,12}}},
/*(58) 121- 7F*/ {9, {{ 1, 0,12}, { 6, 1,12}, { 3, 6,12}, { 2, 3,12}, { 7, 2,12}, {11, 7,12}, {10,11,12}, { 5,10,12}, { 0, 5,12}}},
/*(59) 121- 7E*/ {5, {{ 0, 6, 1}, { 5, 6, 0}, { 5, 2, 3}, { 5, 3, 6}, {10,11, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(60) 121- 7D*/ {9, {{ 5,10,12}, { 0, 5,12}, { 2, 0,12}, { 7, 2,12}, { 3, 7,12}, { 1, 3,12}, { 6, 1,12}, {11, 6,12}, {10,11,12}}},
/*(61) 121- 7C*/ {5, {{ 6,10,11}, { 1,10, 6}, { 1, 3, 7}, { 1, 7,10}, { 0, 5, 2}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(62) 121- 7B*/ {5, {{10, 0, 5}, {11, 0,10}, {11, 7, 2}, {11, 2, 0}, { 6, 1, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(63) 121- 7A*/ {3, {{ 0, 5, 2}, {10,11, 7}, { 3, 6, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}
}; /* subtable7 end */

triangulated_cubes8 subtable10 [] = {
/*(0)   60- 10A*/ {4, {{10, 9, 5}, { 9, 4, 5}, { 7, 1, 6}, { 1, 7, 2}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(1)   60- 10B*/ {8, {{10,12, 7}, { 7,12, 6}, { 6,12, 1}, { 1,12, 2}, { 2,12, 5}, { 5,12, 4}, { 4,12, 9}, { 9,12,10}}},
/*(2)   60- 10C*/ {8, {{10, 9,12}, { 9, 6,12}, { 6, 7,12}, { 7, 2,12}, { 2, 1,12}, { 1, 4,12}, { 4, 5,12}, { 5,10,12}}},
/*(3)   60- 10D*/ {4, {{ 4, 2, 1}, { 4, 5, 2}, { 9, 6,10}, {10, 6, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(4)   90- 10A*/ {4, {{ 8,11, 4}, {11, 6, 4}, { 5, 3, 7}, { 3, 5, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(5)   90- 10B*/ {8, {{ 8,12, 5}, { 5,12, 7}, { 7,12, 3}, { 3,12, 0}, { 0,12, 4}, { 4,12, 6}, { 6,12,11}, {11,12, 8}}},
/*(6)   90- 10C*/ {8, {{ 8,11,12}, {11, 7,12}, { 7, 5,12}, { 5, 0,12}, { 0, 3,12}, { 3, 6,12}, { 6, 4,12}, { 4, 8,12}}},
/*(7)   90- 10D*/ {4, {{ 6, 0, 3}, { 6, 4, 0}, {11, 7, 8}, { 8, 7, 5}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(8)  102- 10A*/ {4, {{11, 3, 9}, { 3, 1, 9}, {10, 0, 2}, { 0,10, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(9)  102- 10B*/ {8, {{11,12,10}, {10,12, 2}, { 2,12, 0}, { 0,12, 8}, { 8,12, 9}, { 9,12, 1}, { 1,12, 3}, { 3,12,11}}},
/*(10) 102- 10C*/ {8, {{11, 3,12}, { 3, 2,12}, { 2,10,12}, {10, 8,12}, { 8, 0,12}, { 0, 1,12}, { 1, 9,12}, { 9,11,12}}},
/*(11) 102- 10D*/ {4, {{ 1, 8, 0}, { 1, 9, 8}, { 3, 2,11}, {11, 2,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}
}; /* subtable10 end */

triangulated_cubes6 subtable12 [] = {
/*(0)   30- 12A*/ {4, {{ 4, 8, 9}, { 6, 0, 1}, { 5, 0, 6}, { 7, 5, 6}, { 0, 0, 0}, { 0, 0, 0}}},
/*(1)   30- 12B*/ {6, {{ 4, 0, 9}, { 5, 8, 7}, { 7, 8, 9}, { 0, 7, 9}, { 0, 1, 7}, { 1, 6, 7}}},
/*(2)   30- 12C*/ {6, {{ 4, 8, 1}, { 5, 0, 7}, { 7, 0, 1}, { 8, 7, 1}, { 8, 9, 7}, { 9, 6, 7}}},
/*(3)   30- 12D*/ {4, {{ 4, 0, 1}, { 5, 8, 9}, { 7, 5, 9}, { 6, 7, 9}, { 0, 0, 0}, { 0, 0, 0}}},
/*(4)   45- 12A*/ {4, {{ 5,10, 8}, { 4, 2, 0}, { 7, 2, 4}, { 6, 7, 4}, { 0, 0, 0}, { 0, 0, 0}}},
/*(5)   45- 12B*/ {6, {{ 5, 2, 8}, { 7,10, 6}, { 6,10, 8}, { 2, 6, 8}, { 2, 0, 6}, { 0, 4, 6}}},
/*(6)   45- 12C*/ {6, {{ 5,10, 0}, { 7, 2, 6}, { 6, 2, 0}, {10, 6, 0}, {10, 8, 6}, { 8, 4, 6}}},
/*(7)   45- 12D*/ {4, {{ 5, 2, 0}, { 7,10, 8}, { 6, 7, 8}, { 4, 6, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(8)   54- 12A*/ {4, {{ 1, 6, 3}, { 2, 4, 0}, { 9, 4, 2}, {10, 9, 2}, { 0, 0, 0}, { 0, 0, 0}}},
/*(9)   54- 12B*/ {6, {{ 1, 4, 3}, { 9, 6,10}, {10, 6, 3}, { 4,10, 3}, { 4, 0,10}, { 0, 2,10}}},
/*(10)  54- 12C*/ {6, {{ 1, 6, 0}, { 9, 4,10}, {10, 4, 0}, { 6,10, 0}, { 6, 3,10}, { 3, 2,10}}},
/*(11)  54- 12D*/ {4, {{ 1, 4, 0}, { 9, 6, 3}, {10, 9, 3}, { 2,10, 3}, { 0, 0, 0}, { 0, 0, 0}}},
/*(12)  57- 12A*/ {4, {{ 2, 3, 7}, {10, 0, 5}, { 1, 0,10}, { 9, 1,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(13)  57- 12B*/ {6, {{ 2, 0, 7}, { 1, 3, 9}, { 9, 3, 7}, { 0, 9, 7}, { 0, 5, 9}, { 5,10, 9}}},
/*(14)  57- 12C*/ {6, {{ 2, 3, 5}, { 1, 0, 9}, { 9, 0, 5}, { 3, 9, 5}, { 3, 7, 9}, { 7,10, 9}}},
/*(15)  57- 12D*/ {4, {{ 2, 0, 5}, { 1, 3, 7}, { 9, 1, 7}, {10, 9, 7}, { 0, 0, 0}, { 0, 0, 0}}},
/*(16)  75- 12A*/ {4, {{ 6, 9,11}, { 7, 1, 3}, { 4, 1, 7}, { 5, 4, 7}, { 0, 0, 0}, { 0, 0, 0}}},
/*(17)  75- 12B*/ {6, {{ 6, 1,11}, { 4, 9, 5}, { 5, 9,11}, { 1, 5,11}, { 1, 3, 5}, { 3, 7, 5}}},
/*(18)  75- 12C*/ {6, {{ 6, 9, 3}, { 4, 1, 5}, { 5, 1, 3}, { 9, 5, 3}, { 9,11, 5}, {11, 7, 5}}},
/*(19)  75- 12D*/ {4, {{ 6, 1, 3}, { 4, 9,11}, { 5, 4,11}, { 7, 5,11}, { 0, 0, 0}, { 0, 0, 0}}},
/*(20)  86- 12A*/ {4, {{ 0, 2, 5}, { 8, 1, 4}, { 3, 1, 8}, {11, 3, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(21)  86- 12B*/ {6, {{ 0, 1, 5}, { 3, 2,11}, {11, 2, 5}, { 1,11, 5}, { 1, 4,11}, { 4, 8,11}}},
/*(22)  86- 12C*/ {6, {{ 0, 2, 4}, { 3, 1,11}, {11, 1, 4}, { 2,11, 4}, { 2, 5,11}, { 5, 8,11}}},
/*(23)  86- 12D*/ {4, {{ 0, 1, 4}, { 3, 2, 5}, {11, 3, 5}, { 8,11, 5}, { 0, 0, 0}, { 0, 0, 0}}},
/*(24)  89- 12A*/ {4, {{ 3, 7, 2}, { 0, 6, 1}, {11, 6, 0}, { 8,11, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(25)  89- 12B*/ {6, {{ 3, 6, 2}, {11, 7, 8}, { 8, 7, 2}, { 6, 8, 2}, { 6, 1, 8}, { 1, 0, 8}}},
/*(26)  89- 12C*/ {6, {{ 3, 7, 1}, {11, 6, 8}, { 8, 6, 1}, { 7, 8, 1}, { 7, 2, 8}, { 2, 0, 8}}},
/*(27)  89- 12D*/ {4, {{ 3, 6, 1}, {11, 7, 2}, { 8,11, 2}, { 0, 8, 2}, { 0, 0, 0}, { 0, 0, 0}}},
/*(28)  99- 12A*/ {4, {{11, 6, 9}, { 8, 4,10}, {10, 4, 1}, {10, 1, 2}, { 0, 0, 0}, { 0, 0, 0}}},
/*(29)  99- 12B*/ {6, {{11, 4, 9}, { 2, 6, 1}, {11, 6, 2}, {11, 2, 4}, { 2, 8, 4}, { 2,10, 8}}},
/*(30)  99- 12C*/ {6, {{ 8, 6, 9}, { 2, 4, 1}, { 8, 4, 2}, { 8, 2, 6}, { 2,11, 6}, { 2,10,11}}},
/*(31)  99- 12D*/ {4, {{ 8, 4, 9}, {11, 6, 1}, {11, 1, 2}, {11, 2,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(32) 101- 12A*/ {4, {{ 8, 5,10}, {11, 4, 9}, { 0, 4,11}, { 3, 0,11}, { 0, 0, 0}, { 0, 0, 0}}},
/*(33) 101- 12B*/ {6, {{ 8, 4,10}, { 0, 5, 3}, { 3, 5,10}, { 4, 3,10}, { 4, 9, 3}, { 9,11, 3}}},
/*(34) 101- 12C*/ {6, {{ 8, 5, 9}, { 0, 4, 3}, { 3, 4, 9}, { 5, 3, 9}, { 5,10, 3}, {10,11, 3}}},
/*(35) 101- 12D*/ {4, {{ 8, 4, 9}, { 0, 5,10}, { 3, 0,10}, {11, 3,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(36) 106- 12A*/ {4, {{11, 6, 9}, { 8, 7,10}, { 3, 7, 8}, { 0, 3, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(37) 106- 12B*/ {6, {{11, 7, 9}, { 3, 6, 0}, { 0, 6, 9}, { 7, 0, 9}, { 7,10, 0}, {10, 8, 0}}},
/*(38) 106- 12C*/ {6, {{11, 6,10}, { 3, 7, 0}, { 0, 7,10}, { 6, 0,10}, { 6, 9, 0}, { 9, 8, 0}}},
/*(39) 106- 12D*/ {4, {{11, 7,10}, { 3, 6, 9}, { 0, 3, 9}, { 8, 0, 9}, { 0, 0, 0}, { 0, 0, 0}}},
/*(40) 108- 12A*/ {4, {{10, 8, 5}, { 2,11, 7}, { 9,11, 2}, { 1, 9, 2}, { 0, 0, 0}, { 0, 0, 0}}},
/*(41) 108- 12B*/ {6, {{10,11, 5}, { 9, 8, 1}, { 1, 8, 5}, {11, 1, 5}, {11, 7, 1}, { 7, 2, 1}}},
/*(42) 108- 12C*/ {6, {{10, 8, 7}, { 9,11, 1}, { 1,11, 7}, { 8, 1, 7}, { 8, 5, 1}, { 5, 2, 1}}},
/*(43) 108- 12D*/ {4, {{10,11, 7}, { 9, 8, 5}, { 1, 9, 5}, { 2, 1, 5}, { 0, 0, 0}, { 0, 0, 0}}},
/*(44) 120- 12A*/ {4, {{ 7, 2, 3}, { 6,10,11}, { 5,10, 6}, { 4, 5, 6}, { 0, 0, 0}, { 0, 0, 0}}},
/*(45) 120- 12B*/ {6, {{ 7,10, 3}, { 5, 2, 4}, { 4, 2, 3}, {10, 4, 3}, {10,11, 4}, {11, 6, 4}}},
/*(46) 120- 12C*/ {6, {{ 7, 2,11}, { 5,10, 4}, { 4,10,11}, { 2, 4,11}, { 2, 3, 4}, { 3, 6, 4}}},
/*(47) 120- 12D*/ {4, {{ 7,10,11}, { 5, 2, 3}, { 4, 5, 3}, { 6, 4, 3}, { 0, 0, 0}, { 0, 0, 0}}}
}; /* subtable12 end */

triangulated_cubes12 subtable13 [] = {
/*(0)  105- 13<0>*/ {4, {{ 5,10, 8}, { 4, 1, 0}, { 6, 9,11}, { 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(1)  105- 13<1>*/ {6, {{11, 7, 9}, { 7, 2, 9}, { 2, 6, 9}, { 2, 3, 6}, { 0, 4, 1}, { 8, 5,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(2)  105- 13<2>*/ {6, {{ 6, 1,11}, { 1, 0,11}, { 0, 9,11}, { 0, 4, 9}, { 5,10, 8}, { 7, 2, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(3)  105- 13<3>*/ {10, {{11,12, 9}, { 6,12,11}, { 1,12, 6}, { 0,12, 1}, { 5,12, 0}, {10,12, 5}, { 8,12,10}, { 4,12, 8}, { 9,12, 4}, { 3, 7, 2}, { 0, 0, 0}, { 0, 0, 0}}},
/*(4)  105- 13<4>*/ {6, {{ 9, 8, 6}, { 8, 5, 6}, { 5,11, 6}, { 5,10,11}, { 2, 3, 7}, { 1, 0, 4}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(5)  105- 13<5>*/ {10, {{ 9,12, 6}, {11,12, 9}, { 7,12,11}, { 2,12, 7}, { 0,12, 2}, { 4,12, 0}, { 1,12, 4}, { 3,12, 1}, { 6,12, 3}, {10, 8, 5}, { 0, 0, 0}, { 0, 0, 0}}},
/*(6)  105- 13<6>*/ {10, {{ 6,12,11}, { 9,12, 6}, { 8,12, 9}, { 5,12, 8}, { 2,12, 5}, { 3,12, 2}, { 7,12, 3}, {10,12, 7}, {11,12,10}, { 4, 1, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(7)  105- 13<7>*/ {6, {{ 5,10, 8}, { 1, 3, 6}, {11, 7, 9}, { 7, 4, 9}, { 7, 2, 4}, { 2, 0, 4}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(8)  105- 13<8>*/ {6, {{ 4, 9, 0}, { 9,11, 0}, {11, 1, 0}, {11, 6, 1}, { 7, 2, 3}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(9)  105- 13<9>*/ {10, {{ 8,12,10}, { 5,12, 8}, { 2,12, 5}, { 3,12, 2}, { 6,12, 3}, { 9,12, 6}, {11,12, 9}, { 7,12,11}, {10,12, 7}, { 0, 4, 1}, { 0, 0, 0}, { 0, 0, 0}}},
/*(10) 105- 13<10>*/ {10, {{ 7,12, 3}, { 2,12, 7}, { 0,12, 2}, { 4,12, 0}, { 9,12, 4}, {11,12, 9}, { 6,12,11}, { 1,12, 6}, { 3,12, 1}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(11) 105- 13<11>*/ {12, {{ 2,12, 7}, { 0,12, 2}, { 5,12, 0}, {10,12, 5}, { 8,12,10}, { 4,12, 8}, { 9,12, 4}, {11,12, 9}, { 6,12,11}, { 1,12, 6}, { 3,12, 1}, { 7,12, 3}}},
/*(12) 105- 13<12>*/ {8, {{ 0, 6, 1}, {11, 6, 0}, { 4,11, 0}, { 9,11, 4}, { 5, 2, 3}, { 8, 5, 3}, { 7, 8, 3}, {10, 8, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(13) 105- 13<13>*/ {12, {{ 5,12, 2}, { 8,12, 5}, { 9,12, 8}, {12, 9, 4}, {12, 4, 0}, {12, 0, 1}, {12, 1, 6}, {11,12, 6}, {10,12,11}, { 7,12,10}, { 3,12, 7}, { 2,12, 3}}},
/*(14) 105- 13<14>*/ {12, {{10,12, 7}, { 8,12,10}, { 4,12, 8}, {12, 4, 9}, {12, 9,11}, {12,11, 6}, {12, 6, 1}, { 0,12, 1}, { 5,12, 0}, { 2,12, 5}, { 3,12, 2}, { 7,12, 3}}},
/*(15) 105- 13<15>*/ {10, {{10,12, 7}, { 8,12,10}, { 4,12, 8}, {12, 4, 9}, {12, 9,11}, {12,11, 6}, {12, 6, 1}, {12, 1, 3}, { 7,12, 3}, { 2, 0, 5}, { 0, 0, 0}, { 0, 0, 0}}},
/*(16) 105- 13<16>*/ {6, {{ 3, 6, 2}, { 6, 9, 2}, { 9, 7, 2}, { 9,11, 7}, { 8, 5,10}, { 0, 4, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(17) 105- 13<17>*/ {10, {{ 8,12,10}, { 5,12, 8}, { 2,12, 5}, { 3,12, 2}, { 6,12, 3}, { 9,12, 6}, {11,12, 9}, { 7,12,11}, {10,12, 7}, { 0, 4, 1}, { 0, 0, 0}, { 0, 0, 0}}},
/*(18) 105- 13<18>*/ {8, {{ 2,11, 7}, { 9,11, 2}, { 3, 9, 2}, { 6, 9, 3}, { 0, 5,10}, { 1, 0,10}, { 8, 1,10}, { 4, 1, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(19) 105- 13<19>*/ {12, {{ 4,12, 8}, { 1,12, 4}, { 3,12, 1}, {12, 3, 6}, {12, 6, 9}, {12, 9,11}, {12,11, 7}, { 2,12, 7}, { 0,12, 2}, { 5,12, 0}, {10,12, 5}, { 8,12,10}}},
/*(20) 105- 13<20>*/ {10, {{ 1,12, 4}, { 0,12, 1}, { 5,12, 0}, {10,12, 5}, {11,12,10}, { 6,12,11}, { 9,12, 6}, { 8,12, 9}, { 4,12, 8}, { 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}}},
/*(21) 105- 13<21>*/ {12, {{ 5,12, 8}, { 2,12, 5}, { 0,12, 2}, { 4,12, 0}, { 1,12, 4}, { 3,12, 1}, { 6,12, 3}, { 9,12, 6}, {11,12, 9}, { 7,12,11}, {10,12, 7}, { 8,12,10}}},
/*(22) 105- 13<22>*/ {12, {{ 0,12, 5}, { 1,12, 0}, { 6,12, 1}, {12, 6, 3}, {12, 3, 2}, {12, 2, 7}, {12, 7,11}, { 9,12,11}, { 4,12, 9}, { 8,12, 4}, {10,12, 8}, { 5,12,10}}},
/*(23) 105- 13<23>*/ {10, {{ 4,12, 8}, { 1,12, 4}, { 3,12, 1}, {12, 3, 6}, {12, 6, 9}, {12, 9,11}, {12,11, 7}, {12, 7,10}, { 8,12,10}, { 5, 2, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(24) 105- 13<24>*/ {10, {{ 0,12, 1}, { 4,12, 0}, { 9,12, 4}, {11,12, 9}, { 7,12,11}, { 2,12, 7}, { 3,12, 2}, { 6,12, 3}, { 1,12, 6}, { 8, 5,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(25) 105- 13<25>*/ {6, {{10, 8, 5}, { 3, 6, 1}, { 4, 9, 0}, { 9, 2, 0}, { 9,11, 2}, {11, 7, 2}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(26) 105- 13<26>*/ {12, {{ 4,12, 9}, { 0,12, 4}, { 2,12, 0}, {12, 2, 5}, {12, 5, 8}, {12, 8,10}, {12,10, 7}, { 3,12, 7}, { 1,12, 3}, { 6,12, 1}, {11,12, 6}, { 9,12,11}}},
/*(27) 105- 13<27>*/ {10, {{ 9,12, 4}, {11,12, 9}, { 7,12,11}, {12, 7,10}, {12,10, 8}, {12, 8, 5}, {12, 5, 2}, {12, 2, 0}, { 4,12, 0}, { 1, 3, 6}, { 0, 0, 0}, { 0, 0, 0}}},
/*(28) 105- 13<28>*/ {12, {{ 8,12, 9}, { 5,12, 8}, { 2,12, 5}, {12, 2, 0}, {12, 0, 4}, {12, 4, 1}, {12, 1, 3}, { 7,12, 3}, {10,12, 7}, {11,12,10}, { 6,12,11}, { 9,12, 6}}},
/*(29) 105- 13<29>*/ {10, {{ 6,12, 3}, { 9,12, 6}, { 8,12, 9}, {12, 8, 4}, {12, 4, 1}, {12, 1, 0}, {12, 0, 5}, {12, 5, 2}, { 3,12, 2}, { 7,10,11}, { 0, 0, 0}, { 0, 0, 0}}},
/*(30) 105- 13<30>*/ {8, {{10, 8, 4}, { 7,10, 4}, { 9, 7, 4}, {11, 7, 9}, { 6, 1, 0}, { 0, 5, 6}, { 5, 3, 6}, { 5, 2, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(31) 105- 13<31>*/ {6, {{ 5, 2, 0}, { 6, 3, 1}, { 9, 7, 4}, {11, 7, 9}, { 7,10, 4}, {10, 8, 4}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(32) 105- 13<32>*/ {6, {{10,11, 5}, {11, 6, 5}, { 6, 8, 5}, { 6, 9, 8}, { 1, 0, 4}, { 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(33) 105- 13<33>*/ {8, {{ 5, 9, 8}, { 6, 9, 5}, {10, 6, 5}, {11, 6,10}, { 2, 0, 4}, { 7, 2, 4}, { 1, 7, 4}, { 3, 7, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(34) 105- 13<34>*/ {10, {{ 7,12, 3}, { 2,12, 7}, { 0,12, 2}, { 4,12, 0}, { 9,12, 4}, {11,12, 9}, { 6,12,11}, { 1,12, 6}, { 3,12, 1}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}},
/*(35) 105- 13<35>*/ {12, {{ 2,12, 0}, { 7,12, 2}, {11,12, 7}, {12,11,10}, {12,10, 5}, {12, 5, 8}, {12, 8, 9}, { 6,12, 9}, { 3,12, 6}, { 1,12, 3}, { 4,12, 1}, { 0,12, 4}}},
/*(36) 105- 13<36>*/ {10, {{ 1,12, 4}, { 0,12, 1}, { 5,12, 0}, {10,12, 5}, {11,12,10}, { 6,12,11}, { 9,12, 6}, { 8,12, 9}, { 4,12, 8}, { 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}}},
/*(37) 105- 13<37>*/ {12, {{ 3,12, 1}, { 7,12, 3}, {10,12, 7}, {12,10,11}, {12,11, 6}, {12, 6, 9}, {12, 9, 8}, { 5,12, 8}, { 2,12, 5}, { 0,12, 2}, { 4,12, 0}, { 1,12, 4}}},
/*(38) 105- 13<38>*/ {12, {{ 0,12, 1}, { 5,12, 0}, { 2,12, 5}, { 3,12, 2}, { 7,12, 3}, {10,12, 7}, {11,12,10}, { 6,12,11}, { 9,12, 6}, { 8,12, 9}, { 4,12, 8}, { 1,12, 4}}},
/*(39) 105- 13<39>*/ {10, {{ 3,12, 1}, { 7,12, 3}, {10,12, 7}, {12,10,11}, {12,11, 6}, {12, 6, 9}, {12, 9, 8}, {12, 8, 4}, { 1,12, 4}, { 0, 5, 2}, { 0, 0, 0}, { 0, 0, 0}}},
/*(40) 105- 13<40>*/ {10, {{ 5,12, 8}, {10,12, 5}, {11,12,10}, { 6,12,11}, { 1,12, 6}, { 0,12, 1}, { 4,12, 0}, { 9,12, 4}, { 8,12, 9}, { 7, 2, 3}, { 0, 0, 0}, { 0, 0, 0}}},
/*(41) 105- 13<41>*/ {12, {{ 7,12,11}, { 2,12, 7}, { 0,12, 2}, {12, 0, 5}, {12, 5,10}, {12,10, 8}, {12, 8, 4}, { 1,12, 4}, { 3,12, 1}, { 6,12, 3}, { 9,12, 6}, {11,12, 9}}},
/*(42) 105- 13<42>*/ {6, {{ 3, 7, 2}, { 4, 9, 8}, {10,11, 5}, {11, 0, 5}, {11, 6, 0}, { 6, 1, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(43) 105- 13<43>*/ {10, {{ 9,12, 4}, {11,12, 9}, { 7,12,11}, {12, 7,10}, {12,10, 8}, {12, 8, 5}, {12, 5, 2}, {12, 2, 0}, { 4,12, 0}, { 1, 3, 6}, { 0, 0, 0}, { 0, 0, 0}}},
/*(44) 105- 13<44>*/ {12, {{10,12,11}, { 5,12,10}, { 0,12, 5}, {12, 0, 2}, {12, 2, 7}, {12, 7, 3}, {12, 3, 1}, { 4,12, 1}, { 8,12, 4}, { 9,12, 8}, { 6,12, 9}, {11,12, 6}}},
/*(45) 105- 13<45>*/ {8, {{ 3, 7,10}, { 1, 3,10}, {11, 1,10}, { 6, 1,11}, { 9, 8, 5}, { 5, 2, 9}, { 2, 4, 9}, { 2, 0, 4}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(46) 105- 13<46>*/ {10, {{11,12,10}, { 6,12,11}, { 1,12, 6}, {12, 1, 3}, {12, 3, 7}, {12, 7, 2}, {12, 2, 0}, {12, 0, 5}, {10,12, 5}, { 8, 4, 9}, { 0, 0, 0}, { 0, 0, 0}}},
/*(47) 105- 13<47>*/ {6, {{ 2, 0, 5}, { 9, 4, 8}, {11, 1,10}, { 6, 1,11}, { 1, 3,10}, { 3, 7,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(48) 105- 13<48>*/ {10, {{ 2,12, 7}, { 3,12, 2}, { 6,12, 3}, { 9,12, 6}, { 8,12, 9}, { 5,12, 8}, {10,12, 5}, {11,12,10}, { 7,12,11}, { 1, 0, 4}, { 0, 0, 0}, { 0, 0, 0}}},
/*(49) 105- 13<49>*/ {12, {{ 3,12, 6}, { 2,12, 3}, { 5,12, 2}, {12, 5, 0}, {12, 0, 1}, {12, 1, 4}, {12, 4, 8}, {10,12, 8}, { 7,12,10}, {11,12, 7}, { 9,12,11}, { 6,12, 9}}},
/*(50) 105- 13<50>*/ {12, {{ 1,12, 6}, { 0,12, 1}, { 5,12, 0}, {12, 5, 2}, {12, 2, 3}, {12, 3, 7}, {12, 7,10}, { 8,12,10}, { 4,12, 8}, { 9,12, 4}, {11,12, 9}, { 6,12,11}}},
/*(51) 105- 13<51>*/ {8, {{ 4, 1, 3}, { 8, 4, 3}, { 6, 8, 3}, { 9, 8, 6}, {11, 7, 2}, { 2, 0,11}, { 0,10,11}, { 0, 5,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(52) 105- 13<52>*/ {6, {{ 4, 1, 0}, {10,11, 7}, { 3, 6, 2}, { 6, 5, 2}, { 6, 9, 5}, { 9, 8, 5}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(53) 105- 13<53>*/ {10, {{ 6,12, 3}, { 9,12, 6}, { 8,12, 9}, {12, 8, 4}, {12, 4, 1}, {12, 1, 0}, {12, 0, 5}, {12, 5, 2}, { 3,12, 2}, { 7,10,11}, { 0, 0, 0}, { 0, 0, 0}}},
/*(54) 105- 13<54>*/ {10, {{11,12,10}, { 6,12,11}, { 1,12, 6}, {12, 1, 3}, {12, 3, 7}, {12, 7, 2}, {12, 2, 0}, {12, 0, 5}, {10,12, 5}, { 8, 4, 9}, { 0, 0, 0}, { 0, 0, 0}}},
/*(55) 105- 13<55>*/ {6, {{ 0, 5, 2}, {11,10, 7}, { 6, 8, 3}, { 9, 8, 6}, { 8, 4, 3}, { 4, 1, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(56) 105- 13<56>*/ {12, {{ 9,12, 6}, { 8,12, 9}, { 4,12, 8}, { 1,12, 4}, { 0,12, 1}, { 5,12, 0}, { 2,12, 5}, { 3,12, 2}, { 7,12, 3}, {10,12, 7}, {11,12,10}, { 6,12,11}}},
/*(57) 105- 13<57>*/ {10, {{ 7,12,11}, { 2,12, 7}, { 0,12, 2}, {12, 0, 5}, {12, 5,10}, {12,10, 8}, {12, 8, 4}, {12, 4, 9}, {11,12, 9}, { 6, 1, 3}, { 0, 0, 0}, { 0, 0, 0}}},
/*(58) 105- 13<58>*/ {10, {{ 1,12, 6}, { 0,12, 1}, { 5,12, 0}, {12, 5, 2}, {12, 2, 3}, {12, 3, 7}, {12, 7,10}, {12,10,11}, { 6,12,11}, { 9, 8, 4}, { 0, 0, 0}, { 0, 0, 0}}},
/*(59) 105- 13<59>*/ {6, {{ 6, 1, 3}, { 5, 0, 2}, {10, 4, 7}, { 8, 4,10}, { 4, 9, 7}, { 9,11, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(60) 105- 13<60>*/ {10, {{ 8,12, 9}, { 5,12, 8}, { 2,12, 5}, {12, 2, 0}, {12, 0, 4}, {12, 4, 1}, {12, 1, 3}, {12, 3, 6}, { 9,12, 6}, {11, 7,10}, { 0, 0, 0}, { 0, 0, 0}}},
/*(61) 105- 13<61>*/ {6, {{11, 7,10}, { 0, 2, 5}, { 4, 3, 8}, { 1, 3, 4}, { 3, 6, 8}, { 6, 9, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(62) 105- 13<62>*/ {6, {{ 9, 8, 4}, { 2, 5, 0}, { 3,10, 1}, { 7,10, 3}, {10,11, 1}, {11, 6, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}},
/*(63) 105- 13<63>*/ {4, {{ 7,10,11}, { 4, 9, 8}, { 5, 2, 0}, { 6, 3, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}
}; /* subtable13 end */

/* For non-ambigious configurations, the first value is the number of
   triangles to tesselate the surface within the cube.  The remaining
   values are the edge numbers of the vertices of each of the triangles.
   Edges are numbered starting from zero.   For ambiguous configurations, 
   the first value is zero to act as a flag that the subtable will be used.
   The next number (where the first edge number would normally be) is the
   the case (to find the correct subtable), then the default subtable location,    
   followed by the number of ambiguous faces in the configuration.  This is 
   followed by the face numbers (also starting from zero since they are really 
   array indices into face []) of each of the ambiguous faces.  These are then
   followed by the base index of the 1st element in the subtable corresponding
   to the various tesselations for that entry (since there can be more 
   subtable entries than will fit here, only the first one is given).  The
   number of subtable entries for each entry here is 2**(# of ambiguous faces).
*/

#define NENTRIES 	127 
triangulated_cubes cell_info [NENTRIES] = {
 { 1, {{ 0, 4, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:   1-*/
 { 1, {{ 0, 2, 5}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:   2-*/
 { 2, {{ 1, 2, 5}, { 5, 4, 1}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:   3-*/
 { 1, {{ 1, 6, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:   4-*/
 { 2, {{ 0, 4, 6}, { 6, 3, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:   5-*/
 { 0, {{ 3, 0, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:   6-*/
 { 3, {{ 3, 2, 5}, { 3, 5, 4}, { 3, 4, 6}, { 0, 0, 0}}}, /* 5:   7-*/
 { 1, {{ 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:   8-*/
 { 0, {{ 3, 2, 1}, { 0, 2, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:   9-*/
 { 2, {{ 0, 3, 7}, { 7, 5, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  10-*/
 { 3, {{ 1, 3, 7}, { 1, 7, 5}, { 1, 5, 4}, { 0, 0, 0}}}, /* 5:  11-*/
 { 2, {{ 1, 6, 2}, { 2, 6, 7}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  12-*/
 { 3, {{ 2, 0, 4}, { 2, 4, 6}, { 2, 6, 7}, { 0, 0, 0}}}, /* 5:  13-*/
 { 3, {{ 0, 1, 6}, { 0, 6, 7}, { 0, 7, 5}, { 0, 0, 0}}}, /* 5:  14-*/
 { 2, {{ 4, 6, 7}, { 4, 7, 5}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8:  15-*/
 { 1, {{ 4, 8, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:  16-*/
 { 2, {{ 0, 8, 9}, { 9, 1, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  17-*/
 { 0, {{ 3, 4, 1}, { 1, 4, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  18-*/
 { 3, {{ 5, 8, 9}, { 5, 9, 1}, { 5, 1, 2}, { 0, 0, 0}}}, /* 5:  19-*/
 { 0, {{ 3, 6, 1}, { 2, 6, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  20-*/
 { 3, {{ 9, 6, 3}, { 9, 3, 0}, { 9, 0, 8}, { 0, 0, 0}}}, /* 5:  21-*/
 { 0, {{ 7, 0, 3}, { 0, 1, 2}, { 0, 0, 0}, { 0, 0, 0}}}, /* 7:  22-*/
 { 4, {{ 9, 6, 3}, { 3, 8, 9}, { 8, 3, 2}, { 2, 5, 8}}}, /* 9:  23-*/
 { 2, {{ 2, 3, 7}, { 4, 8, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4:  24-*/
 { 0, {{ 6, 0, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  25-*/
 { 0, {{ 6, 2, 1}, { 1, 2, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  26-*/
 { 4, {{ 7, 5, 9}, { 5, 8, 9}, { 9, 1, 7}, { 1, 3, 7}}}, /*11:  27-*/
 { 0, {{ 6, 4, 1}, { 2, 4, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  28-*/
 { 4, {{ 2, 0, 7}, { 0, 6, 7}, { 0, 8, 6}, { 8, 9, 6}}}, /*14:  29-*/
 { 0, {{12, 0, 2}, { 1, 2, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /*12:  30-*/
 { 3, {{ 6, 7, 9}, { 7, 5, 9}, { 5, 8, 9}, { 0, 0, 0}}}, /* 5:  31-*/
 { 1, {{ 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:  32-*/
 { 0, {{ 3, 8, 1}, { 1, 8, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  33-*/
 { 2, {{ 0, 2,10}, {10, 8, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  34-*/
 { 3, {{ 8, 4, 1}, { 8, 1, 2}, { 8, 2,10}, { 0, 0, 0}}}, /* 5:  35-*/
 { 2, {{ 1, 6, 3}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4:  36-*/
 { 0, {{ 6, 6, 1}, { 1, 6, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  37-*/
 { 0, {{ 6, 8, 1}, { 0, 8, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  38-*/
 { 4, {{ 3, 2, 6}, { 2, 4, 6}, { 2,10, 4}, {10, 8, 4}}}, /*14:  39-*/
 { 0, {{ 3,10, 1}, { 3,10, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  40-*/
 { 0, {{ 7, 8, 3}, { 0, 1, 3}, { 8, 0, 0}, { 0, 0, 0}}}, /* 7:  41-*/
 { 3, {{ 7,10, 8}, { 7, 8, 0}, { 7, 0, 3}, { 0, 0, 0}}}, /* 5:  42-*/
 { 4, {{ 4, 1, 3}, { 4, 3, 8}, { 3, 7, 8}, { 7,10, 8}}}, /* 9:  43-*/
 { 0, {{ 6,10, 1}, { 3,10, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  44-*/
 { 0, {{12, 4, 2}, { 1, 3, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /*12:  45-*/
 { 4, {{ 6, 7, 8}, { 7,10, 8}, { 8, 0, 6}, { 0, 1, 6}}}, /*11:  46-*/
 { 3, {{ 4, 6, 8}, { 6, 7, 8}, { 7,10, 8}, { 0, 0, 0}}}, /* 5:  47-*/
 { 2, {{ 4, 5,10}, {10, 9, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  48-*/
 { 3, {{ 0, 5,10}, { 0,10, 9}, { 0, 9, 1}, { 0, 0, 0}}}, /* 5:  49-*/
 { 3, {{ 4, 0, 2}, { 4, 2,10}, { 4,10, 9}, { 0, 0, 0}}}, /* 5:  50-*/
 { 2, {{ 1, 2,10}, { 1,10, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8:  51-*/
 { 0, {{ 6,12, 1}, { 2,12, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  52-*/
 { 4, {{10, 9, 3}, { 9, 6, 3}, { 3, 0,10}, { 0, 5,10}}}, /*11:  53-*/
 { 0, {{12, 8, 2}, { 0, 2, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /*12:  54-*/
 { 3, {{ 2,10, 3}, {10, 9, 3}, { 9, 6, 3}, { 0, 0, 0}}}, /* 5:  55-*/
 { 0, {{ 6,14, 1}, { 3,14, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  56-*/
 { 0, {{12,12, 2}, { 0, 3,12}, { 0, 0, 0}, { 0, 0, 0}}}, /*12:  57-*/
 { 4, {{ 4,10, 9}, { 0, 3, 4}, { 4, 3,10}, { 3, 7,10}}}, /*14:  58-*/
 { 3, {{10, 9, 7}, { 9, 1, 7}, { 1, 3, 7}, { 0, 0, 0}}}, /* 5:  59-*/
 { 0, {{10, 0, 2}, { 2, 3, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /*10:  60-*/
 { 0, {{ 6,17, 1}, { 3,16, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  61-*/
 { 0, {{ 6,19, 1}, { 2,18, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  62-*/
 { 2, {{ 6, 7,10}, {10, 9, 6}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  63-*/
 { 1, {{ 6, 9,11}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:  64-*/
 { 0, {{ 3,12, 1}, { 2,12, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  65-*/
 { 2, {{ 0, 2, 5}, { 6, 9,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4:  66-*/
 { 0, {{ 6,20, 1}, { 2,20, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  67-*/
 { 2, {{ 1, 9,11}, {11, 3, 1}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  68-*/
 { 3, {{ 4, 9,11}, { 4,11, 3}, { 4, 3, 0}, { 0, 0, 0}}}, /* 5:  69-*/
 { 0, {{ 6,22, 1}, { 0,22, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  70-*/
 { 4, {{ 5, 4,11}, { 4, 9,11}, {11, 3, 5}, { 3, 2, 5}}}, /*11:  71-*/
 { 0, {{ 3,14, 1}, { 4,14, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  72-*/
 { 0, {{ 7,16, 3}, { 0, 2, 4}, {16, 0, 0}, { 0, 0, 0}}}, /* 7:  73-*/
 { 0, {{ 6,24, 1}, { 4,24, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  74-*/
 { 0, {{12,16, 2}, { 2, 4,16}, { 0, 0, 0}, { 0, 0, 0}}}, /*12:  75-*/
 { 3, {{11, 7, 2}, {11, 2, 1}, {11, 1, 9}, { 0, 0, 0}}}, /* 5:  76-*/
 { 4, {{ 0, 4, 9}, { 0, 9, 2}, { 2, 9,11}, { 2,11, 7}}}, /* 9:  77-*/
 { 4, {{ 0, 1, 5}, { 1, 7, 5}, { 1, 9, 7}, { 9,11, 7}}}, /*14:  78-*/
 { 3, {{ 7, 5,11}, { 5, 4,11}, { 4, 9,11}, { 0, 0, 0}}}, /* 5:  79-*/
 { 2, {{ 4, 8,11}, {11, 6, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  80-*/
 { 3, {{ 6, 1, 0}, { 6, 0, 8}, { 6, 8,11}, { 0, 0, 0}}}, /* 5:  81-*/
 { 0, {{ 6,26, 1}, { 1,26, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  82-*/
 { 4, {{ 5, 8, 2}, { 8, 1, 2}, { 8,11, 1}, {11, 6, 1}}}, /*14:  83-*/
 { 3, {{ 1, 4, 8}, { 1, 8,11}, { 1,11, 3}, { 0, 0, 0}}}, /* 5:  84-*/
 { 2, {{ 0, 8,11}, { 0,11, 3}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8:  85-*/
 { 0, {{12,20, 2}, { 0, 1,20}, { 0, 0, 0}, { 0, 0, 0}}}, /*12:  86-*/
 { 3, {{ 8,11, 5}, {11, 3, 5}, { 3, 2, 5}, { 0, 0, 0}}}, /* 5:  87-*/
 { 0, {{ 6,28, 1}, { 4,28, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  88-*/
 { 0, {{12,24, 2}, { 0, 4,24}, { 0, 0, 0}, { 0, 0, 0}}}, /*12:  89-*/
 { 0, {{10, 4, 2}, { 1, 4, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /*10:  90-*/
 { 0, {{ 6,31, 1}, { 4,30, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  91-*/
 { 4, {{ 4, 8, 7}, { 8,11, 7}, { 7, 2, 4}, { 2, 1, 4}}}, /*11:  92-*/
 { 3, {{ 0, 8, 2}, { 8,11, 2}, {11, 7, 2}, { 0, 0, 0}}}, /* 5:  93-*/
 { 0, {{ 6,33, 1}, { 1,32, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  94-*/
 { 2, {{ 5, 8,11}, {11, 7, 5}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  95-*/
 { 0, {{ 3,16, 1}, { 5,16, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  96-*/
 { 0, {{ 7,24, 3}, { 1, 2, 5}, {24, 0, 0}, { 0, 0, 0}}}, /* 7:  97-*/
 { 0, {{ 6,34, 1}, { 5,34, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6:  98-*/
 { 0, {{12,28, 2}, { 2, 5,28}, { 0, 0, 0}, { 0, 0, 0}}}, /*12:  99-*/
 { 0, {{ 6,36, 1}, { 5,36, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6: 100-*/
 { 0, {{12,32, 2}, { 1, 5,32}, { 0, 0, 0}, { 0, 0, 0}}}, /*12: 101-*/
 { 0, {{10, 8, 2}, { 0, 5, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /*10: 102-*/
 { 0, {{ 6,39, 1}, { 5,38, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6: 103-*/
 { 0, {{ 7,32, 3}, { 3, 4, 5}, {32, 0, 0}, { 0, 0, 0}}}, /* 7: 104-*/
 { 0, {{13, 0, 6}, { 0, 1, 2}, { 3, 4, 5}, { 0, 0, 0}}}, /*13: 105-*/
 { 0, {{12,36, 2}, { 4, 5,36}, { 0, 0, 0}, { 0, 0, 0}}}, /*12: 106-*/
 { 0, {{ 7,47, 3}, { 2, 4, 5}, {40, 0, 0}, { 0, 0, 0}}}, /* 7: 107-*/
 { 0, {{12,40, 2}, { 3, 5,40}, { 0, 0, 0}, { 0, 0, 0}}}, /*12: 108-*/
 { 0, {{ 7,55, 3}, { 1, 3, 5}, {48, 0, 0}, { 0, 0, 0}}}, /* 7: 109-*/
 { 0, {{ 6,41, 1}, { 5,40, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6: 110-*/
 { 0, {{ 3,19, 1}, { 5,18, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 111-*/
 { 3, {{10,11, 4}, {11, 6, 4}, {10, 4, 5}, { 0, 0, 0}}}, /* 5: 112-*/
 { 4, {{ 5,10,11}, {11, 6, 5}, { 5, 6, 0}, { 6, 1, 0}}}, /* 9: 113-*/
 { 4, {{ 0, 2,11}, { 2,10,11}, {11, 6, 0}, { 6, 4, 0}}}, /*11: 114-*/
 { 3, {{ 1, 2, 6}, { 2,10, 6}, {10,11, 6}, { 0, 0, 0}}}, /* 5: 115-*/
 { 4, {{ 5,10,11}, { 5,11, 3}, { 4, 5, 3}, { 4, 3, 1}}}, /*14: 116-*/
 { 3, {{11, 3,10}, { 3, 0,10}, { 0, 5,10}, { 0, 0, 0}}}, /* 5: 117-*/
 { 0, {{ 6,43, 1}, { 0,42, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6: 118-*/
 { 2, {{ 2,10,11}, {11, 3, 2}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 119-*/
 { 0, {{12,44, 2}, { 3, 4,44}, { 0, 0, 0}, { 0, 0, 0}}}, /*12: 120-*/
 { 0, {{ 7,63, 3}, { 0, 3, 4}, {56, 0, 0}, { 0, 0, 0}}}, /* 7: 121-*/
 { 0, {{ 6,45, 1}, { 4,44, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6: 122-*/
 { 0, {{ 3,21, 1}, { 4,20, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 123-*/
 { 0, {{ 6,47, 1}, { 3,46, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 6: 124-*/
 { 0, {{ 3,23, 1}, { 3,22, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 125-*/
 { 2, {{ 7,10,11}, { 0, 1, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4: 126-*/
 { 1, {{ 7,10,11}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 127-*/
};
#ifdef FULL_TABLE 
#define NENTRIES 	255 
triangulated_cubes cell_info [NENTRIES] = {
 { 1, {{ 0, 4, 1}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:   1-*/
 { 1, {{ 0, 2, 5}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:   2-*/
 { 2, {{ 1, 2, 5}, { 5, 4, 1}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:   3-*/
 { 1, {{ 1, 6, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:   4-*/
 { 2, {{ 0, 4, 6}, { 6, 3, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:   5-*/
 { 2, {{ 0, 2, 5}, { 1, 6, 3}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:   6-*/
 { 3, {{ 3, 2, 5}, { 3, 5, 4}, { 3, 4, 6}, { 0, 0, 0}}}, /* 5:   7-*/
 { 1, {{ 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:   8-*/
 { 2, {{ 0, 4, 1}, { 2, 3, 7}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:   9-*/
 { 2, {{ 0, 3, 7}, { 7, 5, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  10-*/
 { 3, {{ 1, 3, 7}, { 1, 7, 5}, { 1, 5, 4}, { 0, 0, 0}}}, /* 5:  11-*/
 { 2, {{ 1, 6, 2}, { 2, 6, 7}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  12-*/
 { 3, {{ 2, 0, 4}, { 2, 4, 6}, { 2, 6, 7}, { 0, 0, 0}}}, /* 5:  13-*/
 { 3, {{ 0, 1, 6}, { 0, 6, 7}, { 0, 7, 5}, { 0, 0, 0}}}, /* 5:  14-*/
 { 2, {{ 4, 6, 7}, { 4, 7, 5}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8:  15-*/
 { 1, {{ 4, 8, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:  16-*/
 { 2, {{ 0, 8, 9}, { 9, 1, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  17-*/
 { 2, {{ 0, 2, 5}, { 4, 8, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  18-*/
 { 3, {{ 5, 8, 9}, { 5, 9, 1}, { 5, 1, 2}, { 0, 0, 0}}}, /* 5:  19-*/
 { 2, {{ 1, 6, 3}, { 4, 8, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  20-*/
 { 3, {{ 9, 6, 3}, { 9, 3, 0}, { 9, 0, 8}, { 0, 0, 0}}}, /* 5:  21-*/
 { 3, {{ 0, 2, 5}, { 1, 6, 3}, { 4, 8, 9}, { 0, 0, 0}}}, /* 7:  22-*/
 { 4, {{ 9, 6, 3}, { 3, 8, 9}, { 8, 3, 2}, { 2, 5, 8}}}, /* 9:  23-*/
 { 2, {{ 2, 3, 7}, { 4, 8, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4:  24-*/
 { 3, {{ 0, 8, 9}, { 9, 1, 0}, { 2, 3, 7}, { 0, 0, 0}}}, /* 6:  25-*/
 { 3, {{ 0, 3, 7}, { 7, 5, 0}, { 4, 8, 9}, { 0, 0, 0}}}, /* 6:  26-*/
 { 4, {{ 7, 5, 9}, { 5, 8, 9}, { 9, 1, 7}, { 1, 3, 7}}}, /*11:  27-*/
 { 3, {{ 1, 6, 7}, { 7, 2, 1}, { 4, 8, 9}, { 0, 0, 0}}}, /* 6:  28-*/
 { 4, {{ 2, 0, 7}, { 0, 6, 7}, { 0, 8, 6}, { 8, 9, 6}}}, /*14:  29-*/
 { 4, {{ 0, 1, 6}, { 0, 6, 7}, { 0, 7, 5}, { 4, 8, 9}}}, /*12:  30-*/
 { 3, {{ 6, 7, 9}, { 7, 5, 9}, { 5, 8, 9}, { 0, 0, 0}}}, /* 5:  31-*/
 { 1, {{ 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:  32-*/
 { 2, {{ 0, 4, 1}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  33-*/
 { 2, {{ 0, 2,10}, {10, 8, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  34-*/
 { 3, {{ 8, 4, 1}, { 8, 1, 2}, { 8, 2,10}, { 0, 0, 0}}}, /* 5:  35-*/
 { 2, {{ 1, 6, 3}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4:  36-*/
 { 3, {{ 0, 4, 6}, { 6, 3, 0}, { 5,10, 8}, { 0, 0, 0}}}, /* 6:  37-*/
 { 3, {{ 0, 2,10}, {10, 8, 0}, { 1, 6, 3}, { 0, 0, 0}}}, /* 6:  38-*/
 { 4, {{ 3, 2, 6}, { 2, 4, 6}, { 2,10, 4}, {10, 8, 4}}}, /*14:  39-*/
 { 2, {{ 2, 3, 7}, { 5,10, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  40-*/
 { 3, {{ 0, 4, 1}, { 2, 3, 7}, { 5,10, 8}, { 0, 0, 0}}}, /* 7:  41-*/
 { 3, {{ 7,10, 8}, { 7, 8, 0}, { 7, 0, 3}, { 0, 0, 0}}}, /* 5:  42-*/
 { 4, {{ 4, 1, 3}, { 4, 3, 8}, { 3, 7, 8}, { 7,10, 8}}}, /* 9:  43-*/
 { 3, {{ 1, 6, 7}, { 7, 2, 1}, { 5,10, 8}, { 0, 0, 0}}}, /* 6:  44-*/
 { 4, {{ 2, 0, 4}, { 2, 4, 6}, { 2, 6, 7}, { 5,10, 8}}}, /*12:  45-*/
 { 4, {{ 6, 7, 8}, { 7,10, 8}, { 8, 0, 6}, { 0, 1, 6}}}, /*11:  46-*/
 { 3, {{ 4, 6, 8}, { 6, 7, 8}, { 7,10, 8}, { 0, 0, 0}}}, /* 5:  47-*/
 { 2, {{ 4, 5,10}, {10, 9, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  48-*/
 { 3, {{ 0, 5,10}, { 0,10, 9}, { 0, 9, 1}, { 0, 0, 0}}}, /* 5:  49-*/
 { 3, {{ 4, 0, 2}, { 4, 2,10}, { 4,10, 9}, { 0, 0, 0}}}, /* 5:  50-*/
 { 2, {{ 1, 2,10}, { 1,10, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8:  51-*/
 { 3, {{ 4, 5,10}, {10, 9, 4}, { 1, 6, 3}, { 0, 0, 0}}}, /* 6:  52-*/
 { 4, {{10, 9, 3}, { 9, 6, 3}, { 3, 0,10}, { 0, 5,10}}}, /*11:  53-*/
 { 4, {{ 4, 0, 2}, { 4, 2,10}, { 4,10, 9}, { 1, 6, 3}}}, /*12:  54-*/
 { 3, {{ 2,10, 3}, {10, 9, 3}, { 9, 6, 3}, { 0, 0, 0}}}, /* 5:  55-*/
 { 3, {{ 4, 5,10}, {10, 9, 4}, { 2, 3, 7}, { 0, 0, 0}}}, /* 6:  56-*/
 { 4, {{ 0, 5,10}, { 0,10, 9}, { 0, 9, 1}, { 2, 3, 7}}}, /*12:  57-*/
 { 4, {{ 4,10, 9}, { 0, 3, 4}, { 4, 3,10}, { 3, 7,10}}}, /*14:  58-*/
 { 3, {{10, 9, 7}, { 9, 1, 7}, { 1, 3, 7}, { 0, 0, 0}}}, /* 5:  59-*/
 { 4, {{ 1, 6, 2}, { 2, 6, 7}, { 4, 5,10}, { 4,10, 9}}}, /*10:  60-*/
 { 3, {{ 5, 2, 0}, { 6, 7,10}, {10, 9, 6}, { 0, 0, 0}}}, /* 6:  61-*/
 { 3, {{ 1, 4, 0}, { 6, 7,10}, {10, 9, 6}, { 0, 0, 0}}}, /* 6:  62-*/
 { 2, {{ 6, 7,10}, {10, 9, 6}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  63-*/
 { 1, {{ 6, 9,11}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1:  64-*/
 { 2, {{ 0, 4, 1}, { 6, 9,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  65-*/
 { 2, {{ 0, 2, 5}, { 6, 9,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4:  66-*/
 { 3, {{ 1, 2, 5}, { 5, 4, 1}, { 6, 9,11}, { 0, 0, 0}}}, /* 6:  67-*/
 { 2, {{ 1, 9,11}, {11, 3, 1}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  68-*/
 { 3, {{ 4, 9,11}, { 4,11, 3}, { 4, 3, 0}, { 0, 0, 0}}}, /* 5:  69-*/
 { 3, {{ 1, 9,11}, {11, 3, 1}, { 0, 2, 5}, { 0, 0, 0}}}, /* 6:  70-*/
 { 4, {{ 5, 4,11}, { 4, 9,11}, {11, 3, 5}, { 3, 2, 5}}}, /*11:  71-*/
 { 2, {{ 2, 3, 7}, { 6, 9,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  72-*/
 { 3, {{ 0, 4, 1}, { 2, 3, 7}, { 6, 9,11}, { 0, 0, 0}}}, /* 7:  73-*/
 { 3, {{ 0, 3, 7}, { 7, 5, 0}, { 6, 9,11}, { 0, 0, 0}}}, /* 6:  74-*/
 { 4, {{ 1, 3, 7}, { 1, 7, 5}, { 1, 5, 4}, { 6, 9,11}}}, /*12:  75-*/
 { 3, {{11, 7, 2}, {11, 2, 1}, {11, 1, 9}, { 0, 0, 0}}}, /* 5:  76-*/
 { 4, {{ 0, 4, 9}, { 0, 9, 2}, { 2, 9,11}, { 2,11, 7}}}, /* 9:  77-*/
 { 4, {{ 0, 1, 5}, { 1, 7, 5}, { 1, 9, 7}, { 9,11, 7}}}, /*14:  78-*/
 { 3, {{ 7, 5,11}, { 5, 4,11}, { 4, 9,11}, { 0, 0, 0}}}, /* 5:  79-*/
 { 2, {{ 4, 8,11}, {11, 6, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  80-*/
 { 3, {{ 6, 1, 0}, { 6, 0, 8}, { 6, 8,11}, { 0, 0, 0}}}, /* 5:  81-*/
 { 3, {{ 4, 8,11}, {11, 6, 4}, { 0, 2, 5}, { 0, 0, 0}}}, /* 6:  82-*/
 { 4, {{ 5, 8, 2}, { 8, 1, 2}, { 8,11, 1}, {11, 6, 1}}}, /*14:  83-*/
 { 3, {{ 1, 4, 8}, { 1, 8,11}, { 1,11, 3}, { 0, 0, 0}}}, /* 5:  84-*/
 { 2, {{ 0, 8,11}, { 0,11, 3}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8:  85-*/
 { 4, {{ 1, 4, 8}, { 1, 8,11}, { 1,11, 3}, { 0, 2, 5}}}, /*12:  86-*/
 { 3, {{ 8,11, 5}, {11, 3, 5}, { 3, 2, 5}, { 0, 0, 0}}}, /* 5:  87-*/
 { 3, {{ 4, 8,11}, {11, 6, 4}, { 2, 3, 7}, { 0, 0, 0}}}, /* 6:  88-*/
 { 4, {{ 6, 1, 0}, { 6, 0, 8}, { 6, 8,11}, { 2, 3, 7}}}, /*12:  89-*/
 { 4, {{ 0, 3, 7}, { 0, 7, 5}, { 4, 8, 6}, { 6, 8,11}}}, /*10:  90-*/
 { 3, {{ 3, 6, 1}, { 5, 8,11}, {11, 7, 5}, { 0, 0, 0}}}, /* 6:  91-*/
 { 4, {{ 4, 8, 7}, { 8,11, 7}, { 7, 2, 4}, { 2, 1, 4}}}, /*11:  92-*/
 { 3, {{ 0, 8, 2}, { 8,11, 2}, {11, 7, 2}, { 0, 0, 0}}}, /* 5:  93-*/
 { 3, {{ 1, 4, 0}, { 5, 8,11}, {11, 7, 5}, { 0, 0, 0}}}, /* 6:  94-*/
 { 2, {{ 5, 8,11}, {11, 7, 5}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2:  95-*/
 { 2, {{ 5,10, 8}, { 6, 9,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3:  96-*/
 { 3, {{ 0, 4, 1}, { 5,10, 8}, { 6, 9,11}, { 0, 0, 0}}}, /* 7:  97-*/
 { 3, {{ 0, 2,10}, {10, 8, 0}, { 6, 9,11}, { 0, 0, 0}}}, /* 6:  98-*/
 { 4, {{ 8, 4, 1}, { 8, 1, 2}, { 8, 2,10}, { 6, 9,11}}}, /*12:  99-*/
 { 3, {{ 1, 9,11}, {11, 3, 1}, { 5,10, 8}, { 0, 0, 0}}}, /* 6: 100-*/
 { 4, {{ 4, 9,11}, { 4,11, 3}, { 4, 3, 0}, { 5,10, 8}}}, /*12: 101-*/
 { 4, {{ 0, 2, 8}, { 2,10, 8}, { 1,11, 3}, { 1, 9,11}}}, /*10: 102-*/
 { 3, {{ 9, 8, 4}, { 2,10,11}, {11, 3, 2}, { 0, 0, 0}}}, /* 6: 103-*/
 { 3, {{ 2, 3, 7}, { 5,10, 8}, { 6, 9,11}, { 0, 0, 0}}}, /* 7: 104-*/
 { 4, {{ 0, 4, 1}, { 2, 3, 7}, { 5,10, 8}, { 6, 9,11}}}, /*13: 105-*/
 { 4, {{ 7,10, 8}, { 7, 8, 0}, { 7, 0, 3}, { 6, 9,11}}}, /*12: 106-*/
 { 3, {{ 1, 3, 6}, { 4, 9, 8}, { 7,10,11}, { 0, 0, 0}}}, /* 7: 107-*/
 { 4, {{11, 7, 2}, {11, 2, 1}, {11, 1, 9}, { 5,10, 8}}}, /*12: 108-*/
 { 3, {{ 0, 5, 2}, { 4, 9, 8}, { 7,10,11}, { 0, 0, 0}}}, /* 7: 109-*/
 { 3, {{10,11, 7}, { 0, 1, 9}, { 9, 8, 0}, { 0, 0, 0}}}, /* 6: 110-*/
 { 2, {{ 4, 9, 8}, { 7,10,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 111-*/
 { 3, {{10,11, 4}, {11, 6, 4}, {10, 4, 5}, { 0, 0, 0}}}, /* 5: 112-*/
 { 4, {{ 5,10,11}, {11, 6, 5}, { 5, 6, 0}, { 6, 1, 0}}}, /* 9: 113-*/
 { 4, {{ 0, 2,11}, { 2,10,11}, {11, 6, 0}, { 6, 4, 0}}}, /*11: 114-*/
 { 3, {{ 1, 2, 6}, { 2,10, 6}, {10,11, 6}, { 0, 0, 0}}}, /* 5: 115-*/
 { 4, {{ 5,10,11}, { 5,11, 3}, { 4, 5, 3}, { 4, 3, 1}}}, /*14: 116-*/
 { 3, {{11, 3,10}, { 3, 0,10}, { 0, 5,10}, { 0, 0, 0}}}, /* 5: 117-*/
 { 3, {{ 1, 4, 0}, { 2,10,11}, {11, 3, 2}, { 0, 0, 0}}}, /* 6: 118-*/
 { 2, {{ 2,10,11}, {11, 3, 2}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 119-*/
 { 4, {{10,11, 6}, {10, 6, 4}, {10, 4, 5}, { 2, 3, 7}}}, /*12: 120-*/
 { 3, {{ 0, 5, 2}, { 1, 3, 6}, { 7,10,11}, { 0, 0, 0}}}, /* 7: 121-*/
 { 3, {{10,11, 7}, { 0, 3, 6}, { 6, 4, 0}, { 0, 0, 0}}}, /* 6: 122-*/
 { 2, {{ 1, 3, 6}, { 7,10,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 123-*/
 { 3, {{10,11, 7}, { 1, 4, 5}, { 5, 2, 1}, { 0, 0, 0}}}, /* 6: 124-*/
 { 2, {{ 0, 5, 2}, { 7,10,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 125-*/
 { 2, {{ 7,10,11}, { 0, 1, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4: 126-*/
 { 1, {{ 7,10,11}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 127-*/
 { 1, {{ 7,11,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 128-*/
 { 2, {{ 0, 4, 1}, { 7,11,10}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4: 129-*/
 { 2, {{ 0, 2, 5}, { 7,11,10}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 130-*/
 { 3, {{ 1, 2, 5}, { 5, 4, 1}, { 7,11,10}, { 0, 0, 0}}}, /* 6: 131-*/
 { 2, {{ 1, 6, 3}, { 7,11,10}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 132-*/
 { 3, {{ 0, 4, 6}, { 6, 3, 0}, { 7,11,10}, { 0, 0, 0}}}, /* 6: 133-*/
 { 3, {{ 0, 2, 5}, { 1, 6, 3}, { 7,11,10}, { 0, 0, 0}}}, /* 7: 134-*/
 { 4, {{ 3, 2, 5}, { 3, 5, 4}, { 3, 4, 6}, { 7,11,10}}}, /*12: 135-*/
 { 2, {{ 2, 3,11}, {11,10, 2}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 136-*/
 { 3, {{ 2, 3,11}, {11,10, 2}, { 0, 4, 1}, { 0, 0, 0}}}, /* 6: 137-*/
 { 3, {{10, 5, 0}, {10, 0, 3}, {10, 3,11}, { 0, 0, 0}}}, /* 5: 138-*/
 { 4, {{ 1, 3, 4}, { 3, 5, 4}, { 3,11, 5}, {11,10, 5}}}, /*14: 139-*/
 { 3, {{ 6,11,10}, { 6,10, 2}, { 6, 2, 1}, { 0, 0, 0}}}, /* 5: 140-*/
 { 4, {{ 4, 6,10}, { 6,11,10}, {10, 2, 4}, { 2, 0, 4}}}, /*11: 141-*/
 { 4, {{ 0, 1, 6}, { 0, 6, 5}, { 5, 6,11}, {11,10, 5}}}, /* 9: 142-*/
 { 3, {{ 5, 4,10}, { 4, 6,10}, { 6,11,10}, { 0, 0, 0}}}, /* 5: 143-*/
 { 2, {{ 4, 8, 9}, { 7,11,10}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 144-*/
 { 3, {{ 0, 8, 9}, { 9, 1, 0}, { 7,11,10}, { 0, 0, 0}}}, /* 6: 145-*/
 { 3, {{ 0, 2, 5}, { 4, 8, 9}, { 7,11,10}, { 0, 0, 0}}}, /* 7: 146-*/
 { 4, {{ 5, 8, 9}, { 5, 9, 1}, { 5, 1, 2}, { 7,11,10}}}, /*12: 147-*/
 { 3, {{ 6, 3, 1}, { 8, 9, 4}, {11,10, 7}, { 0, 0, 0}}}, /* 7: 148-*/
 { 4, {{ 9, 6, 3}, { 9, 3, 0}, { 9, 0, 8}, { 7,11,10}}}, /*12: 149-*/
 { 4, {{ 0, 2, 5}, { 1, 6, 3}, { 4, 8, 9}, { 7,11,10}}}, /*13: 150-*/
 { 3, {{ 7, 3, 2}, { 8,10, 5}, {11, 9, 6}, { 0, 0, 0}}}, /* 7: 151-*/
 { 3, {{ 2, 3,11}, {11,10, 2}, { 4, 8, 9}, { 0, 0, 0}}}, /* 6: 152-*/
 { 4, {{ 0, 8, 2}, { 2, 8,10}, { 1, 3,11}, { 1,11, 9}}}, /*10: 153-*/
 { 4, {{10, 5, 0}, {10, 0, 3}, {10, 3,11}, { 4, 8, 9}}}, /*12: 154-*/
 { 3, {{ 8,10, 5}, { 1, 3,11}, {11, 9, 1}, { 0, 0, 0}}}, /* 6: 155-*/
 { 4, {{ 6,11,10}, { 6,10, 2}, { 6, 2, 1}, { 4, 8, 9}}}, /*12: 156-*/
 { 3, {{11, 9, 6}, { 0, 8,10}, {10, 2, 0}, { 0, 0, 0}}}, /* 6: 157-*/
 { 3, {{ 1, 4, 0}, { 8,10, 5}, {11, 9, 6}, { 0, 0, 0}}}, /* 7: 158-*/
 { 2, {{ 5, 8,10}, { 6,11, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 159-*/
 { 2, {{ 5, 7,11}, {11, 8, 5}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 160-*/
 { 3, {{ 5, 7,11}, {11, 8, 5}, { 0, 4, 1}, { 0, 0, 0}}}, /* 6: 161-*/
 { 3, {{ 2, 7,11}, { 2,11, 8}, { 2, 8, 0}, { 0, 0, 0}}}, /* 5: 162-*/
 { 4, {{ 1, 2,11}, { 2, 7,11}, {11, 8, 1}, { 8, 4, 1}}}, /*11: 163-*/
 { 3, {{ 5, 7,11}, {11, 8, 5}, { 1, 6, 3}, { 0, 0, 0}}}, /* 6: 164-*/
 { 4, {{ 0, 7, 3}, { 0, 5, 7}, { 4, 6, 8}, { 6,11, 8}}}, /*10: 165-*/
 { 4, {{ 2, 7,11}, { 2,11, 8}, { 2, 8, 0}, { 1, 6, 3}}}, /*12: 166-*/
 { 3, {{ 7, 3, 2}, { 4, 6,11}, {11, 8, 4}, { 0, 0, 0}}}, /* 6: 167-*/
 { 3, {{ 5, 2, 3}, { 5, 3,11}, { 5,11, 8}, { 0, 0, 0}}}, /* 5: 168-*/
 { 4, {{ 5, 2, 3}, { 5, 3,11}, { 5,11, 8}, { 0, 4, 1}}}, /*12: 169-*/
 { 2, {{ 0,11, 8}, { 0, 3,11}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8: 170-*/
 { 3, {{ 3,11, 1}, {11, 8, 1}, { 8, 4, 1}, { 0, 0, 0}}}, /* 5: 171-*/
 { 4, {{ 1, 6,11}, { 1,11, 8}, { 2, 1, 8}, { 2, 8, 5}}}, /*14: 172-*/
 { 3, {{ 5, 2, 0}, { 4, 6,11}, {11, 8, 4}, { 0, 0, 0}}}, /* 6: 173-*/
 { 3, {{11, 8, 6}, { 8, 0, 6}, { 0, 1, 6}, { 0, 0, 0}}}, /* 5: 174-*/
 { 2, {{ 4, 6,11}, {11, 8, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 175-*/
 { 3, {{11, 9, 4}, {11, 4, 5}, {11, 5, 7}, { 0, 0, 0}}}, /* 5: 176-*/
 { 4, {{ 7,11, 9}, { 7, 9, 1}, { 5, 7, 1}, { 5, 1, 0}}}, /*14: 177-*/
 { 4, {{ 7,11, 2}, {11, 9, 2}, { 2, 9, 0}, { 9, 4, 0}}}, /* 9: 178-*/
 { 3, {{ 9, 1,11}, { 1, 2,11}, { 2, 7,11}, { 0, 0, 0}}}, /* 5: 179-*/
 { 4, {{11, 9, 4}, {11, 4, 5}, {11, 5, 7}, { 1, 6, 3}}}, /*12: 180-*/
 { 3, {{11, 9, 6}, { 0, 5, 7}, { 7, 3, 0}, { 0, 0, 0}}}, /* 6: 181-*/
 { 3, {{ 1, 4, 0}, { 7, 3, 2}, {11, 9, 6}, { 0, 0, 0}}}, /* 7: 182-*/
 { 2, {{ 2, 7, 3}, { 6,11, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 183-*/
 { 4, {{ 2, 3, 9}, { 3,11, 9}, { 9, 4, 2}, { 4, 5, 2}}}, /*11: 184-*/
 { 3, {{ 5, 2, 0}, { 1, 3,11}, {11, 9, 1}, { 0, 0, 0}}}, /* 6: 185-*/
 { 3, {{ 0, 3, 4}, { 3,11, 4}, {11, 9, 4}, { 0, 0, 0}}}, /* 5: 186-*/
 { 2, {{ 1, 3,11}, {11, 9, 1}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 187-*/
 { 3, {{11, 9, 6}, { 1, 4, 5}, { 5, 2, 1}, { 0, 0, 0}}}, /* 6: 188-*/
 { 2, {{ 0, 5, 2}, { 6,11, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4: 189-*/
 { 2, {{ 0, 1, 4}, { 6,11, 9}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 190-*/
 { 1, {{ 6,11, 9}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 191-*/
 { 2, {{ 6, 9,10}, {10, 7, 6}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 192-*/
 { 3, {{ 6, 9,10}, {10, 7, 6}, { 0, 4, 1}, { 0, 0, 0}}}, /* 6: 193-*/
 { 3, {{ 6, 9,10}, {10, 7, 6}, { 0, 2, 5}, { 0, 0, 0}}}, /* 6: 194-*/
 { 4, {{ 1, 2, 6}, { 7, 6, 2}, {10, 5, 4}, { 4, 9,10}}}, /*10: 195-*/
 { 3, {{ 7, 3, 1}, { 7, 1, 9}, { 7, 9,10}, { 0, 0, 0}}}, /* 5: 196-*/
 { 4, {{10, 7, 3}, {10, 3, 0}, { 9,10, 0}, { 9, 0, 4}}}, /*14: 197-*/
 { 4, {{ 7, 3, 1}, { 7, 1, 9}, { 7, 9,10}, { 0, 2, 5}}}, /*12: 198-*/
 { 3, {{ 7, 3, 2}, { 4, 9,10}, {10, 5, 4}, { 0, 0, 0}}}, /* 6: 199-*/
 { 3, {{ 3, 6, 9}, { 3, 9,10}, { 3,10, 2}, { 0, 0, 0}}}, /* 5: 200-*/
 { 4, {{ 3, 6, 9}, { 3, 9,10}, { 3,10, 2}, { 0, 4, 1}}}, /*12: 201-*/
 { 4, {{ 5, 0, 6}, { 0, 3, 6}, { 6, 9, 5}, { 9,10, 5}}}, /*11: 202-*/
 { 3, {{ 3, 6, 1}, { 4, 9,10}, {10, 5, 4}, { 0, 0, 0}}}, /* 6: 203-*/
 { 2, {{ 1,10, 2}, { 1, 9,10}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8: 204-*/
 { 3, {{ 9,10, 4}, {10, 2, 4}, { 2, 0, 4}, { 0, 0, 0}}}, /* 5: 205-*/
 { 3, {{ 1, 9, 0}, { 9,10, 0}, {10, 5, 0}, { 0, 0, 0}}}, /* 5: 206-*/
 { 2, {{ 4, 9,10}, {10, 5, 4}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 207-*/
 { 3, {{ 8,10, 7}, { 8, 7, 6}, { 8, 6, 4}, { 0, 0, 0}}}, /* 5: 208-*/
 { 4, {{ 1, 0,10}, { 0, 8,10}, {10, 7, 1}, { 7, 6, 1}}}, /*11: 209-*/
 { 4, {{ 8,10, 7}, { 8, 7, 6}, { 8, 6, 4}, { 0, 2, 5}}}, /*12: 210-*/
 { 3, {{ 8,10, 5}, { 1, 2, 7}, { 7, 6, 1}, { 0, 0, 0}}}, /* 6: 211-*/
 { 4, {{ 8,10, 7}, { 8, 7, 3}, { 8, 3, 4}, { 3, 1, 4}}}, /* 9: 212-*/
 { 3, {{ 3, 0, 7}, { 0, 8, 7}, { 8,10, 7}, { 0, 0, 0}}}, /* 5: 213-*/
 { 3, {{ 1, 4, 0}, { 7, 3, 2}, { 8,10, 5}, { 0, 0, 0}}}, /* 7: 214-*/
 { 2, {{ 2, 7, 3}, { 5, 8,10}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 215-*/
 { 4, {{ 4, 8,10}, { 4,10, 2}, { 6, 4, 2}, { 6, 2, 3}}}, /*14: 216-*/
 { 3, {{ 3, 6, 1}, { 0, 8,10}, {10, 2, 0}, { 0, 0, 0}}}, /* 6: 217-*/
 { 3, {{ 8,10, 5}, { 0, 3, 6}, { 6, 4, 0}, { 0, 0, 0}}}, /* 6: 218-*/
 { 2, {{ 1, 3, 6}, { 5, 8,10}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4: 219-*/
 { 3, {{10, 2, 8}, { 2, 1, 8}, { 1, 4, 8}, { 0, 0, 0}}}, /* 5: 220-*/
 { 2, {{ 0, 8,10}, {10, 2, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 221-*/
 { 2, {{ 0, 1, 4}, { 5, 8,10}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 222-*/
 { 1, {{ 5, 8,10}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 223-*/
 { 3, {{ 9, 8, 5}, { 9, 5, 7}, { 9, 7, 6}, { 0, 0, 0}}}, /* 5: 224-*/
 { 4, {{ 9, 8, 5}, { 9, 5, 7}, { 9, 7, 6}, { 0, 4, 1}}}, /*12: 225-*/
 { 4, {{ 6, 9, 8}, { 6, 8, 0}, { 7, 6, 0}, { 7, 0, 2}}}, /*14: 226-*/
 { 3, {{ 9, 8, 4}, { 1, 2, 7}, { 7, 6, 1}, { 0, 0, 0}}}, /* 6: 227-*/
 { 4, {{ 3, 1, 8}, { 1, 9, 8}, { 8, 5, 3}, { 5, 7, 3}}}, /*11: 228-*/
 { 3, {{ 9, 8, 4}, { 0, 5, 7}, { 7, 3, 0}, { 0, 0, 0}}}, /* 6: 229-*/
 { 3, {{ 7, 3, 2}, { 0, 1, 9}, { 9, 8, 0}, { 0, 0, 0}}}, /* 6: 230-*/
 { 2, {{ 2, 7, 3}, { 4, 9, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /* 4: 231*/
 { 4, {{ 8, 5, 2}, { 2, 3, 8}, { 9, 8, 3}, { 3, 6, 9}}}, /* 9: 232-*/
 { 3, {{ 5, 2, 0}, { 3, 6, 1}, { 4, 9, 8}, { 0, 0, 0}}}, /* 7: 233-*/
 { 3, {{ 8, 0, 9}, { 0, 3, 9}, { 3, 6, 9}, { 0, 0, 0}}}, /* 5: 234-*/
 { 2, {{ 1, 3, 6}, { 4, 9, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 235-*/
 { 3, {{ 2, 1, 5}, { 1, 9, 5}, { 9, 8, 5}, { 0, 0, 0}}}, /* 5: 236-*/
 { 2, {{ 0, 5, 2}, { 4, 9, 8}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 237-*/
 { 2, {{ 0, 1, 9}, { 9, 8, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 238-*/
 { 1, {{ 4, 9, 8}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 239-*/
 { 2, {{ 4, 7, 6}, { 4, 5, 7}, { 0, 0, 0}, { 0, 0, 0}}}, /* 8: 240-*/
 { 3, {{ 5, 7, 0}, { 7, 6, 0}, { 6, 1, 0}, { 0, 0, 0}}}, /* 5: 241-*/
 { 3, {{ 7, 6, 2}, { 6, 4, 2}, { 4, 0, 2}, { 0, 0, 0}}}, /* 5: 242-*/
 { 2, {{ 1, 2, 7}, { 7, 6, 1}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 243-*/
 { 3, {{ 4, 5, 1}, { 5, 7, 1}, { 7, 3, 1}, { 0, 0, 0}}}, /* 5: 244-*/
 { 2, {{ 0, 5, 7}, { 7, 3, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 245-*/
 { 2, {{ 0, 1, 4}, { 2, 7, 3}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 246-*/
 { 1, {{ 2, 7, 3}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 247-*/
 { 3, {{ 6, 4, 3}, { 4, 5, 3}, { 5, 2, 3}, { 0, 0, 0}}}, /* 5: 248-*/
 { 2, {{ 0, 5, 2}, { 1, 3, 6}, { 0, 0, 0}, { 0, 0, 0}}}, /* 3: 249-*/
 { 2, {{ 0, 3, 6}, { 6, 4, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 250-*/
 { 1, {{ 1, 3, 6}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 251-*/
 { 2, {{ 1, 4, 5}, { 5, 2, 1}, { 0, 0, 0}, { 0, 0, 0}}}, /* 2: 252-*/
 { 1, {{ 0, 5, 2}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 253-*/
 { 1, {{ 0, 1, 4}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 1: 254-*/
 { 0, {{ 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}, { 0, 0, 0}}}, /* 0: 255-*/
};
#endif /* 0 */

#endif
